/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
    @file cst_cert.c

    Certificate Management Library

    Certificate functions (common)
*/



#include "cst_t.h"
#include <memory.h>
#include <string.h>
#include <openssl/conf.h>
#include <openssl/x509v3.h>
#include <openssl/pkcs12.h>
#include <openssl/err.h>

#define DNS_EMAIL_BUFFER_SIZE 256

static int cert_save_info(CST * st, const t_seqnum certID, CERT * c);
static int cert_load_info(CST * st, const t_seqnum certID, CERT ** c);
static int cst_is_revoked_rescan(CST * st, X509 * cert);

static int cert_save_x(CST * st, const t_seqnum certID, X509 * x);
static int cert_load_x(CST * st, const t_seqnum certID, X509 ** x);

static int cst_export_cert_fmt(X509 * cert, FILE * file, const int fmt);
static int cst_export_cert_by_id_fmt(CST * st, const cst_t_seqnum certID, 
                                     FILE * file, const int fmt);
static int cst_export_all_fmt_lock(CST * st, FILE * file, 
                                   const t_cert_folder folder, 
                                   const int fmt);

int CST_check_purpose(CST * st, const cst_t_seqnum certID, const t_cert_purpose purpose)
{
    if (st && (certID > 0))
    {
        g_assert(st);
        int result = FALSE;
        X509 * x = NULL;
        CST_LOCK_BEGIN_S(LOCK_SH);
            x = cert_get_X509(st, certID);
        CST_LOCK_END;

        if (x)
        {
            result = CST_check_purpose_x(x, purpose);
            X509_free(x);
        }

        return result;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
        return FALSE;
    }
}

int CST_check_purpose_x(X509 * x, const t_cert_purpose purpose)
{
    int result = FALSE;
    int std_found = FALSE;

    if (x)
    {
        int ca = 0;
        if (purpose & CST_PURPOSE_CA)
        {
            ca = 1;
        }

        /* If not only CA purpose */
        if (purpose & !CST_PURPOSE_CA)
        {
            if (purpose & CST_PURPOSE_SSL_SERVER)
            {
                std_found = TRUE;
                result = (1 == X509_check_purpose(x, X509_PURPOSE_SSL_SERVER, ca));
            }

            if (purpose & CST_PURPOSE_SSL_CLIENT)
            {
                std_found = TRUE;
                result = (1 == X509_check_purpose(x, X509_PURPOSE_SSL_CLIENT, ca));
            }

            if (purpose & CST_PURPOSE_SMIME_SGN)
            {
                std_found = TRUE;
                result = (1 == X509_check_purpose(x, X509_PURPOSE_SMIME_SIGN, ca));
            }

            if (purpose & CST_PURPOSE_SMIME_ENC)
            {
                std_found = TRUE;
                result = (1 == X509_check_purpose(x, X509_PURPOSE_SMIME_ENCRYPT, ca));
            }

            if (purpose & CST_PURPOSE_CRL_SIGN)
            {
                std_found = TRUE;
                result = (1 == X509_check_purpose(x, X509_PURPOSE_CRL_SIGN, ca));
            }
        }
        else
        {
            std_found = TRUE;
            if (ca) result = CST_is_CA(x);
        }
    }    

    return std_found ? result : TRUE;
}

static void cert_capability_free(CERT_CAPABILITY * cap)
{
    g_assert(cap);

    if (cap->oid)
    {
        g_free(cap->oid);
    }

    if (cap->data)
    {
        g_free(cap->data);
    }

    g_free(cap);
}

int cert_is_root_x(X509 * x)
{
    g_assert(x);
    return (0 == X509_NAME_cmp(X509_get_issuer_name(x),
                               X509_get_subject_name(x)));
}

int cert_is_root(CERT * c)
{
    g_assert(c);
    return (0 == X509_NAME_cmp(c->name, c->issuer_name));
}

int cert_get_common_name_x(X509_NAME * name, char *buffer, int buffer_size)
{
    g_assert(name && buffer);
    g_assert(buffer_size > 0);
    return X509_NAME_get_text_by_NID(name,
                                     NID_commonName, buffer, buffer_size);
}

int cert_get_subjectAltName_x(X509 * x, char *key,
                              char *buffer, int buffer_size)
{
    /* See example: crypto/x509v3/v3_prn.c X509V3_EXT_print 108 */
    TRACE("*");
    g_assert(x && buffer && key);
    g_assert(buffer_size > 0);

    void *ext_str = NULL;
    STACK_OF(CONF_VALUE) * val_list = NULL;
    CONF_VALUE *val_item = NULL;
    X509V3_EXT_METHOD *m = NULL;

    int result = FALSE;

    int extension_total = X509_get_ext_count(x);
    if (extension_total > 0)
    {
        int i;
        for (i = 0; i < extension_total; i++)
        {
            X509_EXTENSION *e;
            char *e_text;

            e = X509_get_ext(x, i);
            e_text =
                (char *)
                OBJ_nid2sn(OBJ_obj2nid(X509_EXTENSION_get_object(e)));

            if (0 == strcmp(e_text, "subjectAltName"))
            {
                int j;
                unsigned char *data;

                m = X509V3_EXT_get(e);
                if (NULL == m)
                {
                    goto err;
                }

                if (!m->i2v)
                {
                    goto err;
                }
                
                data = e->value->data;
                
                if (m->it)
                {
                    ext_str = ASN1_item_d2i(NULL, &data, e->value->length, ASN1_ITEM_ptr(m->it));
                }
                else
                {
                    ext_str = m->d2i(NULL, &data, e->value->length);
                }
                
                if (NULL == ext_str)
                {
                    goto err;
                }
                
                val_list = m->i2v(m, ext_str, NULL);
               
                g_assert(val_list);
                for (j = 0; j < sk_CONF_VALUE_num(val_list); j++)
                {
                    val_item = sk_CONF_VALUE_value(val_list, j);
                    if (0 == strcmp(val_item->name, key))
                    {
                        strncpy(buffer, val_item->value, buffer_size);
                        TRACE_S(buffer);
                        result = TRUE;
                        goto err;
                    }
                }
                goto err;
            }
        }
    }

err:
    if ((ext_str != NULL) && (m != NULL))
    {
        if (m->it) ASN1_item_free(ext_str, ASN1_ITEM_ptr(m->it));
        else m->ext_free(ext_str);
    }

    if (val_list) 
    {
        sk_CONF_VALUE_pop_free(val_list, X509V3_conf_free);
    }
    
    return result;
}

char *cert_get_domain_name_x(X509 * x)
{
    TRACE("*");
    char buffer[DNS_EMAIL_BUFFER_SIZE];
    memset(&buffer, 0, DNS_EMAIL_BUFFER_SIZE);
        
    g_assert(x);

    if (cert_get_subjectAltName_x(x, "DNS", buffer, DNS_EMAIL_BUFFER_SIZE))
    {
        TRACE_S(buffer);
        return strdup(buffer);
    } else
    {
        if (cert_get_common_name_x
            (X509_get_subject_name(x), buffer, DNS_EMAIL_BUFFER_SIZE))
        {
            return strdup(buffer);
        } else
        {
            return NULL;
        }
    }
}

char *cert_get_email_x(X509 * x)
{
    char buffer[DNS_EMAIL_BUFFER_SIZE];
    memset(&buffer, 0, DNS_EMAIL_BUFFER_SIZE);
        
    g_assert(x);
    if (cert_get_subjectAltName_x
        (x, "email", buffer, DNS_EMAIL_BUFFER_SIZE))
    {
        return strdup(buffer);
    } else
    {
        if (X509_NAME_get_text_by_NID(X509_get_subject_name(x), 
                    NID_pkcs9_emailAddress, buffer, DNS_EMAIL_BUFFER_SIZE))
        {
            return strdup(buffer);
        }
        else
        {
            return NULL;
        }
    }
}

GSList * CST_append_sk_X509(CST * st, STACK_OF(X509) * list)
{
    GSList * result = NULL;
    if (st)
    {
        if (list)
        {
            int i;
            int r;
            for(i = 0; i < sk_X509_num(list); i++)
            {
                r = CST_append_X509(st, sk_X509_value(list, i));
                result = g_slist_append(result, GINT_TO_POINTER(r));
            }
        }
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

int CST_append_X509(CST * st, X509 * cert)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (st && cert)
    {
        result = cert_append(st, cert);
    }
    return result;
}

static int cert_write_capabilities_pack_h(unsigned char *buffer, 
        const t_rtype tp, const t_rsize sz)
{
    memcpy(buffer, &tp, sz_rtype);
    t_rsize size = RSIZE_TO_LE(sz);
    memcpy(buffer + sz_rtype, &size, sz_rsize);
    return sz_rheader;
}

static unsigned char * cert_write_capabilities_pack(CERT * c, t_rsize *length)
{
    g_assert(c && length);

    GSList * i; 
    CERT_CAPABILITY * cap;
    *length = 0;
    
    for(i = c->capabilities; i != NULL; i = i->next)
    {
        cap = (CERT_CAPABILITY *) i->data;
        *length += sz_rheader + cap->oid_length
                 + sz_rheader + cap->data_length;
    }

    if (*length)
    {
        unsigned char *res = g_malloc(*length);
        unsigned char *p = res;
        for (i = c->capabilities; i != NULL; i = i->next)
        {
            cap = (CERT_CAPABILITY *) i->data;
            
            p += cert_write_capabilities_pack_h(p, CST_RT_CAPAB_OID, cap->oid_length);
            memcpy(p, cap->oid, cap->oid_length);
            p += cap->oid_length;
                
            p += cert_write_capabilities_pack_h(p, CST_RT_CAPAB_DATA, cap->data_length);
            memcpy(p, cap->data, cap->data_length);
            p += cap->data_length;
        }
        g_assert((res + *length) == p);
        return res;
    }
    
    return NULL;
}

X509 *cert_get_X509(CST * st, const t_seqnum certID)
{
    g_assert(st);
    X509 * x = NULL;
    if (certID > 0)
    {
        cert_load_x(st, certID, &x);
    }
    else
    {
        CST_error(CST_ERROR_CERT_NOTFOUND);
    }
    return x; 
}

/**
    Fill calculated fields in cert strcture
*/
int cert_calculate(CST * st, CERT * c, X509 * x)
{
    TRACE("*");
    g_assert(st && c && x);

    c->name = X509_NAME_dup(X509_get_subject_name(x));
    c->issuer_name = X509_NAME_dup(X509_get_issuer_name(x));
    c->sn = ASN1_INTEGER_dup(X509_get_serialNumber(x));
    c->domain_name = cert_get_domain_name_x(x);
    TRACE_S(c->domain_name);
    c->email = cert_get_email_x(x);
    TRACE_S(c->email);
    c->serial = CST_get_serial_number_t(x);
    c->fingerprint = CST_get_fingerprint(x);

    return CST_error(CST_ERROR_OK);
}


/**
    Initialize certificate list (constructor)
*/
int cst_cert_list_init(CST * st)
{
    g_assert(st);
    TRACE("*");

    st->certs = g_tree_new(cert_cmp_by_iname_sn);
    st->idx_cert_name = g_tree_new(helper_name_cmp);
    st->idx_cert_dns = g_tree_new(helper_str_cmp);
    st->idx_cert_email = g_tree_new(helper_str_cmp);
    
    st->idx_cert_uid = g_hash_table_new(g_direct_hash, g_direct_equal);    
    
    st->idx_cert_serial      = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    st->idx_cert_fingerprint = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);

    return CST_error(CST_ERROR_OK);
}

gboolean destroy_cert(gpointer key, gpointer value, gpointer data)
{
    g_assert(key && value && data);
    GSList ** plist = (GSList **) data;
    *plist = g_slist_append(*plist, (CERT *) value);
    return FALSE;
}


/**
    Free certificate list (destructor)
    Memory used by cert info free also
*/
int cst_cert_list_destroy(CST * st)
{
    TRACE("*");
    g_assert(st);

    // TODO: Optimize it 
    GSList * certs = NULL;
    g_tree_foreach(st->certs, destroy_cert, &certs);
    GSList *i;
    for (i = certs; i != NULL; i = i->next)
    {
        cert_remove(st, ((CERT *) i->data)->uid);
    }
    g_slist_free(certs);

    g_tree_destroy(st->idx_cert_name);
    g_tree_destroy(st->idx_cert_dns);
    g_tree_destroy(st->idx_cert_email);
    g_tree_destroy(st->certs);
    
    g_hash_table_destroy(st->idx_cert_uid);
    st->idx_cert_uid = NULL;
    
    g_hash_table_destroy(st->idx_cert_serial);
    st->idx_cert_serial = NULL;
    
    g_hash_table_destroy(st->idx_cert_fingerprint);
    st->idx_cert_fingerprint = NULL;

    return CST_error(CST_ERROR_OK);
}

/**
    Append certificate to storage  
    and update indexes
*/
int cert_append(CST * st, X509 * x)
{
    GError * error = NULL;
    cert_append_f(st, x, 0, &error);
    int result = (NULL == error ? CST_ERROR_OK : error->code);
    g_clear_error(&error);
    return result;
}


static void cert_set_default_if_one(CST * st, CERT * c)
{
    g_assert(st && c);

    if (c->email)
    {
        //printf("email: %s\n", c->email);
        GSList * list = (GSList *) g_tree_lookup(st->idx_cert_email, c->email);

        if (list && (g_slist_length(list) == 1))
        {
            cst_set_default(st, c->uid);
        }
    }
}

static int cert_save_info(CST * st, const t_seqnum certID, CERT * c)
{
    TRACE("*");
    g_assert(st && c && (certID > 0));
    
    t_rsize capabilities_length;
    unsigned char * capabilities = cert_write_capabilities_pack(c, &capabilities_length);
    
    DBT info;
    info.size = sz_rheader + sz_cert_purpose /* 1. purpose */
              + sz_rheader + sz_seqnum       /* 2. key_link */ 
              + sz_rheader + sz_cert_folder  /* 3. folder */
              + sz_rheader + sz_bool         /* 4. def */ 
              + sz_rheader + sz_bool         /* 5. is_revoked */
              + capabilities_length;         /* 6. capabilities */

    info.data = g_malloc0(info.size);
    void *i = info.data;

    /* 1. purpose */
    write_header_mem(&i, CST_RT_PURPOSE, sz_cert_purpose);
    t_cert_purpose purpose = CERT_PURPOSE_TO_LE(c->purpose);
    write_mem(&i, &purpose, sz_cert_purpose);

    /* 2. key_link */
    write_header_mem(&i, CST_RT_KEYLINK, sz_seqnum);
    t_seqnum key_link = SEQNUM_TO_LE(c->key_link);
    write_mem(&i, &key_link, sz_seqnum);
    
    /* 3. folder */
    write_header_mem(&i, CST_RT_FOLDER, sz_cert_folder);
    write_mem(&i, &(c->folder), sz_cert_folder);

    /* 4. def */
    write_header_mem(&i, CST_RT_DEF, sz_bool);
    write_mem(&i, &(c->def), sz_bool);

    /* 5. is_revoked */
    write_header_mem(&i, CST_RT_ISREVOKED, sz_bool);
    write_mem(&i, &(c->is_revoked), sz_bool);

    /* 6. capabilities */
    if (capabilities_length > 0)
    {
        g_assert(capabilities);
        write_mem(&i, capabilities, capabilities_length);
        g_free(capabilities);
    }

    g_assert((info.data + info.size) == i);
    
    int r = cst_data_put(st, CST_SUFFIX_CERTINFO, certID, &info);

    g_free(info.data);

    return r;
}

static int cert_load_info(CST * st, const t_seqnum certID, CERT ** ptr_c)
{
    g_assert(st && (certID > 0));

    DBT info;
   
    int result = cst_data_get(st, CST_SUFFIX_CERTINFO, certID, &info);
    if (CST_ERROR_NOT_FOUND == result) result = CST_ERROR_CERT_NOTFOUND;
    
    if (CST_ERROR_OK == result)
    {
        if (cert_parser_info(st, &info, ptr_c) == CST_ERROR_OK)
        {
            (*ptr_c)->uid = certID;
            result = CST_ERROR_OK;
        }    
    }    
    return result;
}

int cert_parser_info(CST * st, DBT * info, CERT ** ptr_c)
{    
    g_assert(st && info);
    
    GSList * list = parse_buffer(info->data, info->size);

    if (!list)
    {
        return CST_error(CST_ERROR_STRUCTURE_CORRUPT);
    }

    PARSE_RES *res = NULL;
    GSList *i = NULL;

    CERT * c = cert_new0(st);
    *ptr_c = c;

    CERT_CAPABILITY * cap = NULL;
    
    for (i = list; i != NULL; i = i->next)
    {
        res = (PARSE_RES *) i->data;

        switch (res->type)
        {
        case CST_RT_PURPOSE:
            memcpy(&(c->purpose), res->data, sizeof(c->purpose));
            c->purpose = CERT_PURPOSE_FROM_LE(c->purpose);
            break;
        case CST_RT_KEYLINK:
            memcpy(&(c->key_link), res->data, sizeof(c->key_link));
            c->key_link = SEQNUM_FROM_LE(c->key_link);
            break;    
        case CST_RT_FOLDER:
            memcpy(&(c->folder), res->data, sizeof(c->folder));
            break;
        case CST_RT_DEF:
            memcpy(&(c->def), res->data, sz_bool);
            break;
        case CST_RT_ISREVOKED:
            memcpy(&(c->is_revoked), res->data, sz_bool);
            break;    
        case CST_RT_CAPAB_OID:
            if (!cap && (res->size > 0))
            {
                cap = (CERT_CAPABILITY *) g_malloc0(sizeof(CERT_CAPABILITY));
                cap->oid = g_malloc(res->size);
                memcpy(cap->oid, res->data, res->size);
                cap->oid_length = res->size;
            }
            break;
        case CST_RT_CAPAB_DATA:
            if (cap)
            {
                g_assert(!cap->data && (0 == cap->data_length));
                if (res->size > 0)
                {
                    cap->data = g_malloc(res->size);
                    memcpy(cap->data, res->data, res->size);
                    cap->data_length = res->size;
                }
                c->capabilities = g_slist_append(c->capabilities, cap);
                cap = NULL;
            }
            break;
        }
    }

    if (cap)
    {
        if (cap->data)
        {
            g_free(cap->data);
        }
        if (cap->oid)
        {
            g_free(cap->oid);
        }
        g_free(cap);
    }

    parse_res_free(list);

    return CST_ERROR_OK;
}

static int cert_save_x(CST * st, const t_seqnum certID, X509 * x)
{
    TRACE("*");
    g_assert(st && (certID > 0) && x);
    
    DBT body;
    body.data = NULL;
    
    body.size = i2d_X509(x, (unsigned char **)&body.data);
    int result =  CST_ERROR_BAD_INTERNAL_FORMAT;
    if (body.size > 0)
    {
        result = cst_data_put(st, CST_SUFFIX_CERTBODY, certID, &body);
        g_free(body.data);
    }
    return result;
}

static int cert_load_x(CST * st, const t_seqnum certID, X509 ** ptr_x)
{
    TRACE("*");
    
    g_assert(st && ptr_x);

    int result = CST_ERROR_CERT_NOTFOUND;
    *ptr_x = NULL;
    
    if (certID > 0)
    {
        DBT body;
        result = cst_data_get(st, CST_SUFFIX_CERTBODY, certID, &body);
        if (CST_ERROR_NOT_FOUND == result) result = CST_ERROR_CERT_NOTFOUND;
        
        if (CST_ERROR_OK == result) 
        {
            /* read man d2i_X509 */
            unsigned char *p = body.data;
            *ptr_x = d2i_X509(NULL, &p, body.size);
            if (NULL != *ptr_x) result = CST_ERROR_OK;
            else result = CST_ERROR_STRUCTURE_CORRUPT;
        }
    }
    return result;
}

/**
    Append certificate to storage, set folder
    and update indexes
*/
t_seqnum cert_append_f(CST * st, X509 * x, t_cert_folder folder, GError **error)
{
    TRACE("*");
    g_assert(st && x);
    int result = CST_ERROR_LOCK;
    CERT * c = NULL;
    int certID = 0;
    int need_rollback = FALSE;
  
    /* Store cert to file an refres indexes */
    CST_LOCK_BEGIN(LOCK_EX);
        c = cert_new(st, x);
        if (c)
        {
            certID = c->uid;
            c->is_revoked = cst_is_revoked_rescan(st, x);

            c->folder = folder;
            if (!cert_put(st, c))
            {
                cert_free(st, c);
                c = NULL;
                result = CST_ERROR_CERT_EXIST;
            }
            else
            {
                result = cert_save_info(st, c->uid, c);
                if (CST_ERROR_OK == result)
                {
                    result = cert_save_x(st, c->uid, x);

                    /* Set default if one with this email */
                    if (CST_ERROR_OK == result)
                    {
                        g_assert(c && (c->uid > 0));
                        cert_set_default_if_one(st, c);
                    } 
                    else 
                    {
                        need_rollback = TRUE;
                    }
                }
                else
                {
                        cert_remove(st, certID);                        
                }
            }
        }
    CST_LOCK_END;

    // Rollback certinfo (BUG 22347)
    if (need_rollback) 
    {
        CST_LOCK_BEGIN(LOCK_EX);
            cst_db_delete(st, CST_SUFFIX_CERTINFO, certID);
        CST_LOCK_END;
        cert_remove(st, certID);
    }

    if (CST_ERROR_OK != result)
    {
        g_set_error(error, CST_ERROR_DOMAIN, result, "Error on append cert to storage");
        return 0;
    }
    else
    {
        return certID;
    }
}

/**
    Create empty CERT structure
*/
CERT *cert_new0(CST * st)
{
    g_assert(st);
    CERT *c = (CERT *) g_malloc0(sizeof(CERT));
    return c;
}

/**
    Create CERT structure
*/
CERT *cert_new(CST * st, X509 * cert)
{
    g_assert(st && cert);

    CERT *c = cert_new0(st);
    c->uid = cst_next_cert_uid(st);

    cert_calculate(st, c, cert);

    return c;
}

/**
    Free CERT structure
*/
void cert_free(CST * st, CERT * c)
{
    g_assert(st);

    if (c)
    {
        if (c->name)
            X509_NAME_free(c->name);
        if (c->issuer_name)
            X509_NAME_free(c->issuer_name);
        if (c->sn)
            ASN1_INTEGER_free(c->sn);
        g_free(c->email);
        g_free(c->domain_name);
        g_free(c->serial);
        g_free(c->fingerprint);

        /* free capabilities list */
        GSList * i;
        for (i = c->capabilities; i != NULL; i = i->next)
        {
            cert_capability_free((CERT_CAPABILITY *) i->data);
        }
        g_slist_free(c->capabilities);

        /* free base structure */
        g_free(c);
    }
}


/**
    Import cert from file
*/
int CST_import_cert(CST * st, FILE * file, unsigned char *password)
{
    return CST_import_cert_f(st, file, password, 0);
}

int CST_import_cert_DER(CST * st, FILE * file)
{
    return CST_import_cert_f_DER(st, file, 0);
}

/**
    Import cert from file and set folder
*/
static t_seqnum CST_import_cert_f_fmt(CST * st, FILE * file, char *password, 
        const t_cert_folder f, const int fmt, GError **error)
{
    t_seqnum result = 0;
    if (st && file)
    {
        X509 *x = NULL;
        
        switch (fmt)
        {
        case IMPORT_FORMAT_PEM:
            x = PEM_read_X509(file, NULL, NULL, password);
            break;
        case IMPORT_FORMAT_DER:
            x = d2i_X509_fp(file, NULL);
            break;
        }
        
        if (!x)
        {
            g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_NOT_FOUND, "Certificate not found, structure corrupt, or other format");
        }
        else
        {
            result = cert_append_f(st, x, f, error);
            X509_free(x);
        }
    }
    else
    {
        g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_PARAM_INCORRECT, "Parameters incorrect");
    }
    return result;
}

int CST_import_cert_f(CST * st, FILE * file, unsigned char *password, const t_cert_folder f)
{
    GError * error = NULL;
    int result = CST_ERROR_OK;
    if (CST_import_cert_f_fmt(st, file, password, f, IMPORT_FORMAT_PEM, &error) <= 0)
    {
        result = (NULL == error ? CST_ERROR_OK : error->code);
    }
    g_clear_error(&error);
    return result;
}

int CST_import_cert_f_DER(CST * st, FILE * file, const t_cert_folder f)
{
    GError * error = NULL;
    int result = CST_ERROR_OK;
    if (CST_import_cert_f_fmt(st, file, NULL, f, IMPORT_FORMAT_DER, &error) <= 0)
    {
        result = (NULL == error ? CST_ERROR_OK : error->code); 
    }
    g_clear_error(&error); 
    return result; 
}

cst_t_seqnum CST_import_cert_adv(CST * st, FILE * file, const t_cert_folder f, GError **error)
{
    return CST_import_cert_f_fmt(st, file, NULL, f, IMPORT_FORMAT_PEM, error);
}

cst_t_seqnum CST_import_cert_adv_DER(CST * st, FILE * file, const t_cert_folder f, GError **error)    
{
    return CST_import_cert_f_fmt(st, file, NULL, f, IMPORT_FORMAT_DER, error);
}

static gboolean export_all_op(gpointer k, gpointer v, gpointer d)
{
    CERT *cert = (CERT *) v;
    CERT_EXPORT *data = (CERT_EXPORT *) d;

    if (cert->folder == data->folder)
    {
        X509 * x = cert_get_X509(data->st, cert->uid);
        data->error = CST_ERROR_CERT_NOTFOUND;
        if (x)
        {
            data->error = cst_export_cert_fmt(x, data->fp, data->fmt);
            X509_free(x);
        }
        
        if (data->error != CST_ERROR_OK)
        {
            return TRUE;
        }
    }

    return FALSE;
}

int CST_export_all_DER(CST * st, FILE * file, const t_cert_folder folder)
{
    return cst_export_all_fmt_lock(st, file, folder, IMPORT_FORMAT_DER);
}

int CST_export_all(CST * st, FILE * file, const t_cert_folder folder)
{
    return cst_export_all_fmt_lock(st, file, folder, IMPORT_FORMAT_PEM);
}

static int cst_export_all_fmt_lock(CST * st, FILE * file, 
                                   const t_cert_folder folder, 
                                   const int fmt)
{
    g_assert(st && file);

    CERT_EXPORT data;
    data.st = st;
    data.fp = file;
    data.folder = folder;
    data.error = CST_ERROR_OK;
    data.fmt = fmt;
    
    CST_LOCK_BEGIN(LOCK_SH);
        data.error = CST_ERROR_OK;
        g_tree_foreach(st->certs, export_all_op, &data);
    CST_LOCK_END;

    return CST_error(data.error);
}



int CST_export_cert_DER(CST * st, X509 * cert, FILE * file)
{
    return cst_export_cert_fmt(cert, file, IMPORT_FORMAT_DER);
}

int CST_export_cert(CST * st, X509 * cert, FILE * file)
{
    return cst_export_cert_fmt(cert, file, IMPORT_FORMAT_PEM);
}

static int cst_export_cert_fmt(X509 * cert, FILE * file, const int fmt)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (cert && file)
    {
        switch (fmt)
        {
        case IMPORT_FORMAT_PEM:
            if (PEM_write_X509(file, cert))
            {
                result = CST_ERROR_OK;
            }
            else
            {
                result = CST_ERROR_EXPORT;
            }
            break;
        case IMPORT_FORMAT_DER:
            if (i2d_X509_fp(file, cert))
            {
                result = CST_ERROR_OK;
            }
            else
            {
                result = CST_ERROR_EXPORT;
            }
            break;    
        }
    }
    return result;
}

int CST_export_cert_by_id(CST * st, const cst_t_seqnum certID, FILE * file)
{
    return cst_export_cert_by_id_fmt(st, certID, file, IMPORT_FORMAT_PEM);
}

int CST_export_cert_by_id_DER(CST * st, const cst_t_seqnum certID, FILE * file)
{
    return cst_export_cert_by_id_fmt(st, certID, file, IMPORT_FORMAT_DER);
}

static int cst_export_cert_by_id_fmt(CST * st, const cst_t_seqnum certID, FILE * file, const int fmt)
{
    if (st && file)
    {
        if (certID > 0)
        {
            X509 * x = CST_get_cert(st, certID);
            if (x)
            {
                int result = cst_export_cert_fmt(x, file, fmt);
                X509_free(x);
                return result;
            }
        }
        return CST_error(CST_ERROR_CERT_NOTFOUND);
    }
    else
    {
        return CST_error(CST_ERROR_PARAM_INCORRECT);
    }
}

/* Compare CERT by Issuer name and Serial number */
gint cert_cmp_by_iname_sn(gconstpointer a, gconstpointer b)
{
    g_assert(a && b);

    CERT *ca = (CERT *) a;
    CERT *cb = (CERT *) b;

    gint res = M_ASN1_INTEGER_cmp(ca->sn, cb->sn);

    if (res)
    {
        return res;
    } else
    {
        return X509_NAME_cmp(ca->issuer_name, cb->issuer_name);
    }
}


/* Certificate folder and purposes (internal) */

typedef int (*fn_set_cert_param)(CERT * c, void * data);
typedef int (*fn_get_cert_param)(CERT * c, void * data, void * result);
typedef struct {t_cert_purpose purpose; int value;} PURPOSE_AND_VALUE;


static int cert_set_param(CST * st, const t_seqnum certID, fn_set_cert_param fn, void * data)
{
    g_assert(st);

    int result = CST_ERROR_OK;
    
    if (certID > 0)
    {
            CERT * c = NULL;
            result = cert_load_info(st, certID, &c);
            if (0 == result)
            {    
                result = fn(c, data);
                if (CST_ERROR_OK == result)
                {    
                    result = cert_save_info(st, certID, c);
                }
                g_free(c);
            }
    }
    else
    {
        result = CST_ERROR_CERT_NOTFOUND;
    }
    
    return CST_error(result);
}

static int cert_get_param(CST * st, const t_seqnum certID, 
            fn_get_cert_param fn, void * data, void * res)
{
    g_assert(st);

    int result = CST_ERROR_OK;
    
    if (certID > 0)
    {
            CERT * c = NULL;
            result = cert_load_info(st, certID, &c);
            if (0 == result)
            {    
                result = fn(c, data, res);
                g_free(c);
            }
    }
    else
    {
        result = CST_ERROR_CERT_NOTFOUND;
    }
    
    return CST_error(result);
}

static int cert_set_purpose(CERT * c, void * data)
{
    g_assert(c);
    g_assert(data);

    PURPOSE_AND_VALUE * pv = (PURPOSE_AND_VALUE *) data;
        
    if (pv->value)
    {
        c->purpose = pv->purpose | c->purpose;
    } else
    {
        c->purpose = pv->purpose ^ c->purpose;
    }

    return CST_ERROR_OK;
}

int cst_is_purpose1(CERT * c, const t_cert_purpose purpose)
{
    return ((c->purpose & purpose) == purpose); 
}

static int cert_is_purpose(CERT * c, void * data, void * res)
{
    g_assert(c && data && res);

    t_cert_purpose * purpose = (t_cert_purpose *) data;
    int * pres = (int *) res;
    
    *pres = cst_is_purpose1(c, *purpose);

    return CST_ERROR_OK;
}

static int cert_set_folder(CERT * c, void * data)
{
    g_assert(c && data);
    t_cert_folder * folder = (t_cert_folder *) data;
    c->folder = *folder;
    return CST_ERROR_OK;
}

static int cert_get_folder(CERT * c, void * data, void * res)
{
    g_assert(c && res);
    t_cert_folder * folder = (t_cert_folder *) res;
    *folder = c->folder;
    return CST_ERROR_OK;
}

static int cert_set_default(CERT * c, void * data)
{
    g_assert(c && data);
    int * def = (int *) data;
    c->def = *def;
    return CST_ERROR_OK;
}

static int cert_get_default(CERT * c, void * data, void * res)
{
    g_assert(c && res);
    int * def = (int *) res;
    *def = c->def;
    return CST_ERROR_OK;
}

static int cert_set_keylink(CERT * c, void * data)
{
    g_assert(c && data);
    t_seqnum * key_link = (t_seqnum *) data;
    c->key_link = *key_link;
    return CST_ERROR_OK;
}
    
static int cert_get_keylink(CERT * c, void * data, void * res)    
{
    g_assert(c && res);
    t_seqnum * key_link = (t_seqnum *) res;
    *key_link = c->key_link;
    return CST_ERROR_OK;
}       

/* Certificate folder and purposes (external) */
int cst_set_keylink(CST * st, const t_seqnum certID, const t_seqnum keyID)
{
    g_assert(st);
    return cert_set_param(st, certID, cert_set_keylink, (void *)&keyID);
}

t_seqnum cst_get_keylink(CST * st, const t_seqnum certID)
{
    g_assert(st);
    t_seqnum result;
    int res = cert_get_param(st, certID, cert_get_keylink, NULL, (void *)&result);
    return (CST_ERROR_OK == res ? result: 0);
}

int cst_set_folder(CST * st, const cst_t_seqnum certID, const t_cert_folder f)
{
    g_assert(st);
    return cert_set_param(st, certID, cert_set_folder, (void *)&f);
}

int CST_set_folder(CST * st, const cst_t_seqnum certID, const t_cert_folder f)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (st)
    {
        CST_LOCK_BEGIN_S(LOCK_EX);
            result = cst_set_folder(st, certID, f);
        CST_LOCK_END;
    }
    return result;
}

t_cert_folder cst_get_folder(CST * st, const cst_t_seqnum certID)
{
    g_assert(st);

    t_cert_folder folder;
    int result = cert_get_param(st, certID, cert_get_folder, NULL, (void *)&folder);

    return (CST_ERROR_OK == result ? folder : -1);
}

t_cert_folder CST_get_folder(CST * st, const cst_t_seqnum certID)
{
    t_cert_folder result = -1;
    if (st)
    {
        CST_LOCK_BEGIN_S(LOCK_SH);
            result = cst_get_folder(st, certID);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

int cst_set_purpose(CST * st, const cst_t_seqnum certID, 
        const t_cert_purpose p, const int value)
{
    g_assert(st);

    PURPOSE_AND_VALUE pv;
    pv.purpose = p;
    pv.value = value;
    
    return cert_set_param(st, certID, cert_set_purpose, &pv);
}

int CST_set_purpose(CST * st, const cst_t_seqnum certID, 
        const t_cert_purpose p, const int value)
{
    int result = CST_ERROR_CERT_NOTFOUND;
    if (st)
    {
        CST_LOCK_BEGIN_S(LOCK_EX);
            result = cst_set_purpose(st, certID, p, value);
        CST_LOCK_END;
    }
    else
    {
        result = CST_ERROR_PARAM_INCORRECT;
    }
    return result;
}

int cst_is_purpose(CST * st, const cst_t_seqnum certID, 
        const t_cert_purpose p)
{
    g_assert(st);

    int res;
    int result = cert_get_param(st, certID, cert_is_purpose, (void *)&p, &res);

    return (CST_ERROR_OK == result ? res : -1);
}

int CST_is_purpose(CST * st, const cst_t_seqnum certID,
        const t_cert_purpose p)
{
    int result = CST_ERROR_OK;
    if (st)
    {
        CST_LOCK_BEGIN_S(LOCK_SH);
            result = cst_is_purpose(st, certID, p);
        CST_LOCK_END;
    }
    else
    {
        result = CST_ERROR_PARAM_INCORRECT;
    }
    return result;
}

/* Certificate revoked */
static int cert_set_revoked(CERT * c, void * data)
{
    g_assert(c && data);
    int * is_revoked = (int *) data;
    c->is_revoked = *is_revoked;
    return CST_ERROR_OK;
}

int cst_set_revoked(CST * st, const t_seqnum certID, const int value)
{
    g_assert(st);
    int r = CST_ERROR_CERT_NOTFOUND;

    CERT * cert = cert_get_by_id(st, certID);

    if (cert)
    {
        r = cert_set_param(st, certID, cert_set_revoked, (void *)&value);
    }

    return CST_error(r);
}

static int cert_get_revoked(CERT * c, void * data, void * res)
{
    g_assert(c && res);
    int * is_revoked = (int *) res;
    *is_revoked = c->is_revoked;
    return CST_ERROR_OK;
}

int cst_is_revoked(CST * st, const cst_t_seqnum certID)
{
    int is_revoked = FALSE;
    if (st)
    {
        cert_get_param(st, certID, cert_get_revoked, NULL, (void *)&is_revoked);
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return is_revoked;
}

/* Certificate default */
int cst_set_default(CST * st, const t_seqnum certID)
{
    g_assert(st);

	CERT * cert = cert_get_by_id(st, certID);
    int rc = CST_ERROR_NOT_FOUND;
	
	if (cert)
	{
        if (cert->email)
        {
            int const temp_true = TRUE;
            int const temp_false = FALSE;
            rc = cert_set_param(st, certID, cert_set_default, (void *)&temp_true);

            if (CST_ERROR_OK == rc)
            {
                GSList *list = (GSList *) g_tree_lookup(st->idx_cert_email, cert->email);

                GSList *i;
                CERT *curr;

                for (i = list; i != NULL; i = i->next)
                {
                    curr = (CERT *) i->data;
                    if (curr->uid != certID)
                    {
			int r = cert_set_param(st, curr->uid, cert_set_default, (void *)&temp_false);
                        if (CST_ERROR_OK != r)
                            rc = r;
                    }
                }
            }
        }
	}
    return CST_error(rc);
	
}

int CST_set_default(CST * st, const t_seqnum certID)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (st)
    {
        CST_LOCK_BEGIN_S(LOCK_EX);
            result = cst_set_default(st, certID);
        CST_LOCK_END;
    }
    return result;
}

t_seqnum cert_get_default_id(CST * st, const char *email)
{
    g_assert(st);

    if (email)
    {
        GSList *list = (GSList *) g_tree_lookup(st->idx_cert_email, email);
        GSList *i;
        CERT *c;
        for (i = list; i != NULL; i = i->next)
        {
            c = (CERT *) i->data;
            if (c->def)
            {
                return c->uid;
            }
        }
    }

    return 0;
}

X509 *CST_default_cert(CST * st, const char *email)
{
    X509 * result = NULL;
    if (st && NAME_IS_NOT_EMPTY(email))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            t_seqnum cid = cert_get_default_id(st, email);
            
            if (cid > 0)
            {
                CST_error(CST_ERROR_OK);
                result = cert_get_X509(st, cid);
            } else
            {
                CST_error(CST_ERROR_CERT_NOTFOUND);
            }
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

t_seqnum CST_default_cert_id(CST * st, const char *email)
{
    t_seqnum result = 0;
    if (st && NAME_IS_NOT_EMPTY(email))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            result = cert_get_default_id(st, email);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

int CST_is_default(CST * st, const cst_t_seqnum certID)
{
    int def = FALSE;
    if (st)
    {
        CST_LOCK_BEGIN_S(LOCK_SH);
            cert_get_param(st, certID, cert_get_default, NULL, (void *)&def);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return def;
}


/* S/MIME Capabilities */

/** 
    Helper function: search capability in list by OID
*/
static CERT_CAPABILITY * cert_get_capability(CERT * c, ASN1_OBJECT * oid)
{
    g_assert(c && oid);

    CERT_CAPABILITY * res = NULL;
    
    if (c->capabilities)
    {
        int length = i2d_ASN1_OBJECT(oid, NULL);
        unsigned char *buf = g_malloc(length);
        g_assert(buf);
        unsigned char *d = buf;
        
        i2d_ASN1_OBJECT(oid, &d);

        GSList * i;
        CERT_CAPABILITY * cap;
        for (i = c->capabilities; i != NULL; i = i->next)
        {
            cap = (CERT_CAPABILITY *) i->data;
          
            if ((length == cap->oid_length) 
                && (memcmp(buf, cap->oid, length) == 0))
            {
                res = cap;
                break;
            }
        }

        g_free(buf);
    }
    
    return res;
}

/** 
    Helper function: convert capability oid to ASN1_OBJECT
*/
static ASN1_OBJECT * cert_capability_oid(CERT_CAPABILITY * cap)
{
    g_assert(cap);
    ASN1_OBJECT *res;
    unsigned char *d = cap->oid;
    res = d2i_ASN1_OBJECT(NULL, &d, cap->oid_length);
    g_assert(res);

    return res;
}

/**
    Set S/MIME Capability for given cert
*/
typedef struct {ASN1_OBJECT * oid; int size; void * data;} CERT_CAPABILITY_HELPER;

static int cst_set_capability1(CERT * c, void * data)
{
    g_assert(c && data);

    CERT_CAPABILITY_HELPER * param = (CERT_CAPABILITY_HELPER *) data;
    g_assert(param->oid);

    CERT_CAPABILITY * cap = cert_get_capability(c, param->oid);
    
    if (!cap)
    {
        cap = (CERT_CAPABILITY *) g_malloc0(sizeof(CERT_CAPABILITY));

        cap->oid_length = i2d_ASN1_OBJECT(param->oid, NULL);
        g_assert(cap->oid_length > 0);
        cap->oid = g_malloc(cap->oid_length);
        g_assert(cap->oid);
        unsigned char * d = cap->oid;
        i2d_ASN1_OBJECT(param->oid, &d);
        
        c->capabilities = g_slist_append(c->capabilities, cap);
    }
    else
    {
        cap->data_length = 0;
        g_free(cap->data);
    }
    
    if (param->data && (param->size > 0))
    {
        cap->data_length = param->size;
        cap->data = g_malloc(cap->data_length);
        memcpy(cap->data, param->data, cap->data_length);
    }
    
    return CST_error(CST_ERROR_OK);
}

int cst_set_capability(CST * st, const cst_t_seqnum certID,
        ASN1_OBJECT * oid, unsigned char *data, int data_length)
{
    g_assert(st && oid);

    CERT_CAPABILITY_HELPER param;
    param.oid  = oid;
    param.data = data;
    param.size = data_length;

    return cert_set_param(st, certID, cst_set_capability1, (void*)&param);
}

int CST_set_capability(CST * st, const cst_t_seqnum certID,
        ASN1_OBJECT * oid, unsigned char *data, int data_length)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (st && oid)
    {
        CST_LOCK_BEGIN_S(LOCK_EX);
            result = cst_set_capability(st, certID, oid, data, data_length);
        CST_LOCK_END;
    }
    return result;
}

static int cst_delete_capability1(CERT * c, void * data)
{
    g_assert(c && data);

    ASN1_OBJECT * oid = (ASN1_OBJECT *) data;

    CERT_CAPABILITY * cap = cert_get_capability(c, oid);

    if (!cap)
    {
        return CST_error(CST_ERROR_CAPABILITY_NOTFOUND);
    }

    c->capabilities = g_slist_remove(c->capabilities, cap);

    cert_capability_free(cap);

    return CST_error(CST_ERROR_OK);
}

int cst_delete_capability(CST * st, const cst_t_seqnum certID,
        ASN1_OBJECT * oid)
{
    g_assert(st && oid);
    return cert_set_param(st, certID, cst_delete_capability1, (void *)oid);
}

int CST_delete_capability(CST * st, const cst_t_seqnum certID,
        ASN1_OBJECT * oid)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (st && oid)
    {
        CST_LOCK_BEGIN_S(LOCK_EX);
            result = cst_delete_capability(st, certID, oid);
        CST_LOCK_END;
    }
    return result;
}

static int cst_is_capability1(CERT * c, void * data, void * res)
{
    g_assert(c && data && res);
    ASN1_OBJECT * oid = (ASN1_OBJECT *) data;
    int * pres = (int *) res; 
    if (cert_get_capability(c, oid))
    {
        *pres = TRUE;
    }
    else
    {
        *pres = FALSE;
    }
    return CST_ERROR_OK;
}

int cst_is_capability(CST * st, const cst_t_seqnum certID, 
        ASN1_OBJECT * oid)
{
    g_assert(st && oid);
    int result;
    if (CST_ERROR_OK == cert_get_param(st, certID, cst_is_capability1, (void*)oid, (void*)&result))
    {
        return result;
    }
    return FALSE;
}

int CST_is_capability(CST * st, const cst_t_seqnum certID, 
         ASN1_OBJECT * oid)
{
    int result = FALSE;
    if (st && oid)
    {
        CST_LOCK_BEGIN_S(LOCK_SH);
            result = cst_is_capability(st, certID, oid);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

/**
    Get S/MIME Capability for given cert
*/

static int cst_get_capability_data1(CERT * c, void * data, void * res)
{
    g_assert(c && data && res);

    ASN1_OBJECT * oid = (ASN1_OBJECT *) data;
    DBT * pres = (DBT *) res;
    
    CERT_CAPABILITY * cap = cert_get_capability(c, oid);

    if (!cap)
    {
        return CST_ERROR_CAPABILITY_NOTFOUND;
    }
    
    pres->size = cap->data_length;

    if (cap->data_length > 0)
    {
        pres->data = g_malloc(cap->data_length);
        memcpy(pres->data, cap->data, cap->data_length);
    }
    
    return CST_ERROR_OK;
}

unsigned char * cst_get_capability_data(CST * st, 
        const cst_t_seqnum certID, 
        ASN1_OBJECT * oid, int *data_length)
{
    g_assert(st && oid && data_length);

    DBT res;
    
    if (CST_ERROR_OK == cert_get_param(st, certID, cst_get_capability_data1, (void*)oid, (void*)&res))
    {
        *data_length = res.size;
        return (unsigned char *) res.data;
    }

    *data_length = 0;
    return NULL;
}

unsigned char * CST_get_capability_data(CST * st, 
        const cst_t_seqnum certID, 
        ASN1_OBJECT * oid, int *data_length)
{
    unsigned char * result = NULL;
    if (st && oid && data_length)
    {
        *data_length = 0;
        CST_LOCK_BEGIN_S(LOCK_SH);
            result = cst_get_capability_data(st, certID, oid, data_length);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

/**
    Get S/MIME Capabilities list for given cert
*/

static int cst_get_capabilities1(CERT * c, void * data, void * r)
{
    g_assert(c && r);

    STACK_OF(ASN1_OBJECT) ** res = (STACK_OF(X509)**) r;

    *res = sk_ASN1_OBJECT_new_null();
    GSList * i;
    CERT_CAPABILITY * cap;
    for (i = c->capabilities; i != NULL; i = i->next)
    {
        cap = (CERT_CAPABILITY *) i->data;
        sk_ASN1_OBJECT_push(*res, cert_capability_oid(cap));
    }
    
    if (sk_ASN1_OBJECT_num(*res) == 0)
    {
        sk_ASN1_OBJECT_free(*res);
        *res = NULL;
        return CST_ERROR_CAPABILITY_NOTFOUND;
    }
   
    return CST_ERROR_OK;
}

STACK_OF(ASN1_OBJECT) * cst_get_capabilities(CST * st, 
        const cst_t_seqnum certID)
{
    g_assert(st);

    STACK_OF(ASN1_OBJECT) * result = NULL;

    if (CST_ERROR_OK == cert_get_param(st, certID, cst_get_capabilities1, NULL, (void*)&result))
    {
        return result;
    }
    return NULL;
}

STACK_OF(ASN1_OBJECT) * CST_get_capabilities(CST * st, 
        const cst_t_seqnum certID)
{
    STACK_OF(ASN1_OBJECT) * result = NULL;
    if (st)
    {
        CST_LOCK_BEGIN_S(LOCK_SH);
            result = cst_get_capabilities(st, certID);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

/* Other functions */

X509 * CST_get_cert(CST * st, const cst_t_seqnum certID)
{
    g_assert(st);
    X509 * x = NULL;
    if (st && (certID > 0))
    {
        CST_LOCK_BEGIN_S(LOCK_SH);
            x = cert_get_X509(st, certID);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return x;
}

void initial_scan_db_cert(CST * st, DBT * key, DBT * data)
{
    CERT * cert;
    X509 * cert_x;
            
    if (0 == cert_parser_info(st, data, &cert))
    {
        t_seqnum id = extract_id(key); 
        /* printf("CERT: %u\n", id); */
        cert->uid = id;
        cert_load_x(st, id, &cert_x);
        if (cert_x)
        {
            cert_calculate(st, cert, cert_x);
            //printf("i: %u, f: %u, p: %u, def: %u, kl: %u, email: %s\n", cert->uid, cert->folder, cert->purpose, cert->def, cert->key_link, cert->email); */
            if (!cert_put(st, cert))
            {
                cert_free(st, cert);
            }
            X509_free(cert_x);
        }
    }
}

static gboolean crl_check(CST * st, X509_NAME * name, ASN1_INTEGER * serial)
{
    g_assert(st && name && serial);
    gboolean result = FALSE;
    GSList * crls = st->simple_crls;
    GSList * i;
    X509_REVOKED *revoked = NULL;
    for (i = crls; i != NULL; i = i->next)
    {
        X509_CRL * x = crl_get_X509_CRL(st, GPOINTER_TO_UINT(i->data));
        if (x)
        {
            if (0 == X509_NAME_cmp(name, X509_CRL_get_issuer(x)))
            {
                STACK_OF(X509_REVOKED) * items = X509_CRL_get_REVOKED(x);    
                int i;    
                for (i = 0; i < sk_X509_REVOKED_num(items); i++)     
                {   
                    revoked = sk_X509_REVOKED_value(items, i);     
                    if (0 == ASN1_INTEGER_cmp(serial, revoked->serialNumber))
                    {
                        result = TRUE;
                        break;
                    }
                }
            }
            X509_CRL_free(x);
        }
        if (result) break;
    }
    return result;
}

int CST_is_revoked(CST * st, X509 * cert)
{
    int result = FALSE;
    if (st && cert)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            cst_t_seqnum certID = cert_search_by_UID(st, 
                    X509_get_issuer_name(cert),
                    X509_get_serialNumber(cert));
            if (certID > 0)
            {
                result = cst_is_revoked(st, certID);
            }
            else
            {
                result = cst_is_revoked_rescan(st, cert);
            }
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;    
}

static int cst_is_revoked_rescan(CST * st, X509 * cert)
{
    int result = FALSE;
    if (st && cert)
    {
        result = crl_check(st, X509_get_issuer_name(cert),
                     X509_get_serialNumber(cert));
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

int CST_delete_cert(CST * st, const cst_t_seqnum certID)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (st && (certID > 0))
    {
        CST_LOCK_BEGIN(LOCK_EX);
            cst_db_delete(st, CST_SUFFIX_CERTBODY, certID);
            result = cst_db_delete(st, CST_SUFFIX_CERTINFO, certID);
            cert_remove(st, certID);
        CST_LOCK_END;
    }

    return CST_error(result);
}

void CST_import_PKCS12(CST * st, 
                    FILE * file, 
                    cst_pkcs12_confirm_cb confirm_cb,
                    cst_pkcs12_error_cb error_cb,
                    unsigned char *password,
                    void * user_data,
                    GError **error)
{
    if (NULL == st 
     || NULL == file 
     || NULL == confirm_cb 
     || NULL == error_cb)
    {
        g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_PARAM_INCORRECT, "Params incorrect");
    }
   
    GError * local_error = NULL;
    int r = 0;
    int error_count = 0;
    
    PKCS12 * data = d2i_PKCS12_fp(file, NULL);

    if (data)
    {
        EVP_PKEY * pkey = NULL;
        X509 * cert = NULL;
        STACK_OF(X509) * ca = NULL;// = sk_X509_new_null();
        
        if (PKCS12_OK == PKCS12_parse(data, password, &pkey, &cert, &ca))
        {    
            cst_t_cert_folder folder = CST_FOLDER_CA;
            cst_t_cert_purpose purpose = CST_PURPOSE_NONE;
            unsigned char *out_pass = NULL;
            int cancel = FALSE;
            int is_pair = FALSE;
            int ok = TRUE;

            t_seqnum certID = 0;
            t_seqnum keyID  = 0;
            
            if (pkey && cert)
            {
                is_pair = TRUE;
                folder = CST_FOLDER_CA;
                purpose = CST_PURPOSE_NONE;
                ok = (*confirm_cb)(cert, &folder, &purpose, &out_pass, is_pair, &cancel, user_data);

                if (ok)
                {
                    // following function lock db
                    certID = cert_append_f(st, cert, folder, &local_error);
                    if (local_error != NULL && local_error->code != CST_ERROR_OK)
                    {
                        error_count++;
                        cancel = (*error_cb)(cert, local_error->code, user_data);
                    }
                    else
                    {
                        r = CST_set_purpose(st, certID, purpose, TRUE);
                        if (CST_ERROR_OK != r)
                        {
                            error_count++;
                            cancel = (*error_cb)(cert, r, user_data);
                        }
                    }
                    g_clear_error(&local_error);

                    if (!cancel)
                    {
                        keyID = key_append(st, 
                                pkey,
                                X509_get_subject_name(cert),
                                FALSE,
                                out_pass, 
                                &local_error);
                        if (local_error != NULL && local_error->code != CST_ERROR_OK)
                        {
                            error_count++;
                            cancel = (*error_cb)(cert, local_error->code, user_data);
                        }
                        else
                        {
                            r = CST_assign(st, certID, keyID, out_pass);
                            if (CST_ERROR_OK != r)
                            {
                                error_count++;
                                cancel = (*error_cb)(cert, r, user_data);
                            }
                        }
                        g_clear_error(&local_error);
                    }
                }
            }

            X509_free(cert);

            is_pair = FALSE;
            X509 * x = NULL;
            if (!cancel && ca)
            {
                int i;
                for (i = 0; i < sk_X509_num(ca); i++)
                {    
                    x = sk_X509_value(ca, i);
                    folder = CST_FOLDER_CA;
                    purpose = CST_PURPOSE_NONE;
                    ok = (*confirm_cb)(x, &folder, &purpose, &out_pass, is_pair, &cancel, user_data); 

                    if (ok)
                    {
                        certID = cert_append_f(st, x, folder, &local_error);
                        if (local_error != NULL && local_error->code != CST_ERROR_OK)
                        {
                            error_count++;
                            cancel = (*error_cb)(x, local_error->code, user_data);
                        }
                        else
                        {
                            r = CST_set_purpose(st, certID, purpose, TRUE);
                            if (CST_ERROR_OK != r)
                            {
                                error_count++;
                                cancel = (*error_cb)(x, r, user_data);
                            }
                        }   
                    }
                    g_clear_error(&local_error);

                    if (cancel) break;
                }
                sk_X509_pop_free(ca, X509_free);
            }

            if (cancel)
            {
                g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_CANCEL, "Import from PKCS12 was stopped after %i errors", error_count);
            }
        }
        else /* if error on parse */
        {
            unsigned long pkcs12_error_code = ERR_get_error();
            g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_PASSWORD_WRONG, "Password wrong on import PKCS12, %lu", pkcs12_error_code);
            error_count = 0;
        }
        PKCS12_free(data);
    }
    else /* if error on read */
    {
        unsigned long pkcs12_error_code = ERR_get_error();
        g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_STRUCTURE_CORRUPT, "File structure is corrupted, %lu", pkcs12_error_code);
        error_count = 0;
    }
}    
