/**
    @file sound.c

    ESD Sound functionality.

    Copyright (c) 2004, 2005 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/

#include "sound.h"
#include <esd.h>

#define COMMON SOUNDSDIR "/game-mahjong_"

gboolean
sound_init(AppSoundData * app_sound_data)
{
    static const gchar *sounds[SOUND_COUNT] = {
        COMMON "successful_selection.wav",
        COMMON "incorrect_selection.wav",
        COMMON "correct_combination.wav",
        COMMON "incorrect_combination.wav",
        COMMON "hint_usage.wav",
        COMMON "shuffle_usage.wav",
        COMMON "undo_usage.wav",
    };

    if (!app_sound_data || app_sound_data->esd > 0)
        return FALSE;

    app_sound_data->esd = esd_open_sound(NULL);

    if (app_sound_data->esd <= 0)
        return FALSE;

    int i;
    for (i = 0; i < SOUND_COUNT; i++)
        app_sound_data->sound_ids[i] = esd_file_cache(app_sound_data->esd,
                                                      "mahjong", sounds[i]);

    return TRUE;
}

void
sound_deinit(AppSoundData * app_sound_data)
{
    if (!app_sound_data)
        return;
    if (app_sound_data->esd <= 0)
        return;

    int i;
    for (i = 0; i < SOUND_COUNT; i++)
        esd_sample_free(app_sound_data->esd, app_sound_data->sound_ids[i]);

    esd_close(app_sound_data->esd);

    app_sound_data->esd = 0;
}

gboolean
sound_play(AppSoundData * app_sound_data, enum SoundType sound)
{
    if (!app_sound_data)
        return FALSE;
    if (app_sound_data->esd <= 0)
        return FALSE;
    if (!app_sound_data->sound_ids)
        return FALSE;
    if (app_sound_data->sound_ids[sound] <= 0)
        return FALSE;

    if (!
        ((app_sound_data->esd)
         &&
         (esd_sample_play
          (app_sound_data->esd, app_sound_data->sound_ids[sound]) >= 0)))
    {
        sound_deinit(app_sound_data);
        sound_init(app_sound_data);
        esd_sample_play(app_sound_data->esd,
                        app_sound_data->sound_ids[sound]);
    }

    return TRUE;
}
