/**
 * @file nitro-ui-utils.c
 *  
 * This file contains most of the UI operations for uploading Rich cores.
 * It uses NITRO Library for uploading
 *
 * This file is part of nitro
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gconf/gconf-client.h>
#include <libgnomevfs/gnome-vfs.h>
#ifdef  MAEMO_CHANGES
#include <hildon/hildon-note.h>
#else
#include <hildon-widgets/hildon-note.h>
#endif
#include <unistd.h>
#include <glib.h>
#include <stdlib.h> 
#include <string.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <glib/gstdio.h>

/*
 * for appending to LZO archive
 */
#define RICH_CORE_TMP_NOTEFILE	"/tmp/rich-core-note.txt"
#define LZOP_PROG		"/usr/bin/lzop"

#include "nitro-ui-utils.h"
#include "nitro-ui-main.h"
#include "nitro-handler.h"

/*It stores the list of files to be uploaded*/
extern GSList *files_list;

/* NITRO UI Application context */
extern osso_context_t *context;

/**
  This function is used to display all the confirmation notes in the appl.

  @param the type confirmation note to be displayed
  @return the response from the user
  */
static gint nitro_ui_display_note(nitroConfirmationType conf_type);

/**
  This function set the connection status callback notifiers 
  
  @param void
  @param void  
*/
void nitro_ui_set_connec_callback_notifiers();


/**
  When NITRO UI application needs to send all core files, this function
  is called to append all file to 'files_list'

  @param gchar * location of dir from which all files shud be appended
  @param TRUE if success else FALSE
*/
static gboolean nitro_ui_collect_all_corefile_names_at_location(const gchar *location);

void usecase_send_button_clicked (GtkWidget *button, GtkTextBuffer *buffer);
void usecase_other_button_clicked (GtkWidget *button, void *arg);
void usecase_options_button_clicked (GtkWidget *button, void *arg);
static GtkWidget *nitro_ui_usecase_dialog(gchar *text);
static void nitro_ui_process_response(gboolean send_all_files_mode, gint response);
static gchar *compose_crashed_note(gchar *str);

/*
 * counts and stats
 */
static gint tried_count = 0, succeeded_count = 0;

/*
 * pointer to usecase/options/send/delete dialog,
 * so that we can call response() on it from button handlers
 */
static GtkDialog *dialog = NULL;
extern gchar server_response[];

static void nitro_ui_process_response(gboolean send_all_files_mode, gint response)
{
    gchar *file_path=NULL;

    osso_log(LOG_DEBUG, "[%s]: enter, send_all_files_mode=%d, response=%d",
	     __FUNCTION__, send_all_files_mode, response);
    switch(response) {
    case GTK_RESPONSE_CANCEL:
	osso_log(LOG_DEBUG, "[%s]: User cancelled\n", __FUNCTION__);
	nitro_ui_exit();
	break;
	
    case GTK_RESPONSE_OK:
	/* Create an internet connection and wait for gconf connec callback*/
	if (!nitro_connect_iap()) {
	    nitro_ui_display_note(CONF_CONNECTION_ERROR);
	    osso_log(LOG_DEBUG, "[%s]: could not connect\n", __FUNCTION__);
	    nitro_ui_exit();
	}
	break;
    /* GTK_RESPONSE_APPLY is the response to delete single Core file, 
     *  or "Delete All" during nitro startup 
     */	
    case GTK_RESPONSE_APPLY:
	if(files_list) {
	    if(send_all_files_mode) {
		GSList *temp = NULL;
		while(files_list){
		    temp = files_list;

		    file_path = gnome_vfs_get_local_path_from_uri((gchar *)files_list->data);
		    if(g_remove(file_path) != 0)
			osso_log(LOG_DEBUG,"Could not remove the core file");
		    g_free(file_path);

		    files_list = g_slist_remove_link(files_list, temp);
		    g_free(temp->data);
		    g_slist_free(temp);
		}
	    } else {
		file_path = gnome_vfs_get_local_path_from_uri((gchar *)files_list->data);
		if(g_remove(file_path) != 0)
		    osso_log(LOG_DEBUG,"Could not remove the core file");
		g_free(file_path);
		
		files_list = g_slist_remove(files_list, files_list->data);
		osso_log(LOG_DEBUG, "[%s]: User choosed to delete\n", __FUNCTION__);
	    }
	}
	nitro_ui_exit();
	break;
    default:
	nitro_ui_exit();
	break;
    }
}

/**
  This function asks and takes the user confirmation
  and establishes the iap connection

  @param gboolean TRUE when need to send all files
		  FALSE when only one file is to be sent
  @param void              
*/
void nitro_ui_ask_user_conf(gboolean send_all_files_mode)
{
    gint response = 0;

    /* Display user confirmation to send rich cores */
    if(!send_all_files_mode) {
	response = nitro_ui_display_note(CONF_USER_CONFIRMATION1);
    }
    else {
	response = nitro_ui_display_note(CONF_USER_CONFIRMATION2);
    }
    nitro_ui_process_response(send_all_files_mode, response);
}

/**
  This is callback called when there are any changes to the variable
  NITRO_GCONF_CONNECTION_STATUS_DIR ie, to get the connection status

  @param the GConfClient that is notifying 
  @param cnxn_id, connection ID from gconf_client_notify_add()
  @param a GConfEntry entry
  @param user data
  @retuen void 
  */
void static nitro_ui_connect_status_cb (GConfClient *gconf_client_connec_dir,
				guint cnxn_id, GConfEntry *entry,
				gpointer user_data)
{
	NITROConnectionState connec_status = -1;
	gint result = -1;

	nitro_libs_gconf_get_connc_status(&connec_status);
	
	
	if(connec_status == CONN_CONNECTED){
		/* Succesfully able to connect */
		result = nitro_ui_upload_files(&tried_count, &succeeded_count);
	
		if(result == NITRO_SUCCESS) {
			/*upload success */
			nitro_ui_display_note(CONF_FILE_SENT);
			if(files_list != NULL) {
				nitro_ui_upload_files(NULL, NULL);
			}
			nitro_ui_exit();	
		} else {
		        /* Upload operation failed */
		    if (dialog) {
			osso_log(LOG_DEBUG, "[%s]: dialog already active, skip showing CURL_ERROR note", 
				 __FUNCTION__);
		    } else {
			osso_log(LOG_DEBUG,"[%s] Upload operation failed, Error=%d", __FUNCTION__, result);
			nitro_ui_display_note(CONF_CURL_ERROR);
			nitro_ui_exit();
		    }
		}
	} else if(connec_status == CONN_DISCONNECTED){
	    if (dialog) {
		osso_log(LOG_DEBUG, "[%s]: dialog already active, skip showing CONNECTION_ERROR note", 
			 __FUNCTION__);
	    } else {
		osso_log(LOG_DEBUG, "[%s] Couldnot create connection, connec_status error is %d", __FUNCTION__, connec_status);
		nitro_ui_display_note(CONF_CONNECTION_ERROR);
		nitro_ui_exit();
	    }
	}
}


/**
  This function set the connection status callback notifiers 
  
  @param void
  @param void  
*/
void nitro_ui_set_connec_callback_notifiers()
{
	GConfClient *gconf_client_connec_dir = NULL;
	gpointer user_data = NULL; 

	/* set callback notifier for gconf variable
	 * NITRO_GCONF_CONNECTION_STATUS*/	
	gconf_client_connec_dir = gconf_client_get_default();

	gconf_client_add_dir(gconf_client_connec_dir, 
			NITRO_GCONF_CONNECTION_STATUS_DIR,
			GCONF_CLIENT_PRELOAD_NONE, NULL	);

	gconf_client_notify_add(gconf_client_connec_dir, 
			NITRO_GCONF_CONNECTION_STATUS, 
			nitro_ui_connect_status_cb, 
			user_data, NULL, NULL);
}



/**
  This function resolves the application name from the executible

  @param executible name
  @return the app name
*/ 
static gchar *nitro_ui_get_appname_from_exe(gchar *exe)
{
	GnomeVFSDirectoryHandle *handle = NULL;
        GnomeVFSFileInfo *fileinfo = NULL;
	gchar *dir_text_uri=NULL;
	gchar *file_path=NULL;
       	gchar *app_name=NULL;
	gchar *exe_name=NULL;
	gchar *exe_base_name=NULL;
	gchar *group = NULL;
	GKeyFile *key = NULL;

	fileinfo = gnome_vfs_file_info_new();

	dir_text_uri = gnome_vfs_make_uri_from_shell_arg(DESKTOP_DIR_PATH);
	if(GNOME_VFS_OK !=
              gnome_vfs_directory_open(&handle, dir_text_uri,
		                      GNOME_VFS_FILE_INFO_DEFAULT)) {
		osso_log(LOG_DEBUG, "[%s] Could not open directory", __FUNCTION__);
		g_free(dir_text_uri);
		return NULL;
	}

	/* Scan all the desktop files to get the appname from exe name*/
	while((GNOME_VFS_OK == gnome_vfs_directory_read_next(handle, fileinfo))) {
		/*Scan in only desktop files*/
		if(g_strrstr(fileinfo->name, ".desktop") != NULL){
			file_path=g_strdup_printf("%s/%s",DESKTOP_DIR_PATH,fileinfo->name);
			key = g_key_file_new();

			if(key == NULL) {
				osso_log(LOG_DEBUG, "[%s] Could not create a key", __FUNCTION__);
				gnome_vfs_directory_close(handle);
				gnome_vfs_file_info_unref(fileinfo);
				g_free(file_path);
				g_free(dir_text_uri);
				return NULL;
			}
			
			if (g_key_file_load_from_file(key, file_path, G_KEY_FILE_NONE, NULL) == FALSE) {
				gnome_vfs_directory_close(handle);
				g_key_file_free(key);
				g_free(file_path);
				g_free(dir_text_uri);
				gnome_vfs_file_info_unref(fileinfo);
				return NULL;
				
			}
			
			group = g_key_file_get_start_group (key);
			exe_name = g_key_file_get_string(key, group, "Exec",NULL);
			exe_base_name = g_filename_display_basename(exe_name);
			/* If exe name match get appname*/
			if(!strcmp(exe_base_name, exe)) {
				app_name=g_key_file_get_string(key, group, "Name", NULL);
				gnome_vfs_directory_close(handle);
				g_key_file_free(key);
				g_free(group);
				g_free(exe_name);
				g_free(exe_base_name);
				g_free(file_path);
				g_free(dir_text_uri);
				gnome_vfs_file_info_unref(fileinfo);
				return app_name;
			}
			g_free(file_path);
			g_key_file_free(key);
			g_free(group);
			g_free(exe_name);
			g_free(exe_base_name);
			file_path=NULL;
			key=NULL;
			group=NULL;
			exe_name=NULL;
				
		}
		gnome_vfs_file_info_clear(fileinfo);
	}

	g_free(dir_text_uri);
	gnome_vfs_file_info_unref(fileinfo);
	gnome_vfs_directory_close(handle);
	return NULL;
}

/**
  this function resolves the application name from the Rich core

  @param text_uri of the Rich core
  @return app name
*/
static gchar * nitro_ui_get_appname_from_core(gchar *text_uri, gchar **sig_num, gchar **exe_name, gchar **process_id)
{
	gchar *app_name=NULL;
	gchar *temp=NULL;
	gchar *p_signum=NULL;
	gchar *p_after_signum=NULL;
	gchar *p_after_name=NULL;
	gchar *p_dot=NULL;
	gchar *p_pid=NULL;
	gchar *core_name = NULL;
	gchar *core_file_path=NULL;

	core_file_path = gnome_vfs_get_local_path_from_uri(text_uri);
	if(core_file_path == NULL)
		return NULL;
	
	core_name = g_filename_display_basename(core_file_path);
	if(core_name == NULL) {
		g_free(core_file_path);
		return NULL;
	}

	/*
	 * The name is something like: osso_pdfviewer-11-3447.rcore
	 * Parsing from start until dash or digits does not work
	 * because both of these may appear as part of executable name;
	 * We walk backwards from .rcore over digits, nondigit, digits
	 * which takes us to endpoint of executable name
	 */
	temp = g_strrstr(core_name, ".rcore");
	if(temp == NULL) {
	    g_free(core_file_path);
	    return NULL;
	}

#if 0
	      /*
	       * starting from nitro 1.17 we drop support 
	       * of plain cores. keeping that line here for historic
	       * reference, how it used to be until that.
	       */
	 /* We also support normal core dumps as rich-core might not
	 * be installed on all systems
	 */


	if(temp == NULL && (temp = g_strrstr(core_name, ".core")) == NULL ) {
		g_free(core_file_path);
		return NULL;
	}

	/* Rich and traditional cores have a different core pattern,
	 * thus we skip the part of the pattern unique to ordinary cores
	 * (non-rich cores have a version number like '-xxxx-xx' appended)
	 */

	if (!g_ascii_strcasecmp(temp, ".core")) {
		while(temp >= core_name && (*--temp) != '-');
		while(temp >= core_name && (*--temp) != '-');
	}
#endif

	p_dot = temp;
	while(temp >= core_name && g_ascii_isdigit(*--temp));
	p_after_signum = temp;
	p_pid = temp + 1;
	while(temp >= core_name && g_ascii_isdigit(*--temp));
	p_after_name = temp;
	p_signum = temp + 1;
	*exe_name = g_strndup(core_name, p_after_name - core_name);
	*sig_num = g_strndup(p_signum, p_after_signum - p_signum);
	*process_id = g_strndup(p_pid, p_dot - p_pid);

	app_name = nitro_ui_get_appname_from_exe(*exe_name);
	g_free(core_file_path);
	g_free(core_name);
	return app_name;
}


static gchar *compose_crashed_note(gchar *str)
{
    gchar *app_name = NULL;
    gchar *text = NULL;
    gchar *text2 = NULL;
    gchar *sig_num = NULL;
    gchar *exe_name = NULL;
    gchar *process_id = NULL;

    app_name = nitro_ui_get_appname_from_core(str, &sig_num, &exe_name, &process_id);
    if(app_name == NULL) {
	app_name = 
	    g_filename_display_basename((gchar *) files_list->data);
	if (sig_num && exe_name && process_id) {
	    text = g_strdup_printf(_("A process '%s' with PID %s died with signal %s."), exe_name, process_id, sig_num);
	} else if (app_name) {
	    text = g_strdup_printf(_("A crash report '%s' was generated."), app_name);
	} else {
	    text = g_strdup_printf(_("A crash file %s was generated."), (gchar *) files_list->data);
	}
    } else {
	if (sig_num && exe_name && process_id) {
	    text = g_strdup_printf(_("A process '%s' with PID %s of application '%s' died with signal %s."), 
				   exe_name, process_id, dgettext("maemo-af-desktop", app_name), sig_num);
	} else {
	    /*
	     * not likely to happen that we found app-name but there is no exename,signum;
	     * but for completeness let's have that 'else' part here
	     */
	    text = g_strdup_printf(_("The application '%s' crashed."), 
				   dgettext("maemo-af-desktop", app_name));
	}
    }	
    text2 = g_strdup_printf(_("%s The system can send the crash report for analysis."), text); 
    g_free(app_name);
    g_free(sig_num);
    g_free(exe_name);
    g_free(process_id);
    g_free(text);

    return text2;
}

/**
  This function is used to display all the confirmation notes in the appl.

  @param the type confirmation note to be displayed
  @return the response from the user
  */
static gint nitro_ui_display_note(nitroConfirmationType conf_type)
{
	GtkWidget *note = NULL;
	gchar *text = NULL;
        gint response = 0;
	
	if (dialog) {
	    osso_log(LOG_DEBUG, "[%s]: dialog already active, skip and return", 
		     __FUNCTION__);
	    return response;
	}
	switch(conf_type) {
		case CONF_USER_CONFIRMATION1:
			if(!files_list) {
				osso_log(LOG_DEBUG, "[%s]: Files list is NULL",__FUNCTION__);
				return GTK_RESPONSE_CANCEL;
			}
			text = compose_crashed_note((gchar *) files_list->data);
			note = nitro_ui_usecase_dialog(text);
			break;
		case CONF_USER_CONFIRMATION2:
			text = g_strdup_printf(_("This system has %u unsent Crash Reporter files. Send for analysis?"), g_slist_length(files_list));
            		note = hildon_note_new_confirmation_add_buttons(
						NULL,text, 
						_("Yes"), GTK_RESPONSE_OK,
					       	_("No"),GTK_RESPONSE_CANCEL, 
					       	_("Delete All"), GTK_RESPONSE_APPLY,
						NULL);
			break;
		case CONF_CONNECTION_ERROR:
			text = g_strdup(_("Nitro failed to connect"));
			note = hildon_note_new_information(NULL, text);
			hildon_note_set_button_text(HILDON_NOTE(note), _("Ok"));
			break;
		case CONF_CURL_ERROR:
		    if (server_response[0]) {
		        text = g_strdup_printf(_("Crash report upload was interrupted or rejected by server.\nServer response was:\n%s"), server_response);
		    } else {
		        text = g_strdup(_("Crash report upload did not succeed. Can not connect to server"));
		    }
			note = hildon_note_new_information(NULL, text);
			hildon_note_set_button_text(HILDON_NOTE(note), _("Ok"));
			break;
		case CONF_FILE_SENT:
		    if (tried_count == 1) {
			if(succeeded_count == 1) {
			    text = g_strdup(_("The crash report file was successfully uploaded"));
			} else {
			    text = g_strdup(_("The crash report file sending failed"));
			}
		    } else {
			    text = g_strdup_printf(_("Crash reports upload result: %d files attempted, %d succeeded"),
						     tried_count, succeeded_count);
		    }
		    
	
			note = hildon_note_new_information(NULL, text);
			hildon_note_set_button_text(HILDON_NOTE(note), _("Ok"));
			break;
		default:
			return 0;
	}
        
	dialog = GTK_DIALOG(note);
	response = gtk_dialog_run(dialog);
	dialog = NULL;
      	gtk_widget_destroy(GTK_WIDGET(note));
	note = NULL;

	g_free(text);
        return response;
}

/* Callback for close button */
void
usecase_send_button_clicked (GtkWidget *button, GtkTextBuffer *buffer)
{
    GtkTextIter start;
    GtkTextIter end;
    gchar *text;
    gchar *nfile;
    gchar *cmd;
    FILE *fp;
    gint len, cmd_len;
    int res;

    /* Obtain iters for the start and end of points of the buffer */
    gtk_text_buffer_get_start_iter (buffer, &start);
    gtk_text_buffer_get_end_iter (buffer, &end);

    /* Get the entire buffer text. */
    text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
    len = strlen(text);

    if (len > 0) {
	fp = fopen(RICH_CORE_TMP_NOTEFILE, "w");
	if(fp){
	    if (fwrite(text, sizeof(char), len, fp) < len) {
		osso_log(LOG_ERR, "[%s]: error writing rich-core-note to file %s", 
			 __FUNCTION__, RICH_CORE_TMP_NOTEFILE);
	    }
	    fclose(fp);
	    nfile = (gchar *)files_list->data + strlen("file://");
	    cmd_len = sizeof(LZOP_PROG) + sizeof(RICH_CORE_TMP_NOTEFILE) + strlen(nfile) + 20;
	    cmd = g_malloc(cmd_len);
	    if (cmd) {
		sprintf(cmd, "%s -c %s >> %s", LZOP_PROG, RICH_CORE_TMP_NOTEFILE, nfile);
		res = system(cmd);
		osso_log(LOG_ERR,"[%s]: system[%s] returned %d\n", __FUNCTION__, cmd, res);
		g_free(cmd);
	    }
	    unlink(RICH_CORE_TMP_NOTEFILE);
	} else {
	    osso_log(LOG_ERR,"[%s]: error opening file %s\n", __FUNCTION__, RICH_CORE_TMP_NOTEFILE);
	}
    }
    g_free (text);

    /*
     * If we are here, then dialog must exist and
     * point to usecase/send/delete/options dialog,
     * but lets check for being really sure 
     */
    if (dialog) {
	gtk_dialog_response(dialog, GTK_RESPONSE_OK);
    }
}

/* Callback for options button */
void
usecase_options_button_clicked (GtkWidget *button, void *arg)
{
    gint res = 0;
    GtkWidget *top_window = (GtkWidget*)arg;
    
    if (context && top_window) {
	res = osso_cp_plugin_execute(context, "libcpnitro.so", top_window, TRUE);
    } else {
	osso_log(LOG_DEBUG, "[%s]: no osso context or topwindow??",__FUNCTION__);
    }
}

/* Callback for other buttons */
void
usecase_other_button_clicked (GtkWidget *button, void *arg)
{
    gint response = (gint)arg;
    /*
     * If we are here, then dialog must exist and
     * point to usecase/send/delete/options dialog,
     * but lets check for being really sure 
     */
    if (dialog) {
	gtk_dialog_response(dialog, response);
    }
}

/**
  Create elements of "describe use case" and 
  send/cancel/del/options dialog
  */
static GtkWidget *nitro_ui_usecase_dialog(gchar *text)
{
    static GtkWidget *window;
    GtkWidget *scrolled_window;
    GtkWidget *label_note;
    GtkWidget *send_button;
    GtkWidget *delete_button;
    GtkWidget *cancel_button;
    GtkWidget *options_button;
    GtkTextBuffer *buffer;
    GtkWidget *text_view;
    GtkWidget *frame;
    
    /* Create a new dialog window for the scrolled window to be
     * packed into.  */
    window = gtk_dialog_new ();
    /***
    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                        (GtkSignalFunc) destroy, NULL);
    **/
    gtk_window_set_title (GTK_WINDOW (window), "Crash Reporter");
    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    gtk_widget_set_size_request(window, 600, 400);
    label_note = gtk_label_new(text);
    gtk_label_set_line_wrap(GTK_LABEL(label_note), TRUE);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), label_note, 
                        TRUE, TRUE, 0);

    /* create a new scrolled window. */
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);

    /* the policy is one of GTK_POLICY AUTOMATIC, or GTK_POLICY_ALWAYS.
     * GTK_POLICY_AUTOMATIC will automatically decide whether you need
     * scrollbars, whereas GTK_POLICY_ALWAYS will always leave the scrollbars
     * there.  The first one is the horizontal scrollbar, the second, 
     * the vertical. */
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    frame = gtk_frame_new("Please describe the use case:");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), frame, 
                        TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (frame), scrolled_window);


    /* Create a multiline text widget. */
    text_view = gtk_text_view_new ();
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_WORD);
    /* Obtaining the buffer associated with the widget. */
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_view));

    /* pack the text widget into the scrolled window */
    gtk_scrolled_window_add_with_viewport (
                   GTK_SCROLLED_WINDOW (scrolled_window), text_view);

    /*------------*/
    /* Add a "Send" button to the bottom of the dialog */
    send_button = gtk_button_new_with_label ("Send");
    g_signal_connect (G_OBJECT (send_button), "clicked",
		      G_CALLBACK (usecase_send_button_clicked),
		      buffer);
    
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
			send_button, TRUE, TRUE, 0);
    
    /*
     * excluding default grab (and CAN_DEFAULT above) to 
     * clarify the state with default action...
     * problem we have currently (nitro 1.20) is that
     * when pressing Enter while in multiline edit field, this warning comes:
     *
     GLIB WARNING ** GLib-GObject - gsignal.c:1019: unable to lookup signal "activate" of unloaded type `GtkEntry'
     *
     * currently no idea why is that, at least excluding those 2 things here does not help..
     */
    /*------------*/
    /* Add a "Options" button to the bottom of the dialog */
    options_button = gtk_button_new_with_label ("Options");
    g_signal_connect (G_OBJECT (options_button), "clicked",
		      G_CALLBACK(usecase_options_button_clicked),
		      window);
    
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
			options_button, TRUE, TRUE, 0);

    /*------------*/
    /* Add a "delete" button to the bottom of the dialog */
    delete_button = gtk_button_new_with_label ("Delete");
    g_signal_connect (G_OBJECT (delete_button), "clicked",
		      G_CALLBACK(usecase_other_button_clicked),
		      (void*)GTK_RESPONSE_APPLY);
    
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
			delete_button, TRUE, TRUE, 0);
    
    /*------------*/
    /* Add a "cancel" button to the bottom of the dialog */
    cancel_button = gtk_button_new_with_label ("Cancel");
    g_signal_connect (G_OBJECT (cancel_button), "clicked",
		      G_CALLBACK(usecase_other_button_clicked),
		      (void*)GTK_RESPONSE_CANCEL);
    
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
			cancel_button, TRUE, TRUE, 0);
    

    gtk_widget_show_all (window);

    return window;
}

/**
  When NITRO UI application needs to send all core files, this function
  is called to append all file to 'files_list'

  @param gchar * location of dir from which all files should be appended
  @param TRUE if success else FALSE
*/
static gboolean nitro_ui_collect_all_corefile_names_at_location(const gchar *location)
{
	gchar *text_uri_location = NULL;
	gchar *file_text_uri = NULL;
	GnomeVFSDirectoryHandle *handle = NULL;
	GnomeVFSFileInfo *fileinfo = NULL;
	
	g_return_val_if_fail(location != NULL, FALSE);
	
	fileinfo = gnome_vfs_file_info_new();
	text_uri_location = gnome_vfs_make_uri_from_shell_arg(location);
	if(GNOME_VFS_OK != 
			gnome_vfs_directory_open(&handle,
						text_uri_location,
					      	GNOME_VFS_FILE_INFO_DEFAULT)) {
		g_free(text_uri_location);
		gnome_vfs_file_info_unref(fileinfo);
		return FALSE;
	}

	while((GNOME_VFS_OK == gnome_vfs_directory_read_next(handle, fileinfo))){
		if( (g_str_has_suffix(fileinfo->name, ".rcore") ||
				g_str_has_suffix(fileinfo->name, ".rcore.lzo") 
#if 0
	      /*
	       * starting from nitro 1.17 we drop support 
	       * of plain cores. keeping that line here for historic
	       * reference, how it used to be until that.
	       */
		     || 
			    	g_str_has_suffix(fileinfo->name, ".core")
#endif
		     )
				&& !g_str_has_prefix(fileinfo->name, ".") ) {
			file_text_uri = g_strdup_printf("%s/%s",text_uri_location, fileinfo->name);
			files_list = g_slist_append(files_list, g_strdup(file_text_uri));
			g_free(file_text_uri);
			file_text_uri = NULL;
		}
		gnome_vfs_file_info_clear(fileinfo);
	}

	gnome_vfs_directory_close(handle);
	g_free(text_uri_location);
	gnome_vfs_file_info_unref(fileinfo);
	
	return TRUE;
}


/**
  The function will collect the names of Rich cores present in both the MMCs
  and append them to the files_list for uploading

  @param void
  @void gboolean
*/
gboolean nitro_ui_collect_all_corefile_names()
{
	gboolean rv=TRUE;


	if(!nitro_ui_collect_all_corefile_names_at_location(MMC1_LOCATION)) {
		osso_log(LOG_DEBUG, "[%s]: failed to append cores from MMC1", __FUNCTION__);
		rv = FALSE;
	}

	if(!nitro_ui_collect_all_corefile_names_at_location(MMC2_LOCATION)) {
		osso_log(LOG_DEBUG, "[%s]: failed to append cores from MMC2", __FUNCTION__);
		rv = FALSE;
	}
	return rv;
}

/**
  This function is called to upload all the files that stored in files_list
  This will call the nitro_libs APIs to upload the each file

  @param void
  @return void
  */
gint nitro_ui_upload_files(gint *tried_cnt, gint *succeeded_cnt)
{
	GSList *temp = NULL;
	temp = files_list;
	gint rv = -1;

	if (tried_cnt) {
	    *tried_cnt = 0;
	}
	if (succeeded_cnt) {
	    *succeeded_cnt = 0;
	}

	while(files_list){
		temp = files_list;
		osso_log(LOG_DEBUG, "[%s] preparing to upload = %s", __FUNCTION__, (gchar *)temp->data);
		if (tried_cnt) {
		    *tried_cnt += 1;
		}
		rv = nitro_upload((gchar *)temp->data);

		if (succeeded_cnt && (rv == NITRO_SUCCESS)) {
		    *succeeded_cnt += 1;
		}

		files_list = g_slist_remove_link(files_list, temp);
		g_free(temp->data);
		g_slist_free(temp);
		temp = NULL;
	}

	return rv;
}


/**
  This function is called to exit the NITRO UI application

  @param void
  @return void
  */
void nitro_ui_exit()
{
	NITROConnectionState connec_status = -1;
	nitro_libs_gconf_get_connc_status(&connec_status);

	if(connec_status == CONN_CONNECTED && files_list) {
		/* Uploads the newly generated
		 * pending Rich cores*/
		nitro_ui_upload_files(NULL, NULL);
	}
	g_slist_foreach (files_list, (GFunc)g_free, NULL);
	g_slist_free(files_list);
	
	if (gnome_vfs_initialized() == TRUE) {
		gnome_vfs_shutdown();
        }
        
	if (gtk_main_level() > 0)
		gtk_main_quit();

        osso_deinitialize(context);

	osso_log(LOG_DEBUG, "[%s]: NITRO UI exited \n", __FUNCTION__);

	exit(0);
}
