/**
 * @file nitro-settings-file.c
 *
 * This file contains the implementation for NITRO settings
 * file functions. 
 *
 * This file is part of nitro
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <glib.h>
#include <string.h>
#include <sys/stat.h>

#include "nitro-settings-file.h"

static GKeyFile *nitro_load_settings_key_file()
{
	gchar filename[256];
	GError *err = NULL;

	GKeyFile *key = g_key_file_new();
	if (key == NULL) {
		return NULL;
	}

	/*
	 * try system location 
	 */
	strcpy(filename, NITRO_SYSTEM_SETTINGS_LOCATION);
	strncat(filename, SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);
	if (g_key_file_load_from_file(key, filename, G_KEY_FILE_NONE, &err) == FALSE) {
	    /*
	     * no config file in system location, give up
	     */
	    if (err) {
		osso_log(LOG_DEBUG, "Failed loading key from %s error=%s", filename, err->message);
		g_error_free(err);
		err = NULL;
	    }
	    g_key_file_free(key);
	    osso_log(LOG_ERR, "Error reading server settings\n");
	    return NULL;
	}

	osso_log(LOG_DEBUG, "server settings loaded from '%s'", filename);
	return key;
}

static GKeyFile *nitro_load_privacy_settings_key_file()
{
    gchar filename[256];
    GError *err = NULL;

    GKeyFile *key = g_key_file_new();
    if (key == NULL) {
	return NULL;
    }

    strncpy(filename, getenv("HOME"), sizeof(filename)-1);
    strncat(filename, PRIVACY_SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);

    if (g_key_file_load_from_file(key, filename, G_KEY_FILE_NONE, &err) == FALSE) {
	if (err) {
	    osso_log(LOG_ERR, "Failed loading key from '%s' error=%s", filename, err->message);
	    g_error_free(err);
	    err = NULL;
	}
	/*
	 * no privacy config file under HOME, try system location
	 */
	strcpy(filename, NITRO_SYSTEM_SETTINGS_LOCATION);
	strncat(filename, PRIVACY_SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);
	if (g_key_file_load_from_file(key, filename, G_KEY_FILE_NONE, &err) == FALSE) {
	    /*
	     * no privacy config file in system location either, give up
	     */
	    if (err) {
		osso_log(LOG_DEBUG, "Failed loading key from %s error=%s", filename, err->message);
		g_error_free(err);
		err = NULL;
	    }
	    g_key_file_free(key);
	    osso_log(LOG_ERR, "Error reading privacy settings\n");
	    return NULL;
	}
    }

    osso_log(LOG_DEBUG, "privacy settings loaded from '%s'", filename);
    return key;
}

/**
  This function reads the Server & Credentials settings file values.

  @param none
  @return the nitroSettings structure pointer
*/
nitroSettings *nitro_read_settings()
{
	GKeyFile *key = NULL;
	GError *err = NULL;
	nitroSettings *details = NULL;

	key = nitro_load_settings_key_file();
	if (!key) {
	    return NULL;
	}

	details = g_new0(nitroSettings, 1);

	if ((details->user_name = g_key_file_get_value(key, SERVER_SECTION,
						       USER_NAME, &err)) == NULL) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", USER_NAME, err->message);
		g_error_free(err);
		err = NULL;
	}

	if ((details->passwd = g_key_file_get_value(key, SERVER_SECTION, PWD, &err)) == NULL) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", PWD, err->message);
		g_error_free(err);
		err = NULL;
	}

	if ((details->server_addr = g_key_file_get_value(key, SERVER_SECTION,
							 SERVER_ADDR, &err)) == NULL) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", SERVER_ADDR, err->message);
		g_error_free(err);
		err = NULL;
	}

	details->use_ssl = g_key_file_get_boolean(key, SERVER_SECTION, USE_SSL, &err); 
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", USE_SSL, err->message);
		g_error_free(err);
		err = NULL;
	}

	g_key_file_free(key);

	/* to be freed by calling function */
	return details;
}

/**
  This function reads the Privacy settings file values.

  @param none
  @return the nitroSettings structure pointer
*/
privacySettings *nitro_read_privacy_settings()
{
	GKeyFile *key = NULL;
	GError *err = NULL;
	privacySettings *details = NULL;

	key = nitro_load_privacy_settings_key_file();
	if (!key) {
	    return NULL;
	}

	details = g_new0(privacySettings, 1);

	details->enabled = g_key_file_get_boolean(key, SETTINGS_SECTION,
						  NITRO_ENABLED, &err);
	if(err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", NITRO_ENABLED, err->message);
		g_error_free(err);
		err = NULL;
	}

	details->include_core = g_key_file_get_boolean(key, PRIVACY_SECTION, 
						       INCLUDE_CORE, &err);
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", INCLUDE_CORE, err->message);
		g_error_free(err);
		err = NULL;
	}
	details->include_syslog = g_key_file_get_boolean(key, PRIVACY_SECTION, 
							 INCLUDE_SYSLOG, &err);
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", INCLUDE_SYSLOG, err->message);
		g_error_free(err);
		err = NULL;
	}
	details->include_pkglist = g_key_file_get_boolean(key, PRIVACY_SECTION, 
							  INCLUDE_PKGLIST, &err);
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", INCLUDE_PKGLIST, err->message);
		g_error_free(err);
		err = NULL;
	}

	g_key_file_free(key);

	/* to be freed by calling function */
	return details;
}

/**
  This function writes the privacy settings file.

  @param sett privacySettings structure pointer
  @return void
*/
void nitro_write_privacy_settings(privacySettings * sett)
{
	GKeyFile *key = NULL;
	FILE *file = NULL;
	gchar filename[256];

	key = nitro_load_privacy_settings_key_file();
	if (!key) {
	    return;
	}

	g_key_file_set_boolean(key, SETTINGS_SECTION, NITRO_ENABLED, sett->enabled);
	g_key_file_set_boolean(key, PRIVACY_SECTION,  INCLUDE_CORE, sett->include_core);
	g_key_file_set_boolean(key, PRIVACY_SECTION,  INCLUDE_SYSLOG, sett->include_syslog);
	g_key_file_set_boolean(key, PRIVACY_SECTION,  INCLUDE_PKGLIST, sett->include_pkglist);

	strncpy(filename, getenv("HOME"), sizeof(filename)-1);
	strncat(filename, PRIVACY_SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);
	file = fopen (filename, "w");
	osso_log(LOG_DEBUG, "[%s]: fopen(%s)=%p", __FUNCTION__, filename, file);
	if (file)
            {
		gsize length = 0;
		char *data = g_key_file_to_data (key, &length, NULL);
		osso_log(LOG_DEBUG, "[%s]: key_file_to_data=%p len=%d", __FUNCTION__, data, length);
                if (length > 0)
                    fwrite (data, length, 1, file);
                g_free (data);
                fclose (file);
            }
	
	g_key_file_free(key);

	/* to be freed by calling function */
}

/**
  this function frees the nitroSettings structure

  @param nitroSettings pointer that is to be freed
  @return void
*/
void nitro_free_settings(nitroSettings * settings)
{
	if (settings == NULL)
		return;

	g_free(settings->user_name);
	g_free(settings->passwd);
	g_free(settings->server_addr);
	g_free(settings);
	return;
}

/**
  this function frees the privacySettings structure

  @param privacySettings pointer that is to be freed
  @return void
*/
void nitro_free_privacy_settings(privacySettings * settings)
{
	g_free(settings);
	return;
}
