/*
 * @file nitro-handler.c
 *
 * This file contains the function defs
 * for  nitro-handler.c.
 *
 * This file is part of nitro
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#define DBUS_API_SUBJECT_TO_CHANGE
#include <sys/stat.h>
#include <conic.h>
#include <glib.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <stdlib.h>
#include <string.h>
#include <sys/syslog.h>
#include <sys/stat.h>
#include <unistd.h>
#include <curl/curl.h>
#include <gtk/gtk.h>
#include <glib/gstdio.h>
#include <gconf/gconf-client.h>
#include <libgnomevfs/gnome-vfs.h>

#include "nitro-handler.h"
#include "nitro-settings-file.h"
#include "nitro-upload.h"

/* this is a connection instance */
static ConIcConnection *connection = NULL;

/**
  This function is for freeing the dbus connection
                                                                                                                      
  @param dbus_connection pointer to the DBusConnection.
  @return none.
  */
static void nitro_dbus_free(DBusConnection * dbus_connection)
{
	if (dbus_connection != NULL) {
		osso_log(LOG_DEBUG, "freeing dbus\n");
#ifdef MAEMO_CHANGES
		dbus_connection_close(dbus_connection);
#else
		dbus_connection_disconnect(dbus_connection);
#endif
		dbus_connection_unref(dbus_connection);
		dbus_connection = NULL;
	}
}

/**
  This function is for obtaining/removing a dbus connection 
                                                                                                                      
  @param create TRUE to get a connection else FALSE
  @return TRUE on success else FALSE.
  */
static gboolean nitro_handle_dbus(gboolean create)
{
	DBusError error;
	static DBusConnection *dbus_connection = NULL;

	if (create == FALSE) {
		nitro_dbus_free(dbus_connection);
		return TRUE;
	}

	dbus_error_init(&error);
	if (dbus_connection == NULL) {
		dbus_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
		if (dbus_connection == NULL) {
			osso_log(LOG_ERR, "Error conecting to system bus %s\n", error.message);
			return FALSE;

		}
	}

	return TRUE;
}


/** 
  This function uploads the files to the configured server

  @param the text_uri of the file to upload
  @return NITRO_ERROR types in case of failure else NITRO_SUCCESS
  */
gint nitro_upload(gchar * text_uri)
{

	nitroSettings *settings = NULL;
	gchar *uploadfile = NULL;
	gchar *base_name = NULL;
	gchar *file_path = NULL;
	gchar *temp = NULL;
	gint result = 0;
	gint rv = -1;
	GnomeVFSURI *file_uri = NULL;
	struct stat s;

	settings = nitro_read_settings();
	if (!settings) {
		osso_log(LOG_DEBUG, "[%s]: Unable to read settings!\n", __FUNCTION__);
		return NITRO_NO_SETTINGS_ERROR;
	} else {
		osso_log(LOG_DEBUG, "[%s]: Settings: '%s'-'%s'-'%s'\n", 
				__FUNCTION__, settings->server_addr,
			settings->user_name, settings->passwd);
	}
	file_uri = gnome_vfs_uri_new(text_uri);
	base_name = gnome_vfs_uri_extract_short_name(file_uri);
	
	uploadfile = g_strdup_printf("%s%s", settings->server_addr, base_name);

	gnome_vfs_uri_unref(file_uri);
	file_uri = NULL;

	temp = text_uri+strlen("file://");
	file_path = g_strdup(temp);
	
	if(stat(file_path, &s) < 0) {
		osso_log(LOG_DEBUG, "[%s]: File %s not Found", __FUNCTION__, file_path);
		return NITRO_CURL_FILE_NOT_FOUND;
	}

	osso_log(LOG_DEBUG, "[%s]: Uploading %s ...\n", __FUNCTION__, file_path);
	result = upload(file_path, uploadfile, settings->user_name,
			settings->passwd, settings->use_ssl);


	if ((result == CURLE_COULDNT_CONNECT) || (result == CURLE_OPERATION_TIMEOUTED) || (result != 0)) {
		rv = NITRO_CURL_COULD_NOT_CONN_ERROR;
	} else if (result == 0) {
		//if(nitro_libs_gconf_get_remove_core_status()) {
			g_remove(file_path);
		//}
		osso_log(LOG_DEBUG, "[%s]: Uploading succesful ", __FUNCTION__);
		rv = NITRO_SUCCESS;
	}
	g_free(base_name);
	g_free(uploadfile);
	g_free(file_path);
	nitro_free_settings(settings);
	return rv;
}


/**
  This function is the callback for connection-event.
  Uploading of the raw data file will be done after successful
  connection to IAP is made.
                                                                                                                      
  @param connection the ConIcConnection object.
  @param event pointer to the ConIcConnectionEvent.
  @param upload_file Raw data file to be uploaded
  @return none.
  */
static void nitro_connection_cb(ConIcConnection * connection,
			       ConIcConnectionEvent * event, gpointer user_data)
{
	ConIcConnectionStatus status = -1;

	status = con_ic_connection_event_get_status(event);
	switch (status) {
	case CON_IC_STATUS_CONNECTED:
		nitro_libs_gconf_set_connc_status(CONN_CONNECTED);
		osso_log(LOG_DEBUG, "[%s]: Event status is CON_IC_STATUS_CONNECTED", __FUNCTION__);
		break;

	case CON_IC_STATUS_DISCONNECTED:
		osso_log(LOG_DEBUG, "[%s]: Event status is CON_IC_STATUS_DISCONNECTED",
			 __FUNCTION__);
		nitro_libs_gconf_set_connc_status(CONN_DISCONNECTED);
		break;

	default:
		osso_log(LOG_DEBUG, "[%s]: Event status is UNKNOWN", __FUNCTION__);
		break;

	}
	return;
}

/**
  This function is for creating/destroying a conic connection 
                                                                                                                      
  @param create TRUE to create a conic connection else FALSE
  @return TRUE on success else FALSE.
  */
static gboolean nitro_handle_ic_connection(gboolean create)
{
	if (create) {
		if (connection) {
			osso_log(LOG_DEBUG, "[%s]: ConIcConnection is already available",
				 __FUNCTION__);
			return TRUE;
		}

		if (nitro_handle_dbus(TRUE) == FALSE) {
			osso_log(LOG_DEBUG, "[%s]: Failure in handling DBUS", __FUNCTION__);
			return FALSE;
		}

		connection = con_ic_connection_new();
		
		if (!connection) {
			osso_log(LOG_DEBUG, "[%s]: Failure in creating a new IC Connection",
				 __FUNCTION__);
			nitro_handle_dbus(FALSE);
			return FALSE;
		}
	}  else {
		if (!connection) {
			osso_log(LOG_DEBUG, "[%s]: ConIcConnection is not available", __FUNCTION__);
		} else {
			g_object_unref(connection);
			connection = NULL;
//modified the following line
			return FALSE;
		}
	}

	return TRUE;
}

/**
  This function is to initiate a connection to the default IAP.
  Uploading of the raw data file will be done in the connection
  callback function after successful connection to IAP is made.

  @param void
  @return TRUE on success else FALSE.
  */
gboolean nitro_connect_iap()
{
	static gulong signal_handler = 0;
	gpointer user_data = NULL;

	nitro_libs_gconf_set_connc_status(CONN_INITIATED);

	if (nitro_handle_ic_connection(TRUE) == FALSE) {
		return FALSE;
	}
	
	
	if (signal_handler != 0) {
		g_signal_handler_disconnect(G_OBJECT(connection), signal_handler);
	}


	signal_handler = g_signal_connect(G_OBJECT(connection),
					  "connection-event",
					   G_CALLBACK(nitro_connection_cb),
					   user_data);

	con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);

	return TRUE;
}


/**
  This function will initialize the nitro_libs

  @param void
  @return if success TRUE  else FALSE
  */
gboolean nitro_libs_init()
{
	GConfClient *client = NULL;

	g_type_init();

	if((client = gconf_client_get_default()) == NULL) {
		return FALSE;
	}

	if(!nitro_libs_gconf_set_connc_status(CONN_NOT_INITIATED))
		return FALSE;
	return TRUE;
}


/**
  This function will store the NITROConnectionState value as gconf data

  @param NITROConnectionState value
  @return TRUE if success else FALSE
  */
gboolean nitro_libs_gconf_set_connc_status(NITROConnectionState val)
{
	GConfClient *client = NULL;

	if((client = gconf_client_get_default()) == NULL)
		return FALSE;
	gconf_client_set_int(client, NITRO_GCONF_CONNECTION_STATUS, val, NULL);
	return TRUE;
}


/**
  This function will get the NITROConnectionState value from gconf file

   @param NITROConnectionState value
   @return TRUE if success else FALSE
   */
gboolean nitro_libs_gconf_get_connc_status(NITROConnectionState *val)
{
	GConfClient *client = NULL;

	if((client = gconf_client_get_default()) == NULL)
		return FALSE;
	*val = gconf_client_get_int(client, NITRO_GCONF_CONNECTION_STATUS, NULL);
	return TRUE;
}

