/**
 * @file nitro-daemon-monitor.c
 *  
 * This file contains the monitoring functions for NITRO device daemon
 * and will invoke UI in case they have to be uploaded to server.
 *
 * This file is part of nitro
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "nitro-daemon-defs.h"
#include "nitro-daemon-monitor.h"

/* is TRUE if daemon is already handling the file creation event*/
static gboolean handling_in_progress = FALSE;

/**
  This function check if the file generated should be rejected to upload
  it will reject if the file starts with "." or is a nitro_ui Rich core

  @param file_text_uri
  @param TRUE if it is to be rejected else FALSE
*/
static gboolean is_core_to_be_ignored(gchar *file_text_uri)
{
	gchar *basename = NULL;
	gboolean rv = FALSE;
	
	g_return_val_if_fail(file_text_uri != NULL, FALSE);	
	basename = g_filename_display_basename(file_text_uri);

	if(! (g_str_has_suffix(basename, ".rcore") ||
		g_str_has_suffix(basename, ".rcore.lzo")
#if 0
	      /*
	       * starting from nitro 1.17 we drop support 
	       * of plain cores. keeping that line here for historic
	       * reference, how it used to be until that.
	       */
	      || g_str_has_suffix(basename, ".core")
#endif
	      )) {
	    /*Ignore if it is not a Rich core*/
		rv = TRUE;
	} else if(g_str_has_prefix(basename, ".")) {
	    /*Ignore the partially created cores,
	     * these have the prefix '.'*/
		rv = TRUE;
	} else if(g_strrstr(file_text_uri, NITRO_UI)) {
	    /* Ignore nitro_ui cores*/
		rv = TRUE;
	} 

	g_free(basename);
	return rv;
}


/**
  This function will invoke NITRO_UI through RPC with 
  method name nitro_send_file

  @param the text_uri of the file that was created
  @return FALSE 
  */
static gboolean send_rich_core(gchar * file_text_uri)
{
	osso_rpc_t rpc_ret;
	osso_return_t retval;

	osso_log(LOG_INFO, "[%s]: The file with uri %s is created", __FUNCTION__, file_text_uri);
	
	if(file_text_uri == NULL) {
		osso_log(LOG_INFO, "[%s]: NULL param passed", __FUNCTION__);
		handling_in_progress = FALSE;
		return FALSE;
	}
	
	if(is_core_to_be_ignored(file_text_uri)) {
		osso_log(LOG_INFO," This file is ignored");
		handling_in_progress = FALSE;
		return FALSE;
	}
		
	retval = osso_rpc_run_with_defaults(context, NITRO_SERVICE, NITRO_METHOD_SEND_FILE, &rpc_ret, DBUS_TYPE_STRING, file_text_uri, DBUS_TYPE_INVALID);

	if(retval != OSSO_OK) {
		if (rpc_ret.type == DBUS_TYPE_STRING) {
			osso_log(LOG_ERR, "Error in service %s rpc method %s:%s",
					NITRO_SERVICE, NITRO_METHOD_SEND_FILE, rpc_ret.value.s);
			g_free((gchar *) rpc_ret.value.s);
		} else {
			osso_log(LOG_ERR, "Error in service %s rpc method %s", NITRO_SERVICE, NITRO_METHOD_SEND_FILE);
		}
	}

	handling_in_progress = FALSE;
	return FALSE;
}

/**
  This function will invoke NITRO_UI through RPC
  with method_name nitro_send_all_files

  @param the text_uri of the file that was created
  @return FALSE 
  */
static gboolean send_all_rich_cores()
{
	osso_rpc_t rpc_ret;
	osso_return_t retval;
	
	retval = osso_rpc_run_with_defaults(context, NITRO_SERVICE, NITRO_METHOD_SEND_ALL, &rpc_ret, DBUS_TYPE_INVALID);

	if(retval != OSSO_OK) {
		if (rpc_ret.type == DBUS_TYPE_STRING) {
			osso_log(LOG_ERR, "Error in service %s rpc method %s:%s",
					NITRO_SERVICE, NITRO_METHOD_SEND_ALL, rpc_ret.value.s);
			g_free((gchar *) rpc_ret.value.s);
		} else {
			osso_log(LOG_ERR, "Error in service %s rpc method %s", NITRO_SERVICE, NITRO_METHOD_SEND_ALL);
		}
	}

	return FALSE;
}


/**
  This function is a callback for gnome_vfs_monitor_add() for both the MMCs
  This function handles GNOME_VFS_MONITOR_EVENT_CREATED event only

  @param handle of the monitor that created the event
  @param the URI of the monitor that was triggered 
  @param the URI of the actual file this event is concerned with
  @param event_type - what happened to info_uri 
  @param user_data
  */
static void monitor_event(GnomeVFSMonitorHandle * handle,
			  const gchar * monitor_uri,
			  const gchar * info_uri,
			  GnomeVFSMonitorEventType event_type,
			  gpointer user_data)
{
	if (event_type == GNOME_VFS_MONITOR_EVENT_CREATED) {
		osso_log(LOG_DEBUG,
			 "[%s]: Rich Core files are generated : Event type : %d",
			 __FUNCTION__, event_type);
		if (handling_in_progress == TRUE) {
			osso_log(LOG_DEBUG,
				 "[%s]: Event handling already in progress!",
				 __FUNCTION__);
			return;
		}

		handling_in_progress = TRUE;
		g_idle_add((GSourceFunc)send_rich_core, (gpointer) info_uri);
	}
}


/**
  This function checks for the presence of core files at a location

  @param gchar - location
  @param gboolean - TRUE if Cores are present else FALSE
*/
static gboolean check_for_cores_at_location(const gchar *location)
{

	gchar *text_uri = NULL;	
	GList *list = NULL;
	GList *temp = NULL;
	gboolean rv = FALSE;
	gint result = -1;
	gint count = 0;
	gchar * file_name = NULL;
	
	if(!location) {
		osso_log(LOG_DEBUG, "[%s] NULL param passed", __FUNCTION__);
		return FALSE;
	}

	text_uri = gnome_vfs_get_uri_from_local_path(location);
	result = gnome_vfs_directory_list_load(&list, text_uri, GNOME_VFS_FILE_INFO_DEFAULT );
	if(result != GNOME_VFS_OK)
		osso_log(LOG_DEBUG, "[%s] Failed to load list result=%d", __FUNCTION__, result);
	
	if(g_list_length(list) > 2) {
	       for (temp = list; temp != NULL; temp = temp->next) {
		       file_name = ((GnomeVFSFileInfo *)temp->data)->name;
		       if((g_str_has_suffix(file_name, ".rcore") ||
			   g_str_has_suffix(file_name, ".rcore.lzo") 
#if 0
	      /*
	       * starting from nitro 1.17 we drop support 
	       * of plain cores. keeping that line here for historic
	       * reference, how it used to be until that.
	       */
			   ||
			   g_str_has_suffix(file_name, ".core")
#endif
			   ))
			       count++;
		       gnome_vfs_file_info_clear((GnomeVFSFileInfo *)temp->data);
	       }
	       if (count > 0 ) {
		       osso_log(LOG_DEBUG, ".rcore or core files are present");
			rv = TRUE;
	       }
	}
	g_list_foreach(list, (GFunc) g_free, NULL);
	g_list_free(list);
	g_free(text_uri);
	
	return rv;
}

/**
  This function sets the monitors for directories in MMC1 and MMC2
  The monitor_event() will be called when any changes
  are done to the monitored directories

  @param gchar* path monitor (which must be MMC1_LOCATION or MMC2_LOCATION)
  @return TRUE if success else FALSE
*/
gboolean set_directory_monitor(gchar * location)
{
	GnomeVFSMonitorHandle **handle = NULL;
	static GnomeVFSMonitorHandle *mmc1_handle = NULL;
	static GnomeVFSMonitorHandle *mmc2_handle = NULL;
	GnomeVFSResult result;
	GnomeVFSMonitorType mtype;
	GnomeVFSFileInfo *info = NULL;
	gchar *text_uri = NULL;

	if(!strcmp(location, MMC1_CORES_LOCATION)) {
	       handle = &mmc1_handle;
	} else if(!strcmp(location, MMC2_CORES_LOCATION)) {
	       handle = &mmc2_handle;
	} else {
		osso_log(LOG_DEBUG,"[%s] Invalid location (= %d) to monitor", __FUNCTION__, location);
		return FALSE;
	}
		
	if (*handle != NULL) {
		result = gnome_vfs_monitor_cancel(*handle);
		if (result != GNOME_VFS_OK) {
			osso_log(LOG_DEBUG,
				 "[%s]: Could not cancel previous monitor",
				 __FUNCTION__);
			return FALSE;
		}
		*handle = NULL;
	}

	text_uri = gnome_vfs_make_uri_from_shell_arg(location);
	if (text_uri == NULL) {
		osso_log(LOG_DEBUG,
			 "[%s]: Could not create uri from location",
			 __FUNCTION__);
		return FALSE;
	}

	info = gnome_vfs_file_info_new();
	if (info == NULL) {
		g_free(text_uri);
		return FALSE;
	}
	result = gnome_vfs_get_file_info(text_uri, info, 0);
	if (result != GNOME_VFS_OK) {
		osso_log(LOG_DEBUG, "[%s]: Could not get file info for %s",
			 __FUNCTION__, text_uri);
		gnome_vfs_file_info_unref(info);
		g_free(text_uri);
		return FALSE;
	}

	osso_log(LOG_DEBUG, "[%s]: Starting monitor for %s\n", __FUNCTION__,
		 text_uri);
	if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		mtype = GNOME_VFS_MONITOR_DIRECTORY;
	} else {
		mtype = GNOME_VFS_MONITOR_FILE;
	}
	result = gnome_vfs_monitor_add(handle, text_uri, mtype,
				       monitor_event, location);
	if (result != GNOME_VFS_OK) {
		osso_log(LOG_ERR, "[%s]: Could not start monitor for %s\n",
			 __FUNCTION__, text_uri);
		gnome_vfs_file_info_unref(info);
		g_free(text_uri);
		return FALSE;
	}

	g_free(text_uri);
	gnome_vfs_file_info_unref(info);

	return TRUE;

}


/**
  This function checks for the presence of core files in 
  'core-dumps' directories in both the MMCs. If present, it will 
  send the cores by calling send_all_rich_cores()

  @param void
  @param void
*/
void check_for_cores()
{
	if( check_for_cores_at_location(MMC1_CORES_LOCATION)
	     ||check_for_cores_at_location(MMC2_CORES_LOCATION) ) {
		send_all_rich_cores();
	}
}
