/*
 * @file nitro-settings-dialog.c
 *
 * This file contains the implementation of
 * settings dialog.
 * 
 * This file is part of nitro
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <sys/wait.h>
#include <stdlib.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gtk/gtkcombobox.h>
#ifdef  MAEMO_CHANGES
#include <hildon/hildon-caption.h>
#include <hildon/hildon-defines.h>
#include <hildon/hildon-banner.h>
#include <hildon-help/hildon/hildon-help.h>
#include <hildon/hildon-note.h>
#else
#include <hildon-widgets/hildon-caption.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-defines.h>
#include <hildon-widgets/hildon-note.h>
#include <osso-helplib.h>
#endif
#include <gconf/gconf-client.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-xfer.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <string.h>

#include "nitro-settings-dialog.h"
#include "nitro-settings-file.h"

static void nitro_cp_show_privacy_disclaimer(gpointer parent);

/**
  This function is used to start/stop the NITRO Daemon.
  It invokes a script that does this job.

  @param gboolean - TRUE to start, FALSE to stop
  @return gboolean - TRUE if success, else FALSE 
  */
static gboolean nitro_cp_execute_invoker_script(gboolean start)
{
	gchar *argv[3];
        gint pid = 0;
        gint status = 0;

	pid = fork();
        if (pid == -1) {
                return FALSE;
        }	
	if (pid == 0) {
		argv[0] = "sh";
		if(start)
			argv[1] = "start";
		else
			argv[1] = "stop";
		argv[2] = 0;
		chmod(SCRIPT, S_IRUSR | S_IWUSR | S_IXUSR);
		if (execve(SCRIPT, argv, NULL) == -1) {
			osso_log(LOG_DEBUG, "[%s]: Error in execve", __FUNCTION__);
		}
		exit(127);
	}
	do{
		if (waitpid(pid, &status, 0) == -1) {
			break;
		}
	}while (1);
	return TRUE;
}


/**
  This function is used to check if the NITRO Daemon is running

  @param void
  @return gboolean - TRUE if the daemon is running, else FALSE
  */
static gboolean nitro_cp_if_nitro_daemon_running()
{
	if(system("pidof nitro_daemon") == 0) {
		osso_log(LOG_DEBUG, "[%s]: daemon is running", __FUNCTION__);
		return TRUE;
	}
	osso_log(LOG_DEBUG, "[%s]: daemon is not running", __FUNCTION__);
	return FALSE;
}


/**
  This function will check the availability of external mmc

  @param void
  @return gboolean TRUE if present else FALSE
*/
static gboolean nitro_check_for_mmc_availability()
{
	GConfClient *gconf_client = NULL;
	GConfValue *gconf_mmc_value = NULL;
	GConfValue *gconf_internal_mmc_value = NULL;
	int mmc_present, internal_mmc_present;

	gconf_client = gconf_client_get_default();
	gconf_mmc_value =
		gconf_client_get(gconf_client, MMC_PRESENT, NULL);
	gconf_internal_mmc_value =
		gconf_client_get(gconf_client, INTERNAL_MMC_PRESENT, NULL);

	if(gconf_mmc_value == NULL) {
		osso_log(LOG_ERR, "[%s]: Unable to read %s",__FUNCTION__, MMC_PRESENT);
		return FALSE;
	}
	if(gconf_internal_mmc_value == NULL) {
		osso_log(LOG_ERR, "[%s]: Unable to read %s",__FUNCTION__, INTERNAL_MMC_PRESENT);
		return FALSE;
	}

	mmc_present = gconf_value_get_bool(gconf_mmc_value);
	internal_mmc_present = gconf_value_get_bool(gconf_internal_mmc_value);

	if(gconf_mmc_value) {
		gconf_value_free(gconf_mmc_value);
	}
	if(gconf_internal_mmc_value) {
		gconf_value_free(gconf_internal_mmc_value);
	}

	osso_log(LOG_DEBUG, "[%s]: mmc_present=%d internal_mmc_present=%d", 
		 __FUNCTION__, mmc_present, internal_mmc_present);

	if(mmc_present || internal_mmc_present) {
		return TRUE;
	} else {
		return FALSE;
	}
}

/**
  this is a callback for gnome_vfs_xfer_delete_list()

  @param GnomeVFSXferProgressInfo structure
  @param callback data

  @return gint
*/
static gint progress_info(GnomeVFSXferProgressInfo * info, gint * data)
{
	if (info->status == GNOME_VFS_XFER_PROGRESS_STATUS_VFSERROR) {
		 osso_log(LOG_ERR, "[%s]: info->status is GNOME_VFS_XFER_PROGRESS_STATUS_VFSERROR", __FUNCTION__);
		*data = info->vfs_status;
		return 0;
	}
	osso_log(LOG_DEBUG, "(%s)info->status=%d",__FUNCTION__,info->status);
	return 1;
}

/** 
  This function will remove all core files and optionally also the 'core-dumps'
  directories from the memory cards.

  @param gboolean TRUE if 'core-dumps' dir to be removed
  		  FALSE otherwise
  @param gchar * path of the directory
  @return void
*/ 
static void nitro_cp_remove_all_core_files(gboolean remove_coredumping, gchar *location)
{
	GnomeVFSURI *source_uri=NULL;
	GList *source_list=NULL;
	GList *file_info_list=NULL;
	GList *temp=NULL;
	gchar *src_text_uri=NULL;
	gchar *file_text_uri=NULL;
	gint error=-1;
	gchar *filename = NULL;
	
	src_text_uri=gnome_vfs_make_uri_from_shell_arg(location);
	
	if(remove_coredumping){
		source_uri = gnome_vfs_uri_new(src_text_uri);
		source_list = g_list_append(source_list, source_uri);
	} else {
		gnome_vfs_directory_list_load(&file_info_list,src_text_uri, GNOME_VFS_FILE_INFO_DEFAULT);
		for(temp=file_info_list; temp; temp=temp->next) {
			filename = (gchar*) ((GnomeVFSFileInfo *)(temp->data))->name;
			if(g_str_has_suffix(filename,".")){
				continue;
			}
			file_text_uri=g_strdup_printf("file://%s/%s",location, (gchar*) ((GnomeVFSFileInfo *)(temp->data))->name );
			
			source_uri = gnome_vfs_uri_new(file_text_uri);
			g_free(file_text_uri);
			file_text_uri=NULL;
			source_list = g_list_append(source_list,source_uri);
		}
	}
	if(source_list != NULL) {
		gnome_vfs_xfer_delete_list(source_list, 
				GNOME_VFS_XFER_ERROR_MODE_QUERY,
				GNOME_VFS_XFER_EMPTY_DIRECTORIES,
				(GnomeVFSXferProgressCallback) progress_info,
				&error);
		osso_log(LOG_DEBUG, "[%s]: Done with deleting", __FUNCTION__);
	}
	
	source_list = gnome_vfs_uri_list_unref(source_list);
	g_list_free(source_list);
}

/** 
  this function will launch the cnfirmation dialog to
  delete the rich cores/core-dumps directories
  
  @param parent widget 
  @return void
*/
static void nitro_cp_launch_conf_dialog(GtkWidget *settings_dialog)
{
	GtkWidget *conf_dialog = NULL;
	GtkWidget *vbox = NULL;
	GtkWidget *hbox = NULL;
	GtkWidget *label_note1 = NULL;
	GtkWidget *label_note2 = NULL;
	GtkWidget *checkbox_coredumping = NULL;
	gint response = -1;
	gboolean remove_dumping = FALSE;

	conf_dialog = gtk_dialog_new_with_buttons(
				_("Remove core files?"), 
				GTK_WINDOW(settings_dialog),
		       		GTK_DIALOG_MODAL,
				_("Remove"), CP_CONF_REMOVE,
				_("Ignore"), CP_CONF_IGNORE,
				NULL);
	gtk_dialog_set_has_separator(GTK_DIALOG(conf_dialog), FALSE);
	gtk_widget_set_size_request(GTK_WIDGET(conf_dialog), 500, 200);
	gtk_window_set_transient_for(GTK_WINDOW(conf_dialog), GTK_WINDOW(settings_dialog));
	vbox = GTK_DIALOG(conf_dialog)->vbox;
	
	label_note1 = gtk_label_new(_("Do you want to remove the existing core files?"));
	gtk_label_set_line_wrap(GTK_LABEL(label_note1), TRUE);
	gtk_box_pack_start(GTK_BOX(vbox), label_note1, FALSE, FALSE,0);
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(vbox)), hbox, FALSE, FALSE, 10);
	
	checkbox_coredumping = gtk_check_button_new();
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_coredumping), FALSE);
	gtk_box_pack_end(GTK_BOX(hbox), checkbox_coredumping, FALSE, FALSE, 35);
	
	label_note2 = gtk_label_new(_("Disable core-dumping \n(remove the core-dumps directories)"));
	gtk_misc_set_alignment(GTK_MISC(label_note2), 0, 1);
	gtk_label_set_line_wrap((GtkLabel *) label_note2, TRUE);
	gtk_box_pack_end(GTK_BOX(hbox), label_note2, FALSE, FALSE, 0);

	gtk_widget_grab_focus(GTK_DIALOG(conf_dialog)->action_area); 
	gtk_widget_show_all(conf_dialog);
	response = gtk_dialog_run(GTK_DIALOG(conf_dialog));

	if(response == CP_CONF_REMOVE) {
		remove_dumping =
			     gtk_toggle_button_get_active(
					   	GTK_TOGGLE_BUTTON(
						   checkbox_coredumping));
			
		nitro_cp_remove_all_core_files(remove_dumping, EXT_MMC);
		nitro_cp_remove_all_core_files(remove_dumping, INT_MMC);
	}
	
	gtk_widget_destroy(conf_dialog);
}

/**
  This function is called as callback when "nitro enabled" button
  gets toggled
  
  @param w is pointer to button widget
  @param data is pointer to _nitroData structure

  */
static gboolean nitro_enabled_toggle_callback(GtkWidget *w, gpointer data) {
    nitroData *ndata = (nitroData*)data;
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))) {
	gtk_widget_set_sensitive(ndata->caption_privacy, TRUE);
	gtk_widget_set_sensitive(ndata->caption_include_core, TRUE);
	gtk_widget_set_sensitive(ndata->caption_include_syslog, TRUE);
	gtk_widget_set_sensitive(ndata->caption_include_pkglist, TRUE);
    } else {
	gtk_widget_set_sensitive(ndata->caption_privacy, FALSE);
	gtk_widget_set_sensitive(ndata->caption_include_core, FALSE);
	gtk_widget_set_sensitive(ndata->caption_include_syslog, FALSE);
	gtk_widget_set_sensitive(ndata->caption_include_pkglist, FALSE);
    }
    return TRUE;
}

/**
  This function is called for creation of the dialog widgets.
  
  @param data is the pointer to the nitroData structure
  @param settings_dialog widget pointer
  @param enable_nitro flag to indicate the nitro service status
  @param context is the osso_context_t pointer for the cp application
  
  @return TRUE on success, else FALSE
  */
static gboolean nitro_create_dialog_widgets(nitroData *data,
					GtkWidget *settings_dialog,
					privacySettings *privsett,
					osso_context_t *context)
{
	GtkWidget *vbox = NULL;
	GtkWidget *caption_control = NULL;
	GtkSizeGroup *size_group = NULL;

	g_return_val_if_fail(data != NULL, FALSE);
	g_return_val_if_fail(settings_dialog != NULL, FALSE);
	
	vbox = (GTK_DIALOG(settings_dialog))->vbox;
	size_group = 
		gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
	
	/*-----------*/
	data->checkbox_nitro = gtk_check_button_new();
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_nitro), 
				     privsett->enabled);

	caption_control =
	    hildon_caption_new(NULL, _("Enable Crash Reporter"),
		       data->checkbox_nitro, NULL, 
		       HILDON_CAPTION_OPTIONAL);
	gtk_box_pack_start(GTK_BOX(vbox), caption_control, 
			   FALSE,FALSE, 10);

	/*---*/
	g_signal_connect (data->checkbox_nitro, "toggled",
			  G_CALLBACK (nitro_enabled_toggle_callback), data);

	/*-----------*/
	data->caption_privacy = gtk_label_new(_("\
NOTE: The following selection does not affect any existing\n\
crash reports, including the current one if you are in the\n\
middle of sending one.\n\
To be included in future crash reports:"));
	gtk_box_pack_start(GTK_BOX(vbox), data->caption_privacy, 
			   FALSE,FALSE, 4);

	/*-----------*/
	data->checkbox_include_core = gtk_check_button_new();
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_include_core), privsett->include_core);

	data->caption_include_core =
	    hildon_caption_new(size_group, _("Core dump"),
		       data->checkbox_include_core, NULL, 
		       HILDON_CAPTION_OPTIONAL);
	gtk_box_pack_start(GTK_BOX(vbox), data->caption_include_core, 
			   FALSE,FALSE, 4);

	/*-----------*/
	data->checkbox_include_syslog = gtk_check_button_new();
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_include_syslog), privsett->include_syslog);

	data->caption_include_syslog =
	    hildon_caption_new(size_group, _("System log (if exists)"),
		       data->checkbox_include_syslog, NULL, 
		       HILDON_CAPTION_OPTIONAL);
	gtk_box_pack_start(GTK_BOX(vbox), data->caption_include_syslog, 
			   FALSE,FALSE, 4);

	/*-----------*/
	data->checkbox_include_pkglist = gtk_check_button_new();
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_include_pkglist), privsett->include_pkglist);

	data->caption_include_pkglist =
	    hildon_caption_new(size_group, _("List of installed packages"),
		       data->checkbox_include_pkglist, NULL, 
		       HILDON_CAPTION_OPTIONAL);
	gtk_box_pack_start(GTK_BOX(vbox), data->caption_include_pkglist, 
			   FALSE,FALSE, 4);

	nitro_enabled_toggle_callback(data->checkbox_nitro, data);
	return TRUE;
}


static void nitro_cp_show_privacy_disclaimer(gpointer parent)
{
    static GtkWidget *window;
    GtkWidget *scrolled_window;
    GtkWidget *label;
    GtkWidget *frame;
    gint response;
    gchar *stmt = NULL;
    gchar *p;
    struct stat st;
    FILE *fp = NULL;
    
    /* Create a new dialog window for the scrolled window to be
     * packed into.  */
    window = gtk_dialog_new_with_buttons(
					 _("Crash Reporter Privacy Note"), 
					 GTK_WINDOW(parent),
					 GTK_DIALOG_MODAL 
					 | GTK_DIALOG_DESTROY_WITH_PARENT
					 | GTK_DIALOG_NO_SEPARATOR,
					 GTK_STOCK_OK, CP_RESPONSE_OK,
					 NULL);
    gtk_widget_set_size_request(window, -1, 300);
    /* create a new scrolled window. */
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);

    /* the policy is one of GTK_POLICY AUTOMATIC, or GTK_POLICY_ALWAYS.
     * GTK_POLICY_AUTOMATIC will automatically decide whether you need
     * scrollbars, whereas GTK_POLICY_ALWAYS will always leave the scrollbars
     * there.  The first one is the horizontal scrollbar, the second, 
     * the vertical. */
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    frame = gtk_frame_new(NULL);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), frame, 
                        TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (frame), scrolled_window);

    if( !stat(PRIVACY_STATEMENT_FILE, &st) 
	&& (stmt = g_malloc(st.st_size + 1))
	&& (fp = fopen(PRIVACY_STATEMENT_FILE, "r"))
	&& (fread(stmt, sizeof(char), st.st_size, fp) == st.st_size)) {
	stmt[st.st_size] = 0;
	/* replace newlines with ' ' because we use wrap option */
        for (p = stmt; *p ; p++) {
	    if (*p == '\n') {
		*p = ' ';
	    }
	}
    } else {
	osso_log(LOG_ERR,"[%s]: error(s) reading file [%s]\n", __FUNCTION__, PRIVACY_STATEMENT_FILE);
	stmt = g_malloc(80);
	if (stmt) {
	    strcpy(stmt, "[can not open privacy statement file!]");
	}
    }
    if (fp) {
	fclose(fp);
    }
    /* Create a text label widget */
    label = gtk_label_new (stmt);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    /* pack the text widget into the scrolled window */
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrolled_window), label);
    gtk_widget_show_all (window);
    response = gtk_dialog_run(GTK_DIALOG(window));
    gtk_widget_destroy(GTK_WIDGET(window));
    g_free(stmt);
}

/**
  The function creates the nitro cp applet dialog

  @param window pointer
  @param osso_context_t pointer 

  @return void
*/
void 
nitro_cp_dialog_create(gpointer window, osso_context_t *context)
{
	GtkWidget *settings_dialog = NULL;
	nitroData *dialog_data = NULL;
	gboolean validate = FALSE;
	gboolean ret = FALSE;
	gboolean enable_nitro = FALSE;
	gboolean nitro_prev_enabled = FALSE;
	CpDialogResponse resp = -1;
	privacySettings* privsettings = NULL;

	g_return_if_fail(context != NULL);
	g_return_if_fail(window != NULL);

	privsettings = nitro_read_privacy_settings();
	if (privsettings) {
	    osso_log(LOG_DEBUG, "[%s]: All settings loaded: enabled=%d inc_core=%d inc_syslog=%d inc_plist=%d", __FUNCTION__, 
		     privsettings->enabled,
		     privsettings->include_core,
		     privsettings->include_syslog,
		     privsettings->include_pkglist);
	}
	else {
	    gchar *text = NULL;
	    GtkWidget *note = NULL;
	    text = g_strdup(_("Error loading configuration. Is NITRO configuration package installed?"));
	    note = hildon_note_new_information(NULL, text);
	    hildon_note_set_button_text(HILDON_NOTE(note), _("Ok"));
	    gtk_dialog_run(GTK_DIALOG(note));
	    gtk_widget_destroy(GTK_WIDGET(note));	    
	    g_free(text);
	    return;
	}

	dialog_data = (nitroData *) g_new0(nitroData, 1);
	
	nitro_prev_enabled = privsettings->enabled;
	settings_dialog = gtk_dialog_new_with_buttons(
			_("Crash Reporter Settings"),
			GTK_WINDOW(window),
			GTK_DIALOG_MODAL,
			_("Ok"),
			CP_RESPONSE_OK,
			_("Privacy statement"),
			CP_RESPONSE_DISCLAIMER, 
			_("Cancel"),
			CP_RESPONSE_CANCEL, 
			NULL);
	
	gtk_widget_set_size_request(GTK_WIDGET(settings_dialog),
			  	    	  	       -1, -1);
	gtk_dialog_set_has_separator(GTK_DIALOG(settings_dialog), 
							TRUE);
	gtk_window_set_transient_for(GTK_WINDOW(settings_dialog), 
			GTK_WINDOW(window));
	ret = nitro_create_dialog_widgets(dialog_data,
					settings_dialog,
					privsettings,
					context);
	if(!ret) {
		osso_log(LOG_DEBUG,"Failure creating dialog widgets\n");
		gtk_widget_destroy(settings_dialog);
		return;
	}
	
#ifdef MAEMO_CHANGES
	if( !hildon_help_dialog_help_enable(GTK_DIALOG(settings_dialog),
		"applications_nitro_settings",
		context) )
		osso_log(LOG_DEBUG, "Help is not enabled");		
#else
	if( !ossohelp_dialog_help_enable(GTK_DIALOG(settings_dialog),
		"applications_nitro_settings",
		context) )
		osso_log(LOG_DEBUG, "Help is not enabled");		
#endif	
	gtk_widget_show_all(settings_dialog);
	while(validate == FALSE) {
	
		resp = 
			gtk_dialog_run(GTK_DIALOG(settings_dialog));

		switch (resp) {
		case CP_RESPONSE_OK: 
		    {
		    gboolean i_core,i_syslog,i_pkglist;
			enable_nitro = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_nitro));

			i_core = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_include_core));
			i_syslog = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_include_syslog));
			i_pkglist = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_include_pkglist));
			if (i_core != privsettings->include_core || 
			    i_syslog != privsettings->include_syslog || 
			    i_pkglist != privsettings->include_pkglist ||
			    nitro_prev_enabled != enable_nitro) {
			    privsettings->enabled         = enable_nitro;
			    privsettings->include_core    = i_core;
			    privsettings->include_syslog  = i_syslog;
			    privsettings->include_pkglist = i_pkglist;
			    nitro_write_privacy_settings(privsettings);
			    nitro_free_privacy_settings(privsettings);
			    privsettings = NULL;
			    }
			if(!nitro_prev_enabled && enable_nitro) {
				osso_log(LOG_DEBUG, "NITRO service is enabled");
				/* NITRO service is enabled*/
				if(!nitro_check_for_mmc_availability()) {
#ifdef MAEMO_CHANGES
					hildon_banner_show_information(GTK_WIDGET(window), NULL, _("Please insert Memory card"));
#else
					
					gtk_infoprint(GTK_WINDOW(window),
							_("Please insert Memory card"));
#endif
					gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_nitro), FALSE);
					osso_log(LOG_DEBUG, "[%s]MMC ABSENT", __FUNCTION__);					
					validate = FALSE;
				} else {
					osso_log(LOG_DEBUG, "[%s]MMC PRESENT", __FUNCTION__);					
					validate = TRUE;
				}
			} else if(nitro_prev_enabled && !enable_nitro) {
				/* NITRO service is disabled */
				nitro_cp_launch_conf_dialog(settings_dialog);
				validate = TRUE;
			} else {
				validate = TRUE;
			}
			

			if(validate == TRUE){

					/* Start nitro_daemon when enable_nitro 
					 * is TRUE AND the daemon is not 
					 * already running Stop it only when 
					 * enable_nitro is FALSE and daemon 
					 * is already running*/
					if(!(enable_nitro && 
					     nitro_cp_if_nitro_daemon_running()) )
						nitro_cp_execute_invoker_script(enable_nitro);
					gtk_widget_destroy(settings_dialog);
			}
			break;
		    }
		case CP_RESPONSE_CANCEL:
			gtk_widget_destroy(settings_dialog);
			validate = TRUE;
			break;
		case CP_RESPONSE_DISCLAIMER:
		    nitro_cp_show_privacy_disclaimer(window);
			break;
		}
	}

	return;
}
