/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is The Browser Profile Migrator.
 *
 * The Initial Developer of the Original Code is Ben Goodger.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *  Ben Goodger <ben@bengoodger.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#ifndef netscapeprofilemigratorbase___h___
#define netscapeprofilemigratorbase___h___

#include "nsILocalFile.h"
#include "nsISupportsArray.h"
#include "nsStringAPI.h"
#include "nsTArray.h"
#include "nsITimer.h"
#include "nsIObserverService.h"
#include "nsISuiteProfileMigrator.h"

class nsIPrefBranch;
class nsIPrefService;

struct fileTransactionEntry {
  nsCOMPtr<nsIFile> srcFile;  // the src path including leaf name
  nsCOMPtr<nsIFile> destFile; // the destination path
  nsString newName; // only valid if the file should be renamed after
                    // getting copied
};

#define FILE_NAME_BOOKMARKS       NS_LITERAL_STRING("bookmarks.html")
#define FILE_NAME_COOKIES         NS_LITERAL_STRING("cookies.txt")
#define FILE_NAME_PREFS           NS_LITERAL_STRING("prefs.js")
#define FILE_NAME_JUNKTRAINING    NS_LITERAL_STRING("training.dat")
#define FILE_NAME_VIRTUALFOLDERS  NS_LITERAL_STRING("virtualFolders.dat")
#define FILE_NAME_USERCONTENT     NS_LITERAL_STRING("userContent.css")
#define FILE_NAME_SEARCH          NS_LITERAL_STRING("search.rdf")
#define FILE_NAME_DOWNLOADS       NS_LITERAL_STRING("downloads.rdf")

#define F(a) nsNetscapeProfileMigratorBase::a

#define MAKEPREFTRANSFORM(pref, newpref, getmethod, setmethod) \
  { pref, newpref, F(Get##getmethod), F(Set##setmethod), PR_FALSE, -1 }

#define MAKESAMETYPEPREFTRANSFORM(pref, method) \
  { pref, 0, F(Get##method), F(Set##method), PR_FALSE, -1 }

class nsNetscapeProfileMigratorBase : public nsISuiteProfileMigrator,
                                      public nsITimerCallback
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSITIMERCALLBACK

  nsNetscapeProfileMigratorBase();
  virtual ~nsNetscapeProfileMigratorBase() { }

  struct PrefTransform;
  typedef nsresult(*prefConverter)(PrefTransform*, nsIPrefBranch*);

  struct PrefTransform {
    char*         sourcePrefName;
    char*         targetPrefName;
    prefConverter prefGetterFunc;
    prefConverter prefSetterFunc;
    PRBool        prefHasValue;
    union {
      PRInt32     intValue;
      PRBool      boolValue;
      char*       stringValue;
    };
  };

  struct PrefBranchStruct {
    char*         prefName;
    PRInt32       type;
    union {
      char*       stringValue;
      PRInt32     intValue;
      PRBool      boolValue;
    };
  };

  typedef nsTArray<PrefBranchStruct*> PBStructArray;

  // nsISuiteProfileMigrator methods
  NS_IMETHOD GetSourceExists(PRBool *aSourceExists);
  NS_IMETHOD GetSourceHasMultipleProfiles(PRBool *aSourceHasMultipleProfiles);
  NS_IMETHOD GetSourceProfiles(nsISupportsArray * *aSourceProfiles);
  NS_IMETHOD GetSourceHomePageURL(nsACString & aSourceHomePageURL);

  // Pref Transform Methods - these seem only to be used by the dogbert migrator
  static nsresult GetString(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  static nsresult SetString(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  static nsresult GetBool(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  static nsresult SetBool(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  static nsresult GetInt(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  static nsresult SetInt(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  static nsresult SetImage(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  static nsresult SetCookie(PrefTransform* aTransform, nsIPrefBranch* aBranch);
  // XXX Bug 381159 When suite uses Toolkit's DM backend, we need to
  // activate this code.
#ifdef SUITE_USING_TOOLKIT_DM
  static nsresult SetDownloadManager(PrefTransform* aTransform,
                                     nsIPrefBranch* aBranch);
#endif

protected:
  // This function is designed to be overriden by derived classes so that
  // the required profile data for the specific application can be obtained.
  virtual nsresult FillProfileDataFromRegistry() = 0;

  // General Utility Methods
  nsresult GetSourceProfile(const PRUnichar* aProfile);
  nsresult GetProfileDataFromProfilesIni(nsILocalFile* aDataDir,
                                         nsISupportsArray* aProfileNames,
                                         nsISupportsArray* aProfileLocations);
  nsresult GetProfileDataFromRegistry(nsILocalFile* aRegistryFile,
                                      nsISupportsArray* aProfileNames,
                                      nsISupportsArray* aProfileLocations);
  nsresult CopyFile(const nsAString& aSourceFileName,
                    const nsAString& aTargetFileName);
  nsresult RecursiveCopy(nsIFile* srcDir, nsIFile* destDir);
  void ReadBranch(const char * branchName, nsIPrefService* aPrefService,
                  PBStructArray &aPrefs);
  void WriteBranch(const char * branchName, nsIPrefService* aPrefService,
                   PBStructArray &aPrefs);

  // Generic Import Functions
  nsresult CopyCookies(PRBool aReplace);
  nsresult CopyFormData(PRBool aReplace);
  nsresult CopyPasswords(PRBool aReplace);
  nsresult CopyUserContentSheet();
  nsresult GetSchemaValueFileName(PRBool aReplace, char** aFileName);
  nsresult GetSignonFileName(PRBool aReplace, char** aFileName);
  nsresult ImportNetscapeCookies(nsIFile* aCookiesFile);
  nsresult LocateWalletFile(const char* aExtension, char** aResult);

  // Browser Import Functions
  nsresult CopyBookmarks(PRBool aReplace);
  nsresult CopyOtherData(PRBool aReplace);
  nsresult ImportNetscapeBookmarks(const nsAString& aBookmarksFileName,
                                   const PRUnichar* aImportSourceNameKey);

  // Mail Import Functions
  nsresult CopyAddressBookDirectories(PBStructArray &aLdapServers,
                                      nsIPrefService* aPrefService);
  nsresult CopySignatureFiles(PBStructArray &aIdentities,
                              nsIPrefService* aPrefService);
  nsresult CopyJunkTraining(PRBool aReplace);
  nsresult CopyMailFolderPrefs(PBStructArray &aMailServers,
                               nsIPrefService* aPrefService);
  void CopyMailFolders();
  void CopyNextFolder();
  void EndCopyFolders();

  // Source & Target profiles
  nsCOMPtr<nsILocalFile> mSourceProfile;
  nsCOMPtr<nsIFile> mTargetProfile;

  // list of src/destination files we still have to copy into the new profile
  // directory
  nsTArray<fileTransactionEntry> mFileCopyTransactions;
  PRUint32 mFileCopyTransactionIndex;

  nsCOMPtr<nsIObserverService> mObserverService;
  PRInt64 mMaxProgress;
  PRInt64 mCurrentProgress;

  nsCOMPtr<nsISupportsArray> mProfileNames;
  nsCOMPtr<nsISupportsArray> mProfileLocations;

  nsCOMPtr<nsITimer> mFileIOTimer;
};
 
#endif
