#include <linux/kernel.h>
#include <linux/mtd/mtd.h>
#include <linux/string.h>
#include <linux/parser.h>
#include <linux/seq_file.h>
#include <linux/jffs2.h>
#include <linux/mount.h>
#include <linux/fs.h>
#include "nodelist.h"

/* Define RP_DEBUG macro to enable RP dbg messages */
#undef RP_DEBUG

#ifdef RP_DEBUG
#define RP_MSG(fmt, ...) printk(KERN_DEBUG "[JFFS2 RP] %s:\t" fmt, __FUNCTION__, ## __VA_ARGS__);
#else
#define RP_MSG(fmt, ...)
#endif

/*
 * The maximal number of arguments which the parser may find in one mount
 * option pattern.
 */
#define JFFS2_MAX_ARGS	1

/*
 * The type of the entry in the list of UIDs/GIDs whic hare allowed to use the
 * reserved pool.
 */
enum jffs2_rp_id_type {
	JFFS2_IDTYPE_UID,
	JFFS2_IDTYPE_GID
};

/*
 * The entry of the list of the UIDs/GIDs which are allowed to use the reserved
 * pool.
 */
struct jffs2_rp_id {
	enum jffs2_rp_id_type type;
	unsigned int id;
	struct jffs2_rp_id *next;
};

/*
 * The mount option identifiers.
 */
enum {
	JFFS2_OPT_RP_SIZE,
	JFFS2_OPT_RP_UID,
	JFFS2_OPT_RP_GID,
	JFFS2_OPT_LAST_FAKE
};

/*
 * The supported mount options to feed to the mount options parser.
 */
static match_table_t tokens = {
	{JFFS2_OPT_RP_SIZE, "rpsize=%u"}, /* The size of the reserved pool */
	{JFFS2_OPT_RP_UID, "rpuid=%u"},   /* UID of a user of the reserved pool */
	{JFFS2_OPT_RP_GID, "rpgid=%u"},   /* GID of a user of the reserved pool */
	{JFFS2_OPT_LAST_FAKE, NULL}       /* End of list marker */
};

/*
 * This is probably a generic function and should be kept somwhere in super.c,
 * but sence our reserved pool hack is not in mainline, we prevere to keep as
 * much as possible in one file in order to fasicilate easier patch
 * management.
 */
int jffs2_rp_parse_options(struct super_block *sb, char *opts)
{
	unsigned int opt;
	int ret = -EINVAL;
	char *p;
	substring_t args[JFFS2_MAX_ARGS];
	struct jffs2_sb_info *c = JFFS2_SB_INFO(sb);
	struct jffs2_rp_id *rp_list_tail = NULL;

	if (!opts)
		return 0;

	while ((p = strsep(&opts, ",")) != NULL) {
		int token;

		if (!*p)
			continue;

		token = match_token(p, tokens, args);

		switch (token) {
			case JFFS2_OPT_RP_SIZE:
				if (match_int(&args[0], &opt))
					goto error;
				c->rp_size = opt * 1024;
				if (c->rp_size > c->mtd->size) {
					printk(KERN_WARNING "Warning: too large reserve pool specified, max is %u KB\n",
							c->mtd->size / 1024);
					goto error;
				}
				RP_MSG("RP size is %u KB\n", opt);
				break;

			case JFFS2_OPT_RP_UID:
			case JFFS2_OPT_RP_GID:
			{
				struct jffs2_rp_id *id;
				if (match_int(&args[0], &opt))
					goto error;

				id = kmalloc(sizeof(*id), GFP_KERNEL);
				if (!id) {
					ret = -ENOMEM;
					goto error;
				}

				id->id = opt;
				id->next = NULL;
				if (token == JFFS2_OPT_RP_UID) {
					id->type = JFFS2_IDTYPE_UID;
					RP_MSG("UID %u may use RP\n", opt);
				} else {
					id->type = JFFS2_IDTYPE_GID;
					RP_MSG("GID %u may use RP\n", opt);
				}

				if (rp_list_tail == NULL)
					c->rp_ids = id;
				else
					rp_list_tail->next = id;

				rp_list_tail = id;
				break;
			}
		}
	}

	return 0;

error:
	jffs2_rp_free(c);

	return ret;
}

void jffs2_rp_free(struct jffs2_sb_info *c)
{
	/* Free the reserved pool UIDs/GIDs */
	while (c->rp_ids) {
		struct jffs2_rp_id *id = c->rp_ids;
		c->rp_ids = c->rp_ids->next;
		kfree(id);
	}
	c->rp_size = 0;
}

int jffs2_rp_show_options(struct seq_file *m, struct vfsmount *mnt)
{
	struct jffs2_sb_info *c = JFFS2_SB_INFO(mnt->mnt_sb);
	struct jffs2_rp_id *id;

	if (c->rp_size)
		seq_printf(m, ",rpsize=%d", c->rp_size/1024);

	id = c->rp_ids;
	while(id) {
		if (id->type == JFFS2_IDTYPE_UID)
			seq_printf(m, ",rpuid=%d", id->id);
		else
			seq_printf(m, ",rpgid=%d", id->id);
		id = id->next;
	}
	return 0;
}

/*
 * Check whether the user is allowed to write.
 */
int jffs2_rp_can_write(struct jffs2_sb_info *c)
{
	uint32_t avail;
	struct jffs2_rp_id *id = c->rp_ids;

	avail = c->dirty_size + c->free_size + c->unchecked_size +
		c->erasing_size - c->resv_blocks_write * c->sector_size
		- c->nospc_dirty_size;

	if (avail < 2*c->rp_size)
		RP_MSG("rpsize %u, dirty_size %u, free_size %u, erasing_size %u, "
			"unchecked_size %u, nr_erasing_blocks %u, avail %u, resrv %u\n",
			c->rp_size, c->dirty_size, c->free_size, c->erasing_size,
			c->unchecked_size, c->nr_erasing_blocks, avail,
			c->nospc_dirty_size);

	if(avail > c->rp_size)
		return 1;

	while (id) {

		if (id->type == JFFS2_IDTYPE_UID) {
			if ((unsigned int)current->user->uid == id->id)
				return 1;
		} else {
			int i;

			/* Walk the groups user belongs to */
			for (i = 0; i< current->group_info->ngroups; i++) {
				int blk = i / NGROUPS_PER_BLOCK;
				int grp = i - blk;

				if ((current->group_info->blocks[blk])[grp] == (gid_t)id->id)
					return 1;
			}
		}

		id = id->next;
	}

	RP_MSG("forbid writing\n");
	return 0;
}
