#ifndef PC2400M_OSAL_H
#define PC2400M_OSAL_H

/*
 * pc2400m_osal.h
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */



#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include "pc2400m_if.h"

/* OS Abstraction layer */
typedef struct net_device wimax_osal_context;
typedef struct sk_buff wimax_osal_packet;

static inline void wimax_osal_ctx_priv_set(wimax_osal_context *ctx, 
					   void *priv) 
{
        struct net_local *nl = netdev_priv(ctx);
	nl->driver_priv = priv;
}

static inline void *wimax_osal_ctx_priv_get(wimax_osal_context *ctx) 
{
	struct net_local *nl = netdev_priv(ctx);
	return nl->driver_priv;
}

#define DRIVER_SUFFIX "PC2400Mdrv: "
#define wimax_osal_mem_cpy(dst, src, len)       memcpy(dst,src,len)
#define wimax_osal_mem_move(dst, src, len)      memmove(dst,src,len)
#define wimax_osal_mem_set(dst, val, len)       memset(dst,val,len)
#define wimax_osal_str_len(str)                 strlen(str)
#define wimax_osal_str_ncpy(tgt, src, len)      strncpy(tgt,src,len)
#define wimax_osal_assert(cond)                 BUG_ON(!(cond))

#define WIMAX_OSAL_MEM_TYPE_NONE                0
#define WIMAX_OSAL_MEM_TYPE_DMA                 1
#define wimax_osal_mem_alloc_ex(size, mask)     wimax_osal_mem_alloc(size)
static inline void *wimax_osal_mem_alloc(u32 size)
{

        void *ptr = NULL;
	if (size == 0) goto out;
	while (!(ptr = kmalloc(size, GFP_KERNEL))) {
	        msleep(1);
	}
 out:
	return ptr;
}
#define wimax_osal_mem_free(ptr) { kfree(*ptr); *ptr = NULL; }


#define wimax_osal_sleep(msec)                  msleep(msec)

/* OSal data types, named after kernel datatypes:
   typedef char s8;
   typedef short s16;
   typedef int s32;
   typedef unsiged s8 u8;
   typedef unsigned s16 u16;
   typedef unsigned s32 u32;
*/
typedef int boolean;

/* boolean definitons */
#ifndef FALSE
#define FALSE 0
#endif /* !FALSE */

#ifndef TRUE
#define TRUE  1
#endif /* !TRUE */

/* variable based conversion */
#define WIMAX_OSAL_U16_TO_LE(x) cpu_to_le16(x)
#define WIMAX_OSAL_LE_TO_U16(x) le16_to_cpu(x)
#define WIMAX_OSAL_U32_TO_LE(x) cpu_to_le32(x)
#define WIMAX_OSAL_LE_TO_U32(x) le32_to_cpu(x)

#define WIMAX_OSAL_U32_TO_BE(x) cpu_to_be32(x)
#define WIMAX_OSAL_BE_TO_U32(x) be32_to_cpu(x)

/* stream extraction */
#define WIMAX_OSAL_GET_LE16(p) ((((u16)((u8*)p)[1])<<8) | (((u16)((u8*)p)[0])))
#define WIMAX_OSAL_GET_LE32(p) ((((u32)((u8*)p)[3])<<24) | \
                                (((u32)((u8*)p)[2])<<16) | \
		                (((u32)((u8*)p)[1])<<8) | \
                                (((u32)((u8*)p)[0])))
#define WIMAX_OSAL_PUT_LE16(p,v) { ((u8*)(p))[0] = (u8)(((u16)v)); \
                                   ((u8*)(p))[1] = (u8)(((u16)v)>>8); }
#define WIMAX_OSAL_PUT_LE32(p,v) { ((u8*)(p))[0] = (u8)(((u32)v)); \
                                   ((u8*)(p))[1] = (u8)(((u32)v)>>8); \
                                   ((u8*)(p))[2] = (u8)(((u32)v)>>16); \
                                   ((u8*)(p))[3] = (u8)(((u32)v)>>24); }
#define WIMAX_OSAL_GET_BE32(p)  ((((u32)((u8*)p)[0])<<24) | \
                                 (((u32)((u8*)p)[1])<<16) | \
		                 (((u32)((u8*)p)[2])<<8) | \
                                 (((u32)((u8*)p)[3])))

static inline wimax_osal_packet *wimax_osal_packet_alloc(u32 size)
{
	struct sk_buff *skb;
#define PACKET_SPARE     (22 + 14) 
	while (!(skb = alloc_skb(size + PACKET_SPARE, GFP_KERNEL))) {
	        msleep(1);
	}
	skb_reserve(skb, PACKET_SPARE);
	return skb;
}

#define wimax_osal_packet_free(pkt)            {kfree_skb(*pkt); *pkt = NULL;}
#define wimax_osal_packet_size(p)              (((wimax_osal_packet*)p)->len)
#define wimax_osal_packet_ptr(p)               (((wimax_osal_packet*)p)->data)
#define wimax_osal_packet_put(p,size)          skb_put(p, size)
#define wimax_osal_packet_trim(p,size)         skb_trim(p, size)
#define wimax_osal_packet_head_cut(p,size)     skb_pull(p, size)
#define wimax_osal_packet_head_put(p,size)     skb_push(p, size)

typedef struct {
	wimax_osal_packet *first;
	wimax_osal_packet *last;
	s32 len;
} wimax_osal_packet_list;

wimax_osal_packet_list *wimax_osal_packet_list_alloc(void);
void wimax_osal_packet_list_free(wimax_osal_packet_list **, boolean);
void wimax_osal_packet_list_initialize(wimax_osal_packet_list*);
void wimax_osal_packet_list_release(wimax_osal_packet_list *, boolean);
void wimax_osal_packet_list_append(wimax_osal_packet_list*, 
				   wimax_osal_packet*);
void wimax_osal_packet_list_push(wimax_osal_packet_list*, wimax_osal_packet*);
void wimax_osal_packet_list_insert(wimax_osal_packet_list*, 
				   wimax_osal_packet*, 
				   void*);
void wimax_osal_packet_list_cmp_fnc_set(wimax_osal_packet_list*, 
					void (*)(wimax_osal_packet*,
						 wimax_osal_packet*,
						 void*));
wimax_osal_packet *wimax_osal_packet_list_pop(wimax_osal_packet_list *list);
#define wimax_osal_packet_list_len(list)  ((list)->len)
#define wimax_osal_packet_list_first(packet_list) ((packet_list)->first)
#define wimax_osal_packet_list_last(packet_list) ((packet_list)->last)
#define wimax_osal_packet_list_next(packet) ((packet)->next)
#define wimax_osal_packet_list_prev(packet) ((packet)->prev)
#define wimax_osal_packet_list_foreach(packetlist, pos) \
                          for ((pos)=(packetlist)->first; \
                          (pos); \
                          (pos)=(pos)->next)

/* timer facilities */
enum WIMAX_OSAL_TR {
        WIMAX_OSAL_TR_10MS,
        WIMAX_OSAL_TR_1MS,
        WIMAX_OSAL_TR_100US,
        WIMAX_OSAL_TR_10US,
        WIMAX_OSAL_TR_1US,
};

typedef void (*wimax_osal_timer_cb)(s32, void*);

extern void wimax_osal_timer_handler(unsigned long);
extern s32 wimax_osal_timer_create(s32 timeout, 
				   enum WIMAX_OSAL_TR resolution, 
				   boolean periodic, 
				   void *data, 
				   wimax_osal_timer_cb cb);
extern s32 wimax_osal_timer_cancel(s32 handle);

/* trace priorities */
#define WIMAX_OSAL_PRIORITY_DIAGNOSTIC            0
#define WIMAX_OSAL_PRIORITY_DEBUG                 1
#define WIMAX_OSAL_PRIORITY_INFO                  2
#define WIMAX_OSAL_PRIORITY_ERROR                 3

/* trace functions */
extern void wimax_osal_trace(
	u32 group_id, u32 trace_id, u8 priority);
extern void wimax_osal_trace_byte(
	u32 group_id, u32 trace_id, u8 priority, u8 param);
extern void wimax_osal_trace_u16(
	u32 group_id, u32 trace_id, u8 priority, u16 param);
extern void wimax_osal_trace_u32(
	u32 group_id, u32 trace_id, u8 priority, u32 param);
extern void wimax_osal_trace_str(
	u32 group_id, u32 trace_id, u8 priority, u8 *param);
extern void wimax_osal_trace_data(
	u32 group_id, u32 trace_id, u8 priority, u8 *param, u32 len);

#endif /* PC2400M_OSAL_H */

