/*
 * pc2400m_if_we.c
 *
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Jouni Lappi <jouni.lappi@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/moduleparam.h>


#include <linux/if_arp.h>
#include <linux/wireless.h>
#include <net/iw_handler.h>

#include "pc2400m_drv_if.h"
#include "pc2400m_if_we.h"
#include "pc2400m_if.h"
#include "pc2400m_com.h"
#include "pc2400m_drv.h"

#define EV_BUFF          2048
#define IW_MAX_NSP_LEN   32
#define ALLA_OKKO        0

static int find_nap_from_stream(char ** stream, int stream_len,	u32 nap_id); 

/* standard IOCTL's */

static int pc2400m_we_get_name(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_set_frag(struct net_device *dev,
                        struct iw_request_info *info,
                        struct iw_param *vwrq,
                        char *extra);

static int pc2400m_we_get_frag(struct net_device *dev,
                        struct iw_request_info *info,
                        struct iw_param *vwrq,
                        char *extra);

static int pc2400m_we_set_txpower(struct net_device *dev,
                        struct iw_request_info *info,
                        struct iw_param *vwrq,
                        char *extra);

static int pc2400m_we_get_txpower(struct net_device *dev,
                        struct iw_request_info *info,
                        struct iw_param *vwrq,
                        char *extra);



/* private IOCTL's */
static int pc2400m_we_connect(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_disconnect(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_set_scan(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_get_scan(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_get_nap(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

#if 0
static int pc2400m_we_get_nsp(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);
#endif

static int pc2400m_we_set_eap_result(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_get_error_code(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_set_debug(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);

static int pc2400m_we_get_debug(struct net_device *dev,
                        struct iw_request_info *info,
                        union iwreq_data *wrqu,
                        char *extra);


/* handlers for the default wireless extension IOCTL's */
static const iw_handler pc2400m_we_handler[] =
{

        (iw_handler)NULL,      			     /* SIOCSIWCOMMIT */
        (iw_handler)pc2400m_we_get_name,             /* SIOCGIWNAME */
        (iw_handler)NULL,                            /* SIOCSIWNWID */
        (iw_handler)NULL,                            /* SIOCGIWNWID */
        (iw_handler)NULL,		             /* SIOCSIWFREQ */
        (iw_handler)NULL,			     /* SIOCGIWFREQ */
        (iw_handler)NULL,		             /* SIOCSIWMODE */
        (iw_handler)NULL,		             /* SIOCGIWMODE */
        (iw_handler)NULL,		             /* SIOCSIWSENS */
        (iw_handler)NULL, 		             /* SIOCGIWSENS */
        (iw_handler)NULL,                            /* SIOCSIWRANGE */
        (iw_handler)NULL, 		             /* SIOCGIWRANGE */
        (iw_handler)NULL,                            /* SIOCSIWPRIV */
        (iw_handler)NULL,                            /* SIOCGIWPRIV */
        (iw_handler)NULL,                            /* SIOCSIWSTATS */
        (iw_handler)pc2400m_get_wireless_stats,      /* SIOCGIWSTATS */
        (iw_handler)NULL,                            /* SIOCSIWSPY */
        (iw_handler)NULL,                            /* SIOCGIWSPY */
        (iw_handler)NULL,                            /* SIOCSIWTHRSPY */
        (iw_handler)NULL,                            /* SIOCGIWTHRSPY */
        (iw_handler)NULL,            		     /* SIOCSIWAP */
        (iw_handler)NULL,            		     /* SIOCGIWAP */
        (iw_handler)NULL,
        (iw_handler)NULL,                            /* SIOCGIWAPLIST */
        (iw_handler)NULL,       	             /* SIOCSIWSCAN */
        (iw_handler)NULL,		             /* SIOCGIWSCAN */
        (iw_handler)NULL,	     	             /* SIOCSIWESSID */
        (iw_handler)NULL,	                     /* SIOCGIWESSID */
        (iw_handler)NULL,                            /* SIOCSIWNICKN */
        (iw_handler)NULL,                            /* SIOCGIWNICKN */
        (iw_handler)NULL,
        (iw_handler)NULL,
        (iw_handler)NULL,            		    /* SIOCSIWRATE */
        (iw_handler)NULL,                           /* SIOCGIWRATE */
        (iw_handler)NULL,	                    /* SIOCSIWRTS */
        (iw_handler)NULL,	                    /* SIOCGIWRTS */
        (iw_handler)pc2400m_we_set_frag,            /* SIOCSIWFRAG */
        (iw_handler)pc2400m_we_get_frag,            /* SIOCGIWFRAG */
        (iw_handler)pc2400m_we_set_txpower,         /* SIOCSIWTXPOW */
        (iw_handler)pc2400m_we_get_txpower,         /* SIOCGIWTXPOW */
        (iw_handler)NULL,                           /* SIOCSIWRETRY */
        (iw_handler)NULL,                           /* SIOCGIWRETRY */
        (iw_handler)NULL,                           /* SIOCSIWENCODE */
        (iw_handler)NULL,                           /* SIOCGIWENCODE */
};

/* argument specification for the own wireless extensions below. one for
each private extension below 
even numbered ioctls are SET, restricted to root, and should not
 * return arguments (get_args = 0) */
static const struct iw_priv_args pc2400m_private_we_args[] =
{
        {
                SIOCIWFIRSTPRIV,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                0,
                "set scan"

        },

        {

                SIOCIWFIRSTPRIV + 1,
                0,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                "get scan"

        },

        {

                SIOCIWFIRSTPRIV + 3,
                0,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                "get nap"

        },

        {

                SIOCIWFIRSTPRIV + 4,
                0,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                "set eap key"

        },

        {

                SIOCIWFIRSTPRIV + 5,
                0,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                "get error code"

        },

        {

                SIOCIWFIRSTPRIV + 6,
                0,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                "disconnect"

        },

        {

                SIOCIWFIRSTPRIV + 7,
                0,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                "connect"

        },

        {

                SIOCIWFIRSTPRIV + 8,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                0,
                "set_debug"

        },

        {

                SIOCIWFIRSTPRIV + 9,
                0,
                IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
                "get_debug"

        },

};

/* handlers for own (WIMAX specific) wireless extensions.
these are allocated in order starting from SIOCIWFIRSTPRIV.*/

/* below are registered simple testing purpose handlers; they
are a starting point for private handlers. */
static const iw_handler pc2400m_private_we_handler[] =
{
	(iw_handler)pc2400m_we_set_scan,	/* SIOCSIWIMAXSCAN */
	(iw_handler)pc2400m_we_get_scan,	/* SIOCGIWIMAXSCAN */
	NULL,
        (iw_handler)pc2400m_we_get_nap,         /* SIOCGIWNAPID */
	(iw_handler)pc2400m_we_set_eap_result,	/* SIOCSIWEAPRESULT */
	(iw_handler)pc2400m_we_get_error_code,	/* SIOCGIWERRORCODE */
	(iw_handler)pc2400m_we_disconnect,	/* SIOCSIWDISCONNECT */
	(iw_handler)pc2400m_we_connect,		/* SIOCSIWCONNECT */
	(iw_handler)pc2400m_we_set_debug,	/* */
        (iw_handler)pc2400m_we_get_debug        /* */

};


/* finally, the wireless extensions definition */
const struct iw_handler_def  pc2400m_we_handler_def =
{

        .num_standard = sizeof(pc2400m_we_handler) / sizeof(iw_handler),
        .num_private = sizeof(pc2400m_private_we_handler) / sizeof(iw_handler),
        .num_private_args =
        sizeof(pc2400m_private_we_args) / sizeof(struct iw_priv_args),
        .standard = (iw_handler *) pc2400m_we_handler,
        .private = (iw_handler *) pc2400m_private_we_handler,
        .private_args = (struct iw_priv_args *) pc2400m_private_we_args,
	.get_wireless_stats = pc2400m_get_wireless_stats,

};


/**
 * pc2400m_send_simple_event - sends custom event to host
 * @dev: pointer to net device
 * @str: IWECUSTOM event to send
 */
void pc2400m_send_simple_event(struct net_device *dev, const char *str)
{
	union iwreq_data wrqu;
	int n = strlen(str);

	wrqu.data.pointer = kmalloc(IW_CUSTOM_MAX, GFP_ATOMIC);
	if (!wrqu.data.pointer)
		return;
	BUG_ON(n > IW_CUSTOM_MAX);
	wrqu.data.length = n;
	strcpy(wrqu.data.pointer, str);
	DEBUG("send evlen:%d\n",n);
	wireless_send_event(dev, IWEVCUSTOM, &wrqu, wrqu.data.pointer);
	kfree(wrqu.data.pointer);
}
/**
 * iw_statistics - returns statistic for interface
 * @dev: pointer to net device
 */
struct iw_statistics *pc2400m_get_wireless_stats(struct net_device *dev)
{

        struct net_local *lp = netdev_priv(dev);
        struct iw_statistics *wstats;

        DEBUG("%s: wireless stats requested!\n", dev->name);

        wstats = &lp->wstats;

        
        wstats->status = 1;
        wstats->qual.qual = lp->quality;

	/* fake some wireless status data */
        wstats->qual.level = 0;
        wstats->qual.noise = 0;
        wstats->qual.updated = 0;
        wstats->discard.nwid = 0;
        wstats->discard.code = 0L;
        wstats->discard.misc = 0L;
	
        return &lp->wstats;

}


/* standard IOCTL's */



/**
 * pc2400m_we_get_name - return protocol name
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_get_name(struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu,
			       char *extra)
{
        struct net_local *nl = netdev_priv(dev);
	int ret = ALLA_OKKO;
        /* this ioctl returns the "name" of the protocol supported by
        this wireless networking driver. */

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

        strncpy(wrqu->name, "IEEE 802.16", IFNAMSIZ);

        return ret;

}

/**
 * pc2400m_we_set_frag - sets socket MTU value
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_set_frag(struct net_device *dev,
			       struct iw_request_info *info,
			       struct iw_param *vwrq,
			       char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	int ret = ALLA_OKKO;
        /* this ioctl is used to set the fragmentation threshold: this is the
        maximum size of fragment being set over the air. this seems to be
        the exact same value as MTU */

        DEBUG("SET FRAG value: %d\n", vwrq->value);

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

	dev->mtu = vwrq->value;

        return ret;

}

/**
 * pc2400m_we_get_frag - gets socket MTU value
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_get_frag(struct net_device *dev,
			       struct iw_request_info *info,
			       struct iw_param *vwrq,
			       char *extra)
{
        struct net_local *nl = netdev_priv(dev);
	int ret = ALLA_OKKO;
        /* this ioctl is used to get the currently used 
	fragmentation threshold value */

        DEBUG("GET FRAG value: %d\n", dev->mtu);

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

	vwrq->value = dev->mtu;

        return ret;

}

/**
 * pc2400m_we_set_txpower - set radio on off
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
/* pc2400m has only support for setting this to enable/disable */
static int pc2400m_we_set_txpower(struct net_device *ndev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{

        struct net_local *nl = netdev_priv(ndev);
	struct drv_cmd cmd;
	int ret = ALLA_OKKO;
	WIMAX_WAIT_RSP;
        /* this ioctl is used to control the transmit power used by the
        chip, or to enable/disable the transmission altogether. */

	if (nl->if_status == UNINITIALIZED) {
		ret = -EIO;
		goto out1;
	}

        /* the requested transmit power is provided in dBm */

        if (vwrq->disabled)
        {
	/* here we should set radio off */
		cmd.cmd_id = E_RADIO_STATE;
		cmd.cmd.radio_state.radio_state = DRV_RADIO_STATE_OFF;


		WIMAX_WAIT_CMD(ndev, &cmd);

	        DEBUG("RADIO OFF\n");

        }
        else 
        {

	/* here we should set radio on */
		cmd.cmd_id = E_RADIO_STATE;
		cmd.cmd.radio_state.radio_state = DRV_RADIO_STATE_ON;


		WIMAX_WAIT_CMD(ndev, &cmd);

	        DEBUG("RADIO ON\n");
        }

	nl->radio_disabled = vwrq->disabled;

	if (pc2400m_complete.pc2400m_result != E_SUCCESS)
		ret = -EINVAL;
out1:
        return ret;

}

/**
 * -
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_get_txpower(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	int ret = ALLA_OKKO;
        /* this ioctl is used to get the current txpower state, on or off*/

        DEBUG("GET TXPOWER \n");

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

	vwrq->disabled = nl->radio_disabled;

        return ret;

}




/* private IOCTL's */

/**
 * pc2400m_we_set_scan - starts network scanning
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_set_scan(struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu,
			       char *extra)
{

        struct net_local *nl = netdev_priv(dev);
        struct iw_wimax_scan_req *wmxscan;
	struct drv_cmd cmd;
	int res, ret = ALLA_OKKO;
	WIMAX_WAIT_RSP;

        DEBUG("set scan ->%s\n",dev->name);

	if (nl->if_status == UNINITIALIZED) {
		ret = -EIO;
		goto out1;
	}
        /* this ioctl is used to request the device to start scanning for
        available networks. the IOCTL simply starts the scanning procedure,
        the scan will complete later and userspace can poll for the results
        via the get method of this ioctl.

        the scan request contains these parameters:
        - wmxscan->flags: IW_SCAN_THIS_NAP, IW_SCAN_NSPS
        - if IW_SCAN_THIS_NAP, then wmxscan->nap contain
        the NAP to scan */
        
	/* here we reset the previous scan list */
	nl->event_stream_len = 0;
	kfree(nl->event_stream);
	nl->event_stream = NULL;

	wmxscan = kmalloc(wrqu->data.length,GFP_KERNEL);
	if (!wmxscan) {
		ret = -ENOMEM;
		goto out1;
	}

	res = copy_from_user(wmxscan,wrqu->data.pointer,wrqu->data.length);

	BUG_ON(res);

	cmd.cmd.search.p_search_plan = 
	        kmalloc(wmxscan->num_channel_plans * 
			sizeof(struct drv_search_plan),
			GFP_KERNEL);
	if (!cmd.cmd.search.p_search_plan) {
		ret = -ENOMEM;
		goto out1;
	}

	cmd.cmd.search.search_plan_count = wmxscan->num_channel_plans;
#define N_I_U 0
	cmd.cmd.search.search_progress = N_I_U;/* progress bar not used */
	cmd.cmd.search.search_type = DRV_SEARCH_TYPE_NORMAL;
	if (wmxscan->num_channel_plans > 0)
		memcpy(cmd.cmd.search.p_search_plan,
	       	       wmxscan->channel_plan,	
	               sizeof(struct drv_search_plan) * 
		       wmxscan->num_channel_plans);

	if (wmxscan->flags & IW_SCAN_NSPS) {
		/* get nsp's for nap's */
		DEBUG("GET NSPS\n");
		cmd.cmd.search.search_nsp_info_req = 
		        DRV_SEARCH_NSP_INFO_REQ_NSP_LIST_AND_NAMES;
      	} 
	else 
	{
		/* no nsp's in scan */
		cmd.cmd.search.search_nsp_info_req = 
		        DRV_SEARCH_NSP_INFO_REQ_NONE;
	}

	cmd.cmd.search.p_known_nap_info = NULL;
	cmd.cmd.search.known_nap_info_count = 0;

	cmd.cmd_id = E_SEARCH;
	WIMAX_NO_WAIT_CMD(dev, &cmd);

	kfree(cmd.cmd.search.p_search_plan);

        DEBUG("scan started num chnlplans %02x\n",wmxscan->num_channel_plans);

	kfree(wmxscan);
out1:
        return ret;

}

/**
 * pc2400m_we_get_scan - host asks for scan results
 * if host doesn't deliver enough memory -E2BIG is returned
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_get_scan(struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu,
			       char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	int res,ret = ALLA_OKKO;

        /* this IOCTL is used to return scan results once scanning for
        networks has been completed or requested */

        DEBUG("GET SCAN\n");

	if (nl->if_status == UNINITIALIZED) {
		ret = -EIO;
		goto out1;
	}
        /* Return scanning result so far with this function 
	   If we do not get enough memory from user space 
	   we will ask for more */
	if (nl->event_stream_len > wrqu->data.length) {
		wrqu->data.length = nl->event_stream_len;	
		ret = -E2BIG;
		goto out1;
	}

	res = copy_to_user(wrqu->data.pointer,
			   nl->event_stream,
		 	   nl->event_stream_len);
	BUG_ON(res);

        wrqu->data.length = nl->event_stream_len;
        wrqu->data.flags = 0;

	DEBUG("returned scan result len:%08x\n",nl->event_stream_len);
out1:
        return ret;

}

/**
 * pc2400m_we_connect - connect network
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_connect(struct net_device *dev,
			      struct iw_request_info *info,
			      union iwreq_data *wrqu,
			      char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	struct drv_cmd cmd;
	struct nap_id *nap_struct = (struct nap_id *)wrqu;	
	int ret = ALLA_OKKO;
	WIMAX_WAIT_RSP;

	if (nl->if_status == UNINITIALIZED) {
		ret = -EIO;
		goto out1;
	}
        /* this ioctl is used by userspace to request a specific nap
        to be used for connectivity. */
	DEBUG("Entering nap:%08x\n",nap_struct->nap_n);

	cmd.cmd.entry.nap_id = nap_struct->nap_n;

	cmd.cmd_id = E_ENTRY;

	WIMAX_WAIT_CMD(dev, &cmd);	
		
	DEBUG("Entry status:%02x\n",
		     pc2400m_complete.pc2400m_result);	

	if (pc2400m_complete.pc2400m_result != E_SUCCESS) {
		ret = -EFAULT;
	}
out1:
        return ret;

}

/**
 * pc2400m_we_disconnect - disconnects from nw
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_disconnect(struct net_device *dev,
				 struct iw_request_info *info,
				 union iwreq_data *wrqu,
				 char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	struct drv_cmd cmd;	
	int ret = ALLA_OKKO;
	WIMAX_WAIT_RSP;

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

	/*disconnect*/

	cmd.cmd_id = E_EXIT;
	WIMAX_WAIT_CMD(dev, &cmd);
	nl->nap = 0xffffffff;
	/* save nap and quality to netlocal */
	if (pc2400m_complete.pc2400m_result != E_SUCCESS) {
			ret = -EINVAL;
			goto out1;
	}		

out1:
        DEBUG("Disconnect\n");

        return ret;

}

/**
 * pc2400m_we_get_error_code - returns codes for failures
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_get_error_code(struct net_device *dev,
				     struct iw_request_info *info,
				     union iwreq_data *wrqu,
				     char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	struct err_cod *errcod = (struct err_cod *)wrqu;
	int ret = ALLA_OKKO;
	
	errcod->code = nl->errcod;
	
        DEBUG("Get error code\n");

        return ret;

}

/**
 * pc2400m_we_get_nap - for getting NAP value
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_get_nap(struct net_device *dev,
			      struct iw_request_info *info,
			      union iwreq_data *wrqu,
			      char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	int ret = ALLA_OKKO;

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

        /* this ioctl is used by userspace to get the currently used NAP */
	DEBUG("Getting nap:%08x\n",nl->nap);

	wrqu->param.value = nl->nap;	

        return ret;

}



/**
 * pc2400m_we_set_eap_result - saves EAP key to device
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_set_eap_result(struct net_device *dev,
				     struct iw_request_info *info,
				     union iwreq_data *wrqu,
				     char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	struct wmx_eap_result *result;
	struct drv_cmd cmd;
	int ret = ALLA_OKKO;
	char *res = NULL;

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

	/*MISSING:no storing of msk in wihal*/

        /* this ioctl is used by userspace to set eap success */
	result = (struct wmx_eap_result *)wrqu->data.pointer;
	if (result->eap_bool) {
		DEBUG("Setting msk\n");
		res = kmalloc(result->msk_length,GFP_KERNEL);
		if (!res) {
			ret = -E2BIG;
			goto out;
		}
		
		ret = copy_from_user(res,
				     (void *)(&result->p_msk[0]),
				     result->msk_length);

		cmd.cmd.eap_result.p_msk = res;
		cmd.cmd.eap_result.msk_length = result->msk_length;
		cmd.cmd.eap_result.result = DRV_TRUE;
	} else {
		DEBUG("Setting eap failure\n");
		cmd.cmd.eap_result.p_msk = NULL;
		cmd.cmd.eap_result.msk_length = 0;
		cmd.cmd.eap_result.result = DRV_FALSE;
	}

	cmd.cmd_id = E_EAP_RESULT;
	WIMAX_NO_WAIT_CMD(dev, &cmd);

	kfree(res);

out:
        return ret;

}

/**
 * pc2400m_we_set_debug - set debugging mode
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_set_debug(struct net_device *dev,
				struct iw_request_info *info,
				union iwreq_data *wrqu,
				char *extra)
{
        struct net_local *nl = netdev_priv(dev);
	int ret = ALLA_OKKO;

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

        DEBUG("%s: SET_INFO accessed.\n", dev->name);

        return ret;

}

/**
 * pc2400m_we_get_debug - get debugging mode
 * @dev: pointer to net device
 * @info: pointer to IOCTL info
 * @wrqu: pointer IOCTL parameters
 * @extra: pointer to extra data
 */
static int pc2400m_we_get_debug(struct net_device *dev,
				struct iw_request_info *info,
				union iwreq_data *wrqu,
				char *extra)
{

        struct net_local *nl = netdev_priv(dev);
	int ret = ALLA_OKKO;

	if (nl->if_status == UNINITIALIZED) ret = -EIO;

        DEBUG("%s: GET_INFO accessed.\n", dev->name);

        return ret;

}


/**
 * pc2400m_we_scan_result - This functions saves received results
 * @resp:result to save
 * @dev: pointer to net device
 */
int pc2400m_we_scan_result(struct drv_ind_search *resp, 
			   struct net_device *dev)
{

        struct net_local *nl = netdev_priv(dev);
	struct iw_event iwe;
	struct iw_event *iwep;
        char *current_ev;
	char *nlevtmp;
	struct drv_search_result *search_result;

	int i,ret = ALLA_OKKO;

	DEBUG("RECEIVED SCAN RESULT\n");

	if (!nl->event_stream)
	{
		nl->event_stream = kmalloc(EV_BUFF,GFP_KERNEL); 
		if (!nl->event_stream) {
			ret = -ENOMEM;
			goto out1;
		}
		nl->event_stream_len = 0;
		nl->event_stream_maxlen = EV_BUFF;
	}


	DEBUG("result count %02x\n",resp->result_count);
	for(i = 0;i < resp->result_count;i++)
	{

		
		current_ev = nl->event_stream;	
		search_result = &(resp->p_search_result[i]);

		/* find this nap from stream or last event */
		if (!find_nap_from_stream(&current_ev,
					  nl->event_stream_len,
					  search_result->nap_id))

		{

			/*if we found nap update quality */
			iwep = (struct iw_event *)current_ev;
			/* signal strength received */
		        iwep->u.qual.level = 
				search_result->signal_strength;

	        	iwep->u.qual.noise = 0;
	        	iwep->u.qual.qual = 0;
		}
		else
		{
			/* check that we have enough buffer*/
			if (nl->event_stream_maxlen < (
				(search_result->number_of_nsp_id *
				IW_MAX_NSP_LEN) +
				IW_EV_ADDR_LEN + IW_EV_QUAL_LEN +
				nl->event_stream_len))
			{
				DEBUG("SCAN BUFFER EXPANDED\n");
				nlevtmp = kmalloc(nl->event_stream_len + 
						  EV_BUFF,
						  GFP_KERNEL);
				if (!nlevtmp) {
					ret = -ENOMEM;
					goto out1;
				}
				nl->event_stream_maxlen = 
					nl->event_stream_len +EV_BUFF;

				memcpy(nlevtmp,
				       nl->event_stream,
				       nl->event_stream_len);
				kfree(nl->event_stream);
				nl->event_stream = nlevtmp;
	
			}

			/*add new event */
			/* add 32bit nap id*/
			iwe.u.param.value = 
				search_result->nap_id;

       			iwe.cmd = SIOCGIWNAP;
	        	current_ev = iwe_stream_add_event(current_ev, 
				nl->event_stream + IW_SCAN_MAX_DATA,
	        		&iwe, IW_EV_ADDR_LEN);

	        	/* signal strength only in quality */
			/* signal strength received */
		        iwe.u.qual.level = 
				search_result->signal_strength; 
  		      	iwe.u.qual.noise = 0;
   		     	iwe.u.qual.qual = 0;
        		iwe.cmd = IWEVQUAL;
        		current_ev = 
				iwe_stream_add_event(current_ev,
						     nl->event_stream + 
						     IW_SCAN_MAX_DATA,
        					     &iwe, IW_EV_QUAL_LEN);

		/* MISSING: here we add nsp's assosiated to nap to stream */
       
			nl->event_stream_len =
				(current_ev - nl->event_stream);	

		}
	}
 out1:
	return ret;

}

/**
 * find_nap_from_stream -
 * this returns 0 if nap was found and pointer stream points to it
 * otherwise returns 1 and stream points to first empty space
 * @stream: pointer to stream of events 
 * @stream_len: Original stream length 
 * @nap_id: uint32
 */
static int find_nap_from_stream(char ** stream, 
				int stream_len,
				u32 nap_id)
{
 
	struct iw_event *iwe;
	char *temp_st;
#define ALLA_OKKO1 1
	int ret = ALLA_OKKO1;
	/* Extract the event header (to get the event id).
	 * Note : the event may be unaligned, therefore copy... */
	temp_st = *stream;
	while( temp_st < (*stream + stream_len))
	{
		iwe = (struct iw_event *)temp_st;
		temp_st += iwe->len;
		if((iwe->cmd == SIOCGIWNAP)&&
			(iwe->u.param.value == nap_id ))
		{
			DEBUG("NAP already in list upd qual\n");
			*stream = temp_st;
			ret = ALLA_OKKO;
			goto out1;
		}
	}


	*stream = temp_st;
 out1:
	return ret;

}


