/*
 * pc2400m_if.c
 *
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Jouni Lappi <jouni.lappi@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/firmware.h>
#include <linux/completion.h>
#include <linux/platform_device.h>
#include <linux/wireless.h>
#include <net/iw_handler.h>

#include "pc2400m_com.h"
#include "pc2400m_if.h"
#include "pc2400m_al.h"
#include "pc2400m_drv_if.h"
#include "pc2400m_if_sysfs.h"
#include "pc2400m_if_we.h"
#include "pc2400m_if_netlink.h"

#define FIRMWARE_FNAME  "pc2400m.bin"

#define ETH_P_EAP	0x888E		/* EAP FRAME */
#define OVERHEAD 	64
#define SL_OFF		0x00
#define SL_ON_FLAG	0x02
#define NO_NAP	0xffffffff
#define IDX_ZR 		0

char firmware_file[32] = FIRMWARE_FNAME;

static int pc2400m_if_open(struct net_device *dev);
static int pc2400m_if_stop(struct net_device *dev);
static int pc2400m_if_xmit(struct sk_buff *skb, struct net_device *dev);
static struct net_device_stats *pc2400m_if_get_stats(struct net_device *dev);

struct wimax_driver_if *pc2400m_driver_if = NULL;

/**
 * pc2400m_if_fetch_firmware - writes firmware to device
 * @ndev: pointer to network device 
 * @fw_id: pointer to firmware filename
 * @dev: pointer to device
 * 
 * kernel's sysfs provides firmware to write to device
 *
 */
static int pc2400m_if_fetch_firmware(struct net_device *ndev, 
				   char *fw_id, 
				   struct device *dev)
{
	const struct firmware *fw_entry = NULL;
	struct net_local *nl = netdev_priv(ndev);
	struct drv_cmd cmd;
	struct drv_fw_data_block fwblock;
	int ret = 0;
	WIMAX_WAIT_RSP;

	ret = request_firmware(&fw_entry, fw_id, dev);

	if (ret) {
		ERROR("WiMAX firmware loading failed (err: %d) "
		      "for '%s' \n", ret, fw_id);
	}
	else
	{
		cmd.cmd_id = E_INITIALIZE;
#define FW_BLK_CNT 1
		cmd.cmd.initialize.block_count = FW_BLK_CNT;
		cmd.cmd.initialize.p_blocks = &fwblock;
		cmd.cmd.initialize.p_blocks->length = fw_entry->size;
		cmd.cmd.initialize.p_blocks->ptr = fw_entry->data;

		WIMAX_WAIT_CMD(ndev, &cmd);

		if (nl->if_status == INIT_DONE) ret = 0;
		else ret = -EIO;

		release_firmware(fw_entry);

	}
	return ret;
}

/**
 * pc2400m_tx - scheduled function to send data to device
 * @work: pointer to work structure containing 
 *        -device and data
 */
static void pc2400m_tx(struct work_struct *work)
{
	struct net_local *nl = container_of(work, struct net_local, tx_work);
	struct net_device_stats *stats = &(nl->stats);
	struct drv_cmd cmd;
	u16 skb_protocol;
	unsigned long flags;
	struct sk_buff *skb;
	struct sk_buff *skb_array[1];

	spin_lock_irqsave(&(nl->tx_lock), flags);
	while(nl->tx_buff && nl->tx_credit) {
		skb = nl->tx_buff;
		nl->tx_buff = nl->tx_buff->next;
		skb->next = skb->prev = NULL;
		spin_unlock_irqrestore(&(nl->tx_lock), flags);
		skb_protocol = htons(skb->protocol);	
		/* check the header for EAP packets */	
		if(skb_protocol == ETH_P_EAP) {
			DEBUG("EAP packet from socket\n");
			cmd.cmd_id = E_EAP_DATA;
			cmd.cmd.eap_data.p_payload =  skb;
			WIMAX_NO_WAIT_CMD(nl->net_dev, &cmd);
		} else {
			cmd.cmd_id = E_DATA_SEND;
			/* MISSING: not used yet*/
#define BSR_NUMBER_ONE 1		
			cmd.cmd.data_send.transport_id = BSR_NUMBER_ONE;
			cmd.cmd.data_send.data_length = BSR_NUMBER_ONE; 
			/*this is number of packets*/
			
			skb_array[0] = skb;
			cmd.cmd.data_send.p_data = skb_array;

			stats->tx_packets++;
			stats->tx_bytes += skb->len;

			/* remove the sent amount of packets from the
			   credit */
			nl->tx_credit -= 1;
			
			WIMAX_NO_WAIT_CMD(nl->net_dev, &cmd);
		}
		spin_lock_irqsave(&(nl->tx_lock), flags);
	}
	nl->tx_flags &= ~(SL_ON_FLAG);
	spin_unlock_irqrestore(&(nl->tx_lock), flags);

	return;
}

/**
 * pc2400m_if_xmit - schedules pc2400m_tx to send data to device
 * @skb: pointer to data to send
 * @dev: pointer to net device
 *
 */
static int pc2400m_if_xmit(struct sk_buff *skb, struct net_device *dev)
{

	struct net_local *nl = netdev_priv(dev);
	struct sk_buff *tmp;
	unsigned long flags;
	int schedule = 1, ret = 0;

	spin_lock_irqsave(&(nl->tx_lock), flags);

	if (!nl->tx_buff)
  		nl->tx_buff = skb;
	else
	{
		tmp = nl->tx_buff;
		while(tmp->next)
		{
			tmp = tmp->next;
		}
		tmp->next = skb;
	}	

	/* this will work on the assumption that there is no actual data
	   transfer while EAP negotiations are performed - otherwise the
	   data transfer could block the EAP negotiations. */

	/* only schedule transmission if none is currently scheduled AND
	   there is credit left */
	if ((nl->tx_flags & SL_ON_FLAG) || nl->tx_credit == 0) 
		schedule = SL_OFF;
	if (nl->tx_credit == 0) netif_stop_queue(dev);
	spin_unlock_irqrestore(&(nl->tx_lock), flags);

	if (schedule) {
		nl->tx_flags |= SL_ON_FLAG;
		INIT_WORK(&nl->tx_work,pc2400m_tx);
		schedule_work(&nl->tx_work);
	}

	return ret;

}

/**
 * net_device_stats - returns net devices statistics
 * @dev: pointer to net device
 *
 */
static struct net_device_stats *pc2400m_if_get_stats(struct net_device *dev)
{
	
	struct net_local *nl = netdev_priv(dev);

	/* the net device stats are in the local data */
	return &(nl->stats);
	
}

/**
 * pc2400m_if_resp - receive a command response from the WiMAX driver
 * @ndev: pointer to network device 
 * @resp: pointer containing response type and parameters
 * @complete: pointer data given in cmd
 * Receive a response from the WiMAX driver. This is a direct result of
 * issuing the driver a command.
 *
 * Do not lock inside this because lock is already used downstrairs
 */
static void pc2400m_if_resp(struct net_device *ndev, 
			    struct drv_resp *resp, 
			    void *pc2400m_complete) 
{

	struct net_local *nl = netdev_priv(ndev);
	struct net_device_stats *stats = &(nl->stats);
	struct sk_buff *skb;
	struct sk_buff *nlskb;

	struct pc2400m_complete_struct *bes;
	int err = 0;

	unsigned long flags;
	int schedule = 1;


	switch (resp->cmd_resp_id) {

	case E_INITIALIZE_RESP:
		/* firmware and init done */
		if (resp->cmd_status == E_SUCCESS)
			nl->if_status = INIT_DONE;
		else nl->if_status = UNINITIALIZED;
		break;

	case E_DATA_SEND_RESP:
		if (resp->cmd_status != E_SUCCESS)
			stats->tx_dropped++;

		spin_lock_irqsave(&(nl->tx_lock), flags);

		/* the response indicates full credit */
		nl->tx_credit = DRV_DATA_SEND_CREDIT_LEN;

		/* determine if a new transfer should be scheduled */
		if ((nl->tx_flags & SL_ON_FLAG) || nl->tx_buff == NULL) 
			schedule = 0;

		spin_unlock_irqrestore(&(nl->tx_lock), flags);

		/* allow incoming data from the kernel */
		netif_wake_queue(ndev);

		if (schedule) {
			nl->tx_flags |= SL_ON_FLAG;
			INIT_WORK(&nl->tx_work, pc2400m_tx);
			schedule_work(&nl->tx_work);
		}
		break;

	case E_DATA_RECEIVE_RESP:
	        /* single packet assumed, as quota provided for only one */
		skb = resp->resp.data_receive.p_data[0];	
		BUG_ON(!skb);

		skb->protocol = htons(ETH_P_IP);

		skb->dev = ndev;		
		skb->ip_summed = CHECKSUM_UNNECESSARY;
		skb->mac.raw = ndev->dev_addr;

		stats->rx_packets++;
		stats->rx_bytes += skb->len;

		err = netif_rx(skb);

		break;

	case E_CONFIGURE_RESP:

		DEBUG("fw vrs:%s\n",
			resp->resp.configure.capabilities.p_fw_version);
		DEBUG("hw vrs:%s\n",
		      	resp->resp.configure.capabilities.p_hw_version);

		DEBUG("Mac address:%02x:%02x:%02x:%02x:%02x:%02x\n",
			resp->resp.configure.capabilities.mac_address[0],
			resp->resp.configure.capabilities.mac_address[1],
			resp->resp.configure.capabilities.mac_address[2],
			resp->resp.configure.capabilities.mac_address[3],
			resp->resp.configure.capabilities.mac_address[4],
			resp->resp.configure.capabilities.mac_address[5]);

		memcpy(nl->fw_vrs, 
			resp->resp.configure.capabilities.p_fw_version,
			VERSION_STR_LEN);
		memcpy(nl->hw_vrs,
			resp->resp.configure.capabilities.p_hw_version,
			VERSION_STR_LEN);

		/* set mac to device */
		memcpy(ndev->dev_addr,
		       resp->resp.configure.capabilities.mac_address,
		       ETH_ALEN);

		break;

	case E_CLOSE_RESP:
		break;


	case E_SEARCH_RESP:
		/* search complete send wireless event for this in we */
		pc2400m_send_simple_event(ndev, SCAN_COMPLETE);

	case E_SEARCH_STOP_RESP:
		break;

	case E_ENTRY_RESP:
		break;

	case E_EXIT_RESP:
		nl->nap = NO_NAP;
		break;

	case E_NETWORK_STATUS_RESP:
		break;

	case E_EAP_DATA_RESP:
		break;

	case E_EAP_RESULT_RESP:
		break;

	case E_FLOW_CREATE_MS_RESP:
		break;

	case E_FLOW_CREATE_BS_RESP:
		break;

	case E_FLOW_DELETE_MS_RESP:
		break;

	case E_FLOW_DELETE_BS_RESP:
		break;

	case E_FLOW_CHANGE_MS_RESP:
		break;

	case E_FLOW_CHANGE_BS_RESP:
		break;

	case E_PROD_TEST_RESP:
		/* send output to prodtest file for now*/
		if (resp->resp.prod_test.data) {
		/*MISSING we should remove this when netlink stuff is in use*/
			if (nl->pt_buffer) kfree(nl->pt_buffer);	
			nl->pt_buffer = 
				kmalloc(resp->resp.prod_test.data_length,
		   	                GFP_KERNEL);
			if (!nl->pt_buffer) goto err1;
			nl->pt_len = resp->resp.prod_test.data_length;
			memcpy(nl->pt_buffer,
			       resp->resp.prod_test.data,
			       resp->resp.prod_test.data_length);
			/* send output to netlink */
			nlskb = dev_alloc_skb(
					resp->resp.prod_test.data_length +
					OVERHEAD);
			if (!nlskb) goto err1;
			skb_reserve(nlskb,OVERHEAD);
			memcpy(skb_put(nlskb,
				       resp->resp.prod_test.data_length),
			       resp->resp.prod_test.data,
			       resp->resp.prod_test.data_length);
			kfree(resp->resp.prod_test.data);
			pc2400m_netlink_send_data(
				ndev, nlskb, NETLINK_PC2400M1);
		}
		break;

	case E_STATISTICS_GET_RESP:

		if(resp->resp.statistics_get.data){
			/* send output to netlink */
			nlskb = dev_alloc_skb(
				resp->resp.statistics_get.data_length + 
				OVERHEAD);
			if (!nlskb) goto err1;
			skb_reserve(nlskb,OVERHEAD);
			memcpy(skb_put(nlskb,
				       resp->resp.statistics_get.data_length),
			       resp->resp.statistics_get.data,
			       resp->resp.statistics_get.data_length);
			kfree(resp->resp.statistics_get.data);
			pc2400m_netlink_send_data(
				ndev, nlskb, NETLINK_PC2400M2);
		}
		break;

	case E_TRACE_RESP:
	{
		struct netlink_pipe_msg_str *pipehdr;

		if (resp->resp.trace.data) {
			/* send output to netlink */
			nlskb = dev_alloc_skb(
				resp->resp.trace.data_length +
				sizeof(struct netlink_pipe_msg_str) +
				OVERHEAD);
			if (!nlskb) goto err1;
			skb_reserve(nlskb, OVERHEAD);

			/* insert multiplex header */
			pipehdr = (struct netlink_pipe_msg_str*)
				skb_put(nlskb, 
					sizeof(struct netlink_pipe_msg_str));
			pipehdr->type = NETLINK_PIPE_TYPE_MESSAGE;
			pipehdr->length = resp->resp.trace.data_length +
				sizeof(struct netlink_pipe_msg_str);
			pipehdr->version = NETLINK_PIPE_VERSION;

			/* insert raw data */
			memcpy(skb_put(nlskb,
				       resp->resp.trace.data_length),
			       resp->resp.trace.data,
			       resp->resp.trace.data_length);
			kfree(resp->resp.trace.data);

			pc2400m_netlink_send_data(
				ndev, nlskb, NETLINK_PC2400M3);
		}
		break;
	}
	case E_RADIO_STATE_RESP:

		break;

	case E_SUSPEND:
	case E_RESUME:

		/* no special handling, these are for synchronization
		   purposes only. */

		break;

	default:
		ERROR("no resp handling\n");
		break;
	}

	/* if after resp there is need to do spi commands use completion 
	   to make things synchronous */
	bes = (struct pc2400m_complete_struct *)pc2400m_complete;
	if (bes) {
		bes->pc2400m_result = resp->cmd_status;
 		complete(&bes->comp);	
	}
 	goto out1;
 err1:
	ERROR("no mem handling resp\n");
 out1:
	return;

}

/**
 * pc2400m_if_ind - receive an indication from the WiMAX driver
 * @ndev: pointer to network device 
 * @ind: pointer containing indication type and parameters
 *
 * Receive an indication from the WiMAX driver module.
 *
 * Do not lock inside ind because lock is already used downstrairs
 */
static void pc2400m_if_ind(struct net_device *ndev, struct drv_ind *ind) 
{

	struct drv_cmd_data_receive data_receive_req;
	struct drv_cmd cmd;
	struct net_local *nl = netdev_priv(ndev);
	struct sk_buff *nlskb = NULL;
	int err;	


	switch (ind->ind_id) {

	case E_DATA_RECV_IND:
		/* not used at the moment */
		data_receive_req.transport_id = 
			ind->ind.data_receive.transport_id;

		/* now we must fetch the data */
		cmd.cmd_id = E_DATA_RECEIVE;
		/* lets get one packet*/
#define ONE_PACKET 1
		cmd.cmd.data_receive.data_length  = ONE_PACKET;
		WIMAX_NO_LOCK_CMD(ndev, &cmd);
		break;

	case E_SEARCH_IND:
		
		if (!pc2400m_we_scan_result(&ind->ind.search, ndev))
			DEBUG("SCAN RESP saved\n");		
		else 
			ERROR("SCAN RESULTS COULD NOT BE SAVED!\n");

		break;

	case E_NETWORK_STATUS_IND:
		nl->quality = ind->ind.network_status.signal_strength;
	        nl->cinr = ind->ind.network_status.cinr;
		nl->bsid = ind->ind.network_status.bsid;

		switch(ind->ind.network_status.network_status){
			/* connecting */
			case E_NETWORK_STATUS_SF:
				DEBUG("Network entry done,\
			       	       waiting for service flows\n");
			break;
			/* connected */
			case E_NETWORK_STATUS_CONNECTED:
				DEBUG("DATA PATH connected\n");
				DEBUG("BSID0 %02x Signal Strength:%d\n",
				      (u8)ind->ind.network_status.bsid[0],
				      ind->ind.network_status.signal_strength);
				/* save nap */
				nl->nap = ((u32)
					   (ind->ind.network_status.bsid[2])|
					   ((u32)
					   (ind->ind.network_status.bsid[1])
					   <<8)|
					   ((u32)
					   (ind->ind.network_status.bsid[0])
					   <<16)
					   );
				/* ready for sending data, 
				   start the interface transmit queue */
				netif_start_queue(ndev);
				/* search complete send wireless 
				   event for this in we */
				pc2400m_send_simple_event
				        (ndev, WIMAX_CONNECTED);
				nl->if_status = CONNECTED;
#if 0 /* TO BE FIXED: the ownership of string pointers is not transferred
	 in WiHAL unless explicitly stated in the interface specification ->
	 in this case they do not. You may not use the pointers directly, but
	 must create own copies. Additionally, memory needs to be freed as 
	 well somewhere -> that seems to be missing in this case. */
  			        nl->nsp_name_len = 
				        ind->ind.network_status.nsp_name_len;
  			      	nl->nsp_realm_len = 
				        ind->ind.network_status.nsp_realm_len;
        			nl->p_nsp_name = 
				        ind->ind.network_status.p_nsp_name;
			        nl->p_nsp_realm = 
				        ind->ind.network_status.p_nsp_realm;
#endif
				break;
			case E_NETWORK_STATUS_DISCONNECTED:
				netif_stop_queue(ndev);
				nl->nap = NO_NAP;
				nl->errcod = No_error;
				pc2400m_send_simple_event
				        (ndev, WIMAX_DISCONNECT);
				break;
			case E_NETWORK_STATUS_HANDOVER:
				pc2400m_send_simple_event(ndev, BS_HO);
				break;
			case E_NETWORK_STATUS_DO_HANDOVER:
				break;	
			case E_NETWORK_STATUS_LOST:	
				nl->errcod = OOZ;
				break;
			case E_NETWORK_STATUS_RECONNECTED:
				nl->errcod = No_error;
				break;
			case E_NETWORK_STATUS_RANGING:
				break;
			case E_NETWORK_STATUS_AUTHENTICATION:
				break;
			case E_NETWORK_STATUS_REGISTERING:
				break;
			default:
				
				break;

		}
		break;
	case E_EAP_DATA_IND:
		nlskb = ind->ind.eap_data.p_payload;
			
		nlskb->protocol = htons(ETH_P_EAP);
		nlskb->dev = ndev;		
		nlskb->ip_summed = CHECKSUM_UNNECESSARY;
		nlskb->mac.raw = ndev->dev_addr;
		err = netif_rx(nlskb);
		break;
	case E_FLOW_CREATE_BS_IND:
		break;
	case E_FLOW_DELETE_BS_IND:
		break;
	case E_FLOW_CHANGE_BS_IND:
		break;
	case E_PROD_TEST_IND:
		if (ind->ind.prod_test.data) {
			if (nl->pt_buffer) kfree(nl->pt_buffer);	
			nl->pt_buffer = kmalloc(ind->ind.prod_test.data_length,
					        GFP_KERNEL);
			if (!nl->pt_buffer) goto err1;
			nl->pt_len = ind->ind.prod_test.data_length;
			memcpy(nl->pt_buffer,ind->ind.prod_test.data,
			       ind->ind.prod_test.data_length);
			/* send output to netlink */
			nlskb = dev_alloc_skb(ind->ind.prod_test.data_length +
					      OVERHEAD);
			if (!nlskb) goto err1;
			skb_reserve(nlskb,OVERHEAD);
			memcpy(skb_put(nlskb,ind->ind.prod_test.data_length),
			       ind->ind.prod_test.data,
			       ind->ind.prod_test.data_length);
			kfree(ind->ind.prod_test.data);
			pc2400m_netlink_send_data(
				ndev, nlskb, NETLINK_PC2400M1);
		}
		break;

	default:
		ERROR("no ind handling\n");
		break;
	}
	goto out1;
 err1:
	ERROR("no mem handling resp\n");
 out1:	
	return;

}

/**
 * pc2400m_if_open - driver is opened
 * @dev: pointer to network device 
 * 
 */
static int pc2400m_if_open(struct net_device *dev)
{

	struct net_local *nl = netdev_priv(dev);
	struct drv_cmd cmd;
	int ret = 0;
	WIMAX_WAIT_RSP;
		
	/* register the new instance in the driver also */
	pc2400m_drv_adapt.cmd_resp = pc2400m_if_resp;
	pc2400m_drv_adapt.ind = pc2400m_if_ind;
	pc2400m_driver_if = WIMAX_OPEN(dev);

	/* prepare the credit counter (this is instance specific) */
	nl->tx_credit = DRV_DATA_SEND_CREDIT_LEN;

	/* the driver structure */
	ret = pc2400m_if_fetch_firmware(dev, firmware_file, 
					&(nl->spi_dev->dev));
	if (ret) {
		cmd.cmd_id = E_CLOSE;
		WIMAX_WAIT_CMD(dev,&cmd);
		goto out1;
	}

	cmd.cmd_id = E_CONFIGURE;
	
	WIMAX_WAIT_CMD(dev,&cmd);

	/*Initialize netlink */	
	pc2400m_netlink_init(dev);
	/* queue is activated after service flows have been established
	E_NETWORK_STATUS_IND == E_NETWORK_STATUS_CONNECTED */
 out1:	
	return ret; /* all OK */
	
}

/**
 * pc2400m_if_stop - driver is stopped
 * @dev: pointer to network device 
 * 
 */
static int pc2400m_if_stop(struct net_device *dev)
{
	struct drv_cmd cmd;
	struct net_local *nl = netdev_priv(dev);
	int ret = 0;
	WIMAX_WAIT_RSP;

	if (nl->if_status == UNINITIALIZED) goto out1;
	/* release resources, such as memory areas */
	cmd.cmd_id = E_CLOSE;
	WIMAX_WAIT_CMD(dev, &cmd);
	
	/* stop the transmit queue (device can no longer transmit data)*/
	netif_stop_queue(dev);

	nl->if_status = UNINITIALIZED;

	pc2400m_netlink_close();
 out1:	
	return ret;
	
}

/**
 * pc2400m_if_setup - if UP from host
 * @dev: pointer to network device 
 *
 */
static void pc2400m_if_setup(struct net_device *dev)
{

	struct net_local *nl;

	nl = netdev_priv(dev);	
	
	/* default ethernet values */
	ether_setup(dev);

	/* kernel interface */
	dev->get_stats          = pc2400m_if_get_stats;
	dev->hard_start_xmit    = pc2400m_if_xmit;
	dev->hard_header_cache  = NULL; /* no ARP caching */
	dev->open               = pc2400m_if_open;
	dev->stop               = pc2400m_if_stop;
	dev->hard_header        = NULL; /* No ARP => must be implemented */

	
	/* wireless extensions */
	dev->wireless_handlers  = (struct iw_handler_def *) 
				    &pc2400m_we_handler_def;
	
	/* no ARP, and no hardware address caching */
	dev->flags             |= IFF_NOARP;
	
	/* broadcasting is not possible (no hardware addresses) */
	dev->flags             &= ~IFF_BROADCAST;
	
	/* multicasting is not possible (no hardware addresses) */
	dev->flags             &= ~IFF_MULTICAST;
	
	/* turn of debugging by default */
	dev->flags             &= ~IFF_DEBUG;
	
	/* set hardware MAC address */
	memset(dev->dev_addr,0,ETH_ALEN);
	
	/* interface status tracking */
	dev->trans_start       = 0;
	dev->last_rx           = 0;
	
}

/**
 * pc2400m_if_new_device - receive an indication from the WiMAX driver
 *
 * this used to set handle to pc2400m drv
 */
struct net_device *pc2400m_if_new_device(void)
{
	
	struct net_device *dev_pc2400m;
	struct net_local *nl;

	dev_pc2400m = alloc_netdev(sizeof(struct net_local), "wimax%d", 
				 pc2400m_if_setup);
	
	if (!dev_pc2400m) {
		
		ERROR("Failed to init net if (out of memory).\n");
		goto err2;
		
	}

	if (register_netdev(dev_pc2400m)) {
		ERROR("Failed to register net if.\n");
		goto err1;
	}		
		
	nl = netdev_priv(dev_pc2400m);
	nl->net_dev = dev_pc2400m;
	nl->if_status = UNINITIALIZED;

	nl->pt_len = 0;
	nl->pt_buffer = NULL;
	nl->tx_buff = NULL;

	spin_lock_init(&(nl->tx_lock));
	nl->tx_flags = 0;
	nl->nap = NO_NAP;

	goto out;

 err1:
	free_netdev(dev_pc2400m);
 err2:
	dev_pc2400m = NULL;

 out:

	return dev_pc2400m;
	
}

/**
 * pc2400m_if_init_device - receive an indication from the WiMAX driver
 *
 */
int pc2400m_if_init_device(struct net_device *dev_pc2400m)
{

	struct net_local *nl;
	int ret = 0;

	nl = netdev_priv(dev_pc2400m);
	nl->if_status = UNINITIALIZED;

	/* initialize sysfs */
	pc2400m_if_sysfs_create_files(&(nl->spi_dev->dev));

	return ret;
}

/**
 * pc2400m_if_free_device - receive an indication from the WiMAX driver
 *
 */
void pc2400m_if_free_device(struct net_device *dev_pc2400m)
{
	
	struct net_local *nl;

	nl = netdev_priv(dev_pc2400m);

	pc2400m_if_sysfs_uncreate_files(&(nl->spi_dev->dev));

	if(nl->pt_buffer) kfree(nl->pt_buffer);
	kfree(nl->event_stream);

  	unregister_netdev(dev_pc2400m);
	free_netdev(dev_pc2400m);
	
} 

/**
 * pc2400m_if_suspend_device - suspend the specified device
 *
 */
void pc2400m_if_suspend_device(struct net_device *dev_pc2400m)
{
	
	struct drv_cmd cmd;
	WIMAX_WAIT_RSP;

	cmd.cmd_id = E_SUSPEND;
	WIMAX_WAIT_CMD(dev_pc2400m, &cmd);
		
} 


/**
 * pc2400m_if_resume_device - resume the specified device
 *
 */
void pc2400m_if_resume_device(struct net_device *dev_pc2400m)
{
	
	struct drv_cmd cmd;
	WIMAX_WAIT_RSP;

	cmd.cmd_id = E_RESUME;
	WIMAX_WAIT_CMD(dev_pc2400m, &cmd);
	
} 

