/*
 * pc2400m_drv_sdio.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */



#include "pc2400m_al.h"
#include "pc2400m_drv_spi.h"
#include "pc2400m_drv_sdio.h"
#include "pc2400m_osal.h"

/**
 * pc2400m_drv_crc7 - calculate a 7 bit crc value
 * @src: pointer to data for which to calculate the crc
 * @cnt: amount of data at src
 *
 * Calculate and return a 7 bit crc for a data buffer
 */
static u8 pc2400m_drv_crc7(const u8* src, s32 cnt) 
{ 
	s32 i, a; 
	u8 crc, data; 

	crc=0; 
	for (a=0; a<cnt; a++) { 
		data=src[a]; 
		for (i=0; i<8; i++) { 
			crc <<= 1; 
			if ((data & 0x80)^(crc & 0x80)) 
				crc ^= 0x09; 
			data <<= 1; 
		} 
	} 
	crc = (crc<<1) | 1; 
	return crc; 
} 

/**
 * pc2400m_drv_build_cmd0 - create an SDIO command 0
 * @dst: buffer to hold the command 0
 *
 * Create a SDIO cmd0 into buffer provided. Cmd0 is to bring SDIO
 * devices to an initial state.
 */
static void pc2400m_drv_build_cmd0(void* dst)
{
	struct SD_CMD *sd_cmd = (struct SD_CMD*)dst;

	sd_cmd->start_bit      = 0;
	sd_cmd->direction      = 1;
	sd_cmd->command_index  = 0;
	sd_cmd->argument       = 0;
	sd_cmd->crc7	       = 0x4A;
	sd_cmd->end_bit	       = 1;
	return;
}

/**
 * pc2400m_drv_send_cmd0 - create and send an SDIO command 0
 * @ctx: handle for the physical device to which to send the cmd0
 *
 * Create a SDIO cmd0 and send it to the physical device specified
 * by the context.
 */
s32 pc2400m_drv_send_cmd0(struct pc2400m_drv_hdi_if *hdi)
{
	u8 cmd[PC2400M_CMD_SIZE];              /* 8 bytes */
        struct R_1 r1;                         /* 1+1 bytes */

        pc2400m_drv_build_cmd0(&cmd[1]);
	cmd[0] = 0xff;
	cmd[PC2400M_CMD_SIZE-1] = 0xff;

	/* queue write of the request */
	pc2400m_drv_adapt.h2d(hdi->ctx, cmd, PC2400M_CMD_SIZE);
	/* queue read of the response */
	pc2400m_drv_adapt.d2h(hdi->ctx, (u8*)&r1, sizeof(struct R_1));
	/* commit the transaction */
	pc2400m_drv_adapt.commit(hdi->ctx, NULL, NULL);

	if (r1.u.as_u8 != 1) {
		wimax_osal_trace_byte(PC2400M_GROUP_SPI_ERROR, 
				      PC2400M_TRACE_CMD0_ERROR, 
				      WIMAX_OSAL_PRIORITY_ERROR, r1.u.as_u8);
		return -EIO;
	}

        return ESABABA;

}

/**
 * pc2400m_drv_build_cmd5 - create an SDIO command 5
 * @dst: buffer to hold the command 5
 *
 * Create a SDIO cmd5 into buffer provided. Cmd5 is for querying and specifying
 * the bus operating conditions.
 */
static void pc2400m_drv_build_cmd5(void* dst, u32 ocr)
{
	struct SD_CMD *sd_cmd = (struct SD_CMD *)dst;
	
	ocr &= 0x00ffffff;
	ocr = WIMAX_OSAL_U32_TO_BE(ocr);

	sd_cmd->start_bit     = 0;
	sd_cmd->direction     = 1;
	sd_cmd->command_index = 5;
	sd_cmd->argument      = ocr;
	sd_cmd->crc7	      = 0;
	sd_cmd->end_bit	      = 1;
	return;
}

/**
 * pc2400m_drv_send_cmd5 - create and send an SDIO command 5
 * @ctx: handle for the physical device to which to send the cmd5
 *
 * Create a SDIO cmd5 and send it to the physical device specified
 * by the context.
 */
u32 pc2400m_drv_send_cmd5(struct pc2400m_drv_hdi_if *hdi, 
			  u32 ocr, 
			  struct R_4* r4)
{

	u8 cmd[PC2400M_CMD_SIZE]; /* 8 bytes */

        pc2400m_drv_build_cmd5(&cmd[1], ocr);
	cmd[0] = 0xff;
	cmd[PC2400M_CMD_SIZE-1] = 0xff;

	/* queue write of the request */
	pc2400m_drv_adapt.h2d(hdi->ctx, cmd, PC2400M_CMD_SIZE);
	/* queue read of the response */
	pc2400m_drv_adapt.d2h(hdi->ctx, (u8*)r4, sizeof(struct R_4));
	/* commit the transaction */
	pc2400m_drv_adapt.commit(hdi->ctx, NULL, NULL);

	return ((r4->u.bits.io_ocr2<<16) & 0xFF0000) + 
		((r4->u.bits.io_ocr1<<8) & 0x00FF00) + 
		(r4->u.bits.io_ocr0 & 0x0000FF);

}

/**
 * pc2400m_drv_build_cmd52 - create an SDIO command 52
 * @dst: buffer to hold the command 52
 *
 * Create a SDIO cmd52 into buffer provided. Cmd5 is for writing or reading
 * one byte of data from a specific address.
 */
static void pc2400m_drv_build_cmd52(void* dst, 
				    unsigned address,
				    u8 data,
				    e_read_write read_write,
				    e_func_num func_num)
{
	struct SD_CMD *sd_cmd = (struct SD_CMD*)dst;
	struct CMD_52 cmd52;
	unsigned crc;
	
	cmd52.u.bits.address          = address;
	cmd52.u.bits.data             = data;
	cmd52.u.bits.function         = (unsigned)func_num;
	cmd52.u.bits.read_after_write = FALSE;
	cmd52.u.bits.reserved1	      = 0;
	cmd52.u.bits.reserved2	      = 0;
	cmd52.u.bits.write_to_device  = (u8)read_write;
	
	crc = pc2400m_drv_crc7((u8*)(&data), sizeof(data));
	
	sd_cmd->start_bit             = 0;
	sd_cmd->direction	      = 1;
	sd_cmd->command_index	      = 52;
	sd_cmd->argument              = WIMAX_OSAL_U32_TO_BE(cmd52.u.as_u32);
	sd_cmd->crc7		      = crc;
	sd_cmd->end_bit		      = 1;
	return;

}

/**
 * pc2400m_drv_send_cmd52 - create and send an SDIO command 52
 * @ctx: handle for the physical device to which to send the cmd52
 *
 * Create a SDIO cmd52 and send it to the physical device specified
 * by the context.
 */
s32 pc2400m_drv_send_cmd52(struct pc2400m_drv_hdi_if *hdi,
			   unsigned address,
			   u8 data,
			   e_read_write	read_write,
			   e_func_num func_num)
{
	u8 cmd[PC2400M_CMD_SIZE]; /* 8 bytes */
        struct R_5 r5;            /* 2 bytes */

        pc2400m_drv_build_cmd52(&cmd[1], address, data, read_write, func_num);
	cmd[0] = 0xff;
	cmd[PC2400M_CMD_SIZE-1] = 0xff;

	/* queue write of the request */
	pc2400m_drv_adapt.h2d(hdi->ctx, cmd, PC2400M_CMD_SIZE);
	/* queue read of the response */
	pc2400m_drv_adapt.d2h(hdi->ctx, (u8*)&r5, sizeof(struct R_5));
	/* commit the transaction */
	pc2400m_drv_adapt.commit(hdi->ctx, NULL, NULL);

	if (r5.u.bits.r1.u.as_u8 != 0) {
		wimax_osal_trace_byte(
			PC2400M_GROUP_SPI_ERROR, 
			PC2400M_TRACE_CMD52_ERROR, 
			WIMAX_OSAL_PRIORITY_ERROR, r5.u.bits.r1.u.as_u8);
		goto err;
	}
	if (read_write == h2d && r5.u.bits.read_data != data) {
		wimax_osal_trace_byte(
			PC2400M_GROUP_SPI_ERROR, 
			PC2400M_TRACE_CMD52_ERROR, 
			WIMAX_OSAL_PRIORITY_ERROR, r5.u.bits.read_data);
		goto err;
	}

	goto out;

 err:
	return -EIO;

 out:
        return r5.u.bits.read_data;

}

/**
 * pc2400m_drv_build_cmd53 - create an SDIO command 53
 * @dst: buffer to hold the command 53
 *
 * Create a SDIO cmd53 into buffer provided. Cmd5 is for writing or reading
 * large amounts of data from/to specific addresses.
 */
static void pc2400m_drv_build_cmd53(void* dst,
				    unsigned byte_block_count,
				    unsigned address,
				    e_read_write read_write)
{
	struct SD_CMD *sd_cmd = (struct SD_CMD*)dst;
	struct CMD_53 cmd53;
	
	cmd53.u.bits.count           = byte_block_count;
	cmd53.u.bits.address         = address;
	cmd53.u.bits.opcode	     = TRUE;		
	cmd53.u.bits.blockmode	     = TRUE;
	cmd53.u.bits.function	     = 1;
	cmd53.u.bits.write_to_device = read_write;

	sd_cmd->start_bit            = 0;
	sd_cmd->direction	     = 1;
	sd_cmd->command_index	     = 53;
	sd_cmd->argument	     = WIMAX_OSAL_U32_TO_BE(cmd53.u.as_u32);
	sd_cmd->crc7		     = 0;
	sd_cmd->end_bit		     = 1;

	return;
}

/**
 * pc2400m_drv_send_write_cmd53 - create an SDIO command 53
 * @ctx: handle to the device instance
 * @byte_block_count: number of blocks to transmit
 * @address: target write address
 * @data: buffer holding data to transmit
 * @len: length of the data to transmit
 *
 * Send a write SDIO CMD53 to the SPI bus, and transmit the
 * accompanied buffer as data.
 */
s32 pc2400m_drv_send_write_cmd53(struct pc2400m_drv_hdi_if *hdi,
				 unsigned byte_block_count,
				 unsigned address,
				 u8 *data, 
				 u32 len)
{

	u8 cmd53[PC2400M_CMD_SIZE];          /* 8 bytes */
        struct R_5 r5;                       /* 4 bytes */
	u32 ret;

        /* queue a write command */
        pc2400m_drv_build_cmd53(&cmd53[1], byte_block_count, address, h2d);
	cmd53[0] = 0xff;
	cmd53[PC2400M_CMD_SIZE-1] = 0xff;
        pc2400m_drv_adapt.h2d(hdi->ctx, cmd53, PC2400M_CMD_SIZE);

        /* queue the response */
        pc2400m_drv_adapt.d2h(hdi->ctx, (u8*)&r5, sizeof(struct R_5));

	/* queue the data */
	wimax_osal_assert(len <= PC2400M_MAX_TRANSACTION);
	pc2400m_drv_adapt.h2d(hdi->ctx, data, len);
	
        /* commit the entire transaction synchronously */
        pc2400m_drv_adapt.commit(hdi->ctx, NULL, NULL);

        /* check on the response to the cmd53 */
	ret = len;
        if (r5.u.bits.read_data != 0 || r5.u.bits.r1.u.as_u8 != 0 ) {
	        wimax_osal_trace_u16(PC2400M_GROUP_SPI_ERROR, 
				     PC2400M_TRACE_TX_CMD_FAIL, 
				     WIMAX_OSAL_PRIORITY_ERROR, r5.u.as_u16);
                ret = -EIO; /* I/O error */
        }

	return ret;

}



/**
 * pc2400m_drv_send_read_cmd53 - create an SDIO command 53
 * @ctx: handle to the device instance
 * @byte_block_count: number of blocks to transmit
 * @address: target write address
 * @data: buffer to store the received data
 * @len: length of the data to receive
 *
 * Send a read SDIO CMD53 to the SPI bus, and receive the
 * data to the accompanied buffer.
 */
s32 pc2400m_drv_send_read_cmd53(struct pc2400m_drv_hdi_if *hdi,
				unsigned byte_block_count,
				unsigned address,
				u8 *data, 
				u32 len)
{

	u8 cmd53[PC2400M_CMD_SIZE];          /* 8 bytes */
        struct R_5 *r5;                      /* 4 bytes */
	u32 ret;

	/* queue a data read command (cmd53) */
        pc2400m_drv_build_cmd53(&cmd53[1], byte_block_count, 0, d2h);
	cmd53[0] = 0xff;
	cmd53[PC2400M_CMD_SIZE-1] = 0xff;
        pc2400m_drv_adapt.h2d(hdi->ctx, cmd53, PC2400M_CMD_SIZE);

	/* queue the read (including the R5 response) */	
	pc2400m_drv_adapt.d2h(hdi->ctx, data, len + R_5_SIZE);

	/* asynchronously commit the transaction */
        pc2400m_drv_adapt.commit(hdi->ctx, NULL, NULL);

	/* check the response for cmd53 */
	ret = len;
	r5 = (struct R_5*)((struct pc2400m_drv_hdi_spi_if*)hdi)->spi_data;
	if (r5->u.bits.read_data != 0 || r5->u.bits.r1.u.as_u8) {
		wimax_osal_trace_byte(
			PC2400M_GROUP_SPI_ERROR, 
			PC2400M_TRACE_RX_CMD_FAIL, 
			WIMAX_OSAL_PRIORITY_ERROR, r5->u.bits.r1.u.as_u8);
		ret = -EIO;
	}

	return ret;

}
