/*
 * pc2400m_drv_if.h
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#ifndef PC2400M_DRV_IF
#define PC2400M_DRV_IF

#include "pc2400m_osal.h"

/**

     INTERFACE IDIOMS
     ----------------

      1. Concurrent access to the operands is prohibited; all access must
         be synchronized.

      2. All access to the operands must be in sleep-able process context.

      3. All responses through this interface can be assumed to be in sleep-
         able process context and concurrency protected.

      4. Ownership of the message structures remain with the caller - the 
         callee may assume the data is valid only for the duration of the
         invoked function call. Exceptions are listed for each operand 
         individually.

         This rule includes all sub pointers pointed to by the main operand
         parameter structures.

*/

/* 
 * bitfield definitions 
 */
/** single EAP authentication */
#define DRV_AUTH_POLICY_SINGLE_EAP   0x01
/** double EAP authentication */
#define DRV_AUTH_POLICY_DOUBLE_EAP   0x02
/** authentication policy bitfield */
typedef u8 drv_auth_policy;

/** band class index 1 */
#define DRV_BAND_CLASS_INDEX_1     0x0001
/** band class index 2 */
#define DRV_BAND_CLASS_INDEX_2     0x0002
/** band class index 3 */
#define DRV_BAND_CLASS_INDEX_3     0x0004
/** band class index 4 */
#define DRV_BAND_CLASS_INDEX_4     0x0008
/** band class index 5 */
#define DRV_BAND_CLASS_INDEX_5     0x0010
/** band class index bitfield */
typedef u16 drv_band_class_index;

/** IPv4 convergence sublayer */
#define DRV_CS_IPV4                  0x01
/** IPv6 convergence sublayer */
#define DRV_CS_IPV6                  0x02
/** IP convergence sublayer support bitfield */
typedef u8 drv_cs;

/** request no NSP info */
#define DRV_ENTRY_NSP_INFO_REQ_NONE  0x00
/** request NSP verbose name info */
#define DRV_ENTRY_NSP_INFO_REQ_NAME  0x01
/** request NSP realm name info */
#define DRV_ENTRY_NSP_INFO_REQ_REALM 0x02
/** NSP information request bitfield */
typedef u8 drv_entry_nsp_info_req;

/*
 * boolean definitions 
 */
/** boolean value TRUE */
#define DRV_TRUE                     1
/** boolean value FALSE */
#define DRV_FALSE                    0
/** boolean datatype */
typedef u8 drv_bool;

/** EDataSend fixed credit size */
#define DRV_DATA_SEND_CREDIT_LEN     10


/*
 * enumerations types 
 */


/** common operation result status codes */
enum drv_status {
        E_SUCCESS = 1,                     /**< operation successfull */
        E_ERR_AUTHENTICATION,              /**< authentication failed */
        E_ERR_CONNECTION_LOST,             /**< connection lost */
        E_ERR_FAILURE,                     /**< generic failure */
        E_ERR_INVALID_STATE,               /**< invalid state for operation */
        E_ERR_INVALID_METHOD,              /**< invalid method used */
        E_ERR_INVALID_PARAMETERS,          /**< invalid parameters for 
                                            *   operation */
        E_ERR_OPERATION_PENDING,           /**< operation is pending */
        E_ERR_TIMEOUT,                     /**< operation timed out */
        E_ERR_BUFFER_FULL,                 /**< no more space in buffer */
        E_ERR_NO_CONNECTION,               /**< no connection */
        E_ERR_RETRY_EXCEEDED,              /**< retried maximum number of 
                                            *   times */
        E_ERR_DEVICE_DROPPED_PACKET,       /**< packet was dropped */
};

/** network/connection status codes */
enum drv_network_status {
        E_NETWORK_STATUS_3WAY_HANDSHAKE = 1,     /**< 3-way handshake */
        E_NETWORK_STATUS_CAPABILITY_NEGOTIATION, /**< capability negotiation */
        E_NETWORK_STATUS_CONNECTED,              /**< connected */
        E_NETWORK_STATUS_DISCONNECTED,           /**< disconnected */
        E_NETWORK_STATUS_HANDOVER,               /**< handover performed */
        E_NETWORK_STATUS_DO_HANDOVER,            /**< handover needed */
        E_NETWORK_STATUS_LOST,                   /**< connection lost */
        E_NETWORK_STATUS_RECONNECTED,            /**< connection resumed */
        E_NETWORK_STATUS_RANGING,                /**< ranging ongoing */
        E_NETWORK_STATUS_AUTHENTICATION,         /**< authentication ongoing */
        E_NETWORK_STATUS_REGISTERING,            /**< registering to nw */
        E_NETWORK_STATUS_SF,                     /**< negotiating service 
                                                  *   flows */
};

/** system state codes */
enum drv_system_state {
        E_SYSTEM_STATE_FATAL_ERROR = 1,          /**< fatal error, reinit 
                                                  *   needed */
};

/** radio state codes */
enum drv_radio_state {
        DRV_RADIO_STATE_OFF = 0,                 /**< radio off */
        DRV_RADIO_STATE_ON,                      /**< radio on */
};

/** channel bandwidth codes */
enum drv_channel_bandwidth {
        DRV_CHANNEL_BANDWIDTH_NA = 0x00,         /**< bandwidth info not 
                                                  *   available */
        DRV_CHANNEL_BANDWIDTH_3_5MHZ,            /**< bandwidth 3.5MHz */
        DRV_CHANNEL_BANDWIDTH_5MHZ,              /**< bandwidth 5MHz */
        DRV_CHANNEL_BANDWIDTH_7MHZ,              /**< bandwidth 7MHz */
        DRV_CHANNEL_BANDWIDTH_8_75MHZ,           /**< bandwidth 8.75MHz */
        DRV_CHANNEL_BANDWIDTH_10MHZ,             /**< bandwidth 10MHz */
};

/** search type codes */
enum drv_search_type {
        DRV_SEARCH_TYPE_NORMAL = 1,              /**< normal search */
        DRV_SEARCH_TYPE_NEIGHBORS,               /**< search for neighbors */
        DRV_SEARCH_TYPE_HANDOVER,                /**< search for handover */
};

/** fft size codes */
enum drv_fft_size {
        DRV_FFT_SIZE_NA = 0x00,                  /**< fft size not available */
        DRV_FFT_SIZE_512,                        /**< fft size 512 */
        DRV_FFT_SIZE_1024,                       /**< fft size 1024 */
};

/** service level prediction codes */
enum drv_service_level_prediction {
        DRV_SERVICE_LEVEL_PREDICTION_NA = 0,     /**< no service level 
                                                  *   prediction */
        DRV_SERVICE_LEVEL_PREDICTION_NONE,       /**< no service predicted */
        /** some services predicted */
        DRV_SERVICE_LEVEL_PREDICTION_SOME,
        DRV_SERVICE_LEVEL_PREDICTION_ALL,        /**< all services predicted */
};

/** search type codes */
enum drv_resp_search_type {
        DRV_RESP_SEARCH_TYPE_NETWORK = 1,        /**< new networks found */
        DRV_RESP_SEARCH_TYPE_PROGRESS,           /**< search progressed */
};

/** handover scheme codes */
enum drv_handover_scheme {
        DRV_HANDOVER_SCHEME_HOST = 1,            /**< host controlled ho's */
        DRV_HANDOVER_SCHEME_NAP_LOCK,            /**< NAP id locked */
        DRV_HANDOVER_SCHEME_BS_LOCK,             /**< BS id locked */
};

/** confirmation codes for service flow operations */
enum drv_confirmation_code {
        DRV_E_SUCCESS = 1,                       /**< SF operation success */
        DRV_E_REJECT_OTHER,                      /**< other */
        DRV_E_REJECT_UNREGOCNIZED_CONFIG_SETTING,/**< illegal configuration 
                                                  *   setting */
        DRV_E_REJECT_TEMPORARY_RESOURCE,         /**< try again later */
        DRV_E_REJECT_PERMANENT_ADMIN,            /**< permanent admin */
        DRV_E_REJECT_NOT_OWNER,                  /**< not owner */
        DRV_E_REJECT_SERVICE_FLOW_NOT_FOUND,     /**< SF not found */
        DRV_E_REJECT_SERVICE_FLOW_EXISTS,        /**< SF already exists */
        /** parameter missing */
        DRV_E_REJECT_REQUIRED_PARAMETER_NOT_PRESENT,
        DRV_E_REJECT_HEADER_SUPPRESSION,         /**< header suppression */
        DRV_E_REJECT_UNKNOWN_TRANSACTION_ID,     /**< unknown transaction id */
        DRV_E_REJECT_AUTH_FAILURE,               /**< authentication failure */
        DRV_E_REJECT_ADD_ABORTED,                /**< add aborted */
        /** exceeded dynamic service limit */
        DRV_E_REJECT_EXCEEDED_DYNAMIC_SERVICE_LIMIT,
        /** not authenticated for requested SAID */
        DRV_E_REJECT_NOT_AUTHED_FOR_REQUESTED_SAID,
        /** failed to establish requested SA */
        DRV_E_REJECT_FAIL_TO_ESTABLISH_REQUESTED_SA,
        /** parameter not supported */
        DRV_E_REJECT_NOT_SUPPORTED_PARAM,
        DRV_E_REJECT_NOT_SUPPORTED_PARAM_VALUE,  /**< parameter value not
                                                  *   supported */
};

/** search result NSP information amount codes */
enum drv_search_nsp_info_req {
        DRV_SEARCH_NSP_INFO_REQ_NONE = 0,        /**< search for no NSP info */
        DRV_SEARCH_NSP_INFO_REQ_NSP_LIST,        /**< search for NSP list */
        /** search for NSP list and associated names */
        DRV_SEARCH_NSP_INFO_REQ_NSP_LIST_AND_NAMES,
};



/* 
 * standard types 
 */

#define DRV_BSID_LENGTH   6                      /**< length of the BSid */
typedef u8 drv_bsid[DRV_BSID_LENGTH];            /**< 48 bit BS identifier */
typedef u8  drv_cinr;                            /**< CINR in dB, range -10db-
                                                  *   53db, encoded as 0x00-
                                                  *   0x3f */
typedef wimax_osal_packet* drv_data_block;       /**< data packet */
typedef u32 drv_frequency;                       /**< frequency in kHz */
typedef u8 drv_idle_mode_retain_info;            /**< idle mode retain info */
typedef u32 drv_nap_id;                          /**< 24bit NAP id (MSB always
                                                  *   zero */
typedef u32 drv_network_info_periodic;           /**< period for status 
                                                  *   messages in milliseconds
                                                  */
typedef u8 drv_network_info_signal_change;       /**< signal strength delta in
                                                  *   dBm causing status 
                                                  *   message */
typedef u32 drv_nsp_id;                          /**< 24bit NSP id (MSB always
                                                  *   zero */
#define DRV_PREAMBLE_SIZE 15                     /**< length of the preamble */
typedef u8 drv_preamble[DRV_PREAMBLE_SIZE];      /**< 114 bit preamble, 
                                                  *   remaining bits zero */
typedef u32 drv_power;                           /**< tx power range -40dBm-
                                                  *   25dBm, represented as 
                                                  *   0x00-0x41 */
typedef void* drv_service_flow_params;           /**< TLV encoded service flow
                                                  *   parameters */
typedef u8 drv_signal_strength;                  /**< signal strength range
                                                  *   -40-(-123) represented as
                                                  *   40-123 */
typedef u8 drv_string;                           /**< string character type */
typedef u8 drv_transport_id;                     /**< transport id, may map 
                                                  *   e.g. to service flow */
typedef u16 drv_uint16;                          /**< generic 16 bit unsigned
                                                  *   integer */
typedef u32 drv_uint32;                          /**< generic 32 bit unsigned
                                                  *   integer */
typedef u8 drv_uint8;                            /**< generic 8 bit unsigned
                                                  *   integer */

/* 
 * message enumerations 
 */

/** interface command opcodes */
enum drv_command_id {
        E_DATA_SEND = 1,                         /**< EDataSend */
        E_DATA_RECEIVE,                          /**< EDataReceive */
        /* E_OPEN is implemented as a function call */
        E_INITIALIZE = 4,                        /**< EInitialize */
        E_CONFIGURE,                             /**< EConfigure */
        E_CLOSE,                                 /**< EClose */
        E_RADIO_STATE,                           /**< ERadioState */
        E_SEARCH,                                /**< ESearch */
        E_SEARCH_STOP,                           /**< ESearchStop */
        E_ENTRY,                                 /**< EEntry */
        E_EXIT,                                  /**< EExit */
        E_NETWORK_STATUS,                        /**< ENetworkStatus */
        E_EAP_DATA,                              /**< EEapData */
        E_EAP_RESULT,                            /**< EEapResult */
        E_FLOW_CREATE_MS,                        /**< EFlowCreateMS */
        E_FLOW_CREATE_BS,                        /**< EFlowCreateBS */
        E_FLOW_DELETE_MS,                        /**< EFlowDeleteMS */
        E_FLOW_DELETE_BS,                        /**< EFlowDeleteBS */
        E_FLOW_CHANGE_MS,                        /**< EFlowChangeMS */
        E_FLOW_CHANGE_BS,                        /**< EFlowChangeBS */
        E_TRACE,                                 /**< ETrace */
        E_STATISTICS_GET,                        /**< EStatisticsGet */
        E_PROD_TEST,                             /**< EProdTest */
	E_SUSPEND,                               /**< ESuspend */
	E_RESUME,                                /**< EResume */
};

/** interface response opcodes */
enum drv_command_response_id {
        E_DATA_SEND_RESP = 1,                    /**< ESendDataResponse */
        E_DATA_RECEIVE_RESP,                     /**< EReceiveDataResponse */
        /* E_OPEN_RESP implemented as a function call */
        E_INITIALIZE_RESP = 4,                   /**< EInitializeResponse */
        E_CONFIGURE_RESP,                        /**< EConfigureResponse */
        E_CLOSE_RESP,                            /**< ECloseResponse */
        E_RADIO_STATE_RESP,                      /**< ERadioStateResponse */
        E_SEARCH_RESP,                           /**< ESearchResponse */
        E_SEARCH_STOP_RESP,                      /**< ESearchStopResponse */
        E_ENTRY_RESP,                            /**< EEntryResponse */
        E_EXIT_RESP,                             /**< EExitResponse */
        E_NETWORK_STATUS_RESP,                   /**< ENetworkStatusResponse */
        E_EAP_DATA_RESP,                         /**< EEapDataResponse */
        E_EAP_RESULT_RESP,                       /**< EEapResultResponse */
        E_FLOW_CREATE_MS_RESP,                   /**< EFlowCreateMSResponse */
        E_FLOW_CREATE_BS_RESP,                   /**< EFlowCreateBSResponse */
        E_FLOW_DELETE_MS_RESP,                   /**< EFlowDeleteMSResponse */
        E_FLOW_DELETE_BS_RESP,                   /**< EFlowDeleteBSResponse */
        E_FLOW_CHANGE_MS_RESP,                   /**< EFlowChangeMSResponse */
        E_FLOW_CHANGE_BS_RESP,                   /**< EFlowChangeBSResponse */
        E_TRACE_RESP,                            /**< ETraceResponse */
        E_STATISTICS_GET_RESP,                   /**< EStatisticsGetResponse */
        E_PROD_TEST_RESP,                        /**< EProdTestResponse */
	E_SUSPEND_RESP,                          /**< ESuspendResp */
	E_RESUME_RESP,                           /**< EResumeResp */
};      

/** interface indication opcodes */
enum drv_indication_id {
        E_DATA_RECV_IND = 1,                     /**< EDataReceiveIndication */
        E_SYSTEM_STATE_IND,                      /**< ESystemStateIndication */
	E_SEARCH_IND,                            /**< ESearchIndication */
        /** ENetworkStatusIndication */
        E_NETWORK_STATUS_IND,
        E_EAP_DATA_IND,                          /**< EEapDataIndication */
        /** EFlowCreateBSIndication */
        E_FLOW_CREATE_BS_IND,
        /** EFlowDeleteBSIndication */
        E_FLOW_DELETE_BS_IND,
        /** EFlowChangeBSIndication */
        E_FLOW_CHANGE_BS_IND,
        E_TRACE_IND,                             /**< ETraceIndication */
        E_PROD_TEST_IND,                         /**< EProdTestIndication */
};      

/** one firmware data block with associated length */
struct drv_fw_data_block {
        /** length of the firmware block pointed to by ptr */
        drv_uint32                     length;
        /** pointer to array of bytes containing firmware block */
        void*                          ptr;
};

/** E_INITIALIZE command parameters */
struct drv_cmd_initialize {
        /** number of blocks in the p_blocks list */
        drv_uint32                     block_count;
        /** pointer to list of firmware blocks */
        struct drv_fw_data_block       *p_blocks;
};

/** E_CONFIGURE command parameters */
struct drv_cmd_configure {
        /** band class index */
        drv_band_class_index           band_class_index;
        /** delta for signal strength to trigger network status ind */
        drv_network_info_signal_change network_info_change_treshold;
        /** interval for periodic network status ind, zero = never */
        drv_network_info_periodic      network_info_periodic_interval;
        /** signal lost time threshold in milliseconds */
        drv_uint32                     signal_lost_time_treshold;
        /** signal recovery time in milliseconds */
        drv_uint32                     signal_recovery_time_limit;
        /** signal recovered indication delay in milliseconds */
        drv_uint32                     signal_recovered_ind_delay;
        /** authentication policy */
        drv_auth_policy                auth_policy;
        /** convergence sublayer support */
        drv_cs                         cs_support;
        /** length of the p_vendor_info byte array */
        drv_uint8                      vendor_info_len;
        /** pointer to array of bytes containing vendor info */
        drv_uint8                      *p_vendor_info;
};

/** capabilities structure for E_CONFIGURE response */
struct drv_capabilities {
#define DRV_VERSION_STR_LEN  20                  /**< length of the version
                                                  *   strings */
        /** null terminated array of characters describing firmware version */
        drv_string                     p_fw_version[DRV_VERSION_STR_LEN];
        /** null terminated array of characters describing hardware version */
        drv_string                     p_hw_version[DRV_VERSION_STR_LEN];
#define DRV_MAC_ADDRESS_LEN  6                   /**< length of the MAC
                                                  *   address */
        /** MAC address of the device */
        drv_uint8                      mac_address[DRV_MAC_ADDRESS_LEN];
};

/** E_CONFIGURE response parameters */
struct drv_resp_configure {
        /** device capabilities structure */
        struct drv_capabilities        capabilities;
};

/** E_RADIO_STATE command parameters */
struct drv_cmd_radio_state {
        /** radio state */
        enum drv_radio_state           radio_state;
};

/** search plan structure for the E_SEARCH command */
struct drv_search_plan {
        /** band class index */
        drv_band_class_index           band_class_index;
        /** carrier center frequency */
        drv_frequency                  carrier_freq;
        /** maximum tx power */
        drv_power                      max_tx_power;
        /** channel bandwidth */
        enum drv_channel_bandwidth     channel_bandwidth;
        /** fast fourier transform size */
        enum drv_fft_size              fft_size;
        /** preambles */
        drv_preamble                   preamble;
};

/** known NAP info structure for the E_SEARCH command */
struct drv_known_nap_info {
        /** NAP identifier */
        drv_nap_id                     nap_id;
        /** NSP change count for NAP */
        drv_uint8                      nsp_change_count;
};

/** E_SEARCH command parameters */
struct drv_cmd_search {
        /** length of the p_search_plan list */
        drv_uint8                      search_plan_count;
        /** length of the p_known_nap_info list */
        drv_uint8                      known_nap_info_count;
        /** search progress percentage interval, zero = no progress reports */
        drv_uint8                      search_progress;
        /** search type */
        enum drv_search_type           search_type;
        /** type of NSP information to include in search */
        enum drv_search_nsp_info_req   search_nsp_info_req;
        /** pointer to a list with search plan definitions */
        struct drv_search_plan         *p_search_plan;
        /** pointer to a list with known nap information */
        struct drv_known_nap_info      *p_known_nap_info;
};

/** nsp id and verbose nsp name information for the drv_search_result 
    structure */
struct drv_nsp {
        /** NSP identifier */
        drv_nsp_id                     nsp_id;
        /** length of character array pointed to by p_nsp_name */
        drv_uint8                      nsp_name_len;
        /** array of characters containing name of NSP */
        drv_string                     *p_nsp_name;
};

/** search result element for the E_SEARCH response */
struct drv_search_result {
        /** service level prediction of result */
        enum drv_service_level_prediction service_level_prediction;
        /** NAP identifier */
        drv_nap_id                     nap_id;
        /** signal strength of best BS */
        drv_signal_strength            signal_strength;
        /** cinr for best BS */
        drv_cinr                       cinr;
        /** number of elements in list pointed to by p_nsp */
        drv_uint8                      number_of_nsp_id;
        /** pointer to array of NSP descriptor structures */
        struct drv_nsp                 *p_nsp;
};

/** E_SEARCH response parameters */
struct drv_ind_search {
        /** search response type (new result or progress) */
        enum drv_resp_search_type      response_type;
        /** pointer to array of search result elements */
        struct drv_search_result       *p_search_result;
        /** number of elements in array pointed to by p_search_result */
        drv_uint8                      result_count;
        /** progress of search in percentage if requested */
        drv_uint8                      search_progress;
};

/* E_ENTRY command parameters */
struct drv_cmd_entry {
        /** NAP identifier to connect to */
        drv_nap_id                     nap_id;
        /** NSP identifier to connect to */
        drv_nsp_id                     nsp_id;
        /** handover scheme to use */
        enum drv_handover_scheme       handover_scheme;
        /** BS id to connect to */
        drv_bsid                       bsid;
        /** type of information needed about the connected NSP */
        drv_entry_nsp_info_req         nsp_info_req;
};

/* E_DATA_SEND command parameters */
struct drv_cmd_data_send {
        /** transport identifier for packet transmission */
        drv_transport_id               transport_id;
        /** length of the array of packets pointed to by p_data */
        drv_uint16                     data_length;
        /** pointers to data packets. ownership of the packets
            in the array will move to the callee, but responsibility
            of the array itself will remain by the caller. */
        drv_data_block                 *p_data;
};

/** E_DATA_SEND response parameters */
struct drv_resp_data_send {
        /** transport identifier for packet transmission */
        drv_transport_id               transport_id;
};

/** E_DATA_RECEIVE indication parameters */
struct drv_ind_data_receive {
        /** reason for the indication */
        enum drv_status                status;
        /** transport identifier for packet transmission */
        drv_transport_id               transport_id;
        /** number of packets available for transmission */
        drv_uint16                     data_length;
};

/** E_DATA_RECEIVE command parameters */
struct drv_cmd_data_receive {
        /** transport identifier for packet transmission */
        drv_transport_id               transport_id;
        /** maximum number of packets allowed to be transmitted */
        drv_uint16                     data_length;
};

/** E_DATA_RECEIVE response parameters */
struct drv_resp_data_receive {
        /** transport identifier for packet transmission */
        drv_transport_id               transport_id;
        /** TRUE if this is the last packet in buffer */
        drv_bool                       last_packet;
        /** number of packets in the array of packets pointed to by p_data */
        drv_uint16                     data_length;
        /** pointer to array of packets. ownership of the packets in the array
           is transferred to the callee, but the responsibility of the array
           itself remains at the caller. */
        drv_data_block                 *p_data;
};

/** E_PROD_TEST command parameters */
struct drv_cmd_prod_test {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_PROD_TEST response parameters */
struct drv_resp_prod_test {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_PROD_TEST indication parameters */
struct drv_ind_prod_test {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_TRACE command parameters */
struct drv_cmd_trace {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_TRACE response parameters */
struct drv_resp_trace {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_TRACE indication parameters */
struct drv_ind_trace {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_STATISTICS_GET command parameters */
struct drv_cmd_statistics_get {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_STATISTICS_GET response parameters */
struct drv_resp_statistics_get {
        /** number of bytes in the byte array pointed to by data */
        drv_uint16                     data_length;
        /** pointer to a byte array. ownership of the bytearray is transferred
            to the callee. */
        void*                          data;
};

/** E_NETWORK_STATUS response parameters */
struct drv_resp_network_status {
        /** current network status */
        enum drv_network_status        network_status;
        /** current signal strength */
        drv_signal_strength            signal_strength;
        /** current CINR */
        drv_cinr                       cinr;
        /** connected-to BS identifier */
        drv_bsid                       bsid;
};

/** E_NETWORK_STATUS indication parameters */
struct drv_ind_network_status {
        /** current network status */
        enum drv_network_status        network_status;
        /** reason for the indication */
        enum drv_status                status;
        /** current signal strength */
        drv_signal_strength            signal_strength;
        /** current CINR */
        drv_cinr                       cinr;
        /** connected-to BS identifier */
        drv_bsid                       bsid;
        /** length of the array of characters pointed to by p_nsp_name */
        drv_uint8                      nsp_name_len;
        /** length of the array of characters pointed to by p_nsp_realm */
        drv_uint8                      nsp_realm_len;
        /** pointer to array of characters providing verbose name for the
            connected-to NSP */
        drv_string                     *p_nsp_name;
        /** pointer to array of characters providing verbose name for the
            connected-to NSP realm*/
        drv_string                     *p_nsp_realm;
};

/** E_SYSTEM_STATE indication parameters */
struct drv_ind_system_state {
        /** current system state/reason for the indication */
        enum drv_system_state          system_state;
};

/** E_EAP_DATA indication parameters */
struct drv_ind_eap_data {
        /** pointer to array of bytes containing the EAP payload. ownership
            of the array is transferred to the callee */
        drv_data_block                 p_payload;
};

/** E_EAP_DATA command parameters */
struct drv_cmd_eap_data {
        /** pointer to array of bytes containing the EAP payload. ownership
            of the array is transferred to the callee */
        drv_data_block                 p_payload;
        /** length of the array of bytes pointed to by p_payload */
        drv_uint16                     payload_length;
};

/** E_EAP_RESULT command parameters */
struct drv_cmd_eap_result {
        /** pointer to an array of bytes containing the calculated MSK */
        void*                          p_msk;
        /** length of the array of bytes pointed to by p_msk */
        drv_uint16                     msk_length;
        /** TRUE if negotiation was successfully completed, FALSE otherwise */
        drv_bool                       result;  
};

/** E_FLOW_CREATE_MS command paramters */
struct drv_cmd_flow_create_ms {
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CREATE_MS response parameters */
struct drv_resp_flow_create_ms {
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** confirmation code for the operation */
        enum drv_confirmation_code     confirmation_code;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CREATE_BS indication parameters */
struct drv_ind_flow_create_bs {
        /** transaction id to map this with the command and response */
        drv_uint8                      transaction_id;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CREATE_BS command parameters */
struct drv_cmd_flow_create_bs {
        /** transaction id to map this with the indication */
        drv_uint8                      transaction_id;
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** confirmation code for the operation */
        enum drv_confirmation_code     confirmation_code;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CREATE_BS response parameters */
struct drv_resp_flow_create_bs {
        /** transaction id to map this with the indication */
        drv_uint8                      transaction_id;
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
};

/** E_FLOW_DELETE_MS command parameters */
struct drv_cmd_flow_delete_ms {
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
};

/** E_FLOW_DELETE_MS response parameters */
struct drv_resp_flow_delete_ms {
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** confirmation code for the operation */
        enum drv_confirmation_code     confirmation_code;
};

/** E_FLOW_DELETE_BS indication parameters */
struct drv_ind_flow_delete_bs {
        /** transaction id to map this with the command and response */
        drv_uint8                      transaction_id;
};

/** E_FLOW_DELETE_BS command parameters */
struct drv_cmd_flow_delete_bs {
        /** transaction id to map this with the indication */
        drv_uint8                      transaction_id;
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** confirmation code for the operation */
        enum drv_confirmation_code     confirmation_code;
};

/** E_FLOW_DELETE_BS response parameters */
struct drv_resp_flow_delete_bs {
        /** transaction id to map this with the indication */
        drv_uint8                      transaction_id;
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
};      

/** E_FLOW_CHANGE_MS command parameters */
struct drv_cmd_flow_change_ms {
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CHANGE_MS response parameters */
struct drv_resp_flow_change_ms {
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** confirmation code for the operation */
        enum drv_confirmation_code     confirmation_code;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CHANGE_BS indication parameters */
struct drv_ind_flow_change_bs {
        /** transaction id to map this with the command and response */
        drv_uint8                      transaction_id;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CHANGE_BS command parameters */
struct drv_cmd_flow_change_bs {
        /** transaction id to map this with the indication */
        drv_uint8                      transaction_id;
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
        /** confirmation code for the operation */
        enum drv_confirmation_code     confirmation_code;
        /** pointer to TLV formatted auxillary data, the length is
            contained within the encoding */
        drv_service_flow_params        p_parameters;
};

/** E_FLOW_CHANGE_BS response parameters */
struct drv_resp_flow_change_bs {
        /** transaction id to map this with the indication */
        drv_uint8                      transaction_id;
        /** transport identifier for the service flow */
        drv_transport_id               transport_id;
};

/** global command structure for passing opcodes and related parameters */
struct drv_cmd {
        /** opcode for the command */
        enum drv_command_id cmd_id;
        /** opcode specific parameters */
        union {
                struct drv_cmd_initialize initialize;
                struct drv_cmd_configure configure;
                struct drv_cmd_radio_state radio_state;
                struct drv_cmd_search search;
                struct drv_cmd_entry entry;
                struct drv_cmd_data_send data_send;
                struct drv_cmd_data_receive data_receive;
                struct drv_cmd_eap_data eap_data;
                struct drv_cmd_eap_result eap_result;
                struct drv_cmd_flow_create_ms flow_create_ms;
                struct drv_cmd_flow_create_bs flow_create_bs;
                struct drv_cmd_flow_delete_ms flow_delete_ms;
                struct drv_cmd_flow_delete_bs flow_delete_bs;
                struct drv_cmd_flow_change_ms flow_change_ms;
                struct drv_cmd_flow_change_bs flow_change_bs;
                struct drv_cmd_prod_test  prod_test;
                struct drv_cmd_trace trace;
                struct drv_cmd_statistics_get statistics_get;
        } cmd;
};

/** global response structure for passing opcodes and related status values and
    parameters */
struct drv_resp {
        /** opcode for the response */
        enum drv_command_response_id cmd_resp_id;
        /** end status for the operation */
        enum drv_status cmd_status;
        /** opcode specific parameters */
        union {
                struct drv_resp_configure configure;
                struct drv_resp_data_send data_send;
                struct drv_resp_data_receive data_receive;
                struct drv_resp_flow_create_ms flow_create_ms;
                struct drv_resp_flow_create_bs flow_create_bs;
                struct drv_resp_flow_delete_ms flow_delete_ms;
                struct drv_resp_flow_delete_bs flow_delete_bs;
                struct drv_resp_flow_change_ms flow_change_ms;
                struct drv_resp_flow_change_bs flow_change_bs;
                struct drv_resp_prod_test prod_test;
                struct drv_resp_network_status network_status;
                struct drv_resp_trace trace;
                struct drv_resp_statistics_get statistics_get;
        } resp;
};

/** global indication structure for passing opcodes and related parameters */
struct drv_ind {
        /** indication opcode */
        enum drv_indication_id ind_id;
        /** opcode specific parameters */
        union {
                struct drv_ind_network_status network_status;
                struct drv_ind_system_state system_state;
                struct drv_ind_search search;
                struct drv_ind_data_receive data_receive;
                struct drv_ind_eap_data eap_data;
                struct drv_ind_flow_create_bs flow_create_bs;
                struct drv_ind_flow_delete_bs flow_delete_bs;
                struct drv_ind_flow_change_bs flow_change_bs;
                struct drv_ind_prod_test prod_test;
                struct drv_ind_trace trace;
        } ind;
};

/*
 * interface function prototypes 
 */

/** function for executing a command opcode */
typedef void (*drv_cmd_fnc)(wimax_osal_context*, struct drv_cmd*, void*);
/** function returning the response opcode */
typedef void (*drv_cmd_response_fnc)(wimax_osal_context*, 
				     struct drv_resp*, 
				     void*);
/** function sending an indication opcode */
typedef void (*drv_indication_fnc)(wimax_osal_context*, struct drv_ind*);

/** structure defining the driver interface to the client */
struct wimax_driver_if {
        /** command exectution function */
	drv_cmd_fnc wimax_cmd;
};

/** declare function for retrieving an instance of the driver module */
struct wimax_driver_if *pc2400m_drv_wimax_open(wimax_osal_context *ctx);

/** macro to remove client symbol dependency to the OPEN function */
#define WIMAX_OPEN(ctx) pc2400m_drv_wimax_open(ctx)

#endif /* PC2400M_DRV_IF */
