/*
 * pc2400m_drv_cd.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */



#include "pc2400m_drv_com.h"
#include "pc2400m_drv_chipif.h"
#include "pc2400m_drv_hdi.h"
#include "pc2400m_drv_dm.h"
#include "pc2400m_drv_cd.h"
#include "pc2400m_drv_sllist.h"

struct cd_ctrl_msg_elem {
	struct sllist list;
	wimax_osal_packet *msg;
	cd_msg_enqueue_cb cb;
	/* data is a carried cookie value for the client, ownership
	   remains at the client side */
	void *data;
	s32 id;
	u16 type;
	s32 timeout_handle;
};

struct cd_ind_elem {
	struct sllist list;
	cd_indication_cb cb;
	u32 id;
	void *data;
};

static void free_msg_elem(void *elem) {
	
        if (elem) {
	        wimax_osal_packet_free(
			&(((struct cd_ctrl_msg_elem*)elem)->msg));
		wimax_osal_mem_free(&elem);
	}

}

static void free_ind_elem(void *elem) {

        wimax_osal_assert(elem);	
	wimax_osal_mem_free(&elem);

}

static void pc2400m_drv_cd_send(struct pc2400m_drv_cd_if *hnd);
static void pc2400m_drv_cd_send_timeout(s32 handle, void* hnddata)
{

	struct pc2400m_drv_cd_if *hnd = (struct pc2400m_drv_cd_if*)hnddata;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(hnd->ctx);

	/* add reference to the main instance */
	priv->ref(priv);

	/* check that there is an event being processed */
	wimax_osal_assert(hnd->ctrl_processed);
	wimax_osal_assert(hnd->ctrl_processed->timeout_handle == handle);

	/* the event of this timer was already processed */
	wimax_osal_trace_u16(PC2400M_GROUP_CD_ERROR,
			     PC2400M_TRACE_MSG_TIMEOUT,
			     WIMAX_OSAL_PRIORITY_DEBUG,
			     hnd->ctrl_processed->type);
	free_msg_elem(hnd->ctrl_processed);
	hnd->ctrl_processed = NULL;
			
	/* stop everyting! */
	priv->dm->abort_mission(priv->dm);

	/* if the system was deleted during the handling of this, 
	   handle it now */
	priv->unref(priv);
	
}

static void pc2400m_drv_cd_send(struct pc2400m_drv_cd_if *hnd)
{

#define PC2400M_DRV_CD_RESPONSE_TIMEOUT 2000000 /* 2 seconds from spec */

	wimax_osal_packet *pkt;
	s32 send = FALSE;

	/* do not execute further if suspended */
	if (hnd->flags & PC2400_DRV_CD_FLAGS_SUSPENDED) goto out;

	if (!hnd->ctrl_processed) {
		hnd->ctrl_processed = (struct cd_ctrl_msg_elem*)
			sllist_pop(&(hnd->ctrl_list));

		if (hnd->ctrl_processed) {
			if (hnd->hdi->put(hnd->hdi, hnd->ctrl_processed->msg, 
					  FRAME_TYPE_CONTROL)) {
				/* the message will not be transmitted */
				wimax_osal_trace_u16
				        (PC2400M_GROUP_CD_ERROR,
					 PC2400M_TRACE_QUEUE_FAILED,
					 WIMAX_OSAL_PRIORITY_DEBUG,
					 WIMAX_OSAL_LE_TO_U16(
						 hnd->ctrl_processed->
							type));
				wimax_osal_packet_free
				        (&(hnd->ctrl_processed->msg));
			} else
				send = TRUE;

			hnd->ctrl_processed->msg = NULL;
			
			hnd->ctrl_processed->timeout_handle = 
				wimax_osal_timer_create(
					PC2400M_DRV_CD_RESPONSE_TIMEOUT,
					WIMAX_OSAL_TR_10MS,
					FALSE,
					(void*)hnd,
					pc2400m_drv_cd_send_timeout);
			
		}

	}

	/* check the data queue */
	while ((pkt = wimax_osal_packet_list_pop(&(hnd->pkt_list)))) {
		if (hnd->hdi->put(hnd->hdi, pkt, FRAME_TYPE_DATA)) {
			/* put back to the front of the list */
			wimax_osal_packet_list_push(&(hnd->pkt_list), pkt);
		} else
			send = TRUE;
	}

	/* transmit the data to the chipset if any was queued */
	if (send) {
		hnd->hdi->write(hnd->hdi);
	}

	/* if data remains in the packet list, send them too */
	if (wimax_osal_packet_list_len(&(hnd->pkt_list)))
		pc2400m_drv_cd_send(hnd);

 out:
	return;
}


/**
 * pc2400m_drv_cd_msg_enqueue - enqueue a control message for dispatch
 * @hnd: handle to the cd instance
 * @msg: packet containing the control message with all headers
 * @cb: call back to call with the response to the message
 * @data: data pointer to pass to the call back
 * @flags: flags to specify the handling of the message
 *
 * Enqueue (and immediately transmit if possible) a control message
 * for sending to the chipset.
 *
 * NOTE! This function is considered fail safe, an assumption also made
 *       by its callers. If the assumption is changed, also the callers must
 *       be revisited.
 */
static s32 pc2400m_drv_cd_msg_enqueue(struct pc2400m_drv_cd_if *hnd, 
				      wimax_osal_packet *msg, 
				      cd_msg_enqueue_cb cb, 
				      void *data, 
				      u32 flags)
{
	struct cd_ctrl_msg_elem *elem;
	struct pc2400m_drv_chipif_ctrl_msg *hdrptr;
	static s32 id = -2;
	static s32 ret = ESABABA;

	hdrptr = (struct pc2400m_drv_chipif_ctrl_msg*)
	        wimax_osal_packet_ptr(msg);
	elem = (struct cd_ctrl_msg_elem*)
		  wimax_osal_mem_alloc(sizeof(struct cd_ctrl_msg_elem));

	id++;
	if (id<0) id = 0;

	elem->msg = msg;
	elem->cb = cb;
	elem->data = data;
	elem->id = id;
	elem->type = WIMAX_OSAL_LE_TO_U16(hdrptr->type);
	ret = elem->id;
	
	if (flags & PC2400M_CD_FLAG_URGENT) {
		sllist_insert(&(hnd->ctrl_list), (struct sllist*)elem);
	} else {
		sllist_append(&(hnd->ctrl_list), (struct sllist*)elem);
	}

	pc2400m_drv_cd_send(hnd);

	return ESABABA;

}

/**
 * pc2400m_drv_cd_msg_remove - remove a queued control message
 * @hnd: handle to the cd instance
 * @msgid: identifier of the message to remove
 *
 * Removes the control message from the queue only if it is not
 * already being handled
 */
static s32 pc2400m_drv_cd_msg_remove(struct pc2400m_drv_cd_if *hnd, s32 msgid)
{
	struct sllist *pos;
	struct sllist *lastpos = NULL;
	s32 ret = -EFAULT;

	sllist_foreach(hnd->ctrl_list, pos) {
		if (((struct cd_ctrl_msg_elem*)pos)->id == msgid) {
			if (lastpos) {
				lastpos->next = pos->next;
				lastpos->last = pos->last;
			} else {
				hnd->ctrl_list = pos->next;
				if (hnd->ctrl_list) 
					hnd->ctrl_list->last = pos->last;
			}
			free_msg_elem(pos);
			ret = ESABABA;
			break;
		}
		lastpos = pos;
	}		

	return ret;
}

/**
 * pc2400m_drv_cd_register_indication - register handler for an indication msg
 * @hnd: handle to the cd instance
 * @indid: identifier of the indication type
 * @cb: the function to call when the indication is received
 *
 * Registers a handler for a specific type of indication from the chipset
 */
static s32 pc2400m_drv_cd_register_indication(struct pc2400m_drv_cd_if* hnd, 
					      u32 indid, 
					      cd_indication_cb cb,
					      void *data) 
{
	struct sllist *pos;
	struct sllist *lastpos = NULL;
	s32 ret = -EFAULT;

	wimax_osal_assert(indid & L3L4_INDICATION_MASK);

	sllist_foreach(hnd->ind_list, pos) {
		if (((struct cd_ind_elem*)pos)->id == indid) {
			if (cb) {
				((struct cd_ind_elem*)pos)->cb = cb;
				((struct cd_ind_elem*)pos)->data = data;
			} else {
				if (lastpos) {
					lastpos->next = pos->next;
					lastpos->last = pos->last;
				} else {
					hnd->ind_list = pos->next;
					if (hnd->ind_list) 
						hnd->ind_list->last = 
							pos->last;
				}
				wimax_osal_mem_free(&pos);
			}
			ret = ESABABA;
			break;

		}
		lastpos = pos;
	}		

	if (ret && cb) {
		pos = (struct sllist*)
		        wimax_osal_mem_alloc(sizeof(struct cd_ind_elem));
		((struct cd_ind_elem*)pos)->id = indid;
		((struct cd_ind_elem*)pos)->cb = cb;
		((struct cd_ind_elem*)pos)->data = data;
		sllist_append(&(hnd->ind_list), pos);
		ret = ESABABA;
	}

	return ret;

}

/**
 * pc2400m_drv_cd_pkt_enqueue - queue a data packet for transmission
 * @hnd: handle to the cd instance
 * @data: packet with the user data and headers
 * @flags: flags affecting the processing of the message
 *
 * Enqueues the data packet for sending to the chipset
 */
static s32 pc2400m_drv_cd_pkt_enqueue(struct pc2400m_drv_cd_if *hnd,
				      wimax_osal_packet *data, 
				      s32 rule_index,
				      u32 flags) 
{

	/* enqueue packet, and add transport header */
	wimax_osal_packet_list_append(&(hnd->pkt_list), data);
	
#define PC2400M_DRV_H2D_PKT_HDR_SIZE  4
	wimax_osal_packet_head_put(data, PC2400M_DRV_H2D_PKT_HDR_SIZE);

	if (rule_index >= 0) {
		WIMAX_OSAL_PUT_LE16(
	                wimax_osal_packet_ptr(data), (u16)rule_index);
		WIMAX_OSAL_PUT_LE16(
			wimax_osal_packet_ptr(data)+sizeof(u16), TRUE);
	} else {
		rule_index = 0; /* not used */
		WIMAX_OSAL_PUT_LE16(
			wimax_osal_packet_ptr(data), (u16)rule_index);
		WIMAX_OSAL_PUT_LE16(
			wimax_osal_packet_ptr(data)+sizeof(u16), FALSE);
	}	

	if (!(flags & PC2400M_CD_FLAG_AGGREGATE))
		pc2400m_drv_cd_send(hnd);

	return 0;

}

/**
 * pc2400m_drv_cd_initialize - initialize the cd instance
 * @hnd: handle to the cd instance
 *
 * Initializes internal variables and prepares the cd for usage. NOTE! the
 * assumption is made that this function never fails. If the assumption changes
 * remember to update function callers as well.
 */
static s32 pc2400m_drv_cd_initialize(struct pc2400m_drv_cd_if *hnd,
				   struct pc2400m_drv_hdi_if *hdi)
{
	
	/* clear flags, such as suspended state */
	hnd->flags = 0;

	/* clear the suspend operator */
	hnd->suspend_cb = NULL;

	hnd->hdi = hdi;
	return ESABABA;

}

/**
 * pc2400m_drv_cd_register_pkt_handler - register a handler function for IP 
 * data
 * @hnd: handle to the cd instance
 * @cb: handler function to call with IP data packets
 *
 * Registers handler function to receive IP data packets received from the 
 * chipset
 */
static s32 pc2400m_drv_cd_register_pkt_handler(struct pc2400m_drv_cd_if *hnd, 
					       cd_pkt_handler_cb cb,
					       void *data)
{

	hnd->pkt_handler = cb;
	hnd->pkt_handler_data = data;
	return 0;

}

/**
 * pc2400m_drv_cd_register_dbg_handler - register a handler function for traces
 * @hnd: handle to the cd instance
 * @cb: handler function to call with debug trace data
 *
 * Registers handler function to debug trace data from the chipset
 */
static s32 pc2400m_drv_cd_register_dbg_handler(struct pc2400m_drv_cd_if *hnd, 
					     cd_dbg_handler_cb cb,
					     void *data)
{

	hnd->dbg_handler = cb;
	hnd->dbg_handler_data = data;
	return 0;

}

/**
 * pc2400m_drv_cd_register_prod_handler - register a handler function for 
 *        production testing reports
 * @hnd: handle to the cd instance
 * @cb: handler function to call with production testing reports
 *
 * Registers handler function for production testing reports data from the 
 * chipset
 */
static s32 pc2400m_drv_cd_register_prod_handler(struct pc2400m_drv_cd_if *hnd, 
						cd_prod_handler_cb cb,
						void *data)
{

	hnd->prod_handler = cb;
	hnd->prod_handler_data = data;
	return 0;

}

static void process_ind(struct pc2400m_drv_cd_if *hnd, wimax_osal_packet *msg)
{
	struct pc2400m_drv_chipif_ctrl_msg *hdrptr = 
		(struct pc2400m_drv_chipif_ctrl_msg*)
	        wimax_osal_packet_ptr(msg);
	struct sllist *pos;
	
	sllist_foreach(hnd->ind_list, pos) {
		if (((struct cd_ind_elem*)pos)->id == 
		    WIMAX_OSAL_LE_TO_U16(hdrptr->type)) {
			((struct cd_ind_elem*)pos)->cb(
				hnd, 
				((struct cd_ind_elem*)pos)->id,
				msg,
				((struct cd_ind_elem*)pos)->data);
			break;
		}
	}

	if (!pos) {
	  
	        /* forward unregistered production testing reports to the
		   production report handler */
	        if (((WIMAX_OSAL_LE_TO_U16(hdrptr->type) & 
		      L3L4_INDICATION_MASK) == L3L4_INDICATION_MASK) &&
		    ((WIMAX_OSAL_LE_TO_U16(hdrptr->type) & 
		      L3L4_MODULE_ID_PRODUCTION) == L3L4_MODULE_ID_PRODUCTION) 
		    && hnd->prod_handler) {
  	                hnd->prod_handler(hnd, msg, hnd->prod_handler_data);
		} else {
		        wimax_osal_trace_u16(PC2400M_GROUP_CD_ERROR,
					     PC2400M_TRACE_UNKNOWN_TYPE,
					     WIMAX_OSAL_PRIORITY_DEBUG,
					     WIMAX_OSAL_LE_TO_U16(hdrptr->type)
					     );
			wimax_osal_packet_free(&msg);
		}

	}

}

/**
 * pc2400m_drv_cd_interrupt - interrupt handler for the chipset
 * @hnd: handle to the cd instance
 *
 * Processes interrupts from the chipset
 */
static void pc2400m_drv_cd_interrupt(struct pc2400m_drv_cd_if *hnd)
{

	struct pc2400m_drv_chipif_ctrl_msg *hdrptr;

	void *data;
	cd_msg_enqueue_cb cb;
	wimax_osal_packet *msg;
	s32 type;
	u8 cstype;

	/* read the data */
	hnd->hdi->read(hnd->hdi);
	while ((msg = hnd->hdi->get(hnd->hdi, &type))) {
		switch (type) {
		case FRAME_TYPE_DATA:
			if (hnd->pkt_handler) {
				/* remove the packet header */
#if 0   /* MISSING: REMOVED FOR THE SAKE OF COMPLIANCE WITH FW2.0 */
				cstype = wimax_osal_packet_ptr(msg)[0];
#define PC2400M_DRV_D2H_PKT_HDR_SIZE  8
				wimax_osal_packet_head_cut(
					msg, PC2400M_DRV_D2H_PKT_HDR_SIZE);
#else
				cstype = 1;
#endif

				/* forward the packet header */
				hnd->pkt_handler(hnd, msg, (s32)cstype,
						 hnd->pkt_handler_data);
			} else {
				wimax_osal_trace(
				        PC2400M_GROUP_CD_ERROR,
					PC2400M_TRACE_PKT_DISCARDED,
				        WIMAX_OSAL_PRIORITY_DEBUG);
				wimax_osal_packet_free(&msg);
			}
			break;
		case FRAME_TYPE_CONTROL:
			/* check against an ongoing control request */
			hdrptr = (struct pc2400m_drv_chipif_ctrl_msg*)
				wimax_osal_packet_ptr(msg);
			
			/* if its a response to a processed message... */
			if (hnd->ctrl_processed && 
			    hnd->ctrl_processed->type == 
			    WIMAX_OSAL_LE_TO_U16(hdrptr->type)) {

				data = hnd->ctrl_processed->data;
				cb = hnd->ctrl_processed->cb;

				/* stop timeout timer */
				wimax_osal_timer_cancel(
					hnd->ctrl_processed->timeout_handle);
				hnd->ctrl_processed->timeout_handle = 0;

				/* call the callback for this message */
				if (cb) 
					cb(hnd, msg, data);
				else
					wimax_osal_packet_free(&msg);

				/* get rid of the item */
				free_msg_elem(hnd->ctrl_processed);
				hnd->ctrl_processed = NULL;

				/* call the suspend handler, if registered */
				if ((hnd->flags & 
				     PC2400_DRV_CD_FLAGS_SUSPENDED) &&
				    hnd->suspend_cb) {
					hnd->suspend_cb(
						hnd, hnd->suspend_cb_data);
					hnd->suspend_cb = NULL;
				}

			} else {
				process_ind(hnd, msg); 
			}
			break;
		case FRAME_TYPE_TRACE_INFO:
			/* forward to the registered handler function */
			if (hnd->dbg_handler) {
				/* forward the packet header */
				hnd->dbg_handler(
					hnd, msg, hnd->dbg_handler_data);
			} else {
				/* lose the data silently */
				wimax_osal_packet_free(&msg);
			}
			break;
		default:
			wimax_osal_packet_free(&msg);
			wimax_osal_trace_byte(
				PC2400M_GROUP_CD_ERROR,
				PC2400M_TRACE_UNKNOWN_FRAME,
				WIMAX_OSAL_PRIORITY_DEBUG,
				type);
			break;
		}


	}

	pc2400m_drv_cd_send(hnd);

	return;

}


/**
 * pc2400m_drv_cd_reset - clean up the control dispatcher
 * @hnd: pointer to the instance of the control dispatcher
 *
 * Free the memory and resources used by the cd cancelling all ongoing
 * operations and bringing it to an initial state
 */
static void pc2400m_drv_cd_reset(struct pc2400m_drv_cd_if *hnd)
{

	/* cancel ongoing timers */
	if (hnd->ctrl_processed) {
		if (hnd->ctrl_processed->timeout_handle) {
			wimax_osal_timer_cancel(
				hnd->ctrl_processed->timeout_handle);
		}
		free_msg_elem(hnd->ctrl_processed);
		hnd->ctrl_processed = NULL;
	}

	/* release flags and suspend handlers */
	hnd->flags = 0;
	hnd->suspend_cb = NULL;

	/* clean up message queue */
	sllist_free(&(hnd->ctrl_list), free_msg_elem);

	/* clean up the enqueued packet list */
	wimax_osal_packet_list_release(&(hnd->pkt_list), TRUE);

}

/**
 * pc2400m_drv_cd_cleanup - remove an instance of the control dispatcher
 * @hnd: pointer to the instance of the control dispatcher
 *
 * Free the memory and resources used up by the specified instance
 * of the control dispatcher.
 */
static void pc2400m_drv_cd_cleanup(struct pc2400m_drv_cd_if *hnd)
{

	pc2400m_drv_cd_reset(hnd);

	/* clean up indication registrations */
	sllist_free(&(hnd->ind_list), free_ind_elem);
	
	wimax_osal_mem_free(&hnd);

}

static s32 pc2400m_drv_cd_suspend(struct pc2400m_drv_cd_if *hnd,
				    cd_suspend_cb cb, 
				    void *data)
{
	
	wimax_osal_assert(cb);

	/* stop the send queue */
	hnd->flags |= PC2400_DRV_CD_FLAGS_SUSPENDED;

	/* wait for a currently possibly executing item to complete */
	if (!hnd->ctrl_processed) {
		cb(hnd, data);
	} else {
		/* have the suspend call back called when the current
		   item is done */
		hnd->suspend_cb = cb;
		hnd->suspend_cb_data = data;
	}

	return ESABABA;

}

static s32 pc2400m_drv_cd_resume(struct pc2400m_drv_cd_if *hnd)
{
	/* go out of suspend state and restart the send queue */
	hnd->flags &= (~PC2400_DRV_CD_FLAGS_SUSPENDED);
	pc2400m_drv_cd_send(hnd);

	return ESABABA;

}

/**
 * pc2400m_drv_cd_if_new - create new instance of the control dispatcher
 * @ctx: the device context to bind this dispatcher instance to
 *
 * Allocate and initialize a new instance of the control dispatcher module,
 * bound to a specific device instance.
 */
struct pc2400m_drv_cd_if *pc2400m_drv_cd_if_new(wimax_osal_context* ctx) 
{

	struct pc2400m_drv_cd_if *iface = NULL;

	iface = (struct pc2400m_drv_cd_if*)
		  wimax_osal_mem_alloc(sizeof(struct pc2400m_drv_cd_if));
	wimax_osal_mem_set(iface, 0x00, 
			   sizeof(struct pc2400m_drv_cd_if));
	iface->ctx = ctx;
	iface->cleanup = pc2400m_drv_cd_cleanup;
	iface->reset = pc2400m_drv_cd_reset;
	iface->msg_enqueue = pc2400m_drv_cd_msg_enqueue;
	iface->msg_remove = pc2400m_drv_cd_msg_remove;
	iface->register_indication = pc2400m_drv_cd_register_indication;
	iface->pkt_enqueue = pc2400m_drv_cd_pkt_enqueue;
	iface->initialize = pc2400m_drv_cd_initialize;
	iface->register_pkt_handler = pc2400m_drv_cd_register_pkt_handler;
	iface->register_dbg_handler = pc2400m_drv_cd_register_dbg_handler;
	iface->register_prod_handler = pc2400m_drv_cd_register_prod_handler;
	iface->interrupt = pc2400m_drv_cd_interrupt;
	iface->suspend = pc2400m_drv_cd_suspend;
	iface->resume = pc2400m_drv_cd_resume;

	wimax_osal_packet_list_initialize(&(iface->pkt_list));

	return iface;

}
