/*
 * drivers/media/video/omap/sensor_tcm825x.c
 *
 * TCM825X Sensor driver for OMAP camera sensor interface
 *
 * Author: David Cohen (david.cohen@indt.org.br)
 *
 * This file is licensed under the terms of the GNU General Public License
 * version 2. This program is licensed "as is" without any warranty of any
 * kind, whether express or implied.
 *
 * This driver was based on ov9640 sensor driver from MontaVista
 */

#include <linux/errno.h>
#include <linux/i2c.h>
#include <linux/videodev.h>
#include <linux/delay.h>
#include <media/video-buf.h>

#include <asm/mach-types.h>

#include <asm/arch/board.h>
#include <asm/arch/gpio.h>

#include "sensor_if.h"
#include "tcm825x.h"

#define OMAP24XX_CAMERA_JAM_HACK

#ifdef OMAP24XX_CAMERA_JAM_HACK
/*
 * We don't need to check every pixel to assume that the frame is
 * corrupt and the sensor is jammed. CHECK_X and CHECK_Y are the
 * number of u32s to check per line / row, plus there are two lines in
 * the bottom of the frame.
 */
#define CHECK_X 8
#define CHECK_Y 6
/*
 * Start checking after this many frames since resetting the sensor.
 * Sometimes the first frame(s) is(/are) black which could trigger
 * unwanted reset(s).
 */
#define JAM_CHECK_AFTER 3
/*
 * If the sensor is quickly brought into bright conditions from dark,
 * it may temporarily be saturated, leaving out the normal background
 * noise. This many saturated frames may go through before the sensor
 * is considered jammed.
 */
#define SATURATED_MAX 30
#endif

#define BYTES_PER_PIXEL 2

/*
 * The sensor has two fps modes: the lower one just gives half the fps
 * at the same xclk than the high one.
 */
#define MAX_FPS 30
#define MIN_FPS 8
#define MAX_HALF_FPS (MAX_FPS / 2)
#define HIGH_FPS_LOWER_LIMIT 14
#define DEFAULT_FPS MAX_HALF_FPS

static struct tcm825x_sensor {
	const struct omap_camera_sensor_config * sensor_config;
	struct i2c_client i2c_client;
	struct i2c_driver *i2c_driver;
	struct v4l2_pix_format pix;
	struct v4l2_fract timeperframe;
#ifdef OMAP24XX_CAMERA_JAM_HACK
	int frames_after_reset;
	int saturated_count;
#endif
} tcm825x;

static struct i2c_driver tcm825x_i2c_driver = {
	.driver = {
		.name	= "TCM825x I2C driver",
	},
	.id		= I2C_DRIVERID_MISC,	/* Experimental ID */
};

/* list of image formats supported by TCM825X sensor */
const static struct v4l2_fmtdesc tcm825x_formats[] = {
	{
		.description = "YUYV (YUV 4:2:2), packed",
		.pixelformat = V4L2_PIX_FMT_UYVY,
	}, {
		/* Note:  V4L2 defines RGB565 as:
		 *
		 *      Byte 0                    Byte 1
		 *      g2 g1 g0 r4 r3 r2 r1 r0   b4 b3 b2 b1 b0 g5 g4 g3
		 *
		 * We interpret RGB565 as:
		 *
		 *      Byte 0                    Byte 1
		 *      g2 g1 g0 b4 b3 b2 b1 b0   r4 r3 r2 r1 r0 g5 g4 g3
		 */
		.description = "RGB565, le",
		.pixelformat = V4L2_PIX_FMT_RGB565,
	},
};

#define TCM825X_NUM_CAPTURE_FORMATS	ARRAY_SIZE(tcm825x_formats)
#define NUM_OVERLAY_FORMATS		2

/* register initialization tables for TCM825X */

/* TCM825X register configuration for all combinations of pixel format and
 * image size
 */
const static struct tcm825x_reg subqcif	=	{ 0x20, TCM825X_PICSIZ };
const static struct tcm825x_reg qcif	=	{ 0x18, TCM825X_PICSIZ };
const static struct tcm825x_reg cif	=	{ 0x14, TCM825X_PICSIZ };
const static struct tcm825x_reg qqvga	=	{ 0x0c, TCM825X_PICSIZ };
const static struct tcm825x_reg qvga	=	{ 0x04, TCM825X_PICSIZ };
const static struct tcm825x_reg vga	=	{ 0x00, TCM825X_PICSIZ };

const static struct tcm825x_reg yuv422	=	{ 0x00, TCM825X_PICFMT };
const static struct tcm825x_reg rgb565	=	{ 0x02, TCM825X_PICFMT };

/* Our own specific controls */
#define V4L2_CID_ALC				V4L2_CID_PRIVATE_BASE
#define V4L2_CID_H_EDGE_EN			V4L2_CID_PRIVATE_BASE + 1
#define V4L2_CID_V_EDGE_EN			V4L2_CID_PRIVATE_BASE + 2
#define V4L2_CID_LENS				V4L2_CID_PRIVATE_BASE + 3
#define V4L2_CID_MAX_EXPOSURE_TIME		V4L2_CID_PRIVATE_BASE + 4
#define V4L2_CID_LAST_PRIV			V4L2_CID_MAX_EXPOSURE_TIME

/*  Video controls  */
static struct vcontrol {
	struct v4l2_queryctrl qc;
	u16 reg;
	u16 start_bit;
} video_control[] = {
	{
		{
			.id = V4L2_CID_GAIN,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Gain",
			.minimum = 0,
			.maximum = 63,
			.step = 1,
		},
		.reg = TCM825X_AG,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_RED_BALANCE,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Red Balance",
			.minimum = 0,
			.maximum = 255,
			.step = 1,
		},
		.reg = TCM825X_MRG,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_BLUE_BALANCE,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Blue Balance",
			.minimum = 0,
			.maximum = 255,
			.step = 1,
		},
		.reg = TCM825X_MBG,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_AUTO_WHITE_BALANCE,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Auto White Balance",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_AWBSW,
		.start_bit = 7,
	},
	{
		{
			.id = V4L2_CID_EXPOSURE,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Exposure Time",
			.minimum = 0,
			.maximum = 0x1fff,
			.step = 1,
		},
		.reg = TCM825X_ESRSPD_U,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_HFLIP,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Mirror Image",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_H_INV,
		.start_bit = 6,
	},
	{
		{
			.id = V4L2_CID_VFLIP,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Vertical Flip",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_V_INV,
		.start_bit = 7,
	},
	/* Private controls */
	{
		{
			.id = V4L2_CID_ALC,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Auto Luminance Control",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_ALCSW,
		.start_bit = 7,
	},
	{
		{
			.id = V4L2_CID_H_EDGE_EN,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Horizontal Edge Enhancement",
			.minimum = 0,
			.maximum = 0xff,
			.step = 1,
		},
		.reg = TCM825X_HDTG,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_V_EDGE_EN,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Vertical Edge Enhancement",
			.minimum = 0,
			.maximum = 0xff,
			.step = 1,
		},
		.reg = TCM825X_VDTG,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_LENS,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Lens Shading Compensation",
			.minimum = 0,
			.maximum = 0x3f,
			.step = 1,
		},
		.reg = TCM825X_LENS,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_MAX_EXPOSURE_TIME,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Maximum Exposure Time",
			.minimum = 0,
			.maximum = 0x3,
			.step = 1,
		},
		.reg = TCM825X_ESRLIM,
		.start_bit = 5,
	},
};


const static struct tcm825x_reg *tcm825x_siz_reg[NUM_IMAGE_SIZES] =
{ &subqcif, &qqvga, &qcif, &qvga, &cif, &vga };

const static struct tcm825x_reg *tcm825x_fmt_reg[NUM_PIXEL_FORMATS] =
{ &yuv422, &rgb565 };

/*
 * Read a value from a register in an TCM825X sensor device.  The value is
 * returned in 'val'.
 * Returns zero if successful, or non-zero otherwise.
 */
static int tcm825x_read_reg(struct i2c_client *client, int reg)
{
	int err;
	struct i2c_msg msg[2];
	u8 reg_buf, data_buf = 0;

	if (!client->adapter)
		return -ENODEV;

	msg[0].addr = client->addr;
	msg[0].flags = 0;
	msg[0].len = 1;
	msg[0].buf = &reg_buf;
	msg[1].addr = client->addr;
	msg[1].flags = I2C_M_RD;
	msg[1].len = 1;
	msg[1].buf = &data_buf;

	reg_buf = reg;

	err = i2c_transfer(client->adapter, msg, 2);
	if (err < 0)
		return err;
	return data_buf;
}

/* Write a value to a register in an TCM825X sensor device.
 * Returns zero if successful, or non-zero otherwise.
 */
static int tcm825x_write_reg(struct i2c_client *client, u8 reg, u8 val)
{
	int err;
	struct i2c_msg msg[1];
	unsigned char data[2];

	if (!client->adapter)
		return -ENODEV;

	msg->addr = client->addr;
	msg->flags = 0;
	msg->len = 2;
	msg->buf = data;
	data[0] = reg;
	data[1] = val;
	err = i2c_transfer(client->adapter, msg, 1);
	if (err >= 0)
		return 0;
	return err;
}

static int __tcm825x_write_reg_mask(struct i2c_client *client,
				    u8 reg, u8 val, u8 mask)
{
	int rc;

	/* need to do read - modify - write */
	rc = tcm825x_read_reg(client, reg);
	if (rc < 0)
		return rc;

	rc &= (~mask);	/* Clear the masked bits */
	val &= mask;	/* Enforce mask on value */
	val |= rc;

	/* write the new value to the register */
	if ((rc = tcm825x_write_reg(client, reg, val)))
		return rc;

	return 0;
}

#define tcm825x_write_reg_mask(client, regmask, val)			\
	__tcm825x_write_reg_mask(client, TCM825X_ADDR((regmask)), val,	\
				 TCM825X_MASK((regmask)))


/* Initialize a list of TCM825X registers.
 * The list of registers is terminated by the pair of values
 * { TCM825X_REG_TERM, TCM825X_VAL_TERM }.
 * Returns zero if successful, or non-zero otherwise.
 */
static int tcm825x_write_default_regs(struct i2c_client *client,
				      const struct tcm825x_reg reglist[])
{
	int err;
	const struct tcm825x_reg *next = reglist;

	while (!((next->reg == TCM825X_REG_TERM)
		 && (next->val == TCM825X_VAL_TERM))) {
		err = tcm825x_write_reg(client, next->reg, next->val);
		udelay(100);
		if (err) {
			printk(KERN_ERR "%s(): Register writing failed\n",
			       __FUNCTION__);
			return err;
		}
		next++;
	}

	return 0;
}

/* Matches the control ID and returns the vcontrol pointer */
static struct vcontrol * find_vctrl(int id)
{
	int i;

	if (id < V4L2_CID_BASE)
		return NULL;

	for (i = 0; i < ARRAY_SIZE(video_control); i++)
		if (video_control[i].qc.id == id)
			return &video_control[i];

	return NULL;
}

/* Configure the TCM825X for a specified image size, pixel format, and frame
 * period.  xclk is the frequency (in Hz) of the xclk input to the TCM825X.
 * fper is the frame period (in seconds) expressed as a fraction.
 * Returns zero if successful, or non-zero otherwise.
 * The actual frame period is returned in fper.
 */
static int tcm825x_configure(struct v4l2_int_device *s,
			     enum image_size isize, u32 pixelformat,
			     u32 xclk, struct v4l2_fract *fper)
{
	int err;
	u32 tgt_fps;
	u8 val;
	enum pixel_format pfmt;
	struct tcm825x_sensor *sensor = s->priv;

	/* common register initialization */
	err = tcm825x_write_default_regs(
		&sensor->i2c_client, sensor->sensor_config->default_regs());
	if (err)
		return err;

	/* configure image size */
	val = tcm825x_siz_reg[isize]->val;
	printk(KERN_DEBUG "%s(): configuring Image Size %d\n",
	       __FUNCTION__, isize);
	err = tcm825x_write_reg_mask(&sensor->i2c_client,
				     tcm825x_siz_reg[isize]->reg, val);
	if (err)
		return err;

	/* configure pixel format */
	switch (pixelformat) {
	default:
	case V4L2_PIX_FMT_RGB565:
		pfmt = RGB565;
		break;
	case V4L2_PIX_FMT_UYVY:
		pfmt = YUV422;
		break;
	}

	val = tcm825x_fmt_reg[pfmt]->val;
	printk(KERN_DEBUG "%s(): configuring Pixel Format %d\n",
	       __FUNCTION__, pfmt);
	err = tcm825x_write_reg_mask(&sensor->i2c_client,
				     tcm825x_fmt_reg[pfmt]->reg, val);
	if (err)
		return err;

	/* for frame rate < 15, the FPS reg (addr 0x02, bit 7) should be set */
	tgt_fps = fper->denominator / fper->numerator;
	if (tgt_fps <= HIGH_FPS_LOWER_LIMIT) {
		val = tcm825x_read_reg(&sensor->i2c_client, 0x02);
		val |= 0x80;
		tcm825x_write_reg(&sensor->i2c_client, 0x02, val);
	}

	return 0;
}

static int tcm825x_detect(struct tcm825x_sensor *sensor)
{
	int r;

	r = tcm825x_read_reg(&sensor->i2c_client, 0x01);
	if (r < 0)
		return r;
	if (r == 0) {
		dev_err(&sensor->i2c_client.dev, "device not detected\n");
		return -EIO;
	}
	return 0;
}


/* This function registers an I2C client via i2c_attach_client() for an TCM825X
 * sensor device.  If 'probe' is non-zero, then the I2C client is only
 * registered if the device can be detected.  If 'probe' is zero, then no
 * device detection is attempted and the I2C client is always registered.
 * Returns zero if an I2C client is successfully registered, or non-zero
 * otherwise.
 */
static int tcm825x_i2c_attach_client(struct i2c_adapter *adap,
				     int addr, int probe)
{
	struct tcm825x_sensor *sensor = &tcm825x;
	struct i2c_client *client = &sensor->i2c_client;
	int err;

	if (client->adapter)
		return -EBUSY;	/* our client is already attached */

	client->addr = addr;
	client->flags = 0;
	client->driver = sensor->i2c_driver;
	client->adapter = adap;
	strlcpy(client->name, "TCM825x I2C driver", sizeof(client->name));

	err = i2c_attach_client(client);
	if (err) {
		client->adapter = NULL;
		return err;
	}

	if (probe) {
		err = tcm825x_detect(sensor);
		if (err < 0) {
			i2c_detach_client(client);
			client->adapter = NULL;
			return err;
		}
	}
	return 0;
}

/* This function is called by i2c_del_adapter() and i2c_del_driver()
 * if the adapter or driver with which this I2C client is associated is
 * removed.  This function unregisters the client via i2c_detach_client().
 * Returns zero if the client is successfully detached, or non-zero
 * otherwise.
 */
static int tcm825x_i2c_detach_client(struct i2c_client *client)
{
	int err;

	if (!client->adapter)
		return -ENODEV;	/* our client isn't attached */

	err = i2c_detach_client(client);
	client->adapter = NULL;

	return err;
}

/* This function will be called for each registered I2C bus adapter when our
 * I2C driver is registered via i2c_add_driver().  It will also be called
 * whenever a new I2C adapter is registered after our I2C driver is registered.
 * This function probes the specified I2C bus adapter to determine if an
 * TCM825X sensor device is present.  If a device is detected, an I2C client
 * is registered for it via tcm825x_i2c_attach_client().  Note that we can't
 * use the standard i2c_probe() function to look for the sensor because the
 * OMAP I2C controller doesn't support probing.
 * Returns zero if an TCM825X device is detected and an I2C client successfully
 * registered for it, or non-zero otherwise.
 */
static int tcm825x_i2c_probe_adapter(struct i2c_adapter *adap)
{
	return tcm825x_i2c_attach_client(adap, TCM825X_I2C_ADDR, 1);
}

/* Find the best match for a requested image capture size.  The best match
 * is chosen as the nearest match that has the same number or fewer pixels
 * as the requested size, or the smallest image size if the requested size
 * has fewer pixels than the smallest image.
 */
static enum image_size tcm825x_find_size(unsigned int width,
					 unsigned int height)
{
	enum image_size isize;
	unsigned long pixels = width * height;

	for (isize = subQCIF; isize < VGA; isize++) {
		if (tcm825x_sizes[isize + 1].height
		    * tcm825x_sizes[isize + 1].width > pixels) {
			printk(KERN_DEBUG "%s(): size %d\n",
			       __FUNCTION__, isize);
			return isize;
		}
	}

	printk(KERN_DEBUG "%s(): format default VGA\n", __FUNCTION__);
	return VGA;
}

/* Given the image capture format in pix, the nominal frame period in
 * timeperframe, calculate the required xclk frequency.
 *
 * TCM825X input frequency characteristics are:
 *     Minimum 11.9 MHz, Typical 24.57 MHz and maximum 25/27 MHz
 */
#define XCLK_MIN 11900000
#define XCLK_MAX 25000000

static u32 vidioc_int_g_ext_clk(struct v4l2_int_device *s)
{
	struct tcm825x_sensor *sensor = s->priv;
	struct v4l2_fract *timeperframe = &sensor->timeperframe;
	u32 tgt_xclk;	/* target xclk */
	u32 tgt_fps;	/* target frames per secound */

	tgt_fps = timeperframe->denominator / timeperframe->numerator;

	tgt_xclk = (tgt_fps <= HIGH_FPS_LOWER_LIMIT) ?
		(2457 * tgt_fps) / MAX_HALF_FPS :
		(2457 * tgt_fps) / MAX_FPS;
	tgt_xclk *= 10000;

	tgt_xclk = min(tgt_xclk, (u32)XCLK_MAX);
	tgt_xclk = max(tgt_xclk, (u32)XCLK_MIN);

	return tgt_xclk;
}

static int vidioc_int_s_ext_clk(struct v4l2_int_device *s, u32 xclk)
{
	if (xclk > XCLK_MAX || xclk < XCLK_MIN)
		return -EINVAL;
	return 0;
}

/* following are sensor interface functions implemented by
 * TCM825X sensor driver.
 */
static int vidioc_int_queryctrl(struct v4l2_int_device *s,
				struct v4l2_queryctrl *qc)
{
	struct vcontrol * control;

	control = find_vctrl(qc->id);

	if (control == NULL)
		return -EINVAL;

	*qc = control->qc;

	return 0;
}

static int vidioc_int_g_ctrl(struct v4l2_int_device *s,
			     struct v4l2_control *vc)
{
	struct tcm825x_sensor *sensor = s->priv;
	struct i2c_client *client = &sensor->i2c_client;
	int val, r;
	struct vcontrol *lvc;

	/* exposure time is special, spread accross 2 registers */
	if (vc->id == V4L2_CID_EXPOSURE) {
		int val_lower, val_upper;

		val_upper = tcm825x_read_reg(client,
					     TCM825X_ADDR(TCM825X_ESRSPD_U));
		if (val_upper < 0)
			return val_upper;
		val_lower = tcm825x_read_reg(client,
					     TCM825X_ADDR(TCM825X_ESRSPD_L));
		if (val_lower < 0)
			return val_lower;

		vc->value = ((val_upper & 0x1f) << 8) | (val_lower);
		return 0;
	}

	lvc = find_vctrl(vc->id);
	if (lvc == NULL)
		return -EINVAL;

	r = tcm825x_read_reg(client, TCM825X_ADDR(lvc->reg));
	if (r < 0)
		return r;
	val = r & TCM825X_MASK(lvc->reg);
	val >>= lvc->start_bit;

	if (val < 0)
		return val;

	if (vc->id == V4L2_CID_HFLIP || vc->id == V4L2_CID_VFLIP)
		val ^= sensor->sensor_config->is_upside_down();

	vc->value = val;
	return 0;
}

static int vidioc_int_s_ctrl(struct v4l2_int_device *s,
			     struct v4l2_control *vc)
{
	struct tcm825x_sensor *sensor = s->priv;
	struct i2c_client *client = &sensor->i2c_client;
	struct vcontrol *lvc;
	int val = vc->value;

	/* exposure time is special, spread accross 2 registers */
	if (vc->id == V4L2_CID_EXPOSURE) {
		int val_lower, val_upper;
		val_lower = val & TCM825X_MASK(TCM825X_ESRSPD_L);
		val_upper = (val >> 8) & TCM825X_MASK(TCM825X_ESRSPD_U);

		if (tcm825x_write_reg_mask(client,
					   TCM825X_ESRSPD_U, val_upper))
			return -EIO;

		if (tcm825x_write_reg_mask(client,
					   TCM825X_ESRSPD_L, val_lower))
			return -EIO;

		return 0;
	}

	lvc = find_vctrl(vc->id);
	if (lvc == NULL)
		return -EINVAL;

	if (vc->id == V4L2_CID_HFLIP || vc->id == V4L2_CID_VFLIP)
		val ^= sensor->sensor_config->is_upside_down();

	val = val << lvc->start_bit;
	if (tcm825x_write_reg_mask(client, lvc->reg, val))
		return -EIO;

	return 0;
}

/* Implement the VIDIOC_ENUM_FMT ioctl for the CAPTURE buffer type.
 */
static int vidioc_int_enum_fmt_cap(struct v4l2_int_device *s,
				   struct v4l2_fmtdesc *fmt)
{
	int index = fmt->index;

	switch (fmt->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		if (index >= TCM825X_NUM_CAPTURE_FORMATS)
			return -EINVAL;
		break;

	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		if (index >= NUM_OVERLAY_FORMATS)
			return -EINVAL;
		break;

	default:
		return -EINVAL;
	}

	fmt->flags = tcm825x_formats[index].flags;
	strlcpy(fmt->description, tcm825x_formats[index].description,
		sizeof(fmt->description));
	fmt->pixelformat = tcm825x_formats[index].pixelformat;

	return 0;
}

/* Implement the VIDIOC_TRY_FMT ioctl for the CAPTURE buffer type.  This
 * ioctl is used to negotiate the image capture size and pixel format
 * without actually making it take effect.
 */
static int vidioc_int_try_fmt_cap(struct v4l2_int_device *s,
				  struct v4l2_format *f)
{
	enum image_size isize;
	int ifmt;
	struct v4l2_pix_format *pix = &f->fmt.pix;

	isize = tcm825x_find_size(pix->width, pix->height);
	printk(KERN_DEBUG "%s(): isize = %d num_capture = %d\n",
	       __FUNCTION__, isize, TCM825X_NUM_CAPTURE_FORMATS);

	pix->width = tcm825x_sizes[isize].width;
	pix->height = tcm825x_sizes[isize].height;

	for (ifmt = 0; ifmt < TCM825X_NUM_CAPTURE_FORMATS; ifmt++)
		if (pix->pixelformat == tcm825x_formats[ifmt].pixelformat)
			break;

	if (ifmt == TCM825X_NUM_CAPTURE_FORMATS)
		ifmt = 0;	/* Default = YUV 4:2:2 */

	pix->pixelformat = tcm825x_formats[ifmt].pixelformat;
	pix->field = V4L2_FIELD_NONE;
	pix->bytesperline = pix->width * BYTES_PER_PIXEL;
	pix->sizeimage = pix->bytesperline * pix->height;
	pix->priv = 0;
	printk(KERN_DEBUG "%s(): format = 0x%08x\n",
	       __FUNCTION__, pix->pixelformat);

	switch (pix->pixelformat) {
	case V4L2_PIX_FMT_UYVY:
	default:
		pix->colorspace = V4L2_COLORSPACE_JPEG;
		break;
	case V4L2_PIX_FMT_RGB565:
		pix->colorspace = V4L2_COLORSPACE_SRGB;
		break;
	}

	return 0;
}

static int vidioc_int_s_fmt_cap(struct v4l2_int_device *s,
				struct v4l2_format *f)
{
	struct tcm825x_sensor *sensor = s->priv;
	struct v4l2_pix_format *pix = &f->fmt.pix;
	int rval;

	if ((rval = vidioc_int_try_fmt_cap(s, f)))
		return rval;

	rval = tcm825x_configure(s,
				 tcm825x_find_size(pix->width, pix->height),
				 pix->pixelformat, vidioc_int_g_ext_clk(s),
				 &sensor->timeperframe);

	sensor->pix = *pix;

	return rval;
}

static int vidioc_int_g_fmt_cap(struct v4l2_int_device *s,
				struct v4l2_format *f)
{
	struct tcm825x_sensor *sensor = s->priv;

	f->fmt.pix = sensor->pix;

	return 0;
}

static int vidioc_int_g_parm(struct v4l2_int_device *s,
			     struct v4l2_streamparm *a)
{
	struct tcm825x_sensor *sensor = s->priv;
	struct v4l2_captureparm *cparm = &a->parm.capture;

	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	memset(a, 0, sizeof(*a));
	a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

	cparm->capability = V4L2_CAP_TIMEPERFRAME;
	cparm->timeperframe = sensor->timeperframe;

	return 0;
}

/* Given a capture format in pix, the frame period in timeperframe, and
 * the xclk frequency, set the capture format of the TCM825X sensor.
 * The actual frame period will be returned in timeperframe.
 */
static int vidioc_int_s_parm(struct v4l2_int_device *s,
			     struct v4l2_streamparm *a)
{
	struct tcm825x_sensor *sensor = s->priv;
	struct v4l2_pix_format *pix = &sensor->pix;
	struct v4l2_fract *timeperframe = &a->parm.capture.timeperframe;
	u32 tgt_fps;	/* target frames per secound */
	int rval;

	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	if ((timeperframe->numerator == 0)
	    || (timeperframe->denominator == 0)) {
		timeperframe->denominator = DEFAULT_FPS;
		timeperframe->numerator = 1;
	}

	tgt_fps = timeperframe->denominator / timeperframe->numerator;

	if (tgt_fps > MAX_FPS) {
		timeperframe->denominator = MAX_FPS;
		timeperframe->numerator = 1;
	} else if (tgt_fps < MIN_FPS) {
		timeperframe->denominator = MIN_FPS;
		timeperframe->numerator = 1;
	}

	rval = tcm825x_configure(s,
				 tcm825x_find_size(pix->width, pix->height),
				 pix->pixelformat, vidioc_int_g_ext_clk(s),
				 timeperframe);

	sensor->timeperframe = *timeperframe;

	return rval;
}

static int power_set(struct v4l2_int_device *s, int on)
{
	struct tcm825x_sensor *sensor = &tcm825x;

#ifdef OMAP24XX_CAMERA_JAM_HACK
	if (on) {
		sensor->saturated_count = 0;
		sensor->frames_after_reset = 0;
	}
#endif
	return sensor->sensor_config->power_set(on);
}

#ifdef OMAP24XX_CAMERA_JAM_HACK
/*
 * Check for jammed sensor, in which case all horizontal lines are
 * equal. Handle also case where sensor could be saturated awhile in
 * case of rapid increase of brightness.
 */
static int vidioc_int_g_needs_reset(struct v4l2_int_device *s, void *buf)
{
	int i, j;
	uint32_t xor, xor2;
	uint32_t offset;
	struct tcm825x_sensor *sensor = s->priv;
	struct v4l2_pix_format *pix = &sensor->pix;
	uint32_t dx_offset;
	uint32_t saturated_pattern;
	int is_saturated = 1;

	switch (pix->pixelformat) {
	default:
	case V4L2_PIX_FMT_RGB565:
		saturated_pattern = 0xffffffff; /* guess */
		break;
	case V4L2_PIX_FMT_UYVY:
		saturated_pattern = 0xe080e080;
		break;
	}

	/*
	 * This won't work for height under 2 at all.
	 */
	if (pix->height < 2)
		return 0;
	/*
	 * Check that there is enough image data.
	 */
	if (pix->width * BYTES_PER_PIXEL < sizeof(uint32_t))
		return 0;
	/*
	 * Don't check for jamming immediately. Sometimes frames
	 * immediately after reset are black.
	 */
	if (sensor->frames_after_reset < JAM_CHECK_AFTER) {
		sensor->frames_after_reset++;
		return 0;
	}
	
	dx_offset = ((pix->width - sizeof(uint32_t) / BYTES_PER_PIXEL)
		     * BYTES_PER_PIXEL) / (CHECK_X - 1);
	dx_offset = dx_offset - dx_offset % BYTES_PER_PIXEL;
	/*
	 * Check two lines in the bottom first. They're unlikely to be
	 * saturated and quick to check.
	 */
	offset = (pix->height - 2) * pix->bytesperline;
	xor = xor2 = 0;
	for (j = 0; j < CHECK_X; j++) {
		uint32_t *val = buf + offset;
		uint32_t *val2 = buf + offset + pix->bytesperline;
		xor ^= *val;
		if (*val != saturated_pattern)
			is_saturated = 0;
		xor2 ^= *val2;
		if (xor2 != xor) {
			sensor->saturated_count = 0;
			return 0;
		}
		offset += dx_offset;
	}
	/*
	 * Check the rest of the picture.
	 */
	offset = 0;
	for (i = 0; i < CHECK_Y; i++) {
		uint32_t offset2 = offset;
		xor2 = 0;
		for (j = 0; j < CHECK_X; j++) {
			uint32_t *val = buf + offset2;
			xor2 ^= *val;
			offset2 += dx_offset;
		}
		if (xor2 != xor) {
			sensor->saturated_count = 0;
			return 0;
		}
		offset += pix->bytesperline * ((pix->height - 2) / CHECK_Y);
	}

	if (is_saturated && sensor->saturated_count++ < SATURATED_MAX)
		return 0;
	
	return -EIO;
}
#endif

static int vidioc_int_reset(struct v4l2_int_device *s)
{
	return -EBUSY;
}

static int vidioc_int_init(struct v4l2_int_device *s)
{
	struct tcm825x_sensor *sensor = s->priv;
	struct v4l2_pix_format *pix = &sensor->pix;

	return tcm825x_configure(s,
				 tcm825x_find_size(pix->width, pix->height),
				 pix->pixelformat, vidioc_int_g_ext_clk(s),
				 &sensor->timeperframe);
}

/* Prepare for the driver to exit.
 * Balances tcm825xsensor_init().
 * This function must de-initialize the sensor and its associated data
 * structures.
 */
static int dev_exit(struct v4l2_int_device *s)
{
	struct tcm825x_sensor *sensor = s->priv;

	i2c_del_driver(sensor->i2c_driver);

	return 0;
}


/* Initialize the TCM825X sensor.
 * This routine allocates and initializes the data structure for the sensor,
 * powers up the sensor, registers the I2C driver, and sets a default image
 * capture format in pix.  The capture format is not actually programmed
 * into the TCM825X sensor by this routine.
 * This function must return a non-NULL value to indicate that
 * initialization is successful.
 */
static int dev_init(struct v4l2_int_device *s)
{
	struct tcm825x_sensor *sensor = s->priv;

	sensor->i2c_driver->attach_adapter = tcm825x_i2c_probe_adapter;
	sensor->i2c_driver->detach_client = tcm825x_i2c_detach_client;

	if (i2c_add_driver(sensor->i2c_driver)) {
		printk(KERN_WARNING
		       "%s(): Failed to register TCM825x I2C client\n",
		       __FUNCTION__);
		goto err_add;
	}

	if (!sensor->i2c_client.adapter) {
		printk(KERN_WARNING
		       "%s(): Failed to detect TCM825x sensor chip\n",
		       __FUNCTION__);
		goto err_detect;
	}

	return 0;

err_detect:
	i2c_del_driver(sensor->i2c_driver);
err_add:
	return -EBUSY;
}

static struct v4l2_int_slave tcm825x_slave = {
	.dev_init = dev_init,
	.dev_exit = dev_exit,
	.power_set = power_set,
	.vidioc_int_g_ext_clk = vidioc_int_g_ext_clk,
	.vidioc_int_s_ext_clk = vidioc_int_s_ext_clk,
#ifdef OMAP24XX_CAMERA_JAM_HACK
	.vidioc_int_g_needs_reset = vidioc_int_g_needs_reset,
#endif
	.vidioc_int_reset = vidioc_int_reset,
	.vidioc_int_init = vidioc_int_init,
	.vidioc_int_enum_fmt_cap = vidioc_int_enum_fmt_cap,
	.vidioc_int_try_fmt_cap = vidioc_int_try_fmt_cap,
	.vidioc_int_g_fmt_cap = vidioc_int_g_fmt_cap,
	.vidioc_int_s_fmt_cap = vidioc_int_s_fmt_cap,
	.vidioc_int_g_parm = vidioc_int_g_parm,
	.vidioc_int_s_parm = vidioc_int_s_parm,
	.vidioc_int_queryctrl = vidioc_int_queryctrl,
	.vidioc_int_g_ctrl = vidioc_int_g_ctrl,
	.vidioc_int_s_ctrl = vidioc_int_s_ctrl,
};

static struct v4l2_int_device tcm825x_if = {
	.module = THIS_MODULE,
	.name = "TCM825x",
	.type = v4l2_int_type_slave,
	.u = { .slave = &tcm825x_slave },
};

int __init sensor_tcm825x_init(void)
{
	struct tcm825x_sensor *sensor = &tcm825x;

	sensor->sensor_config =
		omap_get_config(OMAP_TAG_CAMERA_SENSOR,
				struct omap_camera_sensor_config);

	if (sensor->sensor_config == NULL
	    && !sensor->sensor_config->is_okay())
		return -ENODEV;

	sensor->i2c_driver = &tcm825x_i2c_driver;
	tcm825x_if.priv = sensor;

	/* Make the default capture format QVGA RGB565 */
	sensor->pix.width = tcm825x_sizes[QVGA].width;
	sensor->pix.height = tcm825x_sizes[QVGA].height;
	sensor->pix.pixelformat = V4L2_PIX_FMT_RGB565;

	sensor->timeperframe.denominator = DEFAULT_FPS;
	sensor->timeperframe.numerator = 1;

	return v4l2_int_device_register(&tcm825x_if);
}

void __exit sensor_tcm825x_cleanup(void)
{
	v4l2_int_device_unregister(&tcm825x_if);
}

/*
 * FIXME: Menelaus isn't ready at module_init stage, so use
 * late_initcall for now.
 */
late_initcall(sensor_tcm825x_init);
module_exit(sensor_tcm825x_cleanup);
