/*
 * sensor_if.h
 *
 * An interface for V4L2 sensors.
 *
 * Copyright (C) 2007 Nokia Corporation.
 *
 * Contact: Sakari Ailus <sakari.ailus@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#ifndef SENSOR_IF_H
#define SENSOR_IF_H

#include <linux/module.h>
#include <media/v4l2-common.h>

#define V4L2NAMESIZE 32

enum v4l2_int_type {
	v4l2_int_type_master = 1,
	v4l2_int_type_slave
};

struct v4l2_int_device;

struct v4l2_int_master {
	int (*attach)(struct v4l2_int_device *master,
		      struct v4l2_int_device *slave);
	void (*detach)(struct v4l2_int_device *master);
};

struct v4l2_int_slave {
	/* Don't touch master. */
	struct v4l2_int_device *master;

	char attach_to[V4L2NAMESIZE];

	/*
	 * Initialise and deinitialise the sensor device. The
	 * initialisation is run power and clocks on.
	 */
	int (*dev_init)(struct v4l2_int_device *s);
	int (*dev_exit)(struct v4l2_int_device *s);

	int (*power_set)(struct v4l2_int_device *s, int on);

	/*
	 * Get the desired external clock rate for current format /
	 * parameters.
	 */
	u32 (*vidioc_int_g_ext_clk)(struct v4l2_int_device *s);
	/* Tell what we're actually getting. */
	int (*vidioc_int_s_ext_clk)(struct v4l2_int_device *s, u32 ext_clk);

	/* Ask whether the sensor driver feels it needs to be reset. */
	int (*vidioc_int_g_needs_reset)(struct v4l2_int_device *s, void *buf);

	/* Mimic V4L2 ioctls. Please keep the order same. */
	int (*vidioc_int_g_chip_ident)(struct v4l2_int_device *s,
				       enum v4l2_chip_ident *i);

	int (*vidioc_int_reset)(struct v4l2_int_device *s);
	int (*vidioc_int_init)(struct v4l2_int_device *s);

	int (*vidioc_int_enum_fmt_cap)(struct v4l2_int_device *s,
				       struct v4l2_fmtdesc *fmtdesc);
	int (*vidioc_int_s_fmt_cap)(struct v4l2_int_device *s,
				    struct v4l2_format *f);
	int (*vidioc_int_g_fmt_cap)(struct v4l2_int_device *s,
				    struct v4l2_format *f);
	int (*vidioc_int_try_fmt_cap)(struct v4l2_int_device *s,
				      struct v4l2_format *f);

	int (*vidioc_int_queryctrl)(struct v4l2_int_device *s,
				    struct v4l2_queryctrl *ctrl);
	int (*vidioc_int_g_ctrl)(struct v4l2_int_device *s,
				 struct v4l2_control *ctrl);
	int (*vidioc_int_s_ctrl)(struct v4l2_int_device *s,
				 struct v4l2_control *ctrl);

	int (*vidioc_int_g_parm)(struct v4l2_int_device *s,
				 struct v4l2_streamparm *a);
	int (*vidioc_int_s_parm)(struct v4l2_int_device *s,
				 struct v4l2_streamparm *a);
};

struct v4l2_int_device {
	/* Don't touch head. */
	struct list_head head;

	struct module *module;

	char name[V4L2NAMESIZE];

	enum v4l2_int_type type;
	union {
		struct v4l2_int_master *master;
		struct v4l2_int_slave *slave;
	} u;

	void *priv;
};

int v4l2_int_device_register(struct v4l2_int_device *d);
void v4l2_int_device_unregister(struct v4l2_int_device *d);

#endif
