/*
 * drivers/i2c/chips/tsl2563.c
 *
 * Copyright (C) 2007 Nokia Corporation
 *
 * Written by Timo O. Karjalainen <timo.o.karjalainen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/mutex.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/pm.h>
#include <linux/hwmon.h>
#include <linux/err.h>
#include <linux/tsl2563.h>
#include <asm/arch/board.h>

#define DRIVER_NAME  "tsl2563"

#define DEBUG

#undef pr_info
#undef pr_debug
#define pr_info(fmt, arg...)	printk(KERN_INFO  DRIVER_NAME ": " fmt, ## arg)
#define pr_warn(fmt, arg...)	printk(KERN_WARNING DRIVER_NAME ": " fmt, ## arg)
#define pr_err(fmt, arg...)	printk(KERN_ERR   DRIVER_NAME ": " fmt, ## arg)
#define pr_debug(fmt, arg...)	printk(KERN_DEBUG DRIVER_NAME ": " fmt, ## arg)

/* Use this many bits for fraction part. */
#define ADC_FRAC_BITS		(14)

/* Given number of 1/10000's in ADC_FRAC_BITS precision. */
#define FRAC10K(f)		(((f) * (1L << (ADC_FRAC_BITS))) / (10000))

/* Bits used for fraction in calibration coefficients.*/
#define CALIB_FRAC_BITS		(10)
/* 0.5 in CALIB_FRAC_BITS precision */
#define CALIB_FRAC_HALF		(1 << (CALIB_FRAC_BITS - 1))
/* Make a fraction from a number n that was multiplied with b. */
#define CALIB_FRAC(n,b)		(((n) << CALIB_FRAC_BITS) / (b))
/* Decimal 10^(digits in sysfs presentation) */
#define CALIB_BASE_SYSFS	(1000)

/*
 * Type used to store calibration coefficients. This is a fixed point number
 * having CALIB_FRAC_BITS bits of fraction.
 */
typedef unsigned long calib_t;


#define TSL2563_CMD		(0x80)
#define TSL2563_CLEARINT	(0x40)

#define TSL2563_REG_CTRL	(0x00)
#define TSL2563_REG_TIMING	(0x01)
#define TSL2563_REG_LOWLOW	(0x02) /* data0 low threshold, 2 bytes */
#define TSL2563_REG_LOWHIGH	(0x03)
#define TSL2563_REG_HIGHLOW	(0x04) /* data0 high threshold, 2 bytes */
#define TSL2563_REG_HIGHHIGH	(0x05)
#define TSL2563_REG_INT		(0x06)
#define TSL2563_REG_ID		(0x0a)
#define TSL2563_REG_DATA0LOW	(0x0c) /* broadband sensor value, 2 bytes */
#define TSL2563_REG_DATA0HIGH	(0x0d)
#define TSL2563_REG_DATA1LOW	(0x0e) /* infrared sensor value, 2 bytes */
#define TSL2563_REG_DATA1HIGH	(0x0f)

#define TSL2563_CTRL_POWER_ON	(0x03)
#define TSL2563_CTRL_POWER_OFF	(0x00)
#define TSL2563_CTRL_POWER_MASK (0x03)

#define TSL2563_TIMING_13MS	(0x00)
#define TSL2563_TIMING_100MS	(0x01)
#define TSL2563_TIMING_400MS	(0x02)
#define TSL2563_TIMING_MASK	(0x03)
#define TSL2563_TIMING_GAIN16	(0x10)
#define TSL2563_TIMING_GAIN1	(0x00)

#define TSL2563_INT_LEVEL	(0x10)
#define TSL2563_INT_PERSIST(n)	((n) & 0x0F)

static struct tsl2563_platform_data *tsl2563_pdata;

struct tsl2563_gainlevel_coeff {
	u8 gaintime;
	u16 min;
	u16 max;
};

static struct tsl2563_gainlevel_coeff tsl2563_gainlevel_table[] = {
	{TSL2563_TIMING_400MS | TSL2563_TIMING_GAIN16,	0,	 65534},
	{TSL2563_TIMING_400MS | TSL2563_TIMING_GAIN1,	2048,	 65534},
	{TSL2563_TIMING_100MS | TSL2563_TIMING_GAIN1,	4095,	 37177},
	{TSL2563_TIMING_13MS | TSL2563_TIMING_GAIN1,	3000,	 65535},
};

static unsigned short normal_i2c[] = {
	0x29,	/* 0101 001x, address select low */
	0x39,	/* 0111 001x, address select floats */
	0x49,	/* 1001 001x, address select high */
	I2C_CLIENT_END
};

I2C_CLIENT_INSMOD;

struct tsl2563_chip {
	struct mutex		lock;
	struct i2c_client	client;
	struct class_device	*class_dev;

	int			irq;
	pm_message_t		state;

	void			(*set_power)(struct device *dev, int on);
	int			power_count;

	/*
	 * Remember configuration so we can reconfigure the chip after
	 * a suspend.
	 */
	struct tsl2563_gainlevel_coeff *gainlevel;

	unsigned		low_thres; /* Thresholds are in lux */
	unsigned		high_thres;
	u8			intr;

	/* Calibration coefficients */
	calib_t			calib0;
	calib_t			calib1;

	/* Cache current values, to be returned while suspended */
	u32			data0;
	u32			data1;
};

#define client_to_tsl2563(c)	container_of(c, struct tsl2563_chip, client)
#define dev_to_tsl2563(d)	container_of(d, struct tsl2563_chip, \
					     client.dev)

static int tsl2563_write(struct i2c_client *client, u8 reg, u8 value)
{
	int ret;
	u8 buf[2];

	buf[0] = TSL2563_CMD | reg;
	buf[1] = value;

	ret = i2c_master_send(client, buf, sizeof(buf));
	return (ret == sizeof(buf)) ? 0 : ret;
}

static int tsl2563_read(struct i2c_client *client, u8 reg, void *buf, int len)
{
	int ret;
	u8 cmd = TSL2563_CMD | reg;

	ret = i2c_master_send(client, &cmd, sizeof(cmd));
	if (ret != sizeof(cmd)) {
		return ret;
	}

	return i2c_master_recv(client, buf, len);
}

static int tsl2563_set_power(struct tsl2563_chip *chip, int on)
{
	struct i2c_client *client = &chip->client;
	u8 cmd;

	cmd = on ? TSL2563_CTRL_POWER_ON : TSL2563_CTRL_POWER_OFF;
	return tsl2563_write(client, TSL2563_REG_CTRL, cmd);
}

/*
 * Return value is 0 for off, 1 for on, or a negative error
 * code if reading failed.
 */
static int tsl2563_get_power(struct tsl2563_chip *chip)
{
	struct i2c_client *client = &chip->client;
	int ret;
	u8 val;

	ret = tsl2563_read(client, TSL2563_REG_CTRL, &val, sizeof(val));
	if (ret != sizeof(val))
		return ret;

	return (val & TSL2563_CTRL_POWER_MASK) == TSL2563_CTRL_POWER_ON;
}

static int tsl2563_configure(struct tsl2563_chip *chip)
{
	struct i2c_client *client = &chip->client;
	int r;

	if ((r = tsl2563_write(client, TSL2563_REG_LOWHIGH,
			       chip->low_thres >> 8)))
		goto out;
	if ((r = tsl2563_write(client, TSL2563_REG_LOWLOW,
			       chip->low_thres & 0xff)))
		goto out;
	if ((r = tsl2563_write(client, TSL2563_REG_HIGHHIGH,
			       chip->high_thres >> 8)))
		goto out;
	if ((r = tsl2563_write(client, TSL2563_REG_HIGHLOW,
			       chip->high_thres & 0xff)))
		goto out;

	if ((r = tsl2563_write(client, TSL2563_REG_TIMING,
			       chip->gainlevel->gaintime)))
		goto out;

	r = tsl2563_write(client, TSL2563_REG_INT, chip->intr);

out:
	return r;
}

static int tsl2563_power_inc(struct tsl2563_chip *chip)
{
	int restarted = 0;
	if (!chip->power_count++) {
		if (chip->set_power) {
			chip->set_power(&chip->client.dev, 1);
			/*
			 * This is optimization that we will re-configure the
			 * chip only if it was really unpowered. This is
			 * due that in some platform, platform specific
			 * set_power could only force power on but not off.
			 * Re-configuration is necessary only when the chip
			 * power really went down
			 */
			if (!tsl2563_get_power(chip)) {
				tsl2563_set_power(chip, 1);
				tsl2563_configure(chip);
				restarted = 1;
			}
		} else {
			tsl2563_set_power(chip, 1);
			tsl2563_configure(chip);
			restarted = 1;
		}
	}
	return restarted;
}

static void tsl2563_power_dec(struct tsl2563_chip *chip)
{
	if (!--chip->power_count) {
		if (chip->set_power)
			chip->set_power(&chip->client.dev, 0);
		else
			tsl2563_set_power(chip, 0);
	}
}

static int tsl2563_detect(struct tsl2563_chip *chip)
{
	int ret;

	if ((ret = tsl2563_set_power(chip, 1)))
		return ret;

	ret = tsl2563_get_power(chip);
	if (ret < 0)
		return ret;

	return ret ? 0 : -ENODEV;
}

static int tsl2563_read_id(struct tsl2563_chip *chip, u8 *id)
{
	struct i2c_client *client = &chip->client;
	int ret;

	ret = tsl2563_read(client, TSL2563_REG_ID, id, sizeof(*id));
	if (ret != sizeof(*id))
		return ret;

	return 0;
}

/*
 * "Normalized" ADC value is one obtained with 400ms of integration time and
 * 16x gain. This function returns the number of bits of shift needed to
 * convert between normalized values and HW values obtained using given
 * timing and gain settings.
 */
static int adc_shiftbits(u8 timing)
{
	int shift = 0;

	switch (timing & TSL2563_TIMING_MASK) {
	case TSL2563_TIMING_13MS:
		shift += 5;
		break;
	case TSL2563_TIMING_100MS:
		shift += 2;
		break;
	case TSL2563_TIMING_400MS:
		/* no-op */
		break;
	}

	if (!(timing & TSL2563_TIMING_GAIN16))
		shift += 4;

	return shift;
}

/* Convert a HW ADC value to normalized scale. */
static u32 normalize_adc(u16 adc, u8 timing)
{
	return adc << adc_shiftbits(timing);
}

/* Convert normalized ADC value to HW value. */
static u16 denormalize_adc(u32 normal, u8 timing)
{
	return normal >> adc_shiftbits(timing);
}

static void tsl2563_wait_adc(struct tsl2563_chip *chip)
{
	unsigned int delay;

	switch (chip->gainlevel->gaintime & TSL2563_TIMING_MASK) {
	case TSL2563_TIMING_13MS:
		delay = 14;
		break;
	case TSL2563_TIMING_100MS:
		delay = 101;
		break;
	default:
		delay = 402;
	}
	/*
	 * TODO: Make sure that we wait at least required delay but why we
	 * have to extend it one tick more?
	 */
	schedule_timeout_interruptible(msecs_to_jiffies(delay) + 2);
}

static int tsl2563_adjust_gainlevel(struct tsl2563_chip *chip, u16 adc)
{
	struct i2c_client *client = &chip->client;

	if (adc > chip->gainlevel->max || adc < chip->gainlevel->min) {

		(adc > chip->gainlevel->max) ?
			chip->gainlevel++ : chip->gainlevel--;

		tsl2563_write(client, TSL2563_REG_TIMING,
			      chip->gainlevel->gaintime);

		tsl2563_wait_adc(chip);
		tsl2563_wait_adc(chip);

		return 1;
	} else
		return 0;
}

static int tsl2563_get_adc(struct tsl2563_chip *chip)
{
	struct i2c_client *client = &chip->client;
	u8 buf0[2], buf1[2];
	u16 adc0, adc1;
	int retry = 1;
	int ret = 0;

	/* Wait over first integration cycle if we just powered up */
	if (tsl2563_power_inc(chip))
		tsl2563_wait_adc(chip);

	if (chip->state.event != PM_EVENT_ON)
		goto out;

	while (retry) {
		ret = tsl2563_read(client,
				   TSL2563_REG_DATA0LOW | TSL2563_CLEARINT,
				   buf0, sizeof(buf0));
		if (ret != sizeof(buf0))
			goto out;

		ret = tsl2563_read(client, TSL2563_REG_DATA1LOW,
				   buf1, sizeof(buf1));
		if (ret != sizeof(buf1))
			goto out;

		adc0 = (buf0[1] << 8) + buf0[0];
		adc1 = (buf1[1] << 8) + buf1[0];

		retry = tsl2563_adjust_gainlevel(chip, adc0);
	}

	chip->data0 = normalize_adc(adc0, chip->gainlevel->gaintime);
	chip->data1 = normalize_adc(adc1, chip->gainlevel->gaintime);

	ret = 0;
out:
	tsl2563_power_dec(chip);

	return ret;
}

static inline int calib_to_sysfs(calib_t calib)
{
	return (int) (((calib * CALIB_BASE_SYSFS) +
		       CALIB_FRAC_HALF) >> CALIB_FRAC_BITS);
}

static inline calib_t calib_from_sysfs(int value)
{
	return (((calib_t) value) << CALIB_FRAC_BITS) / CALIB_BASE_SYSFS;
}

/*
 * Conversions between lux and ADC values.
 *
 * The basic formula is lux = c0 * adc0 - c1 * adc1, where c0 and c1 are
 * appropriate constants. Different constants are needed for different
 * kinds of light, determined by the ratio adc1/adc0 (basically the ratio
 * of the intensities in infrared and visible wavelengths). lux_table below
 * lists the upper threshold of the adc1/adc0 ratio and the corresponding
 * constants.
 */

struct tsl2563_lux_coeff {
	unsigned long ch_ratio;
	unsigned long ch0_coeff;
	unsigned long ch1_coeff;
};

static const struct tsl2563_lux_coeff lux_table[] = {
	{ FRAC10K(1300),	FRAC10K(315),	FRAC10K(262)	},
	{ FRAC10K(2600),	FRAC10K(337),	FRAC10K(430)	},
	{ FRAC10K(3900),	FRAC10K(363),	FRAC10K(529)	},
	{ FRAC10K(5200),	FRAC10K(392),	FRAC10K(605)	},
	{ FRAC10K(6500),	FRAC10K(229),	FRAC10K(291)	},
	{ FRAC10K(8000),	FRAC10K(157),	FRAC10K(180)	},
	{ FRAC10K(13000),	FRAC10K(34),	FRAC10K(26)	},
	{ ULONG_MAX,		0,		0		}
};

/*
 * Convert normalized, scaled ADC values to lux.
 */
static unsigned int adc_to_lux(u32 adc0, u32 adc1)
{
	const struct tsl2563_lux_coeff *lp = lux_table;
	unsigned long ratio, lux, ch0 = adc0, ch1 = adc1;

	ratio = ch0 ? ((ch1 << ADC_FRAC_BITS) / ch0) : ULONG_MAX;

	while (lp->ch_ratio < ratio)
		lp++;

	lux = ch0 * lp->ch0_coeff - ch1 * lp->ch1_coeff;

	return (unsigned int) (lux >> ADC_FRAC_BITS);
}

/*
 * Approximate a channel0 ADC value from a lux value.
 * This is needed for setting HW interrupt thresholds as ADC values,
 * whereas userspace requests the thresholds in lux.
 * This reverse conversion cannot be done exactly, since for any given
 * adc0 the lux value depends on adc1. However, in "short range" the
 * kind of light probably stays the same (only the intensity changes).
 * Therefore we do a simple linear estimate from the current lux, as computed
 * from the given ADC values now, to the target lux. This should be reasonably
 * close assuming that the target lux is relatively close to current lux.
 */
static u32 lux_to_adc(unsigned int lux, u32 now0, u32 now1)
{
	unsigned long luxnow, luxtarget = lux;

	luxnow = adc_to_lux(now0, now1);
	return (u32) ((now0 * luxtarget) / luxnow);

}

static ssize_t tsl2563_adc_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int ret;

	mutex_lock(&chip->lock);

	ret = tsl2563_get_adc(chip);
	if (ret)
		return ret;

	ret = snprintf(buf, PAGE_SIZE, "%d %d\n", chip->data0, chip->data1);
	mutex_unlock(&chip->lock);

	return ret;
}

/* Apply calibration coefficient to ADC count. */
static u32 calib_adc(u32 adc, calib_t calib)
{
	unsigned long scaled = adc;

	scaled *= calib;
	scaled >>= CALIB_FRAC_BITS;

	return (u32) scaled;
}

static ssize_t tsl2563_lux_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	u32 calib0, calib1;
	int ret;

	mutex_lock(&chip->lock);

	ret = tsl2563_get_adc(chip);
	if (ret)
		goto out;

	calib0 = calib_adc(chip->data0, chip->calib0);
	calib1 = calib_adc(chip->data1, chip->calib1);

	ret = snprintf(buf, PAGE_SIZE, "%d\n", adc_to_lux(calib0, calib1));

out:
	mutex_unlock(&chip->lock);
	return ret;
}

static ssize_t format_calib(char *buf, int len, calib_t calib)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", calib_to_sysfs(calib));
}

static ssize_t tsl2563_calib0_show(struct device *dev,
				   struct device_attribute *attr, char *buf)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int ret;

	mutex_lock(&chip->lock);
	ret = format_calib(buf, PAGE_SIZE, chip->calib0);
	mutex_unlock(&chip->lock);
	return ret;
}

static ssize_t tsl2563_calib1_show(struct device *dev,
				   struct device_attribute *attr, char *buf)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int ret;

	mutex_lock(&chip->lock);
	ret = format_calib(buf, PAGE_SIZE, chip->calib1);
	mutex_unlock(&chip->lock);
	return ret;
}

#if 0 /* FIXME: disabled until fixed to take into account powering changes */
static ssize_t tsl2563_lowthres_show(struct device *dev,
				     struct device_attribute *attr, char *buf)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int ret;

	mutex_lock(&chip->lock);
	ret = snprintf(buf, PAGE_SIZE, "%d\n", chip->low_thres);
	mutex_unlock(&chip->lock);
	return ret;
}

static ssize_t tsl2563_highthres_show(struct device *dev,
				      struct device_attribute *attr, char *buf)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int ret;

	mutex_lock(&chip->lock);
	ret = snprintf(buf, PAGE_SIZE, "%d\n", chip->high_thres);
	mutex_unlock(&chip->lock);
	return ret;
}

static ssize_t tsl2563_lowthres_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t len)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	struct i2c_client *client = &chip->client;
	int value, ret = 0;

	if (1 != sscanf(buf, "%u", &value)) {
		ret = -EINVAL;
		goto out1;
	}

	mutex_lock(&chip->lock);
	chip->low_thres = value;

	if (chip->state.event == PM_EVENT_ON) {
		u16 adc;

		/*
		 * TODO: should we reread ADC counts from the chip or are
		 * the cached values recent enough?
		 */
		adc = lux_to_adc(value, chip->data0, chip->data1);
		adc = denormalize_adc(adc, chip->gainlevel->gaintime);

		if ((ret = tsl2563_write(client,
					 TSL2563_REG_LOWHIGH, adc >> 8)))
			goto out2;

		if ((ret = tsl2563_write(client,
					 TSL2563_REG_LOWLOW, adc & 0xff)))
			goto out2;

		ret = len;
	}

out2:
	mutex_unlock(&chip->lock);
out1:
	return ret;
}

static ssize_t tsl2563_highthres_store(struct device *dev,
				       struct device_attribute *attr,
				       const char *buf, size_t len)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	struct i2c_client *client = &chip->client;
	int value, ret = 0;

	if (1 != sscanf(buf, "%u", &value)) {
		ret = -EINVAL;
		goto out1;
	}

	mutex_lock(&chip->lock);
	chip->high_thres = value;

	if (chip->state.event == PM_EVENT_ON) {
		u16 adc;

		/*
		 * TODO: should we reread ADC counts from the chip or are
		 * the cached values recent enough?
		 */
		adc = lux_to_adc(value, chip->data0, chip->data1);
		adc = denormalize_adc(adc, chip->gainlevel->gaintime);

		if ((ret = tsl2563_write(client,
					 TSL2563_REG_HIGHHIGH, adc >> 8)))
			goto out2;

		if ((ret = tsl2563_write(client,
					 TSL2563_REG_HIGHLOW, adc & 0xff)))
			goto out2;

		ret = len;
	}

out2:
	mutex_unlock(&chip->lock);

out1:
	return ret;
}
#endif

static int do_calib_store(struct device *dev, const char *buf, size_t len,
			  int ch)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int value;
	calib_t calib;

	if (1 != sscanf(buf, "%d", &value))
		return -EINVAL;

	calib = calib_from_sysfs(value);

	if (ch)
		chip->calib1 = calib;
	else
		chip->calib0 = calib;

	return len;
}

static ssize_t tsl2563_calib0_store(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf, size_t len)
{
	return do_calib_store(dev, buf, len, 0);
}

static ssize_t tsl2563_calib1_store(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf, size_t len)
{
	return do_calib_store(dev, buf, len, 1);
}

static struct device_attribute tsl2563_attr_adc =
	__ATTR(adc, S_IRUGO, tsl2563_adc_show, NULL);

static struct device_attribute tsl2563_attr_lux =
	__ATTR(lux, S_IRUGO, tsl2563_lux_show, NULL);

static struct device_attribute tsl2563_attr_calib0 =
	__ATTR(calib0, S_IRUGO | S_IWUSR,
	       tsl2563_calib0_show, tsl2563_calib0_store);

static struct device_attribute tsl2563_attr_calib1 =
	__ATTR(calib1, S_IRUGO | S_IWUSR,
	       tsl2563_calib1_show, tsl2563_calib1_store);

#if 0 /* FIXME: disabled until fixed to take into account powering changes */
static struct device_attribute tsl2563_attr_lowthres =
	__ATTR(lowthres, S_IRUGO | S_IWUSR,
	       tsl2563_lowthres_show, tsl2563_lowthres_store);

static struct device_attribute tsl2563_attr_highthres =
	__ATTR(highthres, S_IRUGO | S_IWUSR,
	       tsl2563_highthres_show, tsl2563_highthres_store);
#endif

static int tsl2563_register_sysfs(struct i2c_client *client)
{
	struct device *d = &client->dev;
	int r;

	if ((r = device_create_file(d, &tsl2563_attr_adc)))
		goto fail1;

	if ((r = device_create_file(d, &tsl2563_attr_lux)))
		goto fail2;

#if 0 /* FIXME: disabled until fixed to take into account powering changes */
	if ((r = device_create_file(d, &tsl2563_attr_lowthres)))
		goto fail3;

	if ((r = device_create_file(d, &tsl2563_attr_highthres)))
		goto fail4;
#endif

	if ((r = device_create_file(d, &tsl2563_attr_calib0)))
		goto fail5;

	if ((r = device_create_file(d, &tsl2563_attr_calib1)))
		goto fail6;

	return 0;

fail6:
	device_remove_file(d, &tsl2563_attr_calib0);
fail5:
#if 0 /* FIXME: disabled until fixed to take into account powering changes */
	device_remove_file(d, &tsl2563_attr_highthres);
fail4:
	device_remove_file(d, &tsl2563_attr_lowthres);
fail3:
#endif
	device_remove_file(d, &tsl2563_attr_lux);
fail2:
	device_remove_file(d, &tsl2563_attr_adc);
fail1:
	return r;
}

static void tsl2563_unregister_sysfs(struct i2c_client *client)
{
	struct device *d = &client->dev;

	device_remove_file(d, &tsl2563_attr_calib1);
	device_remove_file(d, &tsl2563_attr_calib0);
#if 0 /* FIXME: disabled until fixed to take into account powering changes */
	device_remove_file(d, &tsl2563_attr_highthres);
	device_remove_file(d, &tsl2563_attr_lowthres);
#endif
	device_remove_file(d, &tsl2563_attr_lux);
	device_remove_file(d, &tsl2563_attr_adc);
}

static irqreturn_t tsl2563_irq(int irq, void *_tsl2563)
{
	struct tsl2563_chip *chip = _tsl2563;
	struct i2c_client *client = &chip->client;

	/*
	 * TODO: convert ADC to lux, notify userspace only if out of
	 * thresholds
	 */
	sysfs_notify(&client->dev.kobj, NULL, "adc");
	sysfs_notify(&client->dev.kobj, NULL, "lux");

	/* Failure of these calls is not significant */
	if (sysfs_update_file(&client->dev.kobj, &tsl2563_attr_adc.attr));
	if (sysfs_update_file(&client->dev.kobj, &tsl2563_attr_lux.attr));

	return IRQ_HANDLED;
}

static struct i2c_driver tsl2563_i2c_driver;

static int tsl2563_detect_client(struct i2c_adapter *adapter, int address,
				 int kind)
{
	struct i2c_client *client;
	struct tsl2563_chip *chip;
	struct tsl2563_platform_data *pdata = tsl2563_pdata;
	int err = 0;
	u8 id;

	if (!(chip = kzalloc(sizeof(*chip), GFP_KERNEL)))
		return -ENOMEM;

	client = &chip->client;
	strncpy(client->name, DRIVER_NAME, sizeof(client->name));
	client->addr	= address;
	client->adapter	= adapter;
	client->driver	= &tsl2563_i2c_driver;
	client->flags	= 0;

	if ((err = i2c_attach_client(client)) < 0) {
		pr_err("couldn't attach\n");
		goto fail1;
	}

	mutex_init(&chip->lock);

	chip->set_power = pdata->set_power;
	chip->irq = pdata->irq;
#if 0 /* FIXME: disabled until fixed to take into account powering changes */
	err = request_irq(chip->irq, tsl2563_irq,
			  IRQF_TRIGGER_FALLING | IRQF_DISABLED |
			  IRQF_SAMPLE_RANDOM, DRIVER_NAME, chip);
	if (err) {
		pr_err("could not get IRQ %d\n", chip->irq);
		goto fail2;
	}
#endif

	/* Default values used until userspace says otherwise */
	chip->low_thres = 0x0;
	chip->high_thres = 0xffff;
	chip->gainlevel = tsl2563_gainlevel_table;
	chip->intr = TSL2563_INT_PERSIST(4);
	chip->calib0 = calib_from_sysfs(CALIB_BASE_SYSFS);
	chip->calib1 = calib_from_sysfs(CALIB_BASE_SYSFS);

	/* Probe for the device */
	if (kind < 0) {
		if ((err = tsl2563_detect(chip))) {
			pr_err("device not found, error %d\n", -err);
			goto fail3;
		}
	}

	if ((err = tsl2563_read_id(chip, &id)))
		goto fail3;

	pr_info("model %d, rev. %d\n", id >> 4, id & 0x0f);

	if ((err = tsl2563_configure(chip)))
		goto fail3;

	/*
	 * Keep in power-down and let the tsl2563_power_inc and _dec functions
	 * to manage power according to chip usage
	 */
	if (chip->set_power)
		chip->set_power(&chip->client.dev, 0);
	else
		tsl2563_set_power(chip, 0);

	chip->class_dev = hwmon_device_register(&client->dev);
	if (IS_ERR(chip->class_dev))
		goto fail3;

	if ((err = tsl2563_register_sysfs(client))) {
		pr_err("sysfs registration failed, error %d\n", err);
		goto fail4;
	}

	return 0;
fail4:
	hwmon_device_unregister(chip->class_dev);
fail3:
	free_irq(chip->irq, chip);
fail2:
	i2c_detach_client(client);
fail1:
	kfree(chip);
	return err;
}

static int tsl2563_scan_bus(struct i2c_adapter *bus)
{
	return i2c_probe(bus, &addr_data, tsl2563_detect_client);
}

static int tsl2563_detach_client(struct i2c_client *client)
{
	struct tsl2563_chip *chip = client_to_tsl2563(client);
	int err;

	tsl2563_unregister_sysfs(client);
	hwmon_device_unregister(chip->class_dev);
	free_irq(chip->irq, chip);

	if ((err = i2c_detach_client(client))) {
		pr_err("I2C deregistration failed\n");
		return err;
	}

	kfree(chip);
	return 0;
}

static int tsl2563_suspend(struct device *dev, pm_message_t state)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int ret;

	mutex_lock(&chip->lock);

	if ((ret = tsl2563_set_power(chip, 0)))
		goto out;

	chip->state = state;

out:
	mutex_unlock(&chip->lock);
	return ret;
}

static int tsl2563_resume(struct device *dev)
{
	struct tsl2563_chip *chip = dev_to_tsl2563(dev);
	int ret;

	mutex_lock(&chip->lock);

	if ((ret = tsl2563_set_power(chip, 1)))
		goto out;

	if ((ret = tsl2563_configure(chip)))
		goto out;

	chip->state.event = PM_EVENT_ON;

out:
	mutex_unlock(&chip->lock);
	return ret;
}

static struct i2c_driver tsl2563_i2c_driver = {
	.driver = {
		.name	 = DRIVER_NAME,
		.suspend = tsl2563_suspend,
		.resume	 = tsl2563_resume
	},
	/* FIXME: id according to i2c-ids.h */
	.id		= I2C_DRIVERID_MISC,
	.class		= I2C_CLASS_HWMON,
	.attach_adapter	= tsl2563_scan_bus,
	.detach_client	= tsl2563_detach_client,
};

static int tsl2563_probe(struct platform_device *pdev)
{
	struct tsl2563_platform_data *pdata = pdev->dev.platform_data;

	if (tsl2563_pdata) {
		dev_err(&pdev->dev,
			"only single %s platform device supported\n",
			DRIVER_NAME);
		return -EBUSY;
	}
	tsl2563_pdata = pdata;
	if (tsl2563_pdata->set_power)
		tsl2563_pdata->set_power(&pdev->dev, 1);

	return i2c_add_driver(&tsl2563_i2c_driver);
}

static int tsl2563_remove(struct platform_device *pdev)
{
	int ret;

	ret = i2c_del_driver(&tsl2563_i2c_driver);
	tsl2563_pdata = NULL;

	return ret;
}

static struct platform_driver tsl2563_driver = {
	.driver = {
		.owner	= THIS_MODULE,
		.name	= "tsl2563",
	},
	.probe		= tsl2563_probe,
	.remove		= tsl2563_remove,
};

static int __init tsl2563_init(void)
{
	if (platform_driver_register(&tsl2563_driver)) {
		printk(KERN_ERR "Failed to register %s driver\n",
		       DRIVER_NAME);
		return -ENODEV;
	}

	return 0;
}

static void __exit tsl2563_exit(void)
{
	platform_driver_unregister(&tsl2563_driver);
}

MODULE_AUTHOR("Nokia Corporation");
MODULE_DESCRIPTION("tsl2563 light sensor driver");
MODULE_LICENSE("GPL");

module_init(tsl2563_init);
module_exit(tsl2563_exit);
