/*
 * linux/arch/arm/plat-omap/pa.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Sami Tolvanen <sami.tolvanen@nokia.com>
 *
 * OMAP HS protected application (PA) format handling
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <asm/arch/pa.h>
#include <asm/arch/sec.h>

MODULE_LICENSE("GPL v2");

#define PA_NAME			"omap2_pa"
#define PA_FMT_MAGIC		0x4D464150
#define PA_MAX_NCMD		128

enum {
	PA_TYPE_DATA,
	PA_TYPE_PTR_VIRT,
	PA_TYPE_PTR_PHYS,
	PA_TYPE_PTR_KEYS,
	PA_TYPE_PTR_PAPUB,
};

enum {
	PA_IO_NORMAL,
	PA_IO_RESERVED,
	PA_IO_DEFOUT,
	PA_IO_SIZE_VAL,
	PA_IO_SIZE_PAR,
	PA_IO_ONLY,
};

typedef struct {
	u32 start;
	u32 size;
	u32 spare0;
	u32 spare1;
	u32 spare2;
	u8 filename[12];
} pa_image_toc_entry;


static int error_buffer_size(void)
{
	printk(KERN_ERR PA_NAME ": buffer too small\n");
	return -1;
}

static int error_format_invalid(void)
{
	printk(KERN_ERR PA_NAME ": invalid format\n");
	return -1;
}

static int error_entry_invalid(pa_format_entry *entry)
{
	printk(KERN_ERR PA_NAME ": invalid type %u for ref %u\n",
		entry->type, entry->ref);
	return -1;
}

static int error_entry_unsupported(pa_format_entry *entry)
{
	printk(KERN_ERR PA_NAME ": unsupported type %u\n", entry->type);
	return -1;
}

static int error_entry_reference(pa_format_entry *entry)
{
	printk(KERN_ERR PA_NAME ": invalid parameter reference %u\n",
		entry->value);
	return -1;
}

static int buffer_skip(u8 **buf, u8 *end, u32 size)
{
	if (end < *buf + size)
		return error_buffer_size();
	*buf += size;
	return 0;
}

static int buffer_copy(u8 **buf, u8 *end, u8 *data, u32 size)
{
	if (end < *buf + size)
		return error_buffer_size();
	memcpy(*buf, data, size);
	*buf += size;
	return 0;
}

static int buffer_move(u8 **dbuf, u8 *dend, u8 **sbuf, u8 *send, u32 size)
{
	if (dend < *dbuf + size || send < *sbuf + size)
		return error_buffer_size();
	memcpy(*dbuf, *sbuf, size);
	*dbuf += size;
	*sbuf += size;
	return 0;
}

static inline int buffer_aligned(u8 *ptr)
{
	return ((u32)ptr % sizeof(ptr) == 0);
}

int pa_image_address_get(u8 *base, u8 *name, void **addr)
{
	const pa_image_toc_entry *p;

	for (p = (void *)base; p->filename[0] != 0xFF; ++p) {
		if (!strcmp(name, p->filename)) {
			*addr = base + p->start;
			return 0;
		}
	}
	return -1;
}
EXPORT_SYMBOL(pa_image_address_get);

void pa_format_free(pa_format *p)
{
	u32 i;

	for (i = 0; i < p->ncmd; ++i) {
		kfree(p->cmd[i].par);
		kfree(p->cmd[i].res);
	}

	kfree(p->cmd);
	memset(p, 0, sizeof(*p));
}
EXPORT_SYMBOL(pa_format_free);

static int entry_parse(u8 **base, u8 *end, pa_format_entry ***p, u32 n)
{
	pa_format_entry **entry;
	u32 i;

	if (!n)
		return 0;

	entry = kzalloc(n * sizeof(*entry), GFP_KERNEL);
	if (!entry)
		return -ENOMEM;
	*p = entry;

	for (i = 0; i < n; ++i) {
		entry[i] = (pa_format_entry *)*base;
		if (buffer_skip(base, end, sizeof(pa_format_entry)) < 0)
			return -1;
	}
	return 0;
}

int pa_format_parse(u8 *base, u32 size, pa_format *p)
{
	pa_format_header *hdr = (pa_format_header *)base;
	u32 i;
	u8 *end = base + size;

	if (hdr->magic != PA_FMT_MAGIC || hdr->ncmd > PA_MAX_NCMD)
		return error_format_invalid();

	if (buffer_skip(&base, end, sizeof(*hdr)) < 0)
		return -1;

	p->ncmd = hdr->ncmd;
	p->cmd  = kzalloc(p->ncmd * sizeof(*p->cmd), GFP_KERNEL);

	if (!p->cmd)
		return -ENOMEM;

	for (i = 0; i < p->ncmd; ++i) {
		p->cmd[i].cmd = (pa_format_command *)base;

		if (buffer_skip(&base, end, sizeof(pa_format_command)) < 0 ||
		    entry_parse(&base, end, &p->cmd[i].par,
				p->cmd[i].cmd->npar) < 0 ||
		    entry_parse(&base, end, &p->cmd[i].res,
				p->cmd[i].cmd->nres) < 0) {
			pa_format_free(p);
			return -1;
		}
	}

	return 0;
}
EXPORT_SYMBOL(pa_format_parse);

static inline void copy_value(u32 *dst, u8 *src, u32 n)
{
	memcpy(dst, src, n);
#ifdef __BIG_ENDIAN /* Preserve the numeric value */
	*dst >>= (sizeof(*dst) - n) << 3;
#endif
}

static int entry_io_size(pa_format_entry **p, int n, pa_command *c, u8 *input,
		u32 cinp, u32 *size);

static int entry_input_value(pa_command *c, u32 n, u8 *input, u32 cinp,
		u32 *value)
{
	pa_format_entry *entry = c->par[n];
	u32 i, size;
	u8 *end = input + cinp;
	*value = 0;

	if (entry->type != PA_TYPE_DATA || entry->size > sizeof(*value))
		return error_entry_invalid(entry);

	for (i = 0; i < n; ++i) {
		if (entry_io_size(c->par, i, c, input, cinp, &size) < 0)
			return -1;
		if (buffer_skip(&input, end, size) < 0)
			return -1;
	}

	if (end < input + entry->size)
		return error_buffer_size();

	copy_value(value, input, entry->size);
	return 0;
}

static int entry_io_size(pa_format_entry **p, int n, pa_command *c, u8 *input,
		u32 cinp, u32 *size)
{
	pa_format_entry *entry = p[n];
	*size = 0;

	switch (entry->ref) {
	default:
		if (entry->type == PA_TYPE_DATA ||
		    entry->type == PA_TYPE_PTR_VIRT ||
		    entry->type == PA_TYPE_PTR_PHYS)
			*size = entry->size;
		break;
	case PA_IO_RESERVED:
		if (entry->type != PA_TYPE_DATA)
			return error_entry_invalid(entry);
		break;
	case PA_IO_ONLY:
		if (entry->type != PA_TYPE_DATA)
			return error_entry_invalid(entry);
		*size = entry->size;
		break;
	case PA_IO_SIZE_VAL:
		if (entry->type != PA_TYPE_PTR_VIRT &&
		    entry->type != PA_TYPE_PTR_PHYS)
			return error_entry_invalid(entry);
		if (p == c->par && n <= entry->value)
			return error_entry_reference(entry);
		if (entry_input_value(c, entry->value, input, cinp, size) < 0)
			return -1;
		*size += entry->size;
		break;
	case PA_IO_SIZE_PAR:
		if (entry->type != PA_TYPE_PTR_VIRT &&
		    entry->type != PA_TYPE_PTR_PHYS)
			return error_entry_invalid(entry);
		if (c->cmd->npar < entry->value)
			return error_entry_reference(entry);
		if (entry_io_size(c->par, entry->value, c, input, cinp,
				size) < 0)
			return -1;
		*size += entry->size;
		break;
	}
	return 0;
}

static int command_io_size(pa_command *c, u8 *buf, u32 bufsize, u32 *mininp,
		u32 *maxout)
{
	u32 i, size;
	*mininp = 0;
	*maxout = 0;

	for (i = 0; i < c->cmd->npar; ++i) {
		if (entry_io_size(c->par, i, c, buf, bufsize, &size) < 0)
			return -1;
		*mininp += size;
	}

	for (i = 0; i < c->cmd->nres; ++i) {
		if (entry_io_size(c->res, i, c, buf, bufsize, &size) < 0)
			return -1;
		*maxout += size;
	}

	return 0;
}

static pa_command * command_find(u32 cmd, pa_format *p)
{
	u32 i;

	for (i = 0; i < p->ncmd; ++i)
		if (p->cmd[i].cmd->cmd == cmd)
			return &p->cmd[i];
	return NULL;
}

int pa_command_query(u32 cmd, u8 *input, u32 cinp, u32 *mininp, u32 *maxout,
		pa_format *p)
{
	pa_command *c = command_find(cmd, p);
	if (!c)
		return -1;
	return command_io_size(c, input, cinp, mininp, maxout);
}
EXPORT_SYMBOL(pa_command_query);

static u32 entry_size(pa_format_entry *entry)
{
	if (entry->type != PA_TYPE_DATA)
		return sizeof(u8 *);

	if (entry->ref != PA_IO_ONLY)
		return entry->size;

	return 0;
}

static void command_size(pa_command *c, u32 *cpar, u32 *cres)
{
	u32 i;
	*cpar = 0;
	*cres = 0;

	for (i = 0; i < c->cmd->npar; ++i)
		*cpar += entry_size(c->par[i]);

	for (i = 0; i < c->cmd->nres; ++i)
		*cres += entry_size(c->res[i]);
}

static int prepare_par_data(pa_command_data *p, u32 n, u8 **ppos, u8 *pend,
		u8 **ipos, u8 *iend)
{
	pa_format_entry *entry = p->c->par[n];

	if (entry->ref == PA_IO_ONLY)
		return buffer_skip(ipos, iend, entry->size);

	if (entry->ref == PA_IO_RESERVED)
		return buffer_copy(ppos, pend, (u8 *)&entry->value,
				entry->size);

	return buffer_move(ppos, pend, ipos, iend, entry->size);
}

static int prepare_entry_ptr(pa_command_data *p, pa_format_entry **entry,
		u32 n, u8 **epos, u8 *eend, u8 **spos, u8 *send)
{
	u32 size;
	u8 *buf = *spos;

	if (entry_io_size(entry, n, p->c, p->input, p->cinp, &size) < 0)
		return -1;

	/* Pointers passed to secure mode must be word-aligned */
	if (!buffer_aligned(buf)) {
		if (buffer_skip(&buf, send, size) < 0)
			return -1;
		buf = kmalloc(size, GFP_KERNEL);
		if (!buf)
			return -ENOMEM;
		memcpy(buf, *spos, size);
	}

	if (entry[n]->type == PA_TYPE_PTR_PHYS)
		buf = p->vtp(buf);

	if (buffer_copy(epos, eend, (u8 *)&buf, sizeof(buf)) < 0 ||
	    buffer_skip(spos, send, size) < 0)
		return -1;

	return 0;
}

static int prepare_par(pa_command_data *p)
{
	int rv = 0;
	sec_hal_par_common *sp = (sec_hal_par_common *)p->par;
	u32 i;
	u8 *ipos = p->input;
	u8 *iend = p->input + p->cinp;
	u8 *ppos = p->par + sizeof(*sp);
	u8 *pend = ppos + p->cpar;

	sp->length = p->cpar + sizeof(*sp);
	sp->index  = p->c->cmd->index;

	for (i = 0; i < p->c->cmd->npar && !rv; ++i) {
		switch (p->c->par[i]->type) {
		default:
			return error_entry_unsupported(p->c->par[i]);
		case PA_TYPE_DATA:
			rv = prepare_par_data(p, i, &ppos, pend, &ipos, iend);
			break;
		case PA_TYPE_PTR_VIRT:
		case PA_TYPE_PTR_PHYS:
			rv = prepare_entry_ptr(p, p->c->par, i, &ppos, pend,
				&ipos, iend);
			break;
		case PA_TYPE_PTR_KEYS:
			if (!p->keys)
				return -1;
			rv = buffer_copy(&ppos, pend, (u8 *)&p->keys,
				sizeof(p->keys));
			break;
		case PA_TYPE_PTR_PAPUB:
			if (!p->papub)
				return -1;
			rv = buffer_copy(&ppos, pend, (u8 *)&p->papub,
				sizeof(p->papub));
			break;
		}
	}

	return rv;
}

static int prepare_res_data(pa_command_data *p, u32 n, u8 **rpos, u8 *rend,
		u8 **opos, u8 *oend)
{
	pa_format_entry *entry = p->c->res[n];

	if (entry->ref == PA_IO_RESERVED)
		return buffer_copy(rpos, rend, (u8 *)&entry->value,
				entry->size);

	if (entry->ref == PA_IO_DEFOUT) {
		if (buffer_skip(opos, oend, entry->size) < 0 ||
		    buffer_copy(rpos, rend, (u8 *)&entry->value,
				entry->size) < 0)
			return -1;

		return 0;
	}

	if (buffer_skip(opos, oend, entry->size) < 0 ||
	    buffer_skip(rpos, rend, entry->size) < 0)
		return -1;

	return 0;
}

static int prepare_res(pa_command_data *p)
{
	int rv = 0;
	u32 i;
	u8 *opos = p->output;
	u8 *oend = p->output + p->coup;
	u8 *rpos = p->res + sizeof(sec_hal_res_common);
	u8 *rend = rpos + p->cres;

	for (i = 0; i < p->c->cmd->nres && !rv; ++i) {
		switch (p->c->res[i]->type) {
		default:
			return error_entry_unsupported(p->c->res[i]);
		case PA_TYPE_DATA:
			rv = prepare_res_data(p, i, &rpos, rend, &opos, oend);
			break;
		case PA_TYPE_PTR_VIRT:
		case PA_TYPE_PTR_PHYS:
			rv = prepare_entry_ptr(p, p->c->res, i, &rpos, rend,
				&opos, oend);
			break;
		}
	}

	return rv;
}

static int finish_par_data(pa_command_data *p, u32 n, u8 **ppos, u8 *pend,
		u8 **ipos, u8 *iend)
{
	pa_format_entry *entry = p->c->par[n];

	if (entry->ref == PA_IO_ONLY)
		return buffer_skip(ipos, iend, entry->size);

	if (entry->ref == PA_IO_RESERVED)
		return buffer_skip(ppos, pend, entry->size);

	if  (buffer_skip(ppos, pend, entry->size) < 0 ||
	     buffer_skip(ipos, iend, entry->size) < 0)
		return -1;

	return 0;
}

static int finish_entry_ptr(pa_command_data *p, pa_format_entry **entry,
		u32 n, u8 **epos, u8 *eend, u8 **spos, u8 *send, int copy)
{
	u32 size;
	u8 *buf = *(u8 **)*epos;

	if (entry_io_size(entry, n, p->c, p->input, p->cinp, &size) < 0)
		return -1;

	if (buf) {
		if (entry[n]->type == PA_TYPE_PTR_PHYS)
			buf = p->ptv(buf);

		if (buf != *spos) {
			if (copy)
				memcpy(*spos, buf, size);
			kfree(buf);
		}
	}

	if (buffer_skip(epos, eend, sizeof(buf)) < 0 ||
	    buffer_skip(spos, send, size) < 0)
		return -1;

	return 0;
}

static int finish_par(pa_command_data *p)
{
	int rv = 0;
	u32 i;
	u8 *ipos = p->input;
	u8 *iend = p->input + p->cinp;
	u8 *ppos = p->par + sizeof(sec_hal_par_common);
	u8 *pend = ppos + p->cpar;

	for (i = 0; i < p->c->cmd->npar && !rv; ++i) {
		switch (p->c->par[i]->type) {
		default:
			return error_entry_unsupported(p->c->par[i]);
		case PA_TYPE_DATA:
			rv = finish_par_data(p, i, &ppos, pend, &ipos, iend);
			break;
		case PA_TYPE_PTR_VIRT:
		case PA_TYPE_PTR_PHYS:
			rv = finish_entry_ptr(p, p->c->par, i, &ppos, pend,
				&ipos, iend, 0);
			break;
		case PA_TYPE_PTR_KEYS:
		case PA_TYPE_PTR_PAPUB:
			rv = buffer_skip(&ppos, pend, sizeof(u8 *));
			break;
		}
	}

	return rv;
}

static int finish_res_data(pa_command_data *p, u32 n, u8 **rpos, u8 *rend,
		u8 **opos, u8 *oend)
{
	pa_format_entry *entry = p->c->res[n];

	if (entry->ref == PA_IO_RESERVED)
		return buffer_skip(rpos, rend, entry->size);

	return buffer_move(opos, oend, rpos, rend, entry->size);
}

static int finish_res(pa_command_data *p)
{
	int rv = 0;
	u32 i;
	u8 *opos = p->output;
	u8 *oend = p->output + p->coup;
	u8 *rpos = p->res + sizeof(sec_hal_res_common);
	u8 *rend = rpos + p->cres;

	for (i = 0; i < p->c->cmd->nres && !rv; ++i) {
		switch (p->c->res[i]->type) {
		default:
			return error_entry_unsupported(p->c->res[i]);
		case PA_TYPE_DATA:
			rv = finish_res_data(p, i, &rpos, rend, &opos, oend);
			break;
		case PA_TYPE_PTR_VIRT:
		case PA_TYPE_PTR_PHYS:
			rv = finish_entry_ptr(p, p->c->res, i, &rpos, rend,
				&opos, oend, 1);
			break;
		}
	}

	return rv;
}

int pa_command_prepare(u32 cmd, pa_command_data *p)
{
	int rv = -ENOMEM;

	p->c = command_find(cmd, p->format);
	if (!p->c)
		return -1;

	command_size(p->c, &p->cpar, &p->cres);

	p->par = kzalloc(p->cpar + sizeof(sec_hal_par_common), GFP_KERNEL);
	p->res = kzalloc(p->cres + sizeof(sec_hal_res_common), GFP_KERNEL);

	if (!p->par || !p->res)
		goto out;

	rv = prepare_par(p);
	if (rv < 0)
		goto out;

	rv = prepare_res(p);
	if (rv < 0)
		goto out;

	return 0;
out:
	if (p->par) {
		finish_par(p);
		kfree(p->par);
	}
	if (p->res) {
		finish_res(p);
		kfree(p->res);
	}
	memset(p, 0, sizeof(*p));
	return rv;
}
EXPORT_SYMBOL(pa_command_prepare);

int pa_command_finish(pa_command_data *p)
{
	int rv = 0;

	if (finish_par(p) < 0)
		rv = -1;
	if (finish_res(p) < 0)
		rv = -1;

	kfree(p->res);
	kfree(p->par);
	memset(p, 0, sizeof(*p));
	return rv;
}
EXPORT_SYMBOL(pa_command_finish);
