/*
 * linux/arch/arm/mach-omap2/peeknpoke.c
 *
 * OMAP2 Debug Routines: memory peek 'n poke
 *
 * Copyright (C) 2007 Nokia Corporation
 * Igor Stoppa <igor.stoppa@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/pm.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/sysfs.h>
#include <linux/module.h>
#include <linux/debugfs.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include <asm/mach/time.h>
#include <asm/mach/irq.h>
#include <asm/mach-types.h>

#include <asm/arch/irqs.h>
#include <asm/arch/clock.h>
#include <asm/arch/sram.h>
#include <asm/arch/mux.h>
#include <asm/arch/pm.h>
#include <asm/arch/gpio.h>

static struct dentry *peeknpoke_root;
static struct dentry *address32_f;
static struct dentry *address16_f;
static struct dentry *address8_f;
static struct dentry *data32_f;
static struct dentry *data16_f;
static struct dentry *data8_f;

/* Direct R/W access to phisical memory; 32, 16 an 8 bits */

static u32 address32;

static ssize_t omap_address32_show(struct file *file, char __user *user_buf,
				size_t count, loff_t *ppos)
{
	char buf[12];

	sprintf(buf, "0x%08X\n", address32);
	return simple_read_from_buffer(user_buf, count, ppos, buf, 12);
}

static ssize_t
omap_address32_store(struct file *file, const char __user *user_buf,
		  size_t count, loff_t *ppos)
{
        char buf[12];
	int buf_size;
	unsigned val;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	if (sscanf(buf, "0x%08x", &val) != 1) {
		printk(KERN_ERR "poke 32: Invalid data\n");
		return -EINVAL;
	}

	address32 = val;

	return count;
}

static const struct file_operations fops_address32 = {
        .read =         omap_address32_show,
	.write =        omap_address32_store,
};


static ssize_t omap_data32_show(struct file *file, char __user *user_buf,
				size_t count, loff_t *ppos)
{
	char buf[12];

	sprintf(buf, "0x%08X\n", omap_readl(address32));
	return simple_read_from_buffer(user_buf, count, ppos, buf, 12);
}

static ssize_t
omap_data32_store(struct file *file, const char __user *user_buf,
		  size_t count, loff_t *ppos)
{
        char buf[12];
	int buf_size;
	unsigned val;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	if (sscanf(buf, "0x%08x", &val) != 1) {
		printk(KERN_ERR "poke 32: Invalid data\n");
		return -EINVAL;
	}

	omap_writel((u32)val, address32);
	return count;
}

static const struct file_operations fops_data32 = {
        .read =         omap_data32_show,
	.write =        omap_data32_store,
};

static u32 address16;

static ssize_t omap_address16_show(struct file *file, char __user *user_buf,
				size_t count, loff_t *ppos)
{
	char buf[12];

	sprintf(buf, "0x%08X\n", address16);
	return simple_read_from_buffer(user_buf, count, ppos, buf, 12);
}

static ssize_t
omap_address16_store(struct file *file, const char __user *user_buf,
		  size_t count, loff_t *ppos)
{
        char buf[12];
	int buf_size;
	unsigned val;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	if (sscanf(buf, "0x%08x", &val) != 1) {
		printk(KERN_ERR "poke 16: Invalid data\n");
		return -EINVAL;
	}

	address16 = val;

	return count;
}

static const struct file_operations fops_address16 = {
        .read =         omap_address16_show,
	.write =        omap_address16_store,
};


static ssize_t omap_data16_show(struct file *file, char __user *user_buf,
				size_t count, loff_t *ppos)
{
	char buf[8];

	sprintf(buf, "0x%04X\n", omap_readw(address16));
	return simple_read_from_buffer(user_buf, count, ppos, buf, 8);
}

static ssize_t
omap_data16_store(struct file *file, const char __user *user_buf,
		  size_t count, loff_t *ppos)
{
        char buf[8];
	int buf_size;
	unsigned val;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	if (sscanf(buf, "0x%04x", &val) != 1) {
		printk(KERN_ERR "poke 16: Invalid data\n");
		return -EINVAL;
	}

	omap_writew((u16)val, address16);
	return count;
}

static const struct file_operations fops_data16 = {
        .read =         omap_data16_show,
	.write =        omap_data16_store,
};

static u32 address8;

static ssize_t omap_address8_show(struct file *file, char __user *user_buf,
				size_t count, loff_t *ppos)
{
	char buf[12];

	sprintf(buf, "0x%08X\n", address8);
	return simple_read_from_buffer(user_buf, count, ppos, buf, 12);
}

static ssize_t
omap_address8_store(struct file *file, const char __user *user_buf,
		  size_t count, loff_t *ppos)
{
        char buf[12];
	int buf_size;
	unsigned val;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	if (sscanf(buf, "0x%08x", &val) != 1) {
		printk(KERN_ERR "poke 8: Invalid data\n");
		return -EINVAL;
	}

	address8 = val;

	return count;
}

static const struct file_operations fops_address8 = {
        .read =         omap_address8_show,
	.write =        omap_address8_store,
};


static ssize_t omap_data8_show(struct file *file, char __user *user_buf,
			       size_t count, loff_t *ppos)
{
	char buf[6];

	sprintf(buf, "0x%02X\n", omap_readb(address8));
	return simple_read_from_buffer(user_buf, count, ppos, buf, 6);
}

static ssize_t
omap_data8_store(struct file *file, const char __user *user_buf,
		 size_t count, loff_t *ppos)
{
        char buf[6];
	int buf_size;
	unsigned val;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	if (sscanf(buf, "0x%02x", &val) != 1) {
		printk(KERN_ERR "poke 8: Invalid data\n");
		return -EINVAL;
	}

	omap_writeb((u8)val, address8);
	return count;
}

static const struct file_operations fops_data8 = {
        .read =         omap_data8_show,
	.write =        omap_data8_store,
};

static int __init omap2_peeknpoke_init(void)
{
	printk("Peek and Poke for TI OMAP.\n");

	peeknpoke_root = debugfs_create_dir("peeknpoke", NULL);

	address32  = address16 = address8 = 0x48008000;

	address32_f = debugfs_create_file("address32", 0644, peeknpoke_root,
					  NULL, &fops_address32);
	data32_f    = debugfs_create_file("data32", 0644, peeknpoke_root, NULL,
					  &fops_data32);

	address16_f = debugfs_create_file("address16", 0644, peeknpoke_root,
					  NULL, &fops_address16);
	data16_f    = debugfs_create_file("data16", 0644, peeknpoke_root, NULL,
					  &fops_data16);

	address8_f = debugfs_create_file("address8", 0644, peeknpoke_root, NULL,
					 &fops_address8);

	data8_f     = debugfs_create_file("data8", 0644, peeknpoke_root, NULL,
					  &fops_data8);

	return 0;
}

static void __exit omap2_peeknpoke_exit(void)
{
	debugfs_remove(address32_f);
	debugfs_remove(data32_f);

	debugfs_remove(address16_f);
	debugfs_remove(data16_f);

	debugfs_remove(address8_f);
	debugfs_remove(data8_f);

	debugfs_remove(peeknpoke_root);
}

#ifndef MODULE
late_initcall(omap2_peeknpoke_init);
#else
module_init(omap2_peeknpoke_init);
module_exit(omap2_peeknpoke_exit);

MODULE_DESCRIPTION("8/16/32 bit Memory Peek & Poke");
MODULE_LICENSE("GPL");
#endif
