/*
 * linux/arch/arm/mach-omap2/hs.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Sami Tolvanen <sami.tolvanen@nokia.com>
 *
 * OMAP2420 HS secure mode HAL module
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/firmware.h>
#include <linux/mutex.h>
#include <linux/random.h>
#include <asm/cacheflush.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/arch/cpu.h>
#include <asm/arch/sec.h>
#include <asm/arch/pa.h>

MODULE_LICENSE("GPL v2");

#define HS_NAME					"omap2_hs"

#define OMAP2420_RANDOM_SIZE			512

#define OMAP2420_KEYS_PHYS			0x40204080
#define OMAP2420_ROM_START_PHYS			0x00000000
#define OMAP2420_ROM_SIZE			(64 * 1024)
#define OMAP2420_SDRAM_START_PHYS		0x80000000
#define OMAP2420_SDRAM_END_PHYS			0x9FFFFFFF
#define OMAP2420_TESTRAM_START_PHYS		0x4029FA00
#define OMAP2420_TESTRAM_SIZE			512

#define OMAP2420_REV_10				0x0
#define OMAP2420_REV_20				0x1
#define OMAP2420_REV_205			0x2
#define OMAP2420_REV_21				0x3
#define OMAP2420_REV_211			0x4
#define OMAP2420_REV_22				0x5

#define OMAP2420_REV_10_ROM_ENTRY_OFFSET	(0x00002108 + 1)
#define OMAP2420_REV_20_ROM_ENTRY_OFFSET	(0x00002228 + 1)
#define OMAP2420_REV_21_ROM_ENTRY_OFFSET	(0x00002258 + 1)
#define OMAP2420_REV_22_ROM_ENTRY_OFFSET	(0x00002268 + 1)

#define OMAP2420_REV_21_SDRAM_VIRTUAL_OFFSET	0x20000000

#define OMAP2420_ROM_MMU_ENABLE_MASK		0x0001
#define OMAP2420_ROM_ICACHE_ENABLE_MASK		0x0002
#define OMAP2420_ROM_DCACHE_ENABLE_MASK		0x0004
#define OMAP2420_ROM_IRQ_ENABLE_MASK		0x0008
#define OMAP2420_ROM_FIQ_ENABLE_MASK		0x0010
#define OMAP2420_ROM_UL2_CACHE_ENABLE_MASK	0x0020

#define OMAP2420_ROM_NORMAL_FLAGS_MASK \
		(OMAP2420_ROM_MMU_ENABLE_MASK | \
		 OMAP2420_ROM_ICACHE_ENABLE_MASK | \
		 OMAP2420_ROM_DCACHE_ENABLE_MASK | \
		 OMAP2420_ROM_IRQ_ENABLE_MASK    | \
		 OMAP2420_ROM_FIQ_ENABLE_MASK)

#define OMAP2420_FIRMWARE_PA			"omap2420_pa.bin"
#define OMAP2420_FIRMWARE_PAFMT			"omap2420_pafmt.bin"
#define OMAP2420_FIRMWARE_PAPUB			"omap2420_papub.bin"
#define OMAP2420_FIRMWARE_WRAP			"omap2420_wrap.bin"

#define OMAP2420_PA_OFFSET			0x00000018
#define OMAP2420_PAPUB_OFFSET			0x00000083
#define OMAP2420_WRAP_SETPHYS_OFFSET		(0x00000004 + 1)

#define OMAP2420_ROM_CREATOR_SIZE		12
#define OMAP2420_ROM_APPL_ID_SIZE		3

#define OMAP2420_ROM_COMMON_CREATOR \
	0xAB,0x0F,0xBD,0x96,0xAF,0x2A, \
	0xC2,0x71,0xD2,0x62,0x64,0xAF

#define OMAP2420_ROM_RANDOM_GET_APPL \
	{ { OMAP2420_ROM_COMMON_CREATOR }, \
	  { 0x00,0x00,0x0A }, 0 }
#define OMAP2420_ROM_PA_MSG_SEND_APPL \
	{ { OMAP2420_ROM_COMMON_CREATOR }, \
	  { 0x00,0x00,0x12 }, 0 }
#define OMAP2420_ROM_SECENV_INIT_APPL \
	{ { OMAP2420_ROM_COMMON_CREATOR }, \
	  { 0x00,0x00,0x13 }, 0 }
#define OMAP2420_ROM_PAPUB_IMPORT_APPL \
	{ { OMAP2420_ROM_COMMON_CREATOR }, \
	  { 0x00,0x00,0x1A }, 0 }

typedef struct {
	u8 cpu_rev;
	const char *name;
	u32 entry_offset;
	int (*init_fixup)(struct device *);
} omap2420_revision;

typedef struct {
	u8 creator[OMAP2420_ROM_CREATOR_SIZE];
	u8 appl_id[OMAP2420_ROM_APPL_ID_SIZE];
	u8 sub_appl_id;
} omap2420_rom_appl_id;

typedef u32 (*OMAP2420_ROM_ENTRY)(omap2420_rom_appl_id *, u32 *);
typedef void (*OMAP2420_SETPHYS)(u32);

typedef struct {
	OMAP2420_ROM_ENTRY addr;
	u32 flags;
} hs_rom_entry;

typedef struct {
	void *addr;
	struct resource *rgn;
} hs_iomap;

typedef struct {
	void *data;
	size_t size;
} hs_firmware;

typedef struct {
	struct mutex lock;
	omap2420_revision *cpu;
	hs_firmware pa_bin;
	hs_firmware pafmt_bin;
	hs_firmware papub_bin;
	hs_firmware wrapper_bin;
	hs_iomap rom_map;
	hs_iomap testram_map;
	hs_rom_entry entry;
	pa_format format;
} hs_status;

static hs_status hs;

/*
 * Functions for converting between virtual and physical memory addresses,
 * with workaround fixes for OMAP2420 v2.1 ASIC due to SDRAM virtual address
 * handling error in ROM code.
 */
static void * __omap_vtp(const void *virt)
{
	void *phys = (void *)__pa(virt);

	if ((hs.cpu->cpu_rev == OMAP2420_REV_21 ||
			hs.cpu->cpu_rev == OMAP2420_REV_211) &&
	    phys >= (void *)OMAP2420_SDRAM_START_PHYS &&
			phys <= (void *)OMAP2420_SDRAM_END_PHYS)
		phys += OMAP2420_REV_21_SDRAM_VIRTUAL_OFFSET;

	return phys;
}

static void * __omap_ptv(const void *phys)
{
	if ((hs.cpu->cpu_rev == OMAP2420_REV_21 ||
			hs.cpu->cpu_rev == OMAP2420_REV_211) &&
	    phys >= (void *)(OMAP2420_SDRAM_START_PHYS +
				OMAP2420_REV_21_SDRAM_VIRTUAL_OFFSET) &&
			phys <= (void *)(OMAP2420_SDRAM_END_PHYS +
				OMAP2420_REV_21_SDRAM_VIRTUAL_OFFSET))
		phys -= OMAP2420_REV_21_SDRAM_VIRTUAL_OFFSET;

	return __va(phys);
}

static u32 * setup_stack(u32 n, va_list ap)
{
	u32 i, *p;

	p = kzalloc((n + 1) * sizeof(u32), GFP_KERNEL);
	if (!p)
		return p;

	p[0] = (u32)&p[1];

	for (i = 1; i <= n; ++i)
		p[i] = va_arg(ap, u32);

	return p;
}

static u32 sec_entry(omap2420_rom_appl_id *id, u32 n, ...)
{
	va_list ap;
	u32 rv, *p;

	va_start(ap, n);

	p = setup_stack(n, ap);
	if (!p)
		return SEC_HAL_ENOMEM;

	va_end(ap);
	mutex_lock(&hs.lock);

	/*
	 * Workaround fix for OMAP2420 v2.1 asic due to SDRAM virtual address
 	 * handling error in ROM code: service ID has to be copied to Internal
 	 * SRAM.
 	 */
	if (hs.testram_map.addr)
		memcpy_toio(hs.testram_map.addr, id, sizeof(*id));

	/*
	 * Wrapper handles address conversion for the service ID.
	 */
	if (!hs.wrapper_bin.data)
		id = __omap_vtp(id);

	rv = hs.entry.addr(id, p);

	flush_cache_all();
	mutex_unlock(&hs.lock);
	kfree(p);

	if (rv != SEC_HAL_OK)
		printk(KERN_WARNING HS_NAME ": Error %08x calling ROM\n", rv);

	return rv;
}

static u32 papub_import(void)
{
	omap2420_rom_appl_id id = OMAP2420_ROM_PAPUB_IMPORT_APPL;

	return sec_entry(&id, 2, hs.entry.flags,
			(u32)__omap_vtp(hs.papub_bin.data));
}

static u32 secenv_init(sec_hal_par_secenv_init *params)
{
	omap2420_rom_appl_id id = OMAP2420_ROM_SECENV_INIT_APPL;
	u32 rv;

#ifdef SEC_HACK_SYMBIAN_WRAPPER
	if (hs.wrapper_bin.data) {
		memcpy(params->storage_ptr + params->length,
			hs.wrapper_bin.data, hs.wrapper_bin.size);
		hs.entry.addr =
			(OMAP2420_ROM_ENTRY)(params->storage_ptr +
				params->length + 1);
	}
#endif
	rv = sec_entry(&id, 4, hs.entry.flags,
			(u32)params->rpc_ptr, /* Virtual */
			(u32)__omap_vtp(params->storage_ptr),
			params->length);

	if (rv == SEC_HAL_OK)
		rv = papub_import();

	return rv;
}

static u32 random_get(sec_hal_par_random_get *params)
{
	omap2420_rom_appl_id id = OMAP2420_ROM_RANDOM_GET_APPL;

	return sec_entry(&id, 3, hs.entry.flags,
			(u32)__omap_vtp(params->data),
			params->length);
}

static u32 pa_msg_send(pa_command *c, void *par, void *res)
{
	omap2420_rom_appl_id id = OMAP2420_ROM_PA_MSG_SEND_APPL;
	u32 rv;
	void *pa_addr = NULL;

	rv = pa_image_address_get(hs.pa_bin.data, c->cmd->filename, &pa_addr);
	if (rv < 0)
		return SEC_HAL_ENOPA;

	rv = sec_entry(&id, 6, hs.entry.flags,
			(u32)__omap_vtp(pa_addr),
			0,
			(u32)__omap_vtp(par),
			(u32)__omap_vtp(res),
			(u32)c->cmd->sa);

	if (rv == SEC_HAL_OK)
		rv = ((sec_hal_res_common *)res)->rv; /* PA return value */

	return rv;
}

static u32 pa_query(u32 cmd, u8 *input, u32 cinp, u32 *mininp, u32 *maxout)
{
	if (pa_command_query(cmd, input, cinp, mininp, maxout, &hs.format) < 0)
		return SEC_HAL_ENOPA;

	return SEC_HAL_OK;
}

static u32 pa_service(u32 cmd, u8 *input, u32 cinp, u8 *output, u32 coup)
{
	pa_command_data p;
	u32 rv;

	memset(&p, 0, sizeof(p));

	p.format = &hs.format;
	p.vtp    = __omap_vtp;
	p.ptv    = __omap_ptv;
	p.cinp   = cinp;
	p.coup   = coup;
	p.input  = input;
	p.output = output;
	p.keys   = (void *)OMAP2420_KEYS_PHYS;

	if (hs.papub_bin.data)
		p.papub = __omap_vtp(hs.papub_bin.data);

	if (pa_command_prepare(cmd, &p) < 0)
		return SEC_HAL_FAIL;

	rv = pa_msg_send(p.c, p.par, p.res);

	if (pa_command_finish(&p) < 0 && rv == SEC_HAL_OK)
		return SEC_HAL_FAIL;

	return rv;
}

static u32 rpc_alloc(u32 size)
{
	void *p;

	if (!size)
		return 0;

	p = kzalloc(size, GFP_KERNEL);
	if (!p)
		return 0;

	return (u32)__omap_vtp(p);
}

static void rpc_dealloc(u32 phys)
{
	kfree(__omap_ptv((void *)phys));
}

static int __load_firmware(struct device *dev, const char *name,
		hs_firmware *fw, u32 offset)
{
	const struct firmware *p;
	int rv;

	printk(KERN_INFO HS_NAME ": Loading firmware: %s\n", name);
	rv = request_firmware(&p, name, dev);

	if (rv < 0)
		goto done;

	if (p->size <= offset) {
		rv = -EINVAL;
		goto done;
	}

	fw->size = p->size - offset;
	fw->data = kzalloc(fw->size, GFP_KERNEL);

	if (fw->data)
		memcpy(fw->data, p->data + offset, fw->size);
	else
		rv = -ENOMEM;

	release_firmware(p);
done:
	if (rv < 0)
		printk(KERN_WARNING HS_NAME ": Error loading %s\n", name);
	return rv;
}

static int load_firmware(struct device *dev)
{
	int rv;

	rv = __load_firmware(dev, OMAP2420_FIRMWARE_PA, &hs.pa_bin,
			OMAP2420_PA_OFFSET);

	if (!rv)
		rv = __load_firmware(dev, OMAP2420_FIRMWARE_PAFMT,
				&hs.pafmt_bin, 0);

	if (!rv)
		rv = __load_firmware(dev, OMAP2420_FIRMWARE_PAPUB,
				&hs.papub_bin, OMAP2420_PAPUB_OFFSET);

	return rv;
}

static int load_wrapper(struct device *dev)
{
	int rv;
#ifndef SEC_HACK_SYMBIAN_WRAPPER
	OMAP2420_SETPHYS setphys;
#endif

	rv = __load_firmware(dev, OMAP2420_FIRMWARE_WRAP, &hs.wrapper_bin, 0);

	if (rv < 0)
		return rv;

#ifndef SEC_HACK_SYMBIAN_WRAPPER
	setphys = (OMAP2420_SETPHYS)(hs.wrapper_bin.data +
			OMAP2420_SETPHYS_OFFSET);
	setphys(__omap_vtp(hs.wrapper_bin.data));
#endif

	hs.entry.addr = (OMAP2420_ROM_ENTRY)(hs.wrapper_bin.data + 1);
	return 0;
}

static int map_rom_entry(void)
{
	if (hs.entry.addr)
		return 0;

	hs.rom_map.rgn = request_mem_region(OMAP2420_ROM_START_PHYS,
				OMAP2420_ROM_SIZE, "omap2420_rom");
	if (!hs.rom_map.rgn)
		return -ENODEV;

	hs.rom_map.addr = __ioremap(OMAP2420_ROM_START_PHYS,
				OMAP2420_ROM_SIZE, L_PTE_EXEC);
	if (!hs.rom_map.addr)
		return -ENOMEM;

	hs.entry.addr = (OMAP2420_ROM_ENTRY)(hs.rom_map.addr +
				hs.cpu->entry_offset);
	return 0;
}

static int omap2420vx0_init_fixup(struct device *dev)
{
	hs.entry.flags = OMAP2420_ROM_ICACHE_ENABLE_MASK;
	return 0;
}

static int omap2420v21_init_fixup(struct device *dev)
{
	hs.testram_map.rgn = request_mem_region(OMAP2420_TESTRAM_START_PHYS,
				OMAP2420_TESTRAM_SIZE, "omap2420_testram");
	if (!hs.testram_map.rgn)
		return -ENODEV;

	hs.testram_map.addr = ioremap_nocache(OMAP2420_TESTRAM_START_PHYS,
				OMAP2420_TESTRAM_SIZE);
	if (!hs.testram_map.addr)
		return -ENOMEM;

	return load_wrapper(dev);
}

static int omap2420v22_init_fixup(struct device *dev)
{
	return load_wrapper(dev);
}

static omap2420_revision asic_rev_confs[] = {
	{
		.cpu_rev	= OMAP2420_REV_10,
		.name	 	= "OMAP2420 v1.0",
		.entry_offset	= OMAP2420_REV_10_ROM_ENTRY_OFFSET,
		.init_fixup 	= omap2420vx0_init_fixup
	}, {
		.cpu_rev	= OMAP2420_REV_20,
		.name		= "OMAP2420 v2.0",
		.entry_offset	= OMAP2420_REV_20_ROM_ENTRY_OFFSET,
		.init_fixup	= omap2420vx0_init_fixup
	}, {
		.cpu_rev	= OMAP2420_REV_205,
		.name		= "OMAP2420 v2.05",
		.entry_offset	= OMAP2420_REV_20_ROM_ENTRY_OFFSET,
		.init_fixup	= NULL
	}, {
		.cpu_rev	= OMAP2420_REV_21,
		.name		= "OMAP2420 v2.1",
		.entry_offset	= OMAP2420_REV_21_ROM_ENTRY_OFFSET,
		.init_fixup	= omap2420v21_init_fixup
	}, {
		.cpu_rev	= OMAP2420_REV_211,
		.name		= "OMAP2420 v2.11",
		.entry_offset	= OMAP2420_REV_21_ROM_ENTRY_OFFSET,
		.init_fixup	= omap2420v21_init_fixup
	}, {
		.cpu_rev	= OMAP2420_REV_22,
		.name		= "OMAP2420 v2.2",
		.entry_offset	= OMAP2420_REV_22_ROM_ENTRY_OFFSET,
		.init_fixup	= omap2420v22_init_fixup
	},
};

static void cleanup(void)
{
	kfree(hs.pa_bin.data);
	kfree(hs.pafmt_bin.data);
	kfree(hs.papub_bin.data);
	kfree(hs.wrapper_bin.data);

	if (hs.rom_map.addr)
		iounmap(hs.rom_map.addr);

	if (hs.rom_map.rgn)
		release_mem_region(OMAP2420_ROM_START_PHYS,
			OMAP2420_ROM_SIZE);

	if (hs.testram_map.addr)
		iounmap(hs.testram_map.addr);

	if (hs.testram_map.rgn)
		release_mem_region(OMAP2420_TESTRAM_START_PHYS,
			OMAP2420_TESTRAM_SIZE);

	if (hs.format.cmd)
		pa_format_free(&hs.format);

	memset(&hs, 0, sizeof(hs));
}

static int asic_init(sec_hal_init *init)
{
	int i;
	u8 cpu_rev;

	cpu_rev = omap2_cpu_rev();

	for (i = 0; i < ARRAY_SIZE(asic_rev_confs); ++i) {
		if (cpu_rev == asic_rev_confs[i].cpu_rev) {
			hs.cpu = &asic_rev_confs[i];
			break;
		}
	}

	if (!hs.cpu) {
		printk(KERN_WARNING HS_NAME ": Unsupported ASIC revision\n");
		return -ENODEV;
	}

	if (hs.cpu->name)
		printk(KERN_INFO HS_NAME ": Detected %s\n", hs.cpu->name);

	hs.entry.flags = OMAP2420_ROM_NORMAL_FLAGS_MASK;

	if (hs.cpu->init_fixup)
		return hs.cpu->init_fixup(init->dev);

	return 0;
}

static int firmware_init(sec_hal_init *init)
{
	int rv;

	rv = load_firmware(init->dev);
	if (!rv)
		rv = pa_format_parse(hs.pafmt_bin.data, hs.pafmt_bin.size,
				&hs.format);
	return rv;
}

static int hal_init(sec_hal_init *init)
{
	int rv;
	mutex_lock(&hs.lock);

	rv = asic_init(init);

	if (rv < 0)
		goto out;

	rv = firmware_init(init);

	if (rv < 0)
		goto out;

	rv = map_rom_entry();
out:
	if (rv < 0)
		cleanup();
	mutex_unlock(&hs.lock);
	return rv;
}

void hal_exit(void)
{
	mutex_lock(&hs.lock);
	cleanup();
	mutex_unlock(&hs.lock);
}

static sec_hal_operations omap2420_ops = {
	.hal_init	= hal_init,
	.hal_exit	= hal_exit,
	.rpc_alloc	= rpc_alloc,
	.rpc_dealloc	= rpc_dealloc,
	.secenv_init	= secenv_init,
	.random_get	= random_get,
	.pa_query	= pa_query,
	.pa_service	= pa_service,
};

static int __init hs_init(void)
{
	if (!cpu_is_omap2420()) {
		printk(KERN_WARNING HS_NAME ": Unsupported architecture\n");
		return -ENODEV;
	}

	memset(&hs, 0, sizeof(hs));
	mutex_init(&hs.lock);

	return sec_hal_register(&omap2420_ops);
}

static void __exit hs_exit(void)
{
	sec_hal_deregister(&omap2420_ops);
}

module_init(hs_init);
module_exit(hs_exit);
