/*
 * board-rx44.c - Nokia RX-44 specific initialization
 *
 * Copyright (C) 2007 Nokia Corporation
 * Contact: Jarkko Nikula <jarkko.nikula@nokia.com>
 *
 * Based on board-n800.c by Juha Yrjola <juha.yrjola@nokia.com>
 *
 * RX-44 specific configurations from:
 *
 * Timo Karjalainen
 * Eero Nurkkala
 * Henrik Saari
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/clk.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <linux/spi/tsc2005.h>
#include <linux/input.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/tsl2563.h>
#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/arch/gpio.h>
#include <asm/arch/usb.h>
#include <asm/arch/board.h>
#include <asm/arch/common.h>
#include <asm/arch/mcspi.h>
#include <asm/arch/menelaus.h>
#include <asm/arch/lcd_mipid.h>
#include <asm/arch/clock.h>
#include <asm/arch/gpio-switch.h>
#include <asm/arch/omapfb.h>
#include <asm/arch/blizzard.h>
#include <asm/arch/pm.h>

#include <../drivers/cbus/tahvo.h>

#define RX44_BLIZZARD_POWERDOWN_GPIO	15
#define RX44_STI_GPIO			62

static void __init nokia_rx44_init_irq(void)
{
	omap2_init_common_hw();
	omap_init_irq();
	omap_gpio_init();

#ifdef CONFIG_OMAP_STI
	if (omap_request_gpio(RX44_STI_GPIO) < 0) {
		printk(KERN_ERR "Failed to request GPIO %d for STI\n",
		       RX44_STI_GPIO);
		return;
	}

	omap_set_gpio_direction(RX44_STI_GPIO, 0);
	omap_set_gpio_dataout(RX44_STI_GPIO, 0);
#endif
}

#if defined(CONFIG_MENELAUS) && defined(CONFIG_SENSORS_TMP105)

static int rx44_tmp105_set_power(int enable)
{
	return menelaus_set_vaux(enable ? 2800 : 0);
}

#else

#define rx44_tmp105_set_power NULL

#endif

static struct omap_uart_config rx44_uart_config __initdata = {
	.enabled_uarts = (1 << 0) | (1 << 2),
};

#include "../../../drivers/cbus/retu.h"

static struct omap_fbmem_config rx44_fbmem0_config __initdata = {
	.size = 752 * 1024,
};

static struct omap_fbmem_config rx44_fbmem1_config __initdata = {
	.size = 768 * 1024,
};

static struct omap_fbmem_config rx44_fbmem2_config __initdata = {
	.size = 1536 * 1024,
};

static struct omap_tmp105_config rx44_tmp105_config __initdata = {
	.tmp105_irq_pin = 125,
	.set_power = rx44_tmp105_set_power,
};

static struct omap_mmc_config rx44_mmc_config __initdata = {
	.mmc [0] = {
		.enabled		= 1,
		.wire4			= 1,
	},
};

static void mipid_shutdown(struct mipid_platform_data *pdata)
{
	if (pdata->nreset_gpio != -1) {
		pr_info("shutdown LCD\n");
		omap_set_gpio_dataout(pdata->nreset_gpio, 0);
		msleep(120);
	}
}

static struct mipid_platform_data rx44_mipid_platform_data = {
	.shutdown = mipid_shutdown,
};

static void __init mipid_dev_init(void)
{
	const struct omap_lcd_config *conf;

	conf = omap_get_config(OMAP_TAG_LCD, struct omap_lcd_config);
	if (conf != NULL) {
		rx44_mipid_platform_data.nreset_gpio = conf->nreset_gpio;
		rx44_mipid_platform_data.data_lines = conf->data_lines;
	}
}

static struct {
	struct clk *sys_ck;
} blizzard;

static int blizzard_get_clocks(void)
{
	blizzard.sys_ck = clk_get(0, "osc_ck");
	if (IS_ERR(blizzard.sys_ck)) {
		printk(KERN_ERR "can't get Blizzard clock\n");
		return PTR_ERR(blizzard.sys_ck);
	}
	return 0;
}

static unsigned long blizzard_get_clock_rate(struct device *dev)
{
	return clk_get_rate(blizzard.sys_ck);
}

static void blizzard_enable_clocks(int enable)
{
	if (enable)
		clk_enable(blizzard.sys_ck);
	else
		clk_disable(blizzard.sys_ck);
}

static void blizzard_power_up(struct device *dev)
{
	/* Vcore to 1.475V */
	tahvo_set_clear_reg_bits(0x07, 0, 0xf);
	msleep(10);

	blizzard_enable_clocks(1);
	omap_set_gpio_dataout(RX44_BLIZZARD_POWERDOWN_GPIO, 1);
}

static void blizzard_power_down(struct device *dev)
{
	omap_set_gpio_dataout(RX44_BLIZZARD_POWERDOWN_GPIO, 0);
	blizzard_enable_clocks(0);

	/* Vcore to 1.005V */
	tahvo_set_clear_reg_bits(0x07, 0xf, 0);
}

static struct blizzard_platform_data rx44_blizzard_data = {
	.power_up	= blizzard_power_up,
	.power_down	= blizzard_power_down,
	.get_clock_rate	= blizzard_get_clock_rate,
	.te_connected	= 1,
};

static void __init blizzard_dev_init(void)
{
	int r;

	r = omap_request_gpio(RX44_BLIZZARD_POWERDOWN_GPIO);
	if (r < 0)
		return;
	omap_set_gpio_direction(RX44_BLIZZARD_POWERDOWN_GPIO, 0);
	omap_set_gpio_dataout(RX44_BLIZZARD_POWERDOWN_GPIO, 1);

	blizzard_get_clocks();
	omapfb_set_ctrl_platform_data(&rx44_blizzard_data);
}

#if defined(CONFIG_VIDEO_CAMERA_SENSOR_TCM825X) \
	|| defined(CONFIG_VIDEO_CAMERA_SENSOR_TCM825X_MODULE)
extern struct omap_camera_sensor_config n800_camera_sensor_config;
#endif

#ifdef CONFIG_SENSORS_TSL2563
/* FIXME: Remove when bootloader passes this information */
static struct tsl2563_config rx44_tsl2563_config = {
	.irq_gpio	= 8,
};
#endif

static struct omap_board_config_kernel rx44_config[] = {
	{ OMAP_TAG_UART,	                &rx44_uart_config },
#ifdef SUPPORT_SENSOR
	{ OMAP_TAG_CAMERA_SENSOR,		&rx44_sensor_config },
#endif
	{ OMAP_TAG_FBMEM,			&rx44_fbmem0_config },
	{ OMAP_TAG_FBMEM,			&rx44_fbmem1_config },
	{ OMAP_TAG_FBMEM,			&rx44_fbmem2_config },
	{ OMAP_TAG_TMP105,			&rx44_tmp105_config },
	{ OMAP_TAG_MMC,                         &rx44_mmc_config },
#if defined(CONFIG_VIDEO_CAMERA_SENSOR_TCM825X) \
	|| defined(CONFIG_VIDEO_CAMERA_SENSOR_TCM825X_MODULE)
	{ OMAP_TAG_CAMERA_SENSOR,		&n800_camera_sensor_config },
#endif
#ifdef CONFIG_SENSORS_TSL2563
	{ OMAP_TAG_TSL2563,			&rx44_tsl2563_config },
#endif
};

#ifdef CONFIG_SPI_TSC2005
static struct tsc2005_platform_data tsc2005_config = {
	.reset_gpio = 94,
	.dav_gpio = 106
};
#endif

#ifdef CONFIG_SENSORS_TSL2563
static void rx44_tsl2563_set_power(struct device *dev, int on)
{
	/*
	 * on RX-44, TSL2563 is getting its power from the regulator which is
	 * controlled with OMAP SYS_CLKREQ signal, i.e. regulator is shutdown
	 * when OMAP is in retention. Therefore we'll block the retention when
	 * TSL2563 needs to be powered
	 */
	if (on)
		omap2_block_sleep();
	else
		omap2_allow_sleep();
}

static struct tsl2563_platform_data tsl2563_data = {
	.set_power	= rx44_tsl2563_set_power,
};

static struct platform_device rx44_tsl2563_device = {
	.name		= "tsl2563",
	.id		= -1,
	.dev		= {
		.platform_data = &tsl2563_data,
	},
};

static void tsl2563_dev_init(void)
{
	const struct tsl2563_config *config;
	int gpio;

	config = omap_get_config(OMAP_TAG_TSL2563,
				 struct tsl2563_config);
	if (config != NULL) {
		gpio = config->irq_gpio;
		tsl2563_data.irq = OMAP_GPIO_IRQ(gpio);

		if (omap_request_gpio(gpio) < 0) {
			printk(KERN_ERR "Cannot get TSL2563 GPIO\n");
			return;
		}
		omap_set_gpio_direction(gpio, 1);

		if (platform_device_register(&rx44_tsl2563_device) < 0) {
			printk(KERN_ERR "Cannot register TSL2563\n");
			return;
		}
	}
}
#else
static void tsl2563_dev_init(void) {}
#endif

#ifdef CONFIG_SPI_TSC2005
static struct omap2_mcspi_device_config tsc2005_mcspi_config = {
	.turbo_mode	= 0,
	.single_channel = 1,
};
#endif

static struct omap2_mcspi_device_config mipid_mcspi_config = {
	.turbo_mode	= 0,
	.single_channel	= 1,
};

static struct omap2_mcspi_device_config cx3110x_mcspi_config = {
	.turbo_mode	= 0,
	.single_channel = 1,
};

#ifdef CONFIG_NET_PC2400M_MODULE
static struct omap2_mcspi_device_config pc2400m_mcspi_config = {
	.turbo_mode	= 1,
	.single_channel = 1,
};
#endif

static struct spi_board_info rx44_spi_board_info[] __initdata = {
	[0] = {
		.modalias	= "lcd_mipid",
		.bus_num	= 1,
		.chip_select	= 1,
		.max_speed_hz	= 4000000,
		.controller_data= &mipid_mcspi_config,
		.platform_data	= &rx44_mipid_platform_data,
	}, [1] = {
		.modalias	= "cx3110x",
		.bus_num	= 2,
		.chip_select	= 0,
		.max_speed_hz   = 48000000,
		.controller_data= &cx3110x_mcspi_config,
	},
#ifdef CONFIG_NET_PC2400M_MODULE
	[2] = {
		.modalias	= "pc2400m",
		.bus_num	= 2,
		.chip_select	= 1,
		.max_speed_hz   = 24000000,
		.controller_data= &pc2400m_mcspi_config,
		.irq		= 33,
	},
#endif
#ifdef CONFIG_SPI_TSC2005
	{
		.modalias	= "tsc2005",
		.bus_num	= 1,
		.chip_select	= 0,
		.max_speed_hz   = 6000000,
		.controller_data= &tsc2005_mcspi_config,
		.platform_data  = &tsc2005_config,
	}
#endif	
};

static void __init rx44_ts_set_config(void)
{
	const struct omap_lcd_config *conf;

	conf = omap_get_config(OMAP_TAG_LCD, struct omap_lcd_config);
	if (conf != NULL) {
#ifdef CONFIG_SPI_TSC2005
		if (strcmp(conf->panel_name, "lph8923") == 0) {
			tsc2005_config.ts_x_plate_ohm	= 180;
			tsc2005_config.ts_hw_avg	= 4;
			tsc2005_config.ts_ignore_last	= 1;
			tsc2005_config.ts_max_pressure	= 255;
			tsc2005_config.ts_stab_time	= 100;
		} else if (strcmp(conf->panel_name, "ls041y3") == 0) {
			tsc2005_config.ts_x_plate_ohm	= 280;
			tsc2005_config.ts_hw_avg	= 16;
			tsc2005_config.ts_touch_pressure= 1500;
			tsc2005_config.ts_max_pressure	= 1500;
			tsc2005_config.ts_ignore_last	= 1;
		} else {
			printk(KERN_ERR "Unknown panel type, set default "
			       "touchscreen configuration\n");
			tsc2005_config.ts_x_plate_ohm	= 200;
			tsc2005_config.ts_stab_time	= 100;
		}
#endif
	}
}

static struct omap_gpio_switch rx44_gpio_switches[] __initdata = {
	{
		.name			= "headphone",
		.gpio			= -1,
		.debounce_rising	= 200,
		.debounce_falling	= 200,
	}, {
		.name			= "cam_act",
		.gpio			= -1,
		.debounce_rising	= 200,
		.debounce_falling	= 200,
	}, {
		.name			= "cam_turn",
		.gpio			= -1,
		.debounce_rising	= 100,
		.debounce_falling	= 100,
	}, {
		.name			= "slide",
		.gpio			= -1,
		.debounce_rising	= 200,
		.debounce_falling	= 200,
	}, {
		.name			= "kb_lock",
		.gpio			= -1,
		.debounce_rising	= 200,
		.debounce_falling	= 200,
	},
};

extern void __init n800_flash_init(void);
extern void __init n800_mmc_init(void);
extern void __init n800_bt_init(void);
extern void __init rx44_audio_init(void);
extern void __init n800_dsp_init(void);
extern void __init n800_usb_init(void);
extern void __init n800_cam_init(void);
extern void __init n800_pm_init(void);
extern void __init rx44_keyboard_init(void);

static void __init nokia_rx44_init(void)
{
	n800_flash_init();
	n800_mmc_init();
	n800_bt_init();
	rx44_audio_init();
	n800_dsp_init();
	n800_usb_init();
	n800_cam_init();
	rx44_ts_set_config();
	rx44_keyboard_init();
	spi_register_board_info(rx44_spi_board_info,
				ARRAY_SIZE(rx44_spi_board_info));
	omap_serial_init();
	mipid_dev_init();
	blizzard_dev_init();
	tsl2563_dev_init();
	omap_register_gpio_switches(rx44_gpio_switches,
				    ARRAY_SIZE(rx44_gpio_switches));
	n800_pm_init();
}

static void __init nokia_rx44_map_io(void)
{
	omap_board_config = rx44_config;
	omap_board_config_size = ARRAY_SIZE(rx44_config);

	omap2_map_common_io();
}

MACHINE_START(NOKIA_RX44, "Nokia RX-44")
	.phys_io	= 0x48000000,
	.io_pg_offst	= ((0xd8000000) >> 18) & 0xfffc,
	.boot_params	= 0x80000100,
	.map_io		= nokia_rx44_map_io,
	.init_irq	= nokia_rx44_init_irq,
	.init_machine	= nokia_rx44_init,
	.timer		= &omap_timer,
MACHINE_END
