/*
 * linux/arch/arm/mach-omap2/board-n800-mmc.c
 *
 * Copyright (C) 2006 Nokia Corporation
 * Author: Juha Yrjola
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <asm/arch/mmc.h>
#include <asm/arch/menelaus.h>
#include <asm/arch/gpio.h>

#include <asm/mach-types.h>
#include <linux/delay.h>

#ifdef CONFIG_MMC_OMAP

static const int slot_switch_gpio = 96;

static const int rx44_slot2_pw_vddf = 23;
static const int rx44_slot2_pw_vdd = 9;

static int slot1_cover_open;
static int slot2_cover_open;
static struct device *mmc_device;

/*
 * VMMC   --> slot 1 (N800 & RX44)
 * VDCDC3_APE, VMCS2_APE --> slot 2 on N800
 * GPIO96 --> Menelaus GPIO2
 * GPIO23 --> controls slot2 VSD    (RX44 only)
 * GPIO9  --> controls slot2 VIO_SD (RX44 only)
 */

static int n800_mmc_switch_slot(struct device *dev, int slot)
{
	dev_dbg(dev, "Choose slot %d\n", slot + 1);
	if (slot == 0)
		omap_set_gpio_dataout(slot_switch_gpio, 0);
	else
		omap_set_gpio_dataout(slot_switch_gpio, 1);
	return 0;
}

static int n800_mmc_set_power_menelaus(struct device *dev,
					int slot, int power_on, int vdd)
{
	int mV;

	dev_dbg(dev, "Set slot %d power: %s (vdd %d)\n", slot + 1,
		power_on ? "on" : "off", vdd);

	if (slot == 0) {
		if (!power_on)
			return menelaus_set_vmmc(0);
		switch (1 << vdd) {
		case MMC_VDD_33_34:
		case MMC_VDD_32_33:
		case MMC_VDD_31_32:
			mV = 3100;
			break;
		case MMC_VDD_30_31:
			mV = 3000;
			break;
		case MMC_VDD_28_29:
			mV = 2800;
			break;
		case MMC_VDD_165_195:
			mV = 1850;
			break;
		default:
			BUG();
		}
		return menelaus_set_vmmc(mV);
	} else {
		if (!power_on)
			return menelaus_set_vdcdc(3, 0);
		switch (1 << vdd) {
		case MMC_VDD_33_34:
		case MMC_VDD_32_33:
			mV = 3300;
			break;
		case MMC_VDD_30_31:
		case MMC_VDD_29_30:
			mV = 3000;
			break;
		case MMC_VDD_28_29:
		case MMC_VDD_27_28:
			mV = 2800;
			break;
		case MMC_VDD_24_25:
		case MMC_VDD_23_24:
			mV = 2400;
			break;
		case MMC_VDD_22_23:
		case MMC_VDD_21_22:
			mV = 2200;
			break;
		case MMC_VDD_20_21:
			mV = 2000;
			break;
		case MMC_VDD_165_195:
			mV = 1800;
			break;
		default:
			BUG();
		}
		return menelaus_set_vdcdc(3, mV);
	}
	return 0;
}

static void nokia_mmc_set_power_internal(struct device *dev,
					 int power_on)
{
	dev_dbg(dev, "Set internal slot power %s\n",
		power_on ? "on" : "off");

	if (power_on) {
		omap_set_gpio_dataout(rx44_slot2_pw_vddf, 1);
		msleep(2);
		omap_set_gpio_dataout(rx44_slot2_pw_vdd, 1);
		msleep(1);
	} else {
		omap_set_gpio_dataout(rx44_slot2_pw_vdd, 0);
		msleep(2);
		omap_set_gpio_dataout(rx44_slot2_pw_vddf, 0);
	}
}

static int n800_mmc_set_power(struct device *dev, int slot, int power_on,
				int vdd)
{

	if (machine_is_nokia_n800() || slot == 0)
		return n800_mmc_set_power_menelaus(dev, slot, power_on, vdd);

	nokia_mmc_set_power_internal(dev, power_on);

	return 0;
}

static int n800_mmc_set_bus_mode(struct device *dev, int slot, int bus_mode)
{
	int r;

	dev_dbg(dev, "Set slot %d bus mode %s\n", slot + 1,
		bus_mode == MMC_BUSMODE_OPENDRAIN ? "open-drain" : "push-pull");
	BUG_ON(slot != 0 && slot != 1);
	slot++;
	switch (bus_mode) {
	case MMC_BUSMODE_OPENDRAIN:
		r = menelaus_set_mmc_opendrain(slot, 1);
		break;
	case MMC_BUSMODE_PUSHPULL:
		r = menelaus_set_mmc_opendrain(slot, 0);
		break;
	default:
		BUG();
	}
	if (r != 0 && printk_ratelimit())
		dev_err(dev, "MMC: unable to set bus mode for slot %d\n",
			slot);
	return r;
}

static int n800_mmc_get_cover_state(struct device *dev, int slot)
{
	slot++;
	BUG_ON(slot != 1 && slot != 2);
	if (slot == 1)
		return slot1_cover_open;
	else
		return slot2_cover_open;
}

static void n800_mmc_callback(void *data, u8 card_mask)
{
	int bit, *openp, index;

	if (machine_is_nokia_n800()) {
		bit = 1 << 1;
		openp = &slot2_cover_open;
		index = 1;
	} else {
		bit = 1;
		openp = &slot1_cover_open;
		index = 0;
	}

	if (card_mask & bit)
		*openp = 1;
	else
		*openp = 0;

        omap_mmc_notify_cover_event(mmc_device, index, *openp);
}

void n800_mmc_slot1_cover_handler(void *arg, int closed_state)
{
	if (mmc_device == NULL)
		return;

	slot1_cover_open = !closed_state;
	omap_mmc_notify_cover_event(mmc_device, 0, closed_state);
}

static int n800_mmc_late_init(struct device *dev)
{
	int r, bit, *openp;

	mmc_device = dev;

	r = menelaus_set_slot_sel(1);
	if (r < 0)
		return r;

	r = menelaus_set_mmc_slot(1, 1, 0, 1);
	if (r < 0)
		return r;
	r = menelaus_set_mmc_slot(2, 1, 0, 1);
	if (r < 0)
		return r;

	r = menelaus_get_slot_pin_states();
	if (r < 0)
		return r;

	if (machine_is_nokia_n800()) {
		bit = 1 << 1;
		openp = &slot2_cover_open;
	} else {
		bit = 1;
		openp = &slot1_cover_open;
		slot2_cover_open = 0;
	}

	/* All slot pin bits seem to be inversed until first swith change */
	if (r == 0xf || r == (0xf & ~bit))
		r = ~r;

	if (r & bit)
		*openp = 1;
	else
		*openp = 0;

	r = menelaus_register_mmc_callback(n800_mmc_callback, NULL);

	return r;
}

static void n800_mmc_cleanup(struct device *dev)
{
	menelaus_unregister_mmc_callback();

	omap_free_gpio(slot_switch_gpio);

	if (machine_is_nokia_rx44()) {
		omap_free_gpio(rx44_slot2_pw_vddf);
		omap_free_gpio(rx44_slot2_pw_vdd);
	}
}

static struct omap_mmc_platform_data n800_mmc_data = {
	.nr_slots		= 2,
	.switch_slot		= n800_mmc_switch_slot,
	.init			= n800_mmc_late_init,
	.cleanup		= n800_mmc_cleanup,
	.max_freq               = 24000000,
	.slots[0] = {
		.set_power	= n800_mmc_set_power,
		.set_bus_mode	= n800_mmc_set_bus_mode,
		.get_ro		= NULL,
		.get_cover_state= n800_mmc_get_cover_state,
		.ocr_mask	= MMC_VDD_165_195 |
				  MMC_VDD_28_29 | MMC_VDD_30_31 |
				  MMC_VDD_32_33 | MMC_VDD_33_34,
		.name		= "internal",
	},
	.slots[1] = {
		.set_power	= n800_mmc_set_power,
		.set_bus_mode	= n800_mmc_set_bus_mode,
		.get_ro		= NULL,
		.get_cover_state= n800_mmc_get_cover_state,
		.ocr_mask	= MMC_VDD_165_195 | MMC_VDD_20_21 |
				  MMC_VDD_21_22 | MMC_VDD_22_23 | MMC_VDD_23_24 |
				  MMC_VDD_24_25 | MMC_VDD_27_28 | MMC_VDD_28_29 |
				  MMC_VDD_29_30 | MMC_VDD_30_31 | MMC_VDD_32_33 |
				  MMC_VDD_33_34,
		.name		= "external",
	},
};

void __init n800_mmc_init(void)

{
	if (machine_is_nokia_rx44()) {
		n800_mmc_data.slots[0].name = "external";
		n800_mmc_data.slots[1].name = "internal";
	}

	omap_set_mmc_info(1, &n800_mmc_data);
	if (omap_request_gpio(slot_switch_gpio) < 0)
		BUG();
	omap_set_gpio_dataout(slot_switch_gpio, 0);
	omap_set_gpio_direction(slot_switch_gpio, 0);

	if (machine_is_nokia_rx44()) {
		if (omap_request_gpio(rx44_slot2_pw_vddf) < 0)
			BUG();
		omap_set_gpio_dataout(rx44_slot2_pw_vddf, 0);
		omap_set_gpio_direction(rx44_slot2_pw_vddf, 0);

		if (omap_request_gpio(rx44_slot2_pw_vdd) < 0)
			BUG();
		omap_set_gpio_dataout(rx44_slot2_pw_vdd, 0);
		omap_set_gpio_direction(rx44_slot2_pw_vdd, 0);
	}
}
#else

void __init n800_mmc_init(void)
{
}

void n800_mmc_slot1_cover_handler(void *arg, int state)
{
}

#endif
