/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdsppcmsrc.h"
#include "dspmixer.h"

// Size of G.711 SID_UPDATE frame in bytes
#define G711_DTX_UPDATE_FRAME_SIZE 22

GST_DEBUG_CATEGORY_STATIC (dsppcmsrc_debug);
#define GST_CAT_DEFAULT dsppcmsrc_debug

static const gchar *pcm_node_name = "/dev/dsptask/pcm_rec";

static GstStaticPadTemplate dsppcmsrc_src_template =
    GST_STATIC_PAD_TEMPLATE ("src",
       GST_PAD_SRC,
       GST_PAD_ALWAYS,
       GST_STATIC_CAPS ( "audio/x-raw-int, "
           "endianness = (int) { " G_STRINGIFY (G_BYTE_ORDER) " }, "
           "signed = (boolean) TRUE, "
           "width = (int) 16, "
           "depth = (int) 16, "
           "rate = (int) 8000, "
           "channels = (int) 1;"

           "audio/x-alaw, "
           "rate = (int) 8000, "
           "channels = (int) 1; "

           "audio/x-mulaw, "
           "rate = (int) 8000, "
           "channels = (int) 1"
        )
     );

// Function prototypes

static void gst_dsppcmsrc_class_init (GstDSPPCMSrcClass * klass);
static void gst_dsppcmsrc_base_init (gpointer g_class);
static void gst_dsppcmsrc_init (GstDSPPCMSrc *dspmp3src, GstDSPPCMSrcClass *g_class);
static void gst_dsppcmsrc_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dsppcmsrc_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

static GstClock *gst_dsppcmsrc_provide_clock (GstElement * elem);
static GstClockTime gst_dsppcmsrc_get_time (GstClock * clock, GstDSPPCMSrc * src);


static GstCaps *gst_dsppcmsrc_getcaps (GstBaseSrc * src);
static gboolean gst_dsppcmsrc_setcaps (GstBaseSrc * src, GstCaps *caps);
static GstFlowReturn gst_dsppcmsrc_create (GstPushSrc * psrc, GstBuffer ** buf);
static gboolean gst_dsppcmsrc_start (GstBaseSrc * bsrc);
static gboolean gst_dsppcmsrc_stop (GstBaseSrc * bsrc);
static gboolean gst_dsppcmsrc_event (GstBaseSrc *src, GstEvent *event);
static gboolean gst_dsppcmsrc_unlock (GstBaseSrc *src);
static gboolean gst_dsppcmsrc_unlock_stop (GstBaseSrc *src);


GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPPCMSrc, gst_dsppcmsrc);


/*
 * gst_dsppcmsrc_do_init:
 * @type:
 *
 *
 */
static void
gst_dsppcmsrc_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dsppcmsrc_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dsppcmsrc_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dsppcmsrc_debug, "dsppcmsrc", 0,
                           "DSP PCM audio source");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}


GST_BOILERPLATE_FULL (GstDSPPCMSrc, gst_dsppcmsrc, GstPushSrc,
                      GST_TYPE_PUSH_SRC, gst_dsppcmsrc_do_init);


/*
 * gst_dsppcmsrc_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes PCM src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dsppcmsrc_dispose (GObject * object)
{
  GstDSPPCMSrc *dsp = (GstDSPPCMSrc *) object;

  if (dsp->clock) {
    gst_object_unref (dsp->clock);
    dsp->clock = NULL;
  }

  if (dsp->audio) {
    gst_dspaudio_destroy (dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dsppcmsrc_base_init:
 * @g_class: PCM src class
 *
 * Does the basic initialization of PCM src element class. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsrc_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP PCM Src",
    "Source/Audio",
    "PCM audio src",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
                                      gst_static_pad_template_get (&dsppcmsrc_src_template));
  gst_element_class_set_details (gstelement_class, &details);

}

/*
 * gst_dsppcmsrc_class_init:
 * @klass: PCM Src class object
 *
 * Initializes PCM src element class. This is called by GStreamer
 * framework.
 */
static void
gst_dsppcmsrc_class_init (GstDSPPCMSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_dispose);

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_provide_clock);

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_general_properties (G_OBJECT_CLASS (klass));

  gstbasesrc_class->start       = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_start);
  gstbasesrc_class->stop        = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_stop);
  gstbasesrc_class->get_caps    = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_getcaps);
  gstbasesrc_class->set_caps    = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_setcaps);
  gstbasesrc_class->event       = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_event);
  gstbasesrc_class->unlock      = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_unlock);
  gstbasesrc_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_unlock_stop);

  gstpushsrc_class->create   = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_create);
}


/*
 * gst_dsppcmsrc_init:
 * @dsp: DSP PCM src object
 * @g_class: PCM Class object
 *
 * Initializes the given PCM src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsrc_init (GstDSPPCMSrc * dsppcmsrc, GstDSPPCMSrcClass *g_class)
{
  dsppcmsrc->audio = gst_dspaudio_new ();
  gst_base_src_set_live (GST_BASE_SRC(dsppcmsrc), TRUE);
  GST_BASE_SRC (dsppcmsrc)->blocksize = 320;

  dsppcmsrc->clock = gst_audio_clock_new ("DSPClock",
                                          (GstAudioClockGetTimeFunc)
                                                  gst_dsppcmsrc_get_time,
                                          dsppcmsrc);
  GST_INFO ("clock created");
}


/*
 * gst_dsppcmsrc_getcaps:
 * @src: GstBaseSrc object
 *
 */
static GstCaps *
gst_dsppcmsrc_getcaps (GstBaseSrc * src)
{
  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD(src)));

  return caps;
}


/*
 * gst_dsppcmsrc_round_bsize:
 * @bsize: value to be rounded
 * @min: lower limit of rounding
 * @max: upper limit of rounding
 *
 * Rounds the blocksize to be dividable by 80 (that is 10ms in time)
 * and makes sure that it will be between certain limits
 *
 * Returns: Rounded value
 */
static guint
gst_dsppcmsrc_round_bsize (guint bsize, guint min, guint max)
{
  guint multiplier = bsize / 80;
  guint retval = multiplier * 80;
  return CLAMP (retval, min, max);
}


/*
 * gst_dsppcmsink_setcaps:
 * bsrc: GstBaseSrc object
 * @caps: GstCaps that were offered by the connecting element
 *
 * Returns: TRUE if caps can be accepted, otherwise FALSE
 */
static gboolean
gst_dsppcmsrc_setcaps (GstBaseSrc *bsrc, GstCaps * caps)
{
  GST_DEBUG ("gst_dsppcmsrc_setcaps");

  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (bsrc);
  guint format, sample_rate, codec;
  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if(!strncmp(mimetype, "audio/x-raw-int", 15)) {
    GST_INFO("linear");
    codec = GST_DSPAUDIO_CODEC_RAW;
    dsp->width = 2;
  }
  else if(!strncmp(mimetype, "audio/x-mulaw", 13)) {
    GST_DEBUG("mu-law");
    codec = GST_DSPAUDIO_CODEC_MULAW;
    dsp->width = 1;
  }
  else if(!strncmp(mimetype, "audio/x-alaw", 12)) {
    GST_DEBUG("a-law");
    codec = GST_DSPAUDIO_CODEC_ALAW;
    dsp->width = 1;
  }
  else {
    GST_WARNING ("unknown format");
    return FALSE;
  }

  if(!gst_dspaudio_map_format (dsp->audio, codec, G_LITTLE_ENDIAN, TRUE, 
    dsp->width * 8, &format, NULL)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }

  if(!gst_dspaudio_map_samplerate (8000,&sample_rate)) {
    GST_WARNING ("unsupported sample rate: 8000");
    return FALSE;
  }

  dsp->framesize = gst_dsppcmsrc_round_bsize (bsrc->blocksize, 80, 800);
  dsp->frametimemillis = dsp->framesize / 8;
  GST_DEBUG("DSP frame size: %d bytes (%d millisecs)",
            dsp->framesize * dsp->width, dsp->frametimemillis);

  if (gst_dspaudio_open (dsp->audio, pcm_node_name) == FALSE) {
    gst_dspaudio_ed_microphone (dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_speechparams (dsp->audio,
    format, sample_rate, dsp->framesize)) {
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_update_dsp_settings (dsp->audio);
  gst_dspaudio_play (dsp->audio);

  return TRUE;
}


/*
 * gst_dsppcmsrc_create:
 * @src:
 * @buffer:
 *
 */
static GstFlowReturn
gst_dsppcmsrc_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC(psrc);
  GstFlowReturn retval = GST_FLOW_OK;
  guint recv_frames = 1;

  GST_LOG ("render");

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }

  DSPWaitStatus status;

waitbuf2:
  status = gst_dspaudio_wait_buffer (dsp->audio);

  if (status == DSP_WAIT_ERROR) {
    GST_LOG ("DSP_WAIT_ERROR");
    dsp->audio->mode = DSP_MODE_ERROR;
    retval = GST_FLOW_ERROR;
    goto beach;
  }
  if (status == DSP_WAIT_INTERRUPT) {
    GST_LOG ("DSP_WAIT_INTERRUPT");
    retval = gst_base_src_wait_playing (GST_BASE_SRC_CAST (psrc));
    if (retval != GST_FLOW_OK)
      goto beach;
    goto waitbuf2;
  }

  recv_frames = dsp->audio->readinfo.frame_size / dsp->framesize;
  *buf = gst_buffer_new_and_alloc (recv_frames * dsp->framesize * dsp->width);
  guint8 *data = GST_BUFFER_DATA (*buf);

  if (!gst_dspaudio_read_data (dsp->audio, GST_BUFFER_SIZE (*buf), data)) {
    retval = GST_FLOW_ERROR;
    goto beach;
  }

  GST_BUFFER_DURATION(*buf) = recv_frames * dsp->frametimemillis * GST_MSECOND;
  GST_BUFFER_TIMESTAMP(*buf) = dsp->framecount * dsp->frametimemillis * GST_MSECOND;
  dsp->framecount += recv_frames;

  if(*buf) {
    gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
    retval = GST_FLOW_OK;
  }
beach:
  return retval;
}


/*
 * gst_dsppcmsrc_start:
 *
 */
static gboolean
gst_dsppcmsrc_start (GstBaseSrc * bsrc)
{
  GST_INFO("gst_dsppcmsrc_start");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (bsrc);

  if (gst_dspaudio_ed_microphone (dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  // This is used for clocking
  dsp->framecount = 0;
  GST_DEBUG("gst_dsppcmsrc_start OK");
  return TRUE;
}



/*
 * gst_dsppcmsrc_stop:
 *
 */
static gboolean
gst_dsppcmsrc_stop (GstBaseSrc * bsrc)
{
  GST_INFO("gst_dsppcmsrc_stop");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (bsrc);

  gst_dspaudio_remove_interrupt (dsp->audio);

  if (dsp->audio->all_frames_read ||
      dsp->audio->read_sent)
  {
    GST_DEBUG("Waiting reply for CMD_DATA_READ");
    gst_dspaudio_wait_buffer (dsp->audio);
    GST_DEBUG("Got it");
  }

  gst_dspaudio_dtmf_tone_stop (dsp->audio);
  gst_dspaudio_stop (dsp->audio);
  gst_dspaudio_ed_microphone (dsp->audio, FALSE);
  gst_dspaudio_close (dsp->audio);
  gst_dspaudio_reset (dsp->audio);

  GST_DEBUG ("gst_dsppcmsrc_stop OK");
  return TRUE;
}


/*
 * gst_dsppcmsrc_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dsppcmsrc_set_property (GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (object);

  if (!gst_dspaudio_set_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dsppcmsrc_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dsppcmsrc_get_property (GObject * object,
                            guint prop_id,
                            GValue * value,
                            GParamSpec * pspec)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (object);

  if (!gst_dspaudio_get_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dsppcmsrc_event:
 * @src: GstBaseSrc object
 * @event: GstEvent to be handled
 *
 * Handles incoming events. Checks if the event is special "custom upstream
 * event" and in this case forwards it to dspaudio library for checking if
 * it is DTMF event.
 *
 * Returns: TRUE if the event was handled.
 */
static gboolean
gst_dsppcmsrc_event (GstBaseSrc *src, GstEvent *event)
{
  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (src);
  gboolean ret = FALSE;
    if (type == GST_EVENT_CUSTOM_UPSTREAM) {
    ret = gst_dspaudio_check_upstream_event (dsp->audio, event);
  }
  return ret;
}


/*
 * gst_dsppcmsrc_provide_clock:
 *
 */
static GstClock *
gst_dsppcmsrc_provide_clock (GstElement * elem)
{
  GST_INFO ("gst_dsppcmsrc_provide_clock");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 * gst_dsppcmsrc_get_time:
 *
 */
static GstClockTime
gst_dsppcmsrc_get_time (GstClock * clock, GstDSPPCMSrc * src)
{
  GST_INFO("gst_dsppcmsrc_get_time");
  GstClockTime result = src->framecount * src->frametimemillis * GST_MSECOND;
  return result;
}


/*
 * gst_dsppcmsrc_unlock:
 * @src: GstBaseSrc
 *
 * Interrupts the blocking :create() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dsppcmsrc_unlock (GstBaseSrc *src)
{
    GST_DEBUG ("unlock");
    GstDSPPCMSrc *dsp = GST_DSPPCMSRC (src);
    gst_dspaudio_interrupt_render (dsp->audio);
    return TRUE;
}


/*
 * gst_dsppcmsrc_unlock_stop:
 * @src: GstBaseSrc
 *
 * Removes :create() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dsppcmsrc_unlock_stop (GstBaseSrc *src)
{
    GST_DEBUG ("unlock_stop");
    GstDSPPCMSrc *dsp = GST_DSPPCMSRC (src);
    gst_dspaudio_remove_interrupt (dsp->audio);
    return TRUE;
}

