/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <string.h>
#include <unistd.h>
#include <gst/gst.h>

#include "gstdspmp3sink.h"
#include "dspmixer.h"

GST_DEBUG_CATEGORY_STATIC (dspmp3_debug);
#define GST_CAT_DEFAULT dspmp3_debug

static const gchar *dsp_node_names[3] =
{
  NULL,
  "/dev/dsptask/mp2dec",
  "/dev/dsptask/mp3dec"
};

static gboolean char_in_tmp = FALSE;
static char tmp_char;

// MP3 bitrate lookup tables
static const guint bitrates[2][3][16] =
{
  {
    {0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0},
    {0, 32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384, 0},
    {0, 32, 40, 48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 0}
  },
  {
    {0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0},
    {0,  8, 16, 24, 32, 40, 48,  56,  64,  80,  96, 112, 128, 144, 160, 0},
    {0,  8, 16, 24, 32, 40, 48,  56,  64,  80,  96, 112, 128, 144, 160, 0}
  }
};

static const guint mp3_samplerates[4][3] =
{
  { 11025, 12000,  8000, },   // MPEG version 2.5
  {     0,     0,     0, },   // Reserved
  { 22050, 24000, 16000, },   // MPEG version 2
  { 44100, 48000, 32000, }    // MPEG version 1
};

static const int bs[3] = {384, 1152, 1152};

// Xing MP3 frame header related constants
#define XING_PEEK_BYTES 2048

#define FRAMES_FLAG     0x0001
#define BYTES_FLAG      0x0002
#define TOC_FLAG        0x0004
#define VBR_SCALE_FLAG  0x0008


static GstStaticPadTemplate dspmp3sink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/mpeg, "
                      "mpegversion = (int) 1, "
                      "layer = (int) [2, 3]" ) );

// Function prototypes

static void gst_dspmp3sink_class_init (GstDSPMP3SinkClass * klass);
static void gst_dspmp3sink_base_init (gpointer g_class);
static void gst_dspmp3sink_init (GstDSPMP3Sink *dspmp3sink, GstDSPMP3SinkClass *g_class);
static gboolean gst_dspmp3sink_initialize(GstDSPMP3Sink *dsp);
static GstClockTime gst_dspmp3sink_get_time (GstClock * clock, gpointer sink);
static gboolean gst_dspmp3sink_open(GstDSPMP3Sink *dsp, gint layer);
static gboolean gst_dspmp3sink_loop_initialization(GstDSPMP3Sink *dsp, gboolean check_stream);

// For GObject
static void gst_dspmp3sink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspmp3sink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

//  For GstElement
static gboolean gst_dspmp3sink_event (GstElement * element, GstEvent *event);
static gboolean gst_dspmp3sink_query (GstElement * element, GstQuery *query);
static const GstQueryType * gst_dspmp3sink_get_query_types(GstElement * element);
static GstStateChangeReturn gst_dspmp3sink_change_state (GstElement * element, GstStateChange transition);
static GstClock *gst_dspmp3sink_provide_clock (GstElement * elem);
//static gboolean gst_dspmp3sink_set_clock(GstElement *element, GstClock *clock);

// For GstBaseSink
static GstFlowReturn gst_dspmp3sink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspmp3sink_event2(GstBaseSink *sink, GstEvent *event);
static gboolean gst_dspmp3sink_setcaps(GstBaseSink *sink, GstCaps *caps);
static GstFlowReturn gst_dspmp3sink_preroll(GstBaseSink *sink, GstBuffer * buffer);
static GstStateChangeReturn gst_dspmp3sink_async_play(GstBaseSink *sink);
static gboolean gst_dspmp3sink_unlock (GstBaseSink *sink);
static gboolean gst_dspmp3sink_unlock_stop (GstBaseSink *sink);


GST_IMPLEMENT_DSPMIXER_METHODS(GstDSPMP3Sink, gst_dspmp3sink);

/*
 * gst_dspmp3sink_do_init:
 * @type:
 *
 *
 */
static void
gst_dspmp3sink_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspmp3sink_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspmp3sink_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspmp3_debug, "dspmp3sink", 0,
                           "DSP MP2/MP3 audio sink");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}

GST_BOILERPLATE_FULL (GstDSPMP3Sink, gst_dspmp3sink, GstBaseSink,
                      GST_TYPE_BASE_SINK, gst_dspmp3sink_do_init);


/*
 * gst_dspmp3sink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes MP3 sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspmp3sink_dispose (GObject * object)
{
  GstDSPMP3Sink *dsp = (GstDSPMP3Sink *) object;

  if(dsp->clock) {
    gst_object_unref(dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  g_object_unref(G_OBJECT(dsp->adapter));
  G_OBJECT_CLASS (parent_class)->dispose(object);
}


/*
 * gst_dspmp3sink_base_init:
 * @g_class: MP3 sink class
 *
 * Does the basic initialization of MP3 sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspmp3sink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP MP3 Sink",
    "Sink/Audio",
    "MP3 audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspmp3sink_sink_template));
  gst_element_class_set_details (gstelement_class, &details);
}


/*
 * gst_dspmp3sink_class_init:
 * @klass: MP3 Sink class object
 *
 * Initializes MP3 sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspmp3sink_class_init (GstDSPMP3SinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspmp3sink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspmp3sink_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspmp3sink_dispose);

  gstelement_class->change_state    = GST_DEBUG_FUNCPTR (gst_dspmp3sink_change_state);
  gstelement_class->get_query_types = GST_DEBUG_FUNCPTR (gst_dspmp3sink_get_query_types);
  gstelement_class->send_event      = GST_DEBUG_FUNCPTR (gst_dspmp3sink_event);
  gstelement_class->query           = GST_DEBUG_FUNCPTR (gst_dspmp3sink_query);
  gstelement_class->provide_clock   = GST_DEBUG_FUNCPTR (gst_dspmp3sink_provide_clock);
//   gstelement_class->set_clock       = GST_DEBUG_FUNCPTR (gst_dspmp3sink_set_clock);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_sink_properties (G_OBJECT_CLASS (klass));

  gstbase_sink_class->render      = GST_DEBUG_FUNCPTR (gst_dspmp3sink_render);
  gstbase_sink_class->set_caps    = GST_DEBUG_FUNCPTR (gst_dspmp3sink_setcaps);
  gstbase_sink_class->event       = GST_DEBUG_FUNCPTR (gst_dspmp3sink_event2);
  gstbase_sink_class->preroll     = GST_DEBUG_FUNCPTR (gst_dspmp3sink_preroll);
  gstbase_sink_class->async_play  = GST_DEBUG_FUNCPTR (gst_dspmp3sink_async_play);
  gstbase_sink_class->unlock      = GST_DEBUG_FUNCPTR (gst_dspmp3sink_unlock);
  gstbase_sink_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_dspmp3sink_unlock_stop);
}


/*
 * gst_dspmp3sink_init:
 * @dsp: DSP MP3 sink object
 * @g_class: MP3 sink Class object
 *
 * Initializes the given MP3 sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspmp3sink_init (GstDSPMP3Sink * dspmp3sink, GstDSPMP3SinkClass *g_class)
{
  GstBaseSink *bsink = GST_BASE_SINK(dspmp3sink);
  dspmp3sink->sinkpad = bsink->sinkpad;

  dspmp3sink->audio = gst_dspaudio_new();
  dspmp3sink->adapter = gst_adapter_new();

  dspmp3sink->clock = gst_audio_clock_new ("DSPClock",
                                           (GstAudioClockGetTimeFunc) gst_dspmp3sink_get_time,
                                            dspmp3sink);
  // DSP sinks do not need syncing
  gst_base_sink_set_sync (bsink, FALSE);

}


/*
 * gst_dspmp3sink_get_missing_char:
 * @dsp: DSP MP3 sink element
 *
 * Utility method for handling the troublesome odd character
 * in the end of MP3 frame (MMAP buffer must always be filled
 * with even number of data, so we must store the last odd
 * byte).
 */
static inline void
gst_dspmp3sink_get_missing_char(GstDSPMP3Sink *dsp)
{
  if(char_in_tmp) {
    dsp->audio->codec.mmap_buffer[dsp->mmap_ptr] = tmp_char;
    char_in_tmp = FALSE;
    dsp->total_bytes++;
    dsp->mmap_ptr++;
  }
}


/*
 * gst_dspmp3sink_bpf:
 *
 * Calculate Bits-per-Second (?) value
 *
 */
gdouble
static gst_dspmp3sink_bpf(MP3Frame *frame, guint bitrate)
{
  // Calculate "bytes-per-frame" value
  gdouble bpf = bitrate;
  switch (frame->layer) {
    case 0:
      bpf *= 12000.0 * 4.0;
      bpf /= frame->sample_rate_hz << frame->lsf;
      break;
    case 1:
/*      bpf *= 144000;
      bpf /= frame->sample_rate_hz;
      break;*/
    case 2:
      bpf *= 144000;
      bpf /= frame->sample_rate_hz << frame->lsf;
      break;
    default:
      bpf = 1.0;
  }
  GST_LOG("SR = %d, BR = %d, BPF = %f, lsf = %d", frame->sample_rate_hz,bitrate, bpf, frame->lsf);
  return bpf;
}


/*
 * gst_dspmp3sink_framesize:
 *
 * Calculate frame size in bytes including header.
 *
 */
static int
gst_dspmp3sink_framesize(MP3Frame *frame)
{
  // Calculate "bytes-per-frame" value
  int size = frame->bitrate;
  switch (frame->layer) {
    case 0:
      size *= 12000;
      size /= frame->sample_rate_hz;
      size = (size + frame->pad) << 2;
      break;
    case 1:
      size *= 144000;
      size /= frame->sample_rate_hz;
      size += frame->pad;
      break;
    case 2:
      size *= 144000;
      size /= frame->sample_rate_hz << frame->lsf;
      size += frame->pad;
      break;
    default:
      size = 0;
  }
  return size;
}


/*
 * gst_dspmp3sink_head_check:
 * @head: MP3 header in long int format
 *
 * Checks if the given 32-bit value contains proper MP3 header
 *
 * Returns: TRUE iff the header is valid
 */

gboolean gst_dspmp3sink_head_check(guint32 head)
{
//  if ((head & 0xfff00000) != 0xfff00000)
  if ((head & 0xffe00000) != 0xffe00000)
  return FALSE;
  if (!((head >> 17) & 3))
    return FALSE;
  if (((head >> 12) & 0xf) == 0xf)
    return FALSE;
  if (!((head >> 12) & 0xf))
    return FALSE;
  if (((head >> 10) & 0x3) == 0x3)
    return FALSE;
  if (((head >> 19) & 1) == 1 &&    // MPEG V1 ?
        ((head >> 17) & 3) == 3 &&  // Layer 1 ?
        ((head >> 16) & 1) == 1)
    return FALSE;
  if ((head & 0xffff0000) == 0xfffe0000)
    return FALSE;
  if (((head >> 19) & 0x3) == 0x1)
    return FALSE;

  return TRUE;
}


static inline guint32
gst_dspmp3sink_convert_to_header(const guint8 * buf) {
  return (buf[0] << 24) + (buf[1] << 16) + (buf[2] << 8) + buf[3];
}


/*
 * gst_dspmp3sink_find_sync:
 * @dsp: DSP MP3 sink object
 *
 * Tries to find the MP3 "magic key" inside the incoming
 * data stream. This key is can be used as a synchronization
 * marker, since it is a frame boundary identifier.
 */
static guint32
gst_dspmp3sink_find_sync(GstDSPMP3Sink *dsp, gboolean *gap_detected)
{
  guint32 head = 0;
  const guint8 *data = NULL;
  guint fcount = 0;
  guint avail = gst_adapter_available(dsp->adapter);

  while( avail >= 4 ) {

    data = gst_adapter_peek(dsp->adapter, 4);
    head = gst_dspmp3sink_convert_to_header(data);
    if(gst_dspmp3sink_head_check(head)) {
      if(fcount) {
        GST_DEBUG("Find sync skipped %d bytes", fcount);
      }
      return head;
    }
    gst_adapter_flush(dsp->adapter, 1);
    *gap_detected = TRUE;
    fcount++;
    avail--;
  }
  return 0;
}


/*
 * gst_dspmp3sink_mp3_xing_check:
 * @dsp: DSP MP3 sink object
 * @data: the buffer containing first frame of the MP3 stream
 *
 * Checks if the header contains Xing extension. If so, extract the
 * number of frames.
 *
 * Returns: Number of frames information, or 0 if no Xing header found
 */

#define GET_INT32BE(b) \
  (i = (b[0] << 24) | (b[1] << 16) | b[2] << 8 | b[3], b += 4, i)

static gboolean
gst_dspmp3sink_mp3_xing_check(MP3Frame *frame, XINGTag *tag, gchar *data)
{
  guint i, head_flags;
  gchar mpegid, channelmode;

  // Later we assume that XING header is available if this
  // variable is non-zero
  tag->xframes = 0;

  mpegid      = (data[1] >> 3) & 1;   // Get MPEG version
  channelmode = (data[3] >> 6) & 3;   // Get channel mode
  data += 4;                          // Skip "normal header"

  // Calculate the index for XING
  if(mpegid) {
    if (channelmode != 3)
      data += 32;
    else
      data += 17;
  }
  else {
    if (channelmode != 3)
      data += 17;
    else
      data += 9;
  }

  // Is there a Xing header?
  if(strncmp(data, "Xing", 4)) {
    return 0;
  }

  GST_DEBUG("XING tag found");

  data += 4;
  head_flags = GET_INT32BE(data);

  if(head_flags & FRAMES_FLAG) {
    tag->xframes = GET_INT32BE(data);
  }

  if(head_flags & BYTES_FLAG) {
    tag->xbytes = GET_INT32BE(data);
  }

  if(head_flags & TOC_FLAG) {
    for (i = 0; i < 100; i++) {
      tag->toc[i] = data[i];
      if (i > 0 && tag->toc[i] < tag->toc[i - 1]) {
        return FALSE;
      }
    }
    if (tag->toc[99] == 0) {
      return FALSE;
    }
    data += 100;
  }
  else
    for (i = 0; i < 100; i++)
      tag->toc[i] = (i * 256) / 100;

  return TRUE;
}


/*
 * gst_dspmp3sink_mp3_parse_head:
 *
 */
static void
gst_dspmp3sink_mp3_parse_head(MP3Frame *frame, guint32 head)
{
  short int br_index, sr_index;

  frame->mode     = (head & 0x00180000) >> 19;
  frame->layer    = 3 - ((head & 0x00060000) >> 17);  // between 0-2
  br_index        = (head & 0x0000F000) >> 12;
  sr_index        = (head & 0x00000C00) >> 10;
  frame->pad      = (head & 0x00000200) >> 9;
  frame->channels = ((head & 0x000000C0) >> 6) == 3 ? 1 : 2;

  if(frame->mode == 0) {
    frame->lsf = 1;
    frame->bitrate = bitrates[1][frame->layer][br_index];
  }
  else {
    frame->lsf = frame->mode == 3 ? 0 : 1;
    frame->bitrate = bitrates[frame->lsf][frame->layer][br_index];
  }
  frame->sample_rate_hz = mp3_samplerates[frame->mode][sr_index];
}


/*
 * gst_dspmp3sink_mp3_frame_calc_tpf:
 *
 */
static void inline
gst_dspmp3sink_mp3_frame_calc_tpf(MP3Frame *frame)
{
  frame->tpf = bs[frame->layer];
  frame->tpf /= (frame->sample_rate_hz << (frame->lsf));
  GST_DEBUG("Frame_tpf = %f", frame->tpf);
}


/*
 * gst_dspmp3sink_get_stream_details:
 * @dsp: DSP MP3 sink object
 * @quick_mode: Do not perform deep-scan for the stream. This
 *              also prevents the detection of XING tag
 *
 * Parses the incoming MP3 data stream and extracts
 * stream information from it. This information is then
 * stored into DSP sink object itself.
 *
 * Returns: TRUE if the info was found, otherwise FALSE
 */
static gboolean
gst_dspmp3sink_get_stream_details(GstDSPMP3Sink *dsp, gboolean quick_mode)
{
  MP3Frame first_frame;
  MP3Frame tmp_frame;
  gint same_sr_cnt = 0, sr_old, sr_curr = -1;
  guint count, curr_index = 0;
  gboolean found = FALSE;
  const guint8 *byte;
  guint32 head;
  guint flushamount;

  guint NUM_SAME_SAMPLERATES_NEEDED = quick_mode ? 1 : 5;

  guint avail =  gst_adapter_available(dsp->adapter);

  bzero(&tmp_frame, sizeof(MP3Frame));
  bzero(&first_frame, sizeof(MP3Frame));

  // This ugly loop is needed because we want to find
  // certain amount of consecutive frames with same samplerate value

  GST_DEBUG("Starting samplerate loop");
  while(!found) {

    sr_old = sr_curr;

    if(same_sr_cnt == NUM_SAME_SAMPLERATES_NEEDED) {
      GST_DEBUG("Sync found");
      found = TRUE;
      break;
    }

    count = curr_index + 4;
    if(avail < count) {
      GST_DEBUG("No %d bytes available", count);
      return FALSE;
    }

    byte = gst_adapter_peek(dsp->adapter, count);
    head = gst_dspmp3sink_convert_to_header(&byte[curr_index]);

    if(gst_dspmp3sink_head_check(head)) {
      gst_dspmp3sink_mp3_parse_head(&tmp_frame, head);
      sr_curr = tmp_frame.sample_rate_hz;

      if(curr_index == 0) {
        // Store the first frame
        memcpy(&first_frame, &tmp_frame, sizeof(MP3Frame));
      }

      if(dsp->total_bytes == 0 && dsp->xingtag.xframes == 0 && !quick_mode) {

        GST_DEBUG("Checking for xing");

        // Check for XING extension
        guint amount = count > XING_PEEK_BYTES ? count : XING_PEEK_BYTES;

        if(avail < amount) {
          GST_DEBUG("Cannot check XING, no data");
          return FALSE;
        }

        byte = gst_adapter_peek(dsp->adapter, amount);

        gst_dspmp3sink_mp3_xing_check(&tmp_frame, &dsp->xingtag,
                                       (gchar *) byte+curr_index);
      }

      if(sr_old == sr_curr || sr_old == -1) {
        curr_index += gst_dspmp3sink_framesize(&tmp_frame);
        same_sr_cnt++;
        continue;
      }
    }

    flushamount = curr_index ? curr_index : 1;

    if(avail < flushamount) {
      GST_DEBUG("Cannot flush %d bytes", flushamount);
      return FALSE;
    }
    gst_adapter_flush(dsp->adapter, flushamount);
    avail -= flushamount;
    same_sr_cnt = 0;
    curr_index = 0;
  }

  // WOHOO, valid stream found. Backup frame information
  memcpy(&dsp->frame, &first_frame, sizeof(MP3Frame));
  gst_dspmp3sink_mp3_frame_calc_tpf(&dsp->frame);
  if(!gst_dspaudio_map_samplerate(dsp->frame.sample_rate_hz, &dsp->sample_rate)) {
    GST_ERROR("Not supported sample rate : %d!", dsp->frame.sample_rate_hz);
    return FALSE;
  }


  GST_DEBUG("Samplerate: %d, channels: %d", dsp->frame.sample_rate_hz,
            dsp->frame.channels);

  dsp->sync = TRUE;
  return TRUE;
}


/*
 * gst_dspmp3sink_setcaps:
 * @pad: GstPad that is the target for connect
 * @caps: GStCaps offered
 *
 * Returns: TRUE if caps are accepted
 */
static gboolean
gst_dspmp3sink_setcaps(GstBaseSink *sink, GstCaps *caps)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);
  gboolean ret = TRUE;

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);
  gint version, layer, channels, rate;

  version = layer = channels = rate = -1;

  if(!strncmp(mimetype, "application/x-id3", 17)) {
    return TRUE;
  }

  else if(strncmp(mimetype, "audio/mpeg", 10)) {
    return FALSE;
  }

  if(gst_structure_has_field(structure, "layer")) {
    gst_structure_get_int (structure, "layer", &layer);
  }
  if(gst_structure_has_field(structure, "mpegversion")) {
    gst_structure_get_int (structure, "mpegversion", &version);
  }
  if(gst_structure_has_field(structure, "channels")) {
    gst_structure_get_int (structure, "channels", &channels);
  }
  if(gst_structure_has_field(structure, "rate")) {
    gst_structure_get_int (structure, "rate", &rate);
  }

  if(version != 1 && (layer<2 || layer>3)) {
    GST_ERROR("Not supported version/layer : %d, %d!", version, layer);
    return FALSE;
  }

  // If we have all required information for initializing the node,
  // we do it here.
  if(channels != -1 && rate != -1 && dsp->audio->mode < DSP_MODE_CONFIGURED) {
    dsp->frame.channels = channels;
    dsp->frame.layer = layer-1;
    if(!gst_dspaudio_map_samplerate(rate, &dsp->sample_rate)) {
      GST_ERROR("Not supported sample rate : %d!", rate);
      return FALSE;
    }
    ret = gst_dspmp3sink_loop_initialization(dsp, FALSE);
    dsp->sync = TRUE;
    GST_DEBUG("Initialized in setcaps");
  }
  return ret;
}


/*
 * gst_dspmp3sink_calculate_seekpos:
 * @dsp: DSP MP3 sink object
 * @time_offset: the time where to seek
 *
 * Calculate a seek position (in bytes) according to time_offset
 * stored in DSP structure
 *
 * Returns: seek position in bytes
 */
static gint64
gst_dspmp3sink_calculate_seekpos(GstDSPMP3Sink *dsp, guint64 time_offset)
{

  if (dsp->frame.tpf == 0.0) {
    GST_DEBUG ("failed, stream details not known");
    return -1;
  }

   // XING-based seek
  if(dsp->xingtag.xframes && dsp->xingtag.xbytes) {
    return time_offset / GST_SECOND / dsp->frame.tpf *
        (dsp->xingtag.xbytes / dsp->xingtag.xframes);
  }
  else if (dsp->vbr) {
    GstPad *peer = gst_pad_get_peer(dsp->sinkpad);
    GstFormat fmt = GST_FORMAT_BYTES;
    gboolean ret = FALSE;
    gint64 curr, len;

    ret  = gst_pad_query_position (peer, &fmt, &curr);
    ret |= gst_pad_query_duration (peer, &fmt, &len);
    gst_object_unref(peer); // get_peer() ref'ed it

    if (curr >= 0 && len > 0 && ret) {
      gdouble rel = (gdouble) curr / len;
      gdouble bpf2 = 0.0;
      if(dsp->frame.layer == 1) {
        bpf2 = (gdouble) dsp->total_bytes / (dsp->total_frames << dsp->frame.lsf);
      } else {
        bpf2 = (gdouble) dsp->total_bytes / (dsp->total_frames);
      }

      if(bpf2 != 0.0 && rel != 0.0) {
        gdouble currframes = curr / bpf2;
        gdouble written_time = dsp->frame.tpf * currframes;
        gdouble total = written_time / rel;
        return (gint64) (time_offset/total*len/GST_SECOND);
      }
    }
    return 0;
  }
  else {
    gdouble bpf = gst_dspmp3sink_bpf(&dsp->frame,
                                      dsp->frame.bitrate);

    guint frame = (time_offset /  GST_SECOND) / dsp->frame.tpf;
    guint64 pos = (guint) (frame * bpf);
    return pos;
  }
  return -1;
}


/*
 * gst_dspmp3sink_perform_seek:
 * @dsp: DSP MP3 sink object
 *
 * Perform SEEK operation
 */
static gboolean
gst_dspmp3sink_perform_seek(GstDSPMP3Sink *dsp, guint64 time_offset)
{
  gint64 pos = gst_dspmp3sink_calculate_seekpos(dsp, time_offset);
  if(pos == -1) {
    return FALSE;
  }

  GstEvent *new_event = gst_event_new_seek(1.0, GST_FORMAT_BYTES,
                                           GST_SEEK_FLAG_FLUSH,
                                           GST_SEEK_TYPE_SET, pos,
                                           GST_SEEK_TYPE_NONE,
                                           GST_CLOCK_TIME_NONE);

  if (gst_pad_push_event (dsp->sinkpad, new_event)) {
    GST_DEBUG ("audio mode: %d", dsp->audio->mode);
  }
  else {
    GST_WARNING ("Seeking failed");
    //GST_ELEMENT_ERROR (dsp, CORE, SEEK, (NULL), ("gst_event_new_seek"));
    return FALSE;
  }
  return TRUE;
}


/*
 * gst_dspmp3sink_open:
 *
 * Open the codec
 *
 */
static gboolean
gst_dspmp3sink_open(GstDSPMP3Sink *dsp, gint layer)
{
  const gchar *devname = dsp_node_names[layer]; // between 0-2

  if(!gst_dspaudio_open(dsp->audio, devname)) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }
  return TRUE;
}


/**
 * Initialize the DSP for playback and send PLAY command
 *
 */
static gboolean
gst_dspmp3sink_loop_initialization(GstDSPMP3Sink *dsp, gboolean check_stream)
{
  GST_DEBUG("gst_dspmp3sink_loop_initialization");

  // render() has already gathered enough data for us
  if(check_stream) {
    if(!gst_dspmp3sink_get_stream_details(dsp, FALSE)) {
      return FALSE;
    }
  }

  // Open the DSP audio device if not opened yet
  if (dsp->audio->mode == DSP_MODE_UNINITIALIZED) {
    if (!gst_dspmp3sink_open (dsp, dsp->frame.layer)) {
      return FALSE;
    }
  }

  if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
    if (!gst_dspmp3sink_initialize (dsp)) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                        ("gst_dspmp3sink_initialize"));
      return FALSE;
    }
  }

  // Settings might have changed between PAUSE->PLAY transition
  gst_dspaudio_update_dsp_settings (dsp->audio);
  gst_dspaudio_play (dsp->audio);
  return TRUE;
}


/*
 * gst_dspmp3sink_fill_it:
 *
 */
static gboolean
gst_dspmp3sink_fill_it(GstDSPMP3Sink *dsp, int btr)
{
  const guint8 *data;

  if(gst_adapter_available(dsp->adapter) < btr) {
    GST_DEBUG("fill_it: cannot read %d bytes", btr);
    return FALSE;
  }
  else if(dsp->mmap_ptr + btr >= dsp->audio->codec.mmap_buffer_size) {
    GST_DEBUG("fill_it: mmap buffer full");
    return FALSE;
  }

  data = gst_adapter_peek(dsp->adapter, btr);

  memcpy(&(dsp->audio->codec.mmap_buffer[dsp->mmap_ptr]), data, btr);
  gst_adapter_flush(dsp->adapter, btr);
  dsp->total_bytes += btr;
  dsp->mmap_ptr += btr;
  return TRUE;
}


/*
 * gst_dspmp3sink_copy_frame:
 *
 * Fill DSP's MMAP buffer with MP3 frames
 *
 * Returns: TRUE if operation was successful
 */
static gboolean
gst_dspmp3sink_copy_frame(GstDSPMP3Sink *dsp)
{
  gboolean gap_detected = FALSE;

  dsp->curr_fc = 0;

  guint32 head = gst_dspmp3sink_find_sync(dsp, &gap_detected);
  if(head != 0) {

    // Do we need to take a closer look to the stream?
    if(gap_detected || dsp->sync == FALSE) {
      // This guarantees that we really have found valid stream
      GST_DEBUG("Gap (%d) / Sync (%d) -> closer look",
          gap_detected, dsp->sync);
      if(!gst_dspmp3sink_get_stream_details(dsp, FALSE)) {
        GST_DEBUG("Cannot examine stream");
        return FALSE;
      }
      char_in_tmp = FALSE;
    }
    else {
      gst_dspmp3sink_mp3_parse_head(&dsp->frame, head);
    }
    dsp->sync = TRUE;

    dsp->curr_fc++;
    dsp->frame_count++;
    dsp->total_frames++;
    dsp->bitrate_sum += dsp->frame.bitrate;
    dsp->avg_bitrate = dsp->bitrate_sum / dsp->frame_count;

    // Check if the stream is VBR
    if(dsp->prev_bitrate != dsp->frame.bitrate) {
      if(dsp->vbr_counter++ > 5) {
        dsp->vbr = TRUE;
      }
    }

    dsp->prev_bitrate = dsp->frame.bitrate;

    // Calculate avg. bitrate every 50 frames
    if(dsp->frame_count == 50) {
      dsp->bitrate_sum = 0;
      dsp->frame_count = 0;
    }

    gint btr = gst_dspmp3sink_framesize(&dsp->frame);

    gst_dspmp3sink_get_missing_char(dsp);
    if(!gst_dspmp3sink_fill_it(dsp, btr)) {
      GST_DEBUG("Cannot fill buffer - not enough data");
      return FALSE;
    }
  }
  else {
    GST_WARNING("Cannot find header");
    return FALSE; // Probably EOS
  }

  if((dsp->mmap_ptr % 2) == 1) {
    tmp_char = dsp->audio->codec.mmap_buffer[dsp->mmap_ptr-1];
    char_in_tmp = TRUE;
    dsp->mmap_ptr--;
  }
  return TRUE;
}


/*
 * gst_dspmp3sink_render:
 * @sink:
 * @buffer:
 *
 * This method is automatically called by GStreamer framework, when
 * a scheduler decides to give execution to this element. It's
 * purpose is to process the data and forward it to DSP for playback.
 */

#define STREAM_DETECT_THRESHOLD 10000

static GstFlowReturn
gst_dspmp3sink_render (GstBaseSink *sink, GstBuffer * buffer)
{
  GstFlowReturn retval = GST_FLOW_OK;
  DSPWaitStatus status;

  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);

  GST_LOG ("MP3 render");

  if (dsp->clear_adapter) {
    GST_DEBUG ("clearing adapter after preroll");
    gst_adapter_clear (dsp->adapter);
    dsp->clear_adapter = FALSE;
  }

  // We might be called from EOS handler. In this case buffer is NULL
  if(buffer) {
    gst_buffer_ref(buffer);
    gst_adapter_push (dsp->adapter, buffer);
  }

  // Do the possible initialization if not done yet...
  // NOTE: This initialization might be already be done in _preroll() or
  //       in _setcaps(). in this case the audio->mode == DSP_MODE_CONFIGURED
  if (dsp->audio->mode < DSP_MODE_CONFIGURED) {

    if (gst_adapter_available (dsp->adapter) < STREAM_DETECT_THRESHOLD) {
      // First we need to gather enough data to detect the stream
      return GST_FLOW_OK;
    }

    if (gst_dspmp3sink_loop_initialization (dsp, TRUE) == FALSE) {
      GST_ELEMENT_ERROR (sink, STREAM, TYPE_NOT_FOUND, (NULL),
                            ("gst_dspmp3sink_loop_initialization"));
         GST_WARNING("Render returning gst_flow_error 1");
         return GST_FLOW_ERROR;
    }
  }

  // Initialization done already. Need to sniff the stream info
  else if (dsp->frame.tpf == 0) {
    if (!gst_dspmp3sink_get_stream_details (dsp, TRUE)) {
      GST_ELEMENT_ERROR (sink, STREAM, TYPE_NOT_FOUND, (NULL),
                         ("gst_dspmp3sink_get_stream_details"));
      GST_WARNING ("Render returning gst_flow_error 2");
      return GST_FLOW_ERROR;
    }
  }

  if (!dsp->latency_handled) {
    GstClockTime latency = gst_base_sink_get_latency (sink);
    if (latency) {
      GST_INFO ("Latency: %"GST_TIME_FORMAT, GST_TIME_ARGS (latency));
      g_usleep (latency / GST_USECOND);
    }
    dsp->latency_handled = TRUE;
  }

  while (gst_adapter_available (dsp->adapter) > 1400) {

    status = gst_dspaudio_wait_buffer (dsp->audio);

    if (status == DSP_WAIT_OK) {
      if (gst_dspmp3sink_copy_frame (dsp)) {
        if (gst_dspaudio_write_data (dsp->audio,
                dsp->mmap_ptr / sizeof (short int), FALSE))
        {
          dsp->mmap_ptr = 0;
          continue;
        }
        else {
          GST_ELEMENT_ERROR (sink, RESOURCE, WRITE, (NULL), ("DSP write failed"));
          retval = GST_FLOW_ERROR;
        }
      }
      break;
    }
    else if (status == DSP_WAIT_INTERRUPT) {
      GST_LOG ("DSP_WAIT_INTERRUPT");
      retval = gst_base_sink_wait_preroll (sink);
      if (retval != GST_FLOW_OK)
        break;
      continue;
    }
    else {
      GST_ELEMENT_ERROR (sink, RESOURCE, WRITE, (NULL), ("gst_dspaudio_wait_buffer"));
      retval = GST_FLOW_ERROR;
      break;
    }
  }
  GST_LOG ("MP3 render exit");
  return retval;
}


/*
 * gst_dspmp3sink_initialize:
 * @dsp: DSP MP3 sink object
 *
 * Initializes the underlying audio stream object and sets up the
 * MP3 audio parameters in DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspmp3sink_initialize(GstDSPMP3Sink *dsp)
{
  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_audioparams (dsp->audio,
    0, dsp->sample_rate, dsp->frame.channels)) {
    return FALSE;
  }
  GST_INFO("DSP MP3 configured");
  return TRUE;
}


/*
* gst_dspmp3sink_preroll:
* @pad:
* @buffer:
*
* This method is automatically called by GStreamer framework, when
* a scheduler decides to give execution to this element.
* The purpose is to check the media stream and to open the codec.
*/

static GstFlowReturn
gst_dspmp3sink_preroll (GstBaseSink *sink, GstBuffer * buffer)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);

  // We want this to run only once (to initialize the codec)
  if( dsp->audio->mode == DSP_MODE_UNINITIALIZED ) {

    // We can check the stream only if we really have a buffer
    if(buffer) {
      gst_buffer_ref(buffer);
      gst_adapter_push (dsp->adapter, buffer);

      // Ok, here is the magic; this call will try to examine
      // the contents of the adapter, and if it can detect a
      // valid stream, it will also initialize the DSP codec.
      // This initialization sets the audio->mode accordingly,
      // so this won't be done again in _render() then.
      // If it fails, the initialization is then performed
      // later in _render() function.
      if(gst_dspmp3sink_loop_initialization(dsp,  TRUE)) {
        GST_INFO("Initialized in preroll");
      }
      // Remember to clear adapter before rendering, but use 
      // it if we receive seek event before any buffers
      dsp->clear_adapter = TRUE;
    }
  }
  return GST_FLOW_OK;
}


/*
 * gst_dspmp3sink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspmp3sink_set_property (GObject * object,
                             guint prop_id,
                             const GValue *value,
                             GParamSpec * pspec)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (object);

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspmp3sink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspmp3sink_get_property (GObject * object,
                             guint prop_id,
                             GValue *value,
                             GParamSpec * pspec)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (object);

  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspmp3sink_get_query_types:
 *
 */
static const GstQueryType *
gst_dspmp3sink_get_query_types(GstElement * element)
{
  static const GstQueryType gst_dspmp3sink_query_types[] =
  {
    GST_QUERY_POSITION,
    GST_QUERY_DURATION,
    0
  };

  return gst_dspmp3sink_query_types;
}


/*
 * gst_dspmp3sink_query:
 * @element: GstElement that received this query
 * @type: Type of the query, e.g. TOTAL, POSITION
 * @format: The format of the response, e.g. BYTES, TIME
 * @value: Pointer where the return value should be stored
 *
 * Returns: TRUE, iff the query was successful.
 */
static gboolean
gst_dspmp3sink_query (GstElement * element, GstQuery *query)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (element);
  GstQueryType type = GST_QUERY_TYPE (query);
  gboolean res = FALSE;
  GstFormat format;
  gint64 value = 0;

  // First we check if parent class can handle the query (except LATENCY)
  if (type != GST_QUERY_LATENCY) {
    res = GST_ELEMENT_CLASS (parent_class)->query (element, query);
    if (res) {
      GST_DEBUG ("parent class handled query");
      return res;
    }
  }

  /* We allow the query to be performed only when some data has
  already been written to DSP */
  if(dsp->total_bytes == 0) {
    return FALSE;
  }

  if(type == GST_QUERY_POSITION) {
    GST_DEBUG("Position query");

    gst_query_parse_position(query, &format, NULL);

    if (format != GST_FORMAT_TIME)
      return FALSE;

    if (dsp->audio->mode >= DSP_MODE_CONFIGURED) {
    /* Ask DSP how many frames it has played */
      res = gst_dspaudio_get_info (dsp->audio);
    }

    if (res) {
      /* This calculates the amount of frames played after last SEEK operation */
      guint32 frames_consumed;
#ifdef HAVE_DSP_H
      frames_consumed = (dsp->audio->info.num_frames - dsp->frame_seek_base) << dsp->frame.lsf;
#else
      frames_consumed = 0;
#endif

      /* Time_offset is set during SEEK, so it works as a base time */
      value = dsp->time_offset + (frames_consumed  * dsp->frame.tpf) * GST_SECOND;
      gst_query_set_position (query, GST_FORMAT_TIME, value);
      GST_DEBUG ("Position: %lld", value);
    }
  }

  else if(type == GST_QUERY_DURATION) {
    GST_DEBUG("Duration query");

    gst_query_parse_duration(query, &format, NULL);

    if (format != GST_FORMAT_TIME)
      return FALSE;

    /* XING-based length calculation */
    if (dsp->xingtag.xframes && dsp->frame.tpf != 0) {
      value = dsp->frame.tpf * dsp->xingtag.xframes * GST_SECOND;
      res = TRUE;
    }
    else { /* No XING */
      GstPad *peer = gst_pad_get_peer (dsp->sinkpad);
      GstFormat fmt = GST_FORMAT_BYTES;
      gboolean ret = FALSE;
      gint64 curr, len;

      ret  = gst_pad_query_position (peer, &fmt, &curr);
      ret |= gst_pad_query_duration (peer, &fmt, &len);
      gst_object_unref (peer); // get_peer() ref'ed it

      /* VBR needs different kind of calculation algorithm */
      if (dsp->vbr && len > 0 && curr >= 0) {
        gdouble rel = (gdouble) curr / len;
        gdouble bpf2 = 0;
        if(dsp->frame.layer == 1) {
          bpf2 = (gdouble) dsp->total_bytes / (dsp->total_frames << dsp->frame.lsf);
        }
        else {
          bpf2 = (gdouble) dsp->total_bytes / (dsp->total_frames);
        }

        if (bpf2 != 0.0 && rel != 0.0) {
          guint currframes = curr / bpf2;
          gdouble written_time = dsp->frame.tpf * currframes;
          value = GST_SECOND * written_time / rel;
          res = TRUE;
        }
      }
      else if (len != -1) { /* Constant bitrate */
        gdouble bpf = gst_dspmp3sink_bpf (&dsp->frame, dsp->frame.bitrate);

        if (bpf != 0) {
          value = (guint64) (dsp->frame.tpf * len / bpf) * GST_SECOND;
          res = TRUE;
        }
      }
    }

    if (res) {
      GST_DEBUG ("Duration: %lld", value);
      gst_query_set_duration (query, GST_FORMAT_TIME, value);
    }
  }

  else if (type == GST_QUERY_LATENCY) {
    gboolean live, us_live;
    GstClockTime min_l, max_l;
    GST_DEBUG ("latency query");

    if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
      GST_DEBUG ("we are not yet negotiated, can't report latency yet");
      res = FALSE;
    }
    else if ((res = gst_base_sink_query_latency (GST_BASE_SINK_CAST (element),
              &live, &us_live, &min_l, &max_l)))
    {
      GstClockTime min_latency, max_latency;
      /* upstream is live, adjust the min_latency */
      if (us_live) {
        // We guess that DSP has latency of 5 frames (it buffers some data)
        min_latency = min_l + 5 * (dsp->frame.tpf * GST_SECOND);
        max_latency = min_latency + (max_l == -1 ? 0 : max_l);

        GST_DEBUG ("peer min %" GST_TIME_FORMAT ", our min latency: %"
            GST_TIME_FORMAT, GST_TIME_ARGS (min_l),
                                            GST_TIME_ARGS (min_latency));
      } else {
        GST_DEBUG ("live = %d, us_live = %d", live, us_live);
        GST_DEBUG ("peer or we are not live, don't care about latency");
        min_latency = 0;
        max_latency = -1;
      }
      gst_query_set_latency (query, live, min_latency, max_latency);
    }
  }

  return res;
}


/*
 * gst_dspmp3sink_send_remaining:
 *
 */
static void
gst_dspmp3sink_send_remaining(GstDSPMP3Sink *dsp)
{
  DSPWaitStatus ret = DSP_WAIT_OK;

  if (dsp->audio->mode < DSP_MODE_CONFIGURED || dsp->total_frames == 0) {
    GST_INFO ("DSP not initialized. Leaving.");
    return;
  }

  ret = gst_dspaudio_wait_buffer (dsp->audio);

  if(ret != DSP_WAIT_OK) {
    return;
  }

  if (gst_adapter_available (dsp->adapter)) {
    while (gst_dspmp3sink_copy_frame (dsp)) {
      GST_DEBUG ("EOF frame copied");
    }
  }
  else {
    GST_DEBUG ("No data to send");
    dsp->mmap_ptr = 0;
  }

  // Send remaining bytes to DSP
  GST_DEBUG ("Writing EOF cmd to DSP (%d bytes)", dsp->mmap_ptr);
  gst_dspaudio_write_data (dsp->audio, dsp->mmap_ptr / sizeof(short int), TRUE);
  dsp->total_bytes += dsp->mmap_ptr;
  dsp->audio->mode = DSP_MODE_EOS;
  gst_dspaudio_wait_eos (dsp->audio);
  GST_DEBUG ("Wrote total of %lld bytes to DSP during playback",
             dsp->total_bytes);
}


/*
 * gst_dspmp3sink_event2:
 *
 * This method handles events coming from basesink
 */
static gboolean
gst_dspmp3sink_event2 (GstBaseSink *sink, GstEvent * event)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  if (type == GST_EVENT_EOS) {
    GST_DEBUG ("EOS event");
    gst_dspmp3sink_send_remaining (dsp);
  }
  else if(type == GST_EVENT_FLUSH_START) {
    GST_DEBUG ("FLUSH_START event");
  }
  else if (type == GST_EVENT_FLUSH_STOP) {
    GST_DEBUG ("FLUSH_STOP event");
    if (dsp->audio->mode != DSP_MODE_EOS) {
      // If we reached EOS already, there is no need to send DISCONT
      // since the internal buffer in DSP is already empty
      gst_dspaudio_discont (dsp->audio);
    }
    gst_adapter_clear (dsp->adapter);
    dsp->mmap_ptr = 0;
    dsp->curr_fc = 0;
    dsp->vbr_counter = 0;
    char_in_tmp = FALSE;
    dsp->sync = FALSE;
    char_in_tmp = FALSE;
    GST_DEBUG("MMAP pointer reset");
  }
  return TRUE;
}

/*
 * gst_dspmp3sink_event:
 * @element: GstElement that received this event
 * @event: The GstEvent structure itself
 *
 * Returns: TRUE, iff the event was successful
 */
static gboolean
gst_dspmp3sink_event (GstElement * element, GstEvent * event)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (element);
  gboolean res = FALSE;

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;

  if (type == GST_EVENT_SEEK) {
    gdouble rate;
    GstFormat format = GST_FORMAT_UNDEFINED;
    GstSeekFlags flags;
    GstSeekType cur_type, stop_type;
    gint64 cur, stop;

    GST_DEBUG("Seek event");

    gst_event_parse_seek (event, &rate, &format, &flags,
                          &cur_type, &cur, &stop_type, &stop);

    // Try if previous element can handle the seek event.
    gst_event_ref (event);
    res = gst_pad_send_event (GST_PAD_PEER (dsp->sinkpad), event);

    if (res) {
      GST_DEBUG ("Previous element handled the seek event");
      if (gst_dspaudio_get_info (dsp->audio)) {
#ifdef HAVE_DSP_H
        dsp->frame_seek_base = dsp->audio->info.num_frames;
#endif
      }
      dsp->time_offset = cur;
    }
    else if (format == GST_FORMAT_TIME) {
      gint64 tmp = dsp->time_offset;
      dsp->time_offset = cur;

      if (dsp->frame.bitrate == 0) {
        /* We must have stream details now for the seek to succeed */
        gst_dspmp3sink_get_stream_details (dsp, TRUE);
      }

      if (gst_dspmp3sink_perform_seek (dsp, cur)) {
        if (gst_dspaudio_get_info (dsp->audio)) {
#ifdef HAVE_DSP_H
          dsp->frame_seek_base = dsp->audio->info.num_frames;
#endif
        }
        res = TRUE;
      }
      else {
        dsp->time_offset = tmp;
      }
    }
    gst_event_unref (event);
  }
  else {
    res = gst_pad_event_default (dsp->sinkpad, event);
  }
  return res;
}


/*
 * gst_dspmp3sink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Changes the state of given element. The desired state information
 * is contained by the element itself, and this method is just
 * a callback, so the plugin knows when the state is changing.
 * This method is called by GStreamer framework
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static GstStateChangeReturn
gst_dspmp3sink_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPMP3Sink *dspmp3sink = GST_DSPMP3SINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      GST_DEBUG("NULL TO READY");
      dspmp3sink->total_bytes = 0;
      dspmp3sink->total_frames = 0;
      dspmp3sink->vbr = FALSE;
      dspmp3sink->mmap_ptr = 0;
      dspmp3sink->xingtag.xframes = 0;
      dspmp3sink->frame_seek_base = 0;
      dspmp3sink->latency_handled = FALSE;
      char_in_tmp = FALSE;
      bzero(&dspmp3sink->frame, sizeof(MP3Frame));
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      GST_DEBUG("READY TO PAUSED");
      gst_adapter_clear(dspmp3sink->adapter);
      dspmp3sink->clear_adapter = FALSE;
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      GST_DEBUG ("PAUSED TO PLAYING");
      gst_dspmp3sink_async_play (GST_BASE_SINK_CAST (dspmp3sink));
      GST_DEBUG ("PAUSED TO PLAYING OK");
      break;

    default:
      break;
  }

  gst_dspaudio_clock_change_state (dspmp3sink->audio, transition);
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch(transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      GST_DEBUG ("PLAYING TO PAUSED V2");
      gst_dspaudio_pause (dspmp3sink->audio);
      GST_DEBUG ("PLAYING TO PAUSED V2 OK");
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      GST_DEBUG ("PAUSED TO READY");
      gst_dspaudio_stop (dspmp3sink->audio);
      gst_adapter_clear (dspmp3sink->adapter);
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      GST_DEBUG("READY To NULL");
      gst_dspaudio_close(dspmp3sink->audio);
      gst_dspaudio_reset(dspmp3sink->audio);
      GST_DEBUG("READY TO NULL OK");
      break;

    default:
      break;
  }

  GST_DEBUG("State change successful, retval = %d", ret);
  return ret;
}



/*
 * gst_dspmp3sink_unlock:
 * @sink: GstBaseSink
 *
 * Interrupts the blocking :render() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspmp3sink_unlock (GstBaseSink *sink)
{
  GST_DEBUG("unlock");
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);
  gst_dspaudio_interrupt_render (dsp->audio);
  return TRUE;
}


/*
 * gst_dspmp3sink_unlock_stop:
 * @sink: GstBaseSink
 *
 * Removes :render() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspmp3sink_unlock_stop (GstBaseSink *sink)
{
  GST_DEBUG("unlock_stop");
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);
  gst_dspaudio_remove_interrupt (dsp->audio);
  return TRUE;
}


/*
 * gst_dspmp3sink_async_play:
 * @sink: GstBaseSink
 *
 * Async state change handler. Sends PLAY command to DSP
 *
 * Returns: GstStateChangeReturn
 */
static GstStateChangeReturn
gst_dspmp3sink_async_play (GstBaseSink *sink)
{
  GstDSPMP3Sink *dspmp3sink = GST_DSPMP3SINK (sink);
  GST_DEBUG ("Async play");

  if (dspmp3sink->audio->mode == DSP_MODE_PAUSED ||
      dspmp3sink->audio->mode == DSP_MODE_EOS)
  {
    gst_dspaudio_play (dspmp3sink->audio);
    GST_DEBUG ("Async play success");
  }
  return GST_STATE_CHANGE_SUCCESS;
}


/*
 * gst_dspmp3sink_provide_clock:
 * @elem: GstElement
 *
 * Provides the clock object for pipeline.
 *
 * Returns: GstClock object
 */
static GstClock *
gst_dspmp3sink_provide_clock (GstElement * elem)
{
  GST_INFO("gst_dspmp3sink_provide_clock");
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 * gst_dspmp3sink_set_clock:
 * @element: GstElement
 * @clock: GstClock to be used
 *
 * For testing purposes
 *
static gboolean
gst_dspmp3sink_set_clock(GstElement *element, GstClock *clock)
{
  GST_INFO("gst_dspmp3sink_set_clock: %p", clock);
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (element);
  if(clock) {
    dsp->audio->clock = clock;
    return TRUE;
  }
  return FALSE;
}
*/


/*
 * gst_dspmp3sink_get_time:
 * @clock: GstClock to be used for time query
 * @sink: user data pointing to MP3 sink element
 *
 * Query clock time from DSP
 *
 * Returns: Current clock time, or GST_CLOCK_TIME_NONE if error
 */
static GstClockTime
gst_dspmp3sink_get_time (GstClock * clock, gpointer sink)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);
  return gst_dspaudio_get_dsp_clock (dsp->audio, FALSE);
}


/*
 * plugin_init:
 * @plugin: GstPlugin to be initialized
 *
 * This initializes the plugin when it is loaded by GStreamer
 *
 * Returns: Boolean value, TRUE if initialization was successful
 */
static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dspmp3sink", GST_RANK_PRIMARY,
      GST_TYPE_DSPMP3SINK);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
                   GST_VERSION_MINOR,
                   "dspmp3",
                   "DSP MP3 audio sink",
                   plugin_init,
                   VERSION,
                   GST_LICENSE_UNKNOWN,
                   "dspaudio",
                   "http://www.nokia.com/");
