/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdspilbcsrc.h"
#include "dspmixer.h"

// SID_UPDATE frame size in bytes
#define ILBC_DTX_UPDATE_FRAME_SIZE 22

GST_DEBUG_CATEGORY_STATIC (dspilbcsrc_debug);
#define GST_CAT_DEFAULT dspilbcsrc_debug

static const gchar *devname = "/dev/dsptask/ilbc_enc";

static GstStaticPadTemplate dspilbcsrc_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/x-iLBC, "
                      "rate = (int) 8000, "
                      "channels = (int) 1, "
                      "mode = (int) { 20, 30 }"
                    ) );

static GstStaticPadTemplate dspilbcsrc_cnsrc_template =
GST_STATIC_PAD_TEMPLATE ("cnsrc%d",
    GST_PAD_SRC,
    GST_PAD_REQUEST,
    GST_STATIC_CAPS (
           "audio/CN, "
           "rate = (int) 8000, "
           "channels = (int) 1"
    ) );

// Function prototypes

static void gst_dspilbcsrc_class_init (GstDSPILBCSrcClass * klass);
static void gst_dspilbcsrc_base_init (gpointer g_class);
static void gst_dspilbcsrc_init (GstDSPILBCSrc *dspmp3src, GstDSPILBCSrcClass *g_class);
static void gst_dspilbcsrc_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspilbcsrc_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

static GstCaps *gst_dspilbcsrc_getcaps (GstBaseSrc * src);
static gboolean gst_dspilbcsrc_setcaps (GstBaseSrc * src, GstCaps *caps);
static GstClock *gst_dspilbcsrc_provide_clock (GstElement * elem);
static GstClockTime gst_dspilbcsrc_get_time (GstClock * clock, GstDSPILBCSrc * src);

static GstFlowReturn gst_dspilbcsrc_create (GstPushSrc * psrc, GstBuffer ** buf);
static gboolean gst_dspilbcsrc_start (GstBaseSrc * bsrc);
static gboolean gst_dspilbcsrc_stop (GstBaseSrc * bsrc);
static gboolean gst_dspilbcsrc_event (GstBaseSrc * bsrc, GstEvent *event);
static gboolean gst_dspilbcsrc_unlock (GstBaseSrc *src);
static gboolean gst_dspilbcsrc_unlock_stop (GstBaseSrc *src);

static GstPad *
gst_dspilbcsrc_request_new_pad (GstElement *element,
                                GstPadTemplate *templ,
                                const gchar *name);
static void
gst_dspilbcsrc_release_pad     (GstElement *element,
                                GstPad *pad);

GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPILBCSrc, gst_dspilbcsrc);


/*
 * gst_dspilbcsrc_do_init:
 * @type:
 *
 *
 */
static void
gst_dspilbcsrc_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspilbcsrc_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspilbcsrc_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspilbcsrc_debug, "dspilbcsrc", 0,
                           "DSP iLBC audio source");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}


GST_BOILERPLATE_FULL (GstDSPILBCSrc, gst_dspilbcsrc, GstPushSrc,
                      GST_TYPE_PUSH_SRC, gst_dspilbcsrc_do_init);


#define GST_TYPE_ILBCSRC_MODE (gst_dspilbcsrc_mode_get_type())
static GType
gst_dspilbcsrc_mode_get_type (void)
{
  static GType dspilbcsrc_mode_type = 0;
  static GEnumValue dspilbcsrc_modes[] = {
    {GST_ILBCSRC_MODE_20, "20 ms frames", "20"},
    {GST_ILBCSRC_MODE_30, "30 ms frames", "30"},
    {0, NULL, NULL},
  };

  if (G_UNLIKELY(!dspilbcsrc_mode_type)) {
    dspilbcsrc_mode_type = g_enum_register_static ("GstILBCSrcMode",
        dspilbcsrc_modes);
  }
  return dspilbcsrc_mode_type;
}


/*
 * gst_dspilbcsrc_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes ILBC src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspilbcsrc_dispose (GObject * object)
{
  GstDSPILBCSrc *dsp = (GstDSPILBCSrc *) object;

  if (dsp->clock) {
    gst_object_unref (dsp->clock);
    dsp->clock = NULL;
  }

  if (dsp->audio) {
    gst_dspaudio_destroy (dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspilbcsrc_base_init:
 * @g_class: ILBC src class
 *
 * Does the basic initialization of ILBC src element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspilbcsrc_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP ILBC Src",
    "Source/Audio",
    "ILBC audio src",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspilbcsrc_src_template));
 gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspilbcsrc_cnsrc_template));
 gst_element_class_set_details (gstelement_class, &details);

}


/*
 * gst_dspilbcsrc_class_init:
 * @klass: ILBC Src class object
 *
 * Initializes ILBC src element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspilbcsrc_class_init (GstDSPILBCSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_dispose);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_src_properties (G_OBJECT_CLASS (klass));

  g_object_class_install_property (gobject_class, DSPILBC_PROP_MODE,
                                   g_param_spec_enum ("mode", "iLBC mode",
                                       "iLBC frame size in millisecs.",
                                       GST_TYPE_ILBCSRC_MODE,
                                       GST_ILBCSRC_MODE_20,
                                       G_PARAM_READWRITE));

  gstelement_class->provide_clock   = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_provide_clock);
  gstelement_class->request_new_pad = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_request_new_pad);
  gstelement_class->release_pad     = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_release_pad);

  gstbasesrc_class->start       = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_start);
  gstbasesrc_class->stop        = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_stop);
  gstbasesrc_class->get_caps    = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_getcaps);
  gstbasesrc_class->set_caps    = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_setcaps);
  gstbasesrc_class->event       = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_event);
  gstbasesrc_class->unlock      = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_unlock);
  gstbasesrc_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_unlock_stop);

  gstpushsrc_class->create = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_create);
}


/*
 * gst_dspilbcsrc_init:
 * @dsp: DSP ILBC src object
 * @g_class: ILBC src Class object
 *
 * Initializes the given ILBC src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspilbcsrc_init (GstDSPILBCSrc * dspilbcsrc, GstDSPILBCSrcClass *g_class)
{
  dspilbcsrc->audio = gst_dspaudio_new();
  gst_base_src_set_live (GST_BASE_SRC(dspilbcsrc), TRUE);
  dspilbcsrc->mode = GST_ILBCSRC_MODE_20;
  dspilbcsrc->frametimemillis = 20;

  dspilbcsrc->clock = gst_audio_clock_new ("DSPClock",
                                           (GstAudioClockGetTimeFunc)
                                                gst_dspilbcsrc_get_time,
                                           dspilbcsrc);
  GST_INFO("Clock created");

}


/*
 * gst_dspilbcsrc_setup:
 * @dsp: GstDSPILBCSrc object
 *
 */
static gboolean
gst_dspilbcsrc_setup (GstDSPILBCSrc *dsp)
{
  guint format, sample_rate, frame_samples;

  if (dsp->mode == GST_ILBCSRC_MODE_20) {
    frame_samples = 20*8;      // samples
    dsp->frametimemillis = 20; // msec
    dsp->framesize = 38;       // bytes
  }
  else {
    frame_samples = 30*8;      // samples
    dsp->frametimemillis = 30; // msec
    dsp->framesize = 50;       // bytes
  }

  if(!gst_dspaudio_map_format (dsp->audio, GST_DSPAUDIO_CODEC_ILBC,
    G_LITTLE_ENDIAN, TRUE, 0, &format, NULL)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }
  if(!gst_dspaudio_map_samplerate (8000,&sample_rate)) {
    GST_WARNING ("unsupported sample rate: 8000");
    return FALSE;
  }

  GST_DEBUG ("Frame size: %d, sample: %d", dsp->framesize, frame_samples);

  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_speechparams (dsp->audio,
    format, sample_rate, frame_samples)) {
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_update_dsp_settings (dsp->audio);
  gst_dspaudio_play (dsp->audio);
  GST_DEBUG ("gst_dspilbcsrc_start OK");
  return TRUE;
}


/*
 * gst_dspilbcsrc_getcaps:
 * @src: GstBaseSrc object
 *
 */
static GstCaps *
gst_dspilbcsrc_getcaps (GstBaseSrc * src)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (src);
  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD (src)));

  gst_caps_set_simple (caps, "mode", G_TYPE_INT, dsp->frametimemillis, NULL);
  return caps;
}


/*
 * gst_dspilbcsrc_setcaps:
 *
 */
static gboolean
gst_dspilbcsrc_setcaps (GstBaseSrc * src, GstCaps *caps)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC(src);

  GST_DEBUG("gst_dspilbcsrc_setcaps");

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  gint mode;

  if(gst_structure_has_field(structure, "mode" )) {
    gst_structure_get_int (structure, "mode", &mode);
    GST_DEBUG("Got mode: %d", mode);
    dsp->mode = (mode == 30) ? GST_ILBCSRC_MODE_30 : GST_ILBCSRC_MODE_20;
    dsp->framesize = dsp->mode == GST_ILBCSRC_MODE_20 ? 38 : 50;
    dsp->frametimemillis = dsp->mode == GST_ILBCSRC_MODE_20 ? 20 : 30;
  }
//   gst_pad_use_fixed_caps (GST_BASE_SRC_PAD (src));
//   return gst_dspilbcsrc_setup (dsp);
  return TRUE;
}


/*
 * gst_dspilbcsrc_create:
 * @psrc: GstPushSrc
 * @buf: GstBuffer
 *
 */
static GstFlowReturn
gst_dspilbcsrc_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC(psrc);
  GstFlowReturn ret = GST_FLOW_OK;

  // If DSP is not set up yet (e.g. setcaps not called) we need
  // to do it here
  if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
    gst_dspilbcsrc_setup (dsp);
  }

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }

  for (;;) {
    *buf = gst_dspaudio_read_frame (dsp->audio, dsp->framesize,
                                    ILBC_DTX_UPDATE_FRAME_SIZE,
                                    dsp->frametimemillis);
    if (!*buf) {
      if (dsp->audio->mode == DSP_MODE_ERROR) {
        ret = GST_FLOW_ERROR;
        break;
      }
      ret = gst_base_src_wait_playing (GST_BASE_SRC_CAST (psrc));
      if (ret != GST_FLOW_OK) break;
      else continue;
    }

    if (GST_BUFFER_FLAG_IS_SET (*buf, GST_BUFFER_FLAG_LAST)) {
      // CN frame received
      if (!dsp->cnpad) {
        gst_buffer_unref (*buf);
        continue;
      }
      gst_buffer_set_caps (*buf, GST_PAD_CAPS (dsp->cnpad));
      GST_BUFFER_FLAG_UNSET (*buf, GST_BUFFER_FLAG_LAST);
      gst_pad_push (dsp->cnpad, *buf);
      GST_LOG ("CN buffer created");
      continue;
    }

    gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
    GST_LOG ("Audio buffer created");
    break;
  }
  return ret;
}


/*
 * gst_dspilbcsrc_start:
 * @bsrc: GstBaseSrc
 *
 */
static gboolean
gst_dspilbcsrc_start (GstBaseSrc * bsrc)
{
  GST_INFO ("gst_dspilbcsrc_start");
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (bsrc);

  if (gst_dspaudio_ed_microphone (dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  if (gst_dspaudio_aep_initialize (dsp->audio) == FALSE) {
    gst_dspaudio_ed_microphone (dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_aep_initialize"));
    return FALSE;
  }

  if (gst_dspaudio_open (dsp->audio, devname) == FALSE) {
    gst_dspaudio_ed_microphone (dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }
  gst_dspilbcsrc_setup (dsp);
  return TRUE;
}


/*
 * gst_dspilbcsrc_stop:
 *
 */
static gboolean
gst_dspilbcsrc_stop (GstBaseSrc * bsrc)
{
  GST_INFO("gst_dspilbcsrc_stop");
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (bsrc);

  gst_dspaudio_remove_interrupt (dsp->audio);

  if (dsp->audio->all_frames_read || dsp->audio->read_sent) {
    GST_DEBUG ("Waiting reply for CMD_DATA_READ");
    gst_dspaudio_wait_buffer (dsp->audio);
    GST_DEBUG ("Got it");
  }

  gst_dspaudio_dtmf_tone_stop (dsp->audio);
  gst_dspaudio_stop (dsp->audio);
  gst_dspaudio_ed_microphone (dsp->audio, FALSE);
  gst_dspaudio_close (dsp->audio);
  gst_dspaudio_aep_close (dsp->audio);
  gst_dspaudio_reset (dsp->audio);
  GST_DEBUG ("gst_dspilbcsrc_stop OK");
  return TRUE;
}


/*
 * gst_dspilbcsrc_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspilbcsrc_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (object);

  if (prop_id == DSPILBC_PROP_MODE) {
    dsp->mode = g_value_get_enum(value);
    dsp->framesize = dsp->mode == GST_ILBCSRC_MODE_20 ? 38 : 50;
    dsp->frametimemillis = dsp->mode == GST_ILBCSRC_MODE_20 ? 20 : 30;
    GST_DEBUG("Mode = %d", dsp->mode);
  }
  else if (!gst_dspaudio_set_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspilbcsrc_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspilbcsrc_get_property(GObject * object,
                            guint prop_id,
                            GValue * value,
                            GParamSpec * pspec)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (object);

  if (prop_id == DSPILBC_PROP_MODE) {
    g_value_set_enum (value, dsp->mode);
  }
  else if (!gst_dspaudio_get_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspilbcsrc_event:
 * @src: GstBaseSrc object
 * @event: GstEvent to be handled
 *
 * Handles incoming events. Checks if the event is special "custom upstream
 * event" and in this case forwards it to dspaudio library for checking if
 * it is DTMF event.
 *
 * Returns: TRUE if the event was handled.
 */
static gboolean
gst_dspilbcsrc_event (GstBaseSrc *src, GstEvent *event)
{
  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  GST_DEBUG ("event received: %d", type);
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (src);
  gboolean ret = FALSE;

  if(type == GST_EVENT_CUSTOM_UPSTREAM) {
    GST_DEBUG ("it is upstream event");
    ret = gst_dspaudio_check_upstream_event (dsp->audio, event);
  }
  return ret;
}


/*
 * gst_dspilbcsrc_provide_clock:
 *
 */
static GstClock *
gst_dspilbcsrc_provide_clock (GstElement * elem)
{
  GST_INFO("gst_dspilbcsrc_provide_clock");
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 * gst_dspilbcsrc_get_time:
 *
 */
static GstClockTime
gst_dspilbcsrc_get_time (GstClock * clock, GstDSPILBCSrc * dsp)
{
  GstClockTime result;
  result = dsp->audio->in_framecount * dsp->frametimemillis * GST_MSECOND;
  return result;
}


/*
 * gst_dspilbcsrc_request_new_pad:
 *
 */
static GstPad *
gst_dspilbcsrc_request_new_pad (GstElement     *element,
                                GstPadTemplate *templ,
                                const gchar    *name)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (element);

  if (dsp->cnpad)
    return NULL;

  dsp->cnpad = gst_pad_new_from_template (templ, name);
  gst_pad_set_activatepush_function (dsp->cnpad, gst_dspaudio_activate_push);
  gst_element_add_pad (element, dsp->cnpad);
  return dsp->cnpad;
}


/*
 * gst_dspilbcsrc_release_pad:
 *
 */
static void
gst_dspilbcsrc_release_pad (GstElement *element, GstPad *pad)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (element);

  if (pad == dsp->cnpad)
    dsp->cnpad = NULL;
}


/*
 * gst_dspilbcsrc_unlock:
 * @src: GstBaseSrc
 *
 * Interrupts the blocking :create() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspilbcsrc_unlock (GstBaseSrc *src)
{
    GST_DEBUG ("unlock");
    GstDSPILBCSrc *dsp = GST_DSPILBCSRC (src);
    gst_dspaudio_interrupt_render (dsp->audio);
    return TRUE;
}


/*
 * gst_dspilbcsrc_unlock_stop:
 * @src: GstBaseSrc
 *
 * Removes :create() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspilbcsrc_unlock_stop (GstBaseSrc *src)
{
    GST_DEBUG ("unlock_stop");
    GstDSPILBCSrc *dsp = GST_DSPILBCSRC (src);
    gst_dspaudio_remove_interrupt (dsp->audio);
    return TRUE;
}

