/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdspg711src.h"
#include "dspmixer.h"

// G.711 VOICE_FRAME size in bytes
#define G711_VOICE_FRAME_SIZE 160

// G.711 SID_UPDATE frame size in bytes
#define G711_DTX_UPDATE_FRAME_SIZE 22

// G.711 voice frame duration in milliseconds
#define G711_FRAMETIMEMILLIS 20


GST_DEBUG_CATEGORY_STATIC (dspg711src_debug);
#define GST_CAT_DEFAULT dspg711src_debug

static const gchar *devname = "/dev/dsptask/g711_enc";

static GstStaticPadTemplate dspg711src_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (
           "audio/x-alaw, "
           "rate = (int) 8000, "
           "channels = (int) 1; "

           "audio/x-mulaw, "
           "rate = (int) 8000, "
           "channels = (int) 1"
    ) );

static GstStaticPadTemplate dspg711src_cnsrc_template =
GST_STATIC_PAD_TEMPLATE ("cnsrc%d",
    GST_PAD_SRC,
    GST_PAD_REQUEST,
    GST_STATIC_CAPS (
           "audio/CN, "
           "rate = (int) 8000, "
           "channels = (int) 1"
    ) );

// Function prototypes

static void gst_dspg711src_class_init (GstDSPG711SrcClass * klass);
static void gst_dspg711src_base_init (gpointer g_class);
static void gst_dspg711src_init (GstDSPG711Src *dspmp3src, GstDSPG711SrcClass *g_class);
static void gst_dspg711src_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspg711src_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

static GstCaps *gst_dspg711src_getcaps (GstBaseSrc * src);
static gboolean gst_dspg711src_setcaps (GstBaseSrc * src, GstCaps *caps);
static GstClock *gst_dspg711src_provide_clock (GstElement * elem);
static GstClockTime gst_dspg711src_get_time (GstClock * clock, GstDSPG711Src * src);

static GstFlowReturn gst_dspg711src_create (GstPushSrc * psrc, GstBuffer ** buf);
static gboolean gst_dspg711src_start (GstBaseSrc * bsrc);
static gboolean gst_dspg711src_stop (GstBaseSrc * bsrc);
static gboolean gst_dspg711src_event (GstBaseSrc * bsrc, GstEvent *event);
static gboolean gst_dspg711src_unlock (GstBaseSrc *src);
static gboolean gst_dspg711src_unlock_stop (GstBaseSrc *src);

static GstPad *
gst_dspg711src_request_new_pad (GstElement *element,
                                GstPadTemplate *templ,
                                const gchar *name);

static void
gst_dspg711src_release_pad     (GstElement *element,
                                GstPad *pad);

GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPG711Src, gst_dspg711src);


/*
 * gst_dspg711src_do_init:
 * @type:
 *
 *
 */
static void
gst_dspg711src_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspg711src_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspg711src_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspg711src_debug, "dspg711src", 0,
                           "DSP G.711 audio source");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}


GST_BOILERPLATE_FULL (GstDSPG711Src, gst_dspg711src, GstPushSrc,
                      GST_TYPE_PUSH_SRC, gst_dspg711src_do_init);


/*
 * gst_dspg711src_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes G711 src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspg711src_dispose (GObject * object)
{
  GstDSPG711Src *dsp = (GstDSPG711Src *) object;

  if (dsp->clock) {
    gst_object_unref (dsp->clock);
    dsp->clock = NULL;
  }

  if (dsp->audio) {
    gst_dspaudio_destroy (dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspg711src_base_init:
 * @g_class: G711 src class
 *
 * Does the basic initialization of G711 src element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspg711src_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP G.711 src",
    "Source/Audio",
    "G.711 speech audio source element",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspg711src_src_template));
  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspg711src_cnsrc_template));
  gst_element_class_set_details (gstelement_class, &details);

}


/*
 * gst_dspg711src_class_init:
 * @klass: G711 Src class object
 *
 * Initializes G711 src element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspg711src_class_init (GstDSPG711SrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspg711src_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspg711src_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspg711src_dispose);

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_src_properties (G_OBJECT_CLASS (klass));

  gstelement_class->provide_clock   = GST_DEBUG_FUNCPTR (gst_dspg711src_provide_clock);
  gstelement_class->request_new_pad = GST_DEBUG_FUNCPTR (gst_dspg711src_request_new_pad);
  gstelement_class->release_pad     = GST_DEBUG_FUNCPTR (gst_dspg711src_release_pad);

  gstbasesrc_class->start       = GST_DEBUG_FUNCPTR (gst_dspg711src_start);
  gstbasesrc_class->stop        = GST_DEBUG_FUNCPTR (gst_dspg711src_stop);
  gstbasesrc_class->get_caps    = GST_DEBUG_FUNCPTR (gst_dspg711src_getcaps);
  gstbasesrc_class->set_caps    = GST_DEBUG_FUNCPTR (gst_dspg711src_setcaps);
  gstbasesrc_class->event       = GST_DEBUG_FUNCPTR (gst_dspg711src_event);
  gstbasesrc_class->unlock      = GST_DEBUG_FUNCPTR (gst_dspg711src_unlock);
  gstbasesrc_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_dspg711src_unlock_stop);

  gstpushsrc_class->create   = GST_DEBUG_FUNCPTR (gst_dspg711src_create);
}


/*
 * gst_dspg711src_init:
 * @dsp: DSP G711 src object
 * @g_class: G711 src Class object
 *
 * Initializes the given G711 src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspg711src_init (GstDSPG711Src * dspg711src, GstDSPG711SrcClass *g_class)
{
  dspg711src->audio = gst_dspaudio_new();
  gst_base_src_set_live (GST_BASE_SRC(dspg711src), TRUE);
  GST_BASE_SRC(dspg711src)->blocksize = 160;  // equivalent to 20 ms

  dspg711src->clock = gst_audio_clock_new ("clock",
                                           (GstAudioClockGetTimeFunc)
                                                gst_dspg711src_get_time,
                                           dspg711src);
  GST_INFO("Clock created");

}


/*
 * gst_dspg711src_getcaps:
 * @src: GstBaseSrc object
 *
 */
static GstCaps *
gst_dspg711src_getcaps (GstBaseSrc * src)
{
  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD(src)));

  return caps;
}


/*
 * gst_dspg711src_setcaps:
 *
 */
static gboolean
gst_dspg711src_setcaps (GstBaseSrc * bsrc, GstCaps *caps)
{
  GstDSPG711Src *dsp = GST_DSPG711SRC(bsrc);
  guint format, sample_rate,codec;

  GST_DEBUG("gst_dspg711src_getcaps");

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if(!strncmp(mimetype, "audio/x-alaw", 12)) {
    GST_DEBUG("a-law");
    codec = GST_DSPAUDIO_CODEC_ALAW;
  }
  else if(!strncmp(mimetype, "audio/x-mulaw", 13)) {
    GST_DEBUG("mu-law");
    codec = GST_DSPAUDIO_CODEC_MULAW;
  }
  else {
    GST_WARNING("Unknown format");
    return FALSE;
  }

  if(!gst_dspaudio_map_format (dsp->audio, codec, G_LITTLE_ENDIAN, TRUE, 0,
    &format, NULL)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }
  if(!gst_dspaudio_map_samplerate (8000,&sample_rate)) {
    GST_WARNING ("unsupported sample rate: 8000");
    return FALSE;
  }

  if (gst_dspaudio_open (dsp->audio, devname) == FALSE) {
    gst_dspaudio_ed_microphone (dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if (!gst_dspaudio_set_speechparams (dsp->audio,
    format, sample_rate, 160)) {
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_update_dsp_settings (dsp->audio);
  gst_dspaudio_play (dsp->audio);
  return TRUE;
}


/*
 * gst_dspg711src_create:
 * @psrc: GstPushSrc
 * @buf: GstBuffer
 *
 */
static GstFlowReturn
gst_dspg711src_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GstDSPG711Src *dsp = GST_DSPG711SRC (psrc);
  GstFlowReturn ret = GST_FLOW_OK;

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }

  for (;;) {
    *buf = gst_dspaudio_read_frame (dsp->audio, G711_VOICE_FRAME_SIZE,
                                    G711_DTX_UPDATE_FRAME_SIZE,
                                    G711_FRAMETIMEMILLIS);

    if (!*buf) {
      if (dsp->audio->mode == DSP_MODE_ERROR) {
        ret = GST_FLOW_ERROR;
        break;
      }
      ret = gst_base_src_wait_playing (GST_BASE_SRC_CAST (psrc));
      if (ret != GST_FLOW_OK) break;
      else continue;
    }

    if (GST_BUFFER_FLAG_IS_SET (*buf, GST_BUFFER_FLAG_LAST)) {
      // CN frame received
      if (!dsp->cnpad) {
        gst_buffer_unref (*buf);
        continue;
      }
      gst_buffer_set_caps (*buf, GST_PAD_CAPS (dsp->cnpad));
      GST_BUFFER_FLAG_UNSET (*buf, GST_BUFFER_FLAG_LAST);
      gst_pad_push (dsp->cnpad, *buf);
      GST_LOG ("CN buffer created");
      continue;
    }

    gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
    GST_LOG ("Audio buffer created");
    break;
  }
  return ret;
}


/*
 * gst_dspg711src_start:
 * @bsrc: GstBaseSrc
 *
 */
static gboolean
gst_dspg711src_start (GstBaseSrc * bsrc)
{
  GST_INFO ("gst_dspg711src_start");
  GstDSPG711Src *dsp = GST_DSPG711SRC (bsrc);

  if (gst_dspaudio_ed_microphone (dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  if (gst_dspaudio_aep_initialize (dsp->audio) == FALSE) {
    gst_dspaudio_ed_microphone (dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_aep_initialize"));
    return FALSE;
  }

  if (gst_dspaudio_open (dsp->audio, devname) == FALSE) {
    gst_dspaudio_ed_microphone (dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  GST_DEBUG("gst_dspg711src_start OK");
  return TRUE;
}


/*
 * gst_dspg711src_stop:
 *
 */
static gboolean
gst_dspg711src_stop (GstBaseSrc * bsrc)
{
  GST_INFO("gst_dspg711src_stop");
  GstDSPG711Src *dsp = GST_DSPG711SRC (bsrc);

  gst_dspaudio_remove_interrupt (dsp->audio);

  if (dsp->audio->all_frames_read || dsp->audio->read_sent) {
    GST_DEBUG ("waiting reply for CMD_DATA_READ");
    gst_dspaudio_wait_buffer (dsp->audio);
    GST_DEBUG ("got it");
  }

  gst_dspaudio_dtmf_tone_stop (dsp->audio);
  gst_dspaudio_stop (dsp->audio);
  gst_dspaudio_ed_microphone (dsp->audio, FALSE);
  gst_dspaudio_close (dsp->audio);
  gst_dspaudio_aep_close (dsp->audio);
  gst_dspaudio_reset (dsp->audio);
  GST_DEBUG ("gst_dspg711src_stop OK");
  return TRUE;
}


/*
 * gst_dspg711src_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspg711src_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPG711Src *dsp = GST_DSPG711SRC (object);

  if (!gst_dspaudio_set_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg711src_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspg711src_get_property (GObject * object,
                             guint prop_id,
                             GValue * value,
                             GParamSpec * pspec)
{
  GstDSPG711Src *dsp = GST_DSPG711SRC (object);

  if (!gst_dspaudio_get_property (dsp->audio, prop_id, value)) {
    GST_WARNING ("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg711src_event:
 * @src: GstBaseSrc object
 * @event: GstEvent to be handled
 *
 * Handles incoming events. Checks if the event is special "custom upstream
 * event" and in this case forwards it to dspaudio library for checking if
 * it is DTMF event.
 *
 * Returns: TRUE if the event was handled.
 */
static gboolean
gst_dspg711src_event (GstBaseSrc *src, GstEvent *event)
{
  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  GST_DEBUG ("event received: %d", type);
  GstDSPG711Src *dsp = GST_DSPG711SRC (src);
  gboolean ret = FALSE;

  if(type == GST_EVENT_CUSTOM_UPSTREAM) {
    GST_DEBUG ("it is upstream event");
    ret = gst_dspaudio_check_upstream_event (dsp->audio, event);
  }
  return ret;
}


/*
 * gst_dspg711src_provide_clock:
 *
 */
static GstClock *
gst_dspg711src_provide_clock (GstElement * elem)
{
  GST_INFO("gst_dspg711src_provide_clock");
  GstDSPG711Src *dsp = GST_DSPG711SRC (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 * gst_dspg711src_get_time:
 *
 */
static GstClockTime
gst_dspg711src_get_time (GstClock * clock, GstDSPG711Src * dsp)
{
  GstClockTime result;
  result = dsp->audio->in_framecount * G711_FRAMETIMEMILLIS * GST_MSECOND;
  return result;
}


/*
 * gst_dspg711src_request_new_pad:
 *
 */
static GstPad *
gst_dspg711src_request_new_pad (GstElement     *element,
                                GstPadTemplate *templ,
                                const gchar    *name)
{
  GstDSPG711Src *dsp = GST_DSPG711SRC (element);

  if (dsp->cnpad)
    return NULL;

  dsp->cnpad = gst_pad_new_from_template (templ, name);
  gst_pad_set_activatepush_function (dsp->cnpad, gst_dspaudio_activate_push);
  gst_element_add_pad (element, dsp->cnpad);
  return dsp->cnpad;
}


/*
 * gst_dspg711src_release_pad:
 *
 */
static void
gst_dspg711src_release_pad (GstElement *element, GstPad *pad)
{
  GstDSPG711Src *dsp = GST_DSPG711SRC (element);

  if (pad == dsp->cnpad)
    dsp->cnpad = NULL;
}


/*
 * gst_dspg711src_unlock:
 * @src: GstBaseSrc
 *
 * Interrupts the blocking :create() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspg711src_unlock (GstBaseSrc *src)
{
    GST_DEBUG ("unlock");
    GstDSPG711Src *dsp = GST_DSPG711SRC (src);
    gst_dspaudio_interrupt_render (dsp->audio);
    return TRUE;
}


/*
 * gst_dspg711src_unlock_stop:
 * @src: GstBaseSrc
 *
 * Removes :create() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspg711src_unlock_stop (GstBaseSrc *src)
{
    GST_DEBUG ("unlock_stop");
    GstDSPG711Src *dsp = GST_DSPG711SRC (src);
    gst_dspaudio_remove_interrupt (dsp->audio);
    return TRUE;
}

