/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/*
 * gst_dspaudio_open_node:
 * @dsp: DSP node info object
 * @devname: device name to be opened & initialized
 *
 * Initializes given DSP node.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_open_node (DSPNodeInfo *info, const gchar *devname)
{
  AUDIO_INIT_STATUS init_status;
  AUDIO_STATUS_INFO statInfo;
  guint trycount = 0;

  GST_INFO ("dspaudio_open_node : %s", devname);

  g_return_val_if_fail (info, FALSE);
  g_return_val_if_fail (devname, FALSE);

  while (TRUE) {
    info->fd = open (devname, O_RDWR);
    if (info->fd == -1) {
        GST_INFO ("Opening failed, errno: %d", errno);
        if (trycount++ > 4 || errno != EAGAIN) {
            GST_ERROR ("Opening %s failed (%d tries)", devname, trycount);
            return FALSE;
        }
        GST_WARNING ("Opening returned EAGAIN, will try again");
        usleep (200000);
        continue;
    }
    break;
  }

  //if (ioctl (info->fd, OMAP_DSP_TASK_IOCTL_LOCK) < 0) {
  //  GST_INFO ("Locking of %s failed", devname);
  //  return FALSE;
  //}

  // Read all the junk out from the node first
  gst_dspaudio_flush_unlocked (info);

  statInfo.dsp_cmd = DSP_CMD_STATE;
  if (write (info->fd, &statInfo, sizeof (short int)) < 0) {
    GST_ERROR ("failed to query %s state", devname);
    return FALSE;
  }

  if (read (info->fd, &statInfo, sizeof (AUDIO_STATUS_INFO)) < 0) {
    GST_ERROR ("failed to read CMD_STATE return value");
    return FALSE;
  }

  if (statInfo.status == STATE_UNINITIALISED) {

    short int tmp = DSP_CMD_INIT;
    if (write (info->fd, &tmp, sizeof (short int)) < 0) {
      GST_ERROR ("Cannot write to DSP");
      return FALSE;
    }

    if (read (info->fd, &init_status, sizeof (AUDIO_INIT_STATUS)) < 0) {
      GST_ERROR ("Error reading INIT status from %s", devname);
      return FALSE;
    }
    else if (init_status.init_status != DSP_OK) {
      GST_ERROR ("Init_status fail: %d", init_status.init_status);
      return FALSE;
    }

    info->bridge_buffer_size = init_status.bridge_buffer_size;
    info->mmap_buffer_size = init_status.mmap_buffer_size * sizeof (short int);
  }
  else {
    info->bridge_buffer_size = statInfo.bridge_buffer_size;
    info->mmap_buffer_size = statInfo.mmap_buffer_size * sizeof (short int);
  }

  info->mmap_buffer = (char *)
      mmap ((void *) 0, info->mmap_buffer_size, PROT_READ | PROT_WRITE,
            MAP_SHARED, info->fd, 0);

  if (info->mmap_buffer == NULL) {
    GST_ERROR ("Cannot mmap data buffer");
    return FALSE;
  }

  GST_DEBUG ("%s MMAP buffer size: %d bytes", devname, info->mmap_buffer_size);
  GST_INFO ("%s opened", devname);
  return TRUE;
}


/*
 * gst_dspaudio_close_node:
 * @dsp: DSP node info object
 *
 * Closes the specific DSP node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_close_node (DSPNodeInfo *info)
{
  DSP_CMD_STATUS cmd;
  GST_INFO ("dspaudio_close_node");

  g_return_val_if_fail (info, FALSE);

  // We may get called even when DSP is not opened yet
  if (info->fd != -1) {

    gst_dspaudio_flush_unlocked (info);

    cmd.dsp_cmd = DSP_CMD_CLOSE;
    if (write (info->fd, &cmd, sizeof (short int)) > 0) {
      read (info->fd, &cmd, sizeof (DSP_CMD_STATUS));
    }

    if (info->mmap_buffer) {
      munmap (info->mmap_buffer, info->mmap_buffer_size);
      info->mmap_buffer = NULL;
    }

    // we need to give DSP some time to send the last messages
    usleep (200000);

    // flush the message buffer once again to make sure it is empty
    gst_dspaudio_flush_unlocked (info);

    //ioctl (info->fd, OMAP_DSP_TASK_IOCTL_UNLOCK);
    close (info->fd);
    info->fd = -1;
  }

  GST_DEBUG ("Node closed");
  return TRUE;
}


/*
 * gst_dspaudio_flush_unlocked:
 * @dsp: DSP node info object
 *
 * Reads all unread data from node. Caller needs to grab DSP lock first.
 */
void
gst_dspaudio_flush_unlocked (DSPNodeInfo *info)
{
  short int tmp;

  while (gst_dspaudio_check_read (info->fd)) {
    if (read (info->fd, &tmp, sizeof (short int)) < 0) break;
  }
}


/*
 * gst_dspaudio_setparams:
 * @dsp: DSP audio object
 * @data: pointer to AUDIO_PARAMS_DATA or SPEECH_PARAMS_DATA to be set
 * @datasize: size of the parameter struct (in bytes)
 *
 * This sets the given parameters to audio node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_setparams (GstDSPAudio *dsp, const char *data, guint datasize)
{
  DSP_CMD_STATUS cmd_status;
  gint len;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_ERROR || dsp->codec.fd < 0)
      return FALSE;

  GST_INFO ("setparams called");

  len = write (dsp->codec.fd, data, datasize);
  if (len == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  GST_DEBUG ("setparams command written");

  len = read (dsp->codec.fd, &cmd_status, sizeof (DSP_CMD_STATUS));
  if (len == -1 || cmd_status.status != DSP_OK) {
    dsp->error_cmd = cmd_status.dsp_cmd;
    dsp->error_status = cmd_status.status;
    dsp->mode = DSP_MODE_ERROR;
    GST_ERROR ("Set_params failed with status %d", cmd_status.status);
    return FALSE;
  }

  dsp->mode = DSP_MODE_CONFIGURED;
  GST_INFO ("setparams successful");
  return TRUE;

}


/*
 * gst_dspaudio_set_audioparams:
 * @dsp: DSP audio object
 * @format: audio format constant
 * @sample_rate: sample rate constant
 * @channels: number of audio channels
 *
 * This sets the given audio parameters to audio node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_set_audioparams (GstDSPAudio *dsp,
                              guint format,
                              guint sample_rate,
                              guint channels)
{
  AUDIO_PARAMS_DATA init_data;

  init_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  init_data.audio_fmt = format;
  init_data.sample_rate = sample_rate;
  init_data.number_channels = channels;
  init_data.stream_priority = dsp->priority;
  init_data.ds_stream_ID = 0;

  GST_INFO ("format: %d srate: %d, channels: %d",
            format, sample_rate, channels);

  if (!gst_dspaudio_setparams (dsp, (char *)&init_data,
                               sizeof(AUDIO_PARAMS_DATA)))
  {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  return TRUE;
}


/*
 * gst_dspaudio_set_speechparams:
 * @dsp: DSP audio object
 * @format: audio format constant
 * @sample_rate: sample rate constant
 * @frame_size: size of sample frames
 *
 * This sets the given audio parameters to audio node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_set_speechparams (GstDSPAudio *dsp,
                               guint format,
                               guint sample_rate,
                               guint frame_size)
{
  SPEECH_PARAMS_DATA init_data;

  init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
  init_data.audio_fmt = format;
  init_data.sample_rate = sample_rate;
  init_data.frame_size = frame_size;
  init_data.stream_priority = dsp->priority;
  init_data.ds_stream_ID = 0;

  GST_INFO ("format: %d srate: %d, framesize: %d",
            format, sample_rate, frame_size);

  if (!gst_dspaudio_setparams (dsp, (char *)&init_data,
                               sizeof(SPEECH_PARAMS_DATA)))
  {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  return TRUE;
}



/*
 * gst_dspaudio_discont:
 * @dsp: DSP audio object
 *
 * Informs the DSP about discontinuity of the stream. Must be used
 * when a seek operation is performed on a compressed stream.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_discont (GstDSPAudio *dsp)
{
  short int cmd = DSP_CMD_DISCONT;
  gboolean retval = FALSE;

  GST_DEBUG ("Discont called");

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->discont_sent ||
      dsp->mode == DSP_MODE_ERROR ||
      dsp->codec.fd < 0)
      return FALSE;

  gst_dspaudio_borrow_lock (dsp);
  if (write (dsp->codec.fd, &cmd, sizeof (short int)) > 0) {
    if (gst_dspaudio_read_cmd (dsp, 0) == DSP_WAIT_OK) {
      dsp->discont_sent = TRUE;
      retval = TRUE;
    }
  }
  gst_dspaudio_return_lock (dsp);

  if (!retval)
    dsp->mode = DSP_MODE_ERROR;

  GST_DEBUG ("Discont sent");
  return retval;
}


/*
 * gst_dspaudio_set_volume:
 * @dsp: DSP audio object
 *
 * Sends the volume command to DSP. Volume value is retrieved from
 * "volume" attribute of the audio object. The value can be between
 * 0 - 0xFFFF.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_volume (GstDSPAudio *dsp)
{
  VOLUME_DATA volume_data;
  gint ret;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_UNINITIALIZED ||
      dsp->mode == DSP_MODE_ERROR ||
      dsp->codec.fd < 0)
      return FALSE;

  if (dsp->volume > DSP_VOLUME_MAX_INT) {
    dsp->volume = DSP_VOLUME_MAX_INT;
  }

  volume_data.dsp_cmd = DSP_CMD_SET_VOLUME;

  g_mutex_lock (dsp->prop_mutex);
  guint foo = dsp->volume / 0xCCC;  // foo scale: 0x00 - 0x14 --> 21 levels
  volume_data.scale = volumescaletable[foo];
  volume_data.power2 = volumepower2table[foo];
  dsp->volume_changed = FALSE;
  g_mutex_unlock (dsp->prop_mutex);

  GST_DEBUG ("Set_volume called: power2 = %d, scale = 0x%04X",
    volume_data.power2, volume_data.scale );

  ret = write (dsp->codec.fd, &volume_data, sizeof (volume_data));
  if (ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  GST_DEBUG ("Set_volume successful");
  return TRUE;
}


/*
 * gst_dspaudio_set_panning:
 * @dsp: DSP audio object
 *
 * Sends the balance command to DSP. Panning value is retrieved from
 * "panning" attribute of the audio object. This value must be between
 * -0x4000 (left panning) and +0x4000 (right panning)
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_panning (GstDSPAudio *dsp)
{
  PANNING_DATA panning_data;
  gint ret;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_UNINITIALIZED ||
      dsp->mode == DSP_MODE_ERROR ||
      dsp->codec.fd < 0)
      return FALSE;

  g_mutex_lock (dsp->prop_mutex);
  if (dsp->panning > DSP_PANNING_MAX) {
    dsp->panning = DSP_PANNING_MAX;
  }
  else if (dsp->panning < DSP_PANNING_MIN) {
    dsp->panning = DSP_PANNING_MIN;
  }

  panning_data.dsp_cmd = DSP_CMD_SET_PANNING;
  panning_data.left_gain = DSP_PANNING_MAX - (dsp->panning>0 ? dsp->panning : 0);
  panning_data.right_gain = DSP_PANNING_MAX + (dsp->panning<0 ? dsp->panning : 0);
  panning_data.steps = 1;
  dsp->panning_changed = FALSE;
  g_mutex_unlock (dsp->prop_mutex);

  GST_DEBUG ("Set_panning called: left = 0x%04X, right = 0x%04X",
    panning_data.left_gain, panning_data.right_gain );

  ret = write (dsp->codec.fd, &panning_data, sizeof (panning_data));
  if (ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  GST_DEBUG ("Set_panning successful");
  return TRUE;
}


/*
 * gst_dspaudio_set_mute:
 * @dsp: DSP audio object
 *
 * Send MUTE/UNMUTE command to DSP. The value is retrieved from
 * "mute" attribute of the audio object.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_mute (GstDSPAudio *dsp)
{
  DSP_CMD_STATUS status;
  gint ret;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_UNINITIALIZED ||
      dsp->mode == DSP_MODE_ERROR ||
      dsp->codec.fd < 0)
      return FALSE;

  GST_DEBUG ("Set (un)mute called, value = %d", dsp->mute);

  g_mutex_lock (dsp->prop_mutex);
  status.dsp_cmd = dsp->mute ? DSP_CMD_MUTE : DSP_CMD_UNMUTE;
  dsp->mute_changed = FALSE;
  g_mutex_unlock (dsp->prop_mutex);

  ret = write (dsp->codec.fd, &status, sizeof (short int));

  if (ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  GST_DEBUG ("Set_mute successful");
  return TRUE;
}


/*
 * gst_dspaudio_play:
 * @dsp: DSP audio object
 *
 * Sends a PLAY command to DSP. After this the DSP will start sending
 * DSP_CMD_DATA_WRITE requests, that can be read with wait_buffer ()
 * method.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_play (GstDSPAudio *dsp)
{
  short int cmd = DSP_CMD_PLAY;
  gboolean retval = FALSE;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_UNINITIALIZED ||
      dsp->mode == DSP_MODE_ERROR ||
      dsp->codec.fd < 0)
      return FALSE;

  GST_DEBUG ("Sending play to dsp");

  gst_dspaudio_borrow_lock (dsp);
  if (write (dsp->codec.fd, &cmd, sizeof (short int)) > 0) {
    dsp->mode = DSP_MODE_PLAYING;
    retval = TRUE;
  }
  gst_dspaudio_return_lock (dsp);

  if (retval == FALSE)
    dsp->mode = DSP_MODE_ERROR;

  GST_DEBUG ("play %s", retval ? "successful" : "failed");
  dsp->rw_pending = 0;
  return retval;
}


/*
 * gst_dspaudio_pause:
 * @dsp: DSP audio object
 *
 * Sends a PAUSE command to DSP.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_pause (GstDSPAudio *dsp)
{
  short int cmd = DSP_CMD_PAUSE;
  gboolean retval = FALSE;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_ERROR ||
      dsp->mode == DSP_MODE_EOS ||
      dsp->codec.fd < 0)
      return FALSE;

  if (dsp->mode != DSP_MODE_PLAYING) {
    GST_DEBUG ("Not in playing mode -> cannot pause");
    return FALSE;
  }

  gst_dspaudio_borrow_lock (dsp);
  if (write (dsp->codec.fd, &cmd, sizeof (short int)) > 0) {
    if (gst_dspaudio_read_cmd (dsp, 0) == DSP_WAIT_OK) {
      dsp->mode = DSP_MODE_PAUSED;
      dsp->rw_pending = 0;
      retval = TRUE;
    }
  }
  gst_dspaudio_return_lock (dsp);

  if (retval == FALSE)
    dsp->mode = DSP_MODE_ERROR;

  GST_DEBUG ("node pause %s", retval ? "successful" : "failed");
  return retval;
}


/*
 * gst_dspaudio_stop:
 * @dsp: DSP audio object
 *
 * Send a STOP command to DSP. This will stop the audio playback and
 * set the audio object mode into DSP_MODE_STOPPED.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_stop (GstDSPAudio *dsp)
{
  short int cmd = DSP_CMD_STOP;
  gboolean retval = FALSE;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_ERROR ||
      dsp->mode == DSP_MODE_EOS ||
      dsp->codec.fd < 0)
      return FALSE;

  GST_DEBUG ("Stop called");

  // We can stop only from PLAY / PAUSE mode
  if (dsp->mode == DSP_MODE_PLAYING || dsp->mode == DSP_MODE_PAUSED) {

    gst_dspaudio_borrow_lock (dsp);
    if (write (dsp->codec.fd, &cmd, sizeof (short int)) > 0) {
      if (gst_dspaudio_read_cmd (dsp, 0) == DSP_WAIT_OK) {
        dsp->mode = DSP_MODE_STOPPED;
        retval = TRUE;
      }
    }
    gst_dspaudio_return_lock (dsp);

    if (retval == FALSE)
      dsp->mode = DSP_MODE_ERROR;
  }
  GST_DEBUG ("stop %s", retval ? "successful" : "failed");
  return retval;
}


/*
 * gst_dspaudio_map_samplerate:
 * @rate: Source rate to be converted
 * @dest_rate: Pointer where to store DSP-specific rate definition
 *
 * Converts sample rate value into DSP-specific enumeration value
 *
 * Returns: TRUE if conversion was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_map_samplerate (guint rate, guint *dest_rate)
{
  *dest_rate = 0xffff;

  switch (rate) {
    case 96000: *dest_rate = SAMPLE_RATE_96KHZ; break;
    case 88200: *dest_rate = SAMPLE_RATE_88_2KHZ; break;
    case 64000: *dest_rate = SAMPLE_RATE_64KHZ; break;
    case 48000: *dest_rate = SAMPLE_RATE_48KHZ; break;
    case 44100: *dest_rate = SAMPLE_RATE_44_1KHZ; break;
    case 32000: *dest_rate = SAMPLE_RATE_32KHZ; break;
    case 24000: *dest_rate = SAMPLE_RATE_24KHZ; break;
    case 22050: *dest_rate = SAMPLE_RATE_22_05KHZ; break;
    case 16000: *dest_rate = SAMPLE_RATE_16KHZ; break;
    case 12000: *dest_rate = SAMPLE_RATE_12KHZ; break;
    case 11025: *dest_rate = SAMPLE_RATE_11_025KHZ; break;
    case  8000: *dest_rate = SAMPLE_RATE_8KHZ; break;
  }

  return *dest_rate != 0xffff ? TRUE : FALSE;
}


/*
 * gst_dspaudio_map_channels:
 * @channels: Number of channels to convert
 * @dest_channels: Pointer where to store DSP-specific channels definition
 *
 * Converts channel number value into DSP-specific enumeration value
 *
 * Returns: TRUE if conversion was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_map_channels (guint channels, guint *dest_channels)
{
  *dest_channels = 0xffff;

  switch (channels) {
    case 1: *dest_channels = CHANNELS_1; break;
    case 2: *dest_channels = CHANNELS_2; break;
  }

  return *dest_channels != 0xffff ? TRUE : FALSE;
}

/*
 * gst_dspaudio_map_format:
 * @dsp: DSP audio object
 * @codec: codec type
 * @endianness: endianness of the sample (G_BIG_ENDIAN / G_LITTLE_ENDIAN)
 * @sign: sign mode. FALSE=unsigned, TRUE=signed
 * @width: Width of the samples. Should be 8 or 16
 * @format: Pointer to a variable where format value should be placed
 * @bps: Pointer to a variable where to put bytes-per-sample value (or %NULL if not needed)
 *
 * Translates various audio-related information into one format definition
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_map_format (GstDSPAudio *dsp, GstDSPAudioCodec codec, gint endianness,
    gboolean sign, gint width, guint * format, guint * bps)
{
  if(bps) *bps=1;

  switch(codec) {
    case GST_DSPAUDIO_CODEC_RAW:
      if (width == 16) {
        if (sign == TRUE) {
          if (endianness == G_BIG_ENDIAN) {
            *format = DSP_AFMT_S16_BE;
          } else {
            *format = DSP_AFMT_S16_LE;
          }
        }
        else {
          if (endianness == G_BIG_ENDIAN) {
            *format = DSP_AFMT_U16_BE;
          } else {
            *format = DSP_AFMT_U16_LE;
          }
        }
        if(bps) *bps = 2;
      }
      else if (width == 8) {
        if (sign == TRUE) {
          *format = DSP_AFMT_S8;
        } else {
          *format = DSP_AFMT_U8;
        }
      }
      else {
        GST_DEBUG("unknown sample width");
        return FALSE;
      }
      break;
    case GST_DSPAUDIO_CODEC_MULAW:
      if(dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_OFF) {
        GST_DEBUG ("DTX off");
        *format = DSP_AFMT_ULAW;
      }
      else {
        GST_DEBUG ("DTX on");
        *format = DSP_AFMT_ULAW_DTX;
      }
      break;
    case GST_DSPAUDIO_CODEC_ALAW:
      if(dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_OFF) {
        GST_DEBUG ("DTX off");
        *format = DSP_AFMT_ALAW;
      }
      else {
        GST_DEBUG ("DTX on");
        *format = DSP_AFMT_ALAW_DTX;
      }
      break;
    case GST_DSPAUDIO_CODEC_ILBC:
      if(dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_OFF) {
        GST_DEBUG ("DTX off");
        *format = DSP_AFMT_ILBC;
      }
      else {
        GST_DEBUG ("DTX on");
        *format = DSP_AFMT_ILBC_DTX;
      }
      break;
    case GST_DSPAUDIO_CODEC_G729:
      if(dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_OFF) {
        GST_DEBUG ("DTX off");
        *format = DSP_AFMT_G729;
      }
      else {
        GST_DEBUG ("DTX on");
        *format = DSP_AFMT_G729_DTX;
      }
      break;
    case GST_DSPAUDIO_CODEC_AMR:
      // would need to know the rate!
      //*format = DSP_AFMT_AMR, DSP_AFMT_AMRNB;
      break;
    default:
      GST_DEBUG ("unknown codec-mode : %d", codec);
      return FALSE;
  }

  return TRUE;
}



/*
 * gst_dspaudio_get_info_unlocked:
 * @dsp: DSP audio object
 *
 * Send a status query to DSP. The status reply needs to be
 * read with _wait_buffer () or _read_cmd () call.
 *
 * Returns: TRUE, if operation was successful. Otherwise FALSE
 */
gboolean
gst_dspaudio_get_info_unlocked (GstDSPAudio *dsp)
{
  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_ERROR || dsp->codec.fd < 0)
      return FALSE;

  unsigned short int cmd = DSP_CMD_STATE;
  gint len;

  GST_LOG ("dspaudio_get_info");

  len = write (dsp->codec.fd, &cmd, sizeof (short int));
  if (len == -1) {
    dsp->error_cmd = DSP_CMD_STATE;
    dsp->mode = DSP_MODE_ERROR;
    dsp->error_status = -1;
    return FALSE;
  }
  return TRUE;
}


/*
 * gst_dspaudio_read_data:
 * @dsp: DSPAudio object
 * @data_size: Amount of data in MMAP buffer (number of short ints)
 * @eof_mode: Inform DSP that this is the last audio data chunk 
 *
 * Informs DSP that application wants to read more data by sending
 * a DSP_CMD_DATA_READ command. If eof_mode was selected, DSP_CMD_EOF will
 * be sent instead.
 *
 * Returns: TRUE if operation was successful
 */
gboolean
gst_dspaudio_read_data (GstDSPAudio *dsp, guint data_size, guint8 *ptr)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  short int tmp = DSP_CMD_DATA_READ;
  gboolean retval = TRUE;

  memcpy (ptr, dsp->codec.mmap_buffer, data_size);

  g_mutex_lock (dsp->dsp_mutex);
  if (write (dsp->codec.fd, &tmp, sizeof (short int)) < 1) {
    GST_WARNING ("write() failed");
    dsp->mode = DSP_MODE_ERROR;
    retval = FALSE;
  }
  dsp->rw_pending = 0;
  g_mutex_unlock (dsp->dsp_mutex);
  return retval;
}


/*
 * gst_dspaudio_read_frame:
 * @dsp: DSPAudio object
 * @voice_framesize: size of the VOICE_FRAME to be read
 * @update_framesize: size of the SID_FRAME_UPDATE (CN) frame
 * @frame_duration_millis: duration of one frame in milliseconds
 *
 * Reads one DTX audio frame from DSP. CN frames will be marked with
 * GST_BUFFER_FLAG_LAST-flag.
 *
 * Returns: GstBuffer that contains the frame, or NULL. If NULL is
 * returned, it means that either an error is occured, or pipeline is
 * set to PAUSED. On error the GstDSPAudio::mode will be set to DSP_MODE_ERROR.
 */
GstBuffer *
gst_dspaudio_read_frame (GstDSPAudio *dsp,
                         guint voice_framesize,
                         guint update_framesize,
                         guint frame_duration_millis)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  short int tmp = DSP_CMD_DATA_READ;
  short int *mmap_ptr = (short int *) dsp->codec.mmap_buffer;
  GstBuffer *newbuf = NULL;
  DSPWaitStatus status;

  do {
    if (dsp->all_frames_read) {
      status = gst_dspaudio_wait_buffer (dsp);
      if (status == DSP_WAIT_INTERRUPT) {
        // Engine is running the pipeline down. We need to exit now
        GST_DEBUG ("READ_DTX_FRAME exit");
        return NULL;
      }
      else if (status == DSP_WAIT_ERROR) {
        dsp->mode = DSP_MODE_ERROR;
        return NULL;
      }
      GST_LOG ("Framesize: %d", dsp->readinfo.frame_size);
      dsp->all_frames_read = FALSE;
      dsp->readindex = 0;
    }

    switch (mmap_ptr[dsp->readindex++]) {
      case VOICE_FRAME:
        //GST_LOG ("VOICE FRAME");
        newbuf = dsp->handle_voice_frame (&(mmap_ptr[dsp->readindex]),
                                          voice_framesize);
        dsp->readindex += (voice_framesize / sizeof (short int));
        dsp->silence_detected = FALSE;
        break;
      case SID_FRAME_UPDATE:
        //GST_LOG ("SID_FRAME_UPDATE");
        if (dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_ON_SEND_CN) {
          newbuf = dsp->handle_sid_frame (&(mmap_ptr[dsp->readindex]),
                                          update_framesize);
          if (dsp->lastcn) {
            gst_buffer_unref (dsp->lastcn);
          }
          gst_buffer_ref (newbuf);
          dsp->lastcn = newbuf;
        }
        else if (dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_ON_SEND_EMPTY) {
          newbuf = gst_buffer_new ();
        }
        dsp->readindex += (update_framesize / sizeof (short int));
        dsp->silence_detected = TRUE;
        break;
      case SID_FRAME_NO_UPDATE:
        //GST_LOG ("SID_FRAME_NO_UPDATE");
        if (dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_ON_SEND_CN &&
            dsp->lastcn != NULL)
        {
          newbuf = dsp->lastcn;
          gst_buffer_ref (newbuf);
        }
        else if (dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_ON_SEND_EMPTY) {
          newbuf = gst_buffer_new ();
        }
        dsp->silence_detected = TRUE;
        break;
      default:
        GST_WARNING ("Unknown header: %d", mmap_ptr[dsp->readindex-1]);
    }

    if (newbuf) {
      GST_BUFFER_DURATION (newbuf) = frame_duration_millis * GST_MSECOND;
      GST_BUFFER_TIMESTAMP (newbuf) = dsp->in_framecount * frame_duration_millis * GST_MSECOND;
      if (dsp->silence_detected) {
        GST_BUFFER_FLAG_SET (newbuf, GST_BUFFER_FLAG_LAST);
      }
    }

    if (dsp->readindex == dsp->readinfo.frame_size) {
      g_mutex_lock (dsp->dsp_mutex);
      if (write (dsp->codec.fd, &tmp, sizeof (short int)) < 0) {
        dsp->mode = DSP_MODE_ERROR;
        break;
      }
      dsp->rw_pending = 0;
      g_mutex_unlock (dsp->dsp_mutex);
      dsp->all_frames_read = TRUE;
      dsp->read_sent = TRUE;
    }
    dsp->in_framecount++;
  }
  while (!newbuf);

  return newbuf;
}


/*
 * gst_dspaudio_write_data:
 * @dsp: DSPAudio object
 * @data_size: Amount of data in MMAP buffer (number of short ints)
 * @eof_mode: Inform DSP that this is the last audio data chunk 
 *
 * Informs DSP that new audio data has been copied to MMAP buffer by sending
 * a DSP_CMD_DATA_WRITE command. If eof_mode was selected, DSP_CMD_EOF will
 * be sent instead.
 *
 * Returns: TRUE if operation was successful
 */
gboolean
gst_dspaudio_write_data (GstDSPAudio *dsp, guint data_size, gboolean eof_mode)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  gboolean retval = FALSE;

  if (eof_mode) {
    dsp->outbuf.dsp_cmd = DSP_CMD_EOF;
    dsp->eof_received = FALSE;
  } else {
    dsp->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;
  }
  dsp->outbuf.data_size = data_size;

  g_mutex_lock (dsp->dsp_mutex);
  if (write (dsp->codec.fd, &dsp->outbuf, sizeof (DSP_DATA_WRITE)) > 0) {
    dsp->discont_sent = FALSE;
    dsp->rw_pending = 0;
    retval = TRUE;
  }
  else {
    GST_WARNING ("write() failed");
    dsp->mode = DSP_MODE_ERROR;
  }
  g_mutex_unlock (dsp->dsp_mutex);
  return retval;
}


/*
 * gst_dspaudio_write_frame:
 * @dsp: DSPAudio object
 * @buffer: Audio data buffer to be rendered
 * @framesize: size of the VOICE_FRAME to be read
 * @update_framesize: size of the SID_FRAME_UPDATE frame
 * @use_lfi: Set to TRUE if codec requires Last Frame Indicator flags
 *
 * Writes DTX audio frame(s) to DSP.
 *
 * Returns: GST_FLOW_OK if operation was successful, GST_FLOW_WRONG_STATE if
 * interrupted and GST_FLOW_ERROR otherwise
 */
GstFlowReturn
gst_dspaudio_write_frame (GstDSPAudio *dsp, GstBuffer *buffer,
                          guint framesize, guint update_framesize,
                          gboolean use_lfi)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (buffer, FALSE);

  const guint8 *data = GST_BUFFER_DATA (buffer);
  guint datasize = GST_BUFFER_SIZE (buffer);
  GstFlowReturn ret = GST_FLOW_OK;
  gboolean voice_mode = FALSE;
  gboolean do_write = FALSE;
  gboolean do_break = FALSE;
  DSPWaitStatus status;
  short int *mmap_ptr = (short int *) dsp->codec.mmap_buffer;
  guint dataIndex = 0, mmapIndex, tw;

  dsp->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;

  while (TRUE) {
    status = gst_dspaudio_wait_buffer (dsp);

    if (status == DSP_WAIT_INTERRUPT) {
      GST_DEBUG ("DSP_WAIT_INTERRUPT");
      return GST_FLOW_WRONG_STATE;
    }
    if (status == DSP_WAIT_ERROR) {
      dsp->mode = DSP_MODE_ERROR;
      ret = GST_FLOW_ERROR;
      break;
    }
    mmapIndex = 0;

    mmap_ptr[mmapIndex++] = 0;                // Bad frame indication
    if (use_lfi) mmap_ptr[mmapIndex++] = 0;   // Last frame indication

    // FIXME: This is not very reliable method
    if (datasize > update_framesize || voice_mode) {
      mmap_ptr[mmapIndex++] = VOICE_FRAME;
      tw = MIN (framesize - dsp->out_index, datasize);
      memcpy ((char *) &(mmap_ptr[mmapIndex]) + dsp->out_index, data+dataIndex, tw);
      dsp->out_index += tw;
      dataIndex += tw;
      datasize -= tw;
      voice_mode = TRUE;

      if (dsp->out_index == framesize) {
        // Full frame
        dsp->outbuf.data_size = mmapIndex + (framesize / sizeof(short int));
        do_break = (datasize == 0);
        do_write = TRUE;
      }
      else {
        // It was incomplete frame
        do_write = FALSE;
        do_break = TRUE;
      }

    }
    else if (datasize == update_framesize) {
        mmap_ptr[mmapIndex++] = SID_FRAME_UPDATE;
        memcpy(&mmap_ptr[mmapIndex], data, update_framesize);
        dsp->outbuf.data_size = mmapIndex + (update_framesize / sizeof(short int));
        do_write = do_break = TRUE;
    }
    else {
        // Zero-length buffer means no-update
        mmap_ptr[mmapIndex++] = SID_FRAME_NO_UPDATE;
        dsp->outbuf.data_size = mmapIndex;
        do_write = do_break = TRUE;
    }

    if (do_write) {
      g_mutex_lock (dsp->dsp_mutex);
      if (write (dsp->codec.fd, &dsp->outbuf, sizeof(DSP_DATA_WRITE)) < 0) {
        g_mutex_unlock (dsp->dsp_mutex);
        dsp->mode = DSP_MODE_ERROR;
        ret = GST_FLOW_ERROR;
        break;
      }
      dsp->rw_pending = 0;
      g_mutex_unlock (dsp->dsp_mutex);

      GST_LOG ("Wrote %d bytes to DSP", dsp->outbuf.data_size * sizeof(short int));
      dsp->out_index = 0;
      do_write = FALSE;
    }
    if (do_break) break;
  }
  return ret;
}


/*
 * gst_dspaudio_get_dsp_time:
 * @dsp: DSPAudio object
 *
 * Reads the current time from DSP.
 *
 * Returns: GstClockTime representing DSP time, or GST_CLOCK_TIME_NONE if
 * something went wrong.
 */
static GstClockTime
gst_dspaudio_get_dsp_time (GstDSPAudio *dsp)
{
  if (!gst_dspaudio_get_info (dsp)) {
    return GST_CLOCK_TIME_NONE;
  }

  dsp->clock_time = gst_dspaudio_info_to_time (dsp);
  return dsp->clock_time;
}


/*
 * gst_dspaudio_map_dtmf:
 * @tone_id: DTMF id to be mapped.
 * @dsp_tone_id: output parameter for DSP tone ID
 *
 * Map the given character to DSP tone ID. The DTMF codes can be found
 * from http://www.iana.org/assignments/audio-telephone-event-registry
 *
 * Returns: TRUE if mapping found, otherwise FALSE.
 */
gboolean
gst_dspaudio_map_dtmf (gint tone_id, unsigned short int *dsp_tone_id)
{
  gboolean ret = TRUE;

  switch(tone_id) {
    case 0:  *dsp_tone_id = DTMF_0; break;
    case 1:  *dsp_tone_id = DTMF_1; break;
    case 2:  *dsp_tone_id = DTMF_2; break;
    case 3:  *dsp_tone_id = DTMF_3; break;
    case 4:  *dsp_tone_id = DTMF_4; break;
    case 5:  *dsp_tone_id = DTMF_5; break;
    case 6:  *dsp_tone_id = DTMF_6; break;
    case 7:  *dsp_tone_id = DTMF_7; break;
    case 8:  *dsp_tone_id = DTMF_8; break;
    case 9:  *dsp_tone_id = DTMF_9; break;
    case 10: *dsp_tone_id = DTMF_STAR; break;
    case 11: *dsp_tone_id = DTMF_HASH; break;
    case 12: *dsp_tone_id = DTMF_A; break;
    case 13: *dsp_tone_id = DTMF_B; break;
    case 14: *dsp_tone_id = DTMF_C; break;
    case 15: *dsp_tone_id = DTMF_D; break;
    default: ret = FALSE; break;
  }
  return ret;
}


/*
 * gst_dspaudio_dtmf_start_unlocked:
 * @dsp: DSP audio object
 * @tone_char: DTMF tone character to be sent (start) or -1 (stop)
 *
 * Send DTMF_START command to DSP. Caller need to grab DSP lock first
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_dtmf_tone_start_unlocked (GstDSPAudio *dsp, gint tone_id)
{
  TONE_STATUS status;

  g_return_val_if_fail (dsp, FALSE);

  if (dsp->mode == DSP_MODE_ERROR ||
      dsp->mode != DSP_MODE_PLAYING ||
      dsp->codec.fd < 0)
      return FALSE;

  if (dsp->dtmf_playing) {
    // Stop the previous tone before playing new
    GST_LOG ("turning off old tone");
    gst_dspaudio_dtmf_tone_stop_unlocked (dsp);
  }

  if (gst_dspaudio_map_dtmf (tone_id, &status.tone_id)) {
    GST_DEBUG ("starting DTMF tone: %d", status.tone_id);
    status.dsp_cmd = DSP_CMD_DTMF_START;
    status.amplitude = 29205; // FIXME: Hardcoded value
    if (write (dsp->codec.fd, &status, sizeof (TONE_STATUS)) < 0) {
        dsp->mode = DSP_MODE_ERROR;
        return FALSE;
    }
    gst_dspaudio_read_cmd (dsp, 0);
    dsp->dtmf_playing = TRUE;
  }
  GST_LOG ("tone command sent");
  return TRUE;
}


/*
 * gst_dspaudio_dtmf_stop_unlocked:
 * @dsp: DSP audio object
 *
 * Send DTMF_STOP command to DSP. Caller need to grab DSP lock first
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_dtmf_tone_stop_unlocked (GstDSPAudio *dsp)
{
  short int cmd = DSP_CMD_DTMF_STOP;

  if (dsp->mode == DSP_MODE_ERROR || dsp->codec.fd < 0)
    return FALSE;

  if (dsp->dtmf_playing) {
    GST_DEBUG ("stopping DTMF tone");
    if (write (dsp->codec.fd, &cmd, sizeof (short int)) < 0) {
      dsp->mode = DSP_MODE_ERROR;
      return FALSE;
    }
    dsp->dtmf_playing = FALSE;
    gst_dspaudio_read_cmd (dsp, 0);
    GST_LOG ("DTMF tone stopped");
  }
  return TRUE;
}


