/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/poll.h>

#include <gst/gst.h>

#include "dspaudio.h"
#include "apps_voltable.h"

#ifdef HAVE_DSP_H
#include <dsp/interface_common.h>
#if DSP_API_VERSION != 10
#error Expected DSP API version 10
#endif
#endif /* HAVE_DSP_H */

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

#define AUDIO_PM_SERVICE            "com.nokia.osso_audio_pm"
#define AUDIO_PM_PLAYBACK_RESOURCE  "/com/nokia/osso/pm/audio/playback"
#define AUDIO_PM_RECORD_RESOURCE    "/com/nokia/osso/pm/audio/record"
#define RESOURCE_INTERFACE          "com.nokia.osso_resource_manager"
#define RESOURCE_TIMEOUT            200

GST_DEBUG_CATEGORY_STATIC (dspaudio_debug);
#define GST_CAT_DEFAULT dspaudio_debug

static gboolean 
gst_dspaudio_open_node (DSPNodeInfo *info, const gchar *devname);

static gboolean
gst_dspaudio_close_node (DSPNodeInfo *info);

static gboolean
gst_dspaudio_check_read (int dev_fd);

static gboolean
gst_dspaudio_set_volume (GstDSPAudio *dsp);

static gboolean
gst_dspaudio_set_panning (GstDSPAudio *dsp);

static gboolean
gst_dspaudio_set_mute (GstDSPAudio *dsp);

static gboolean
gst_dspaudio_reset_node (GstDSPAudio *dsp);

static gboolean
gst_dspaudio_mic_ctl (GstDSPAudio *dsp, gboolean mic_enable);

static GstClockTime
gst_dspaudio_get_dsp_time (GstDSPAudio *dsp);

static GstBuffer *
gst_dspaudio_default_voice_frame_handler (short int *mmap_ptr, guint fsize);

static GstBuffer *
gst_dspaudio_default_sid_frame_handler (short int *mmap_ptr, guint fsize);

void
gst_dspaudio_borrow_lock (GstDSPAudio *dsp);

void
gst_dspaudio_return_lock (GstDSPAudio *dsp);

void
gst_dspaudio_flush_unlocked (DSPNodeInfo *info);

gboolean
gst_dspaudio_get_info_unlocked (GstDSPAudio *dsp);

gboolean
gst_dspaudio_dtmf_tone_stop_unlocked (GstDSPAudio *dsp);



/*
 * gst_dspaudio_dtx_mode_get_type:
 *
 * Give a special GType that defines DTX mode.
 *
 */
GType
gst_dspaudio_dtx_mode_get_type (void)
{
  static GType dspaudio_dtx_mode_type = 0;
  static GEnumValue dspaudio_dtx_modes[] = {
    {GST_DSPAUDIO_DTX_MODE_OFF, "Don't use DTX", "off"},
    {GST_DSPAUDIO_DTX_MODE_ON_NOSEND, "Use DTX, but don't send anything when silent", "on"},
    {GST_DSPAUDIO_DTX_MODE_ON_SEND_EMPTY, "Use DTX and send empty buffers when silent", "on-empty"},
    {GST_DSPAUDIO_DTX_MODE_ON_SEND_CN, "Use DTX and send CN frames", "on-cn"},
    {0, NULL, NULL},
  };

  if (G_UNLIKELY(!dspaudio_dtx_mode_type)) {
    dspaudio_dtx_mode_type = g_enum_register_static ("GstDSPAudioDTXMode",
        dspaudio_dtx_modes);
  }
  return dspaudio_dtx_mode_type;
}


/*
 * gst_dspaudio_init:
 *
 * Initialize DSP audio library.
 *
 * Should be called before using these functions from source linking
 * to this source file.
 */
void
gst_dspaudio_init (void)
{
  static gboolean _dspaudio_initialized = FALSE;

  if (_dspaudio_initialized)
    return;

  _dspaudio_initialized = TRUE;

  GST_DEBUG_CATEGORY_INIT (dspaudio_debug, "dspaudio",
                           0, "DSP audio utility");
}


/*
 * gst_dspaudio_reset:
 * @dsp: DSP audio object
 *
 * Method for resetting the audio object
 */
void
gst_dspaudio_reset (GstDSPAudio * dsp)
{
  dsp->codec.fd                 = dsp->aep.fd = -1;
  dsp->codec.bridge_buffer_size = dsp->aep.bridge_buffer_size = 0;
  dsp->codec.mmap_buffer_size   = dsp->aep.mmap_buffer_size = 0;
  dsp->codec.mmap_buffer        = dsp->aep.mmap_buffer = NULL;

  dsp->mode = DSP_MODE_UNINITIALIZED;
  dsp->dtxmode = GST_DSPAUDIO_DTX_MODE_OFF;
  dsp->volume_changed = FALSE;
  dsp->mute_changed = FALSE;
  dsp->panning_changed = FALSE;
  dsp->discont_sent = TRUE;
  dsp->rw_pending = 0;
  dsp->all_frames_read = TRUE;
  dsp->readindex = 0;
  dsp->read_sent = FALSE;
  dsp->interrupted = FALSE;
  dsp->silence_detected = FALSE;
  dsp->out_index = 0;
  dsp->in_framecount = 0;
  dsp->eof_received = FALSE;
  dsp->reset_needed = FALSE;

  FD_ZERO (&dsp->read_fdset);

  if (dsp->lastcn) {
    gst_buffer_unref (dsp->lastcn);
    dsp->lastcn = NULL;
  }

  dsp->handle_voice_frame = gst_dspaudio_default_voice_frame_handler;
  dsp->handle_sid_frame = gst_dspaudio_default_sid_frame_handler;
}


/*
 * gst_dspaudio_new:
 *
 * Creates new audio object instance and initializes it
 * with default parameters. After this the device must be
 * separately opened with dspaudio_open () method.
 *
 * Returns: New #GstDSPAudio instance
 */
GstDSPAudio *
gst_dspaudio_new ()
{
  GstDSPAudio *dsp = g_new0(GstDSPAudio, 1);
  dsp->volume = DSP_VOLUME_DEFAULT_INT;
  dsp->volume_f = DSP_VOLUME_DEFAULT_FLOAT;
  dsp->panning = DSP_PANNING_DEFAULT;
  dsp->priority = DSP_PRIORITY_DEFAULT;
  dsp->mute = FALSE;
  dsp->dsp_mutex = g_mutex_new ();
  dsp->prop_mutex = g_mutex_new ();
  dsp->clock_mutex = g_mutex_new ();
  dsp->dsp_cond = g_cond_new ();
  gst_dspaudio_reset (dsp);

  if (pipe (dsp->pipe_fds) != 0) {
    g_mutex_free (dsp->dsp_mutex);
    g_mutex_free (dsp->prop_mutex);
    g_mutex_free (dsp->clock_mutex);
    g_cond_free (dsp->dsp_cond);
    g_free (dsp);
    return NULL;
  }

  /* Spec says we should just proceed with playback/record if power mgmt
   * fails, so we don't care about the dbus_bus_get return value here */
  dsp->dbus_connection = (gpointer *) dbus_bus_get (DBUS_BUS_SYSTEM, NULL);

  return dsp;
}


/*
 * gst_dspaudio_destroy:
 * @dsp: DSP audio object
 *
 * Delete given audio object instance
 */
void
gst_dspaudio_destroy (GstDSPAudio * dsp)
{
  if (dsp->dsp_mutex != NULL) {
    g_mutex_free (dsp->dsp_mutex);
    dsp->dsp_mutex = NULL;
  }
  if (dsp->prop_mutex != NULL) {
    g_mutex_free (dsp->prop_mutex);
    dsp->prop_mutex = NULL;
  }
  if (dsp->clock_mutex != NULL) {
    g_mutex_free (dsp->clock_mutex);
    dsp->clock_mutex = NULL;
  }
  if (dsp->dsp_cond != NULL) {
    g_cond_free (dsp->dsp_cond);
    dsp->dsp_cond = NULL;
  }
  if (dsp->tracklist) {
    g_list_foreach (dsp->tracklist, (GFunc) g_object_unref, NULL);
    g_list_free (dsp->tracklist);
    dsp->tracklist = NULL;
  }
  close (dsp->pipe_fds[0]);
  close (dsp->pipe_fds[1]);
  //if (dsp->dbus_connection != NULL) {
  //   dbus_connection_close ((DBusConnection *)dsp->dbus_connection);
  //}
  g_free (dsp);
}


#ifdef HAVE_DSP_H
#include "dspaudioreal.h"
#else
#include "dspaudiofake.h"
#endif


/*
 * gst_dspaudio_flush:
 * @dsp: DSP node info object
 *
 * Reads all unread data from node
 */
void
gst_dspaudio_flush (GstDSPAudio *dsp)
{
  g_return_if_fail (dsp);
  g_return_if_fail (dsp->mode != DSP_MODE_ERROR);

  g_mutex_lock (dsp->dsp_mutex);
  gst_dspaudio_flush_unlocked (&(dsp->codec));
  g_mutex_unlock (dsp->dsp_mutex);
}


/*
 * gst_dspaudio_open:
 * @dsp: DSP audio object
 * @devname: device name to be opened & initialized
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_open (GstDSPAudio *dsp, const gchar *devname)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (devname, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  if (dsp->mode > DSP_MODE_UNINITIALIZED) {
    GST_WARNING ("DSP node '%s' already open", devname);
    return TRUE;
  }

  g_mutex_lock (dsp->dsp_mutex);
  if (gst_dspaudio_open_node (&dsp->codec, devname) == FALSE) {
    dsp->mode = DSP_MODE_ERROR;
    g_mutex_unlock (dsp->dsp_mutex);
    return FALSE;
  }
  dsp->mode = DSP_MODE_INITIALIZED;
  g_mutex_unlock (dsp->dsp_mutex);
  return TRUE;
}


/*
 * gst_dspaudio_close:
 * @dsp: DSP audio object
 *
 * Closes DSP audio
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_close (GstDSPAudio *dsp)
{
  g_return_val_if_fail (dsp, FALSE);

  g_mutex_lock (dsp->dsp_mutex);
  gboolean ret = gst_dspaudio_close_node (&dsp->codec);
  gst_dspaudio_close_node (&dsp->aep);
  dsp->mode = DSP_MODE_UNINITIALIZED;
  g_mutex_unlock (dsp->dsp_mutex);
  return ret;
}


/*
 * gst_dspaudio_check_read:
 * @dev_fd: file descriptor to be checked
 *
 * Checks if there is some data available from the given file descriptor
 *
 * Returns: TRUE if data is available, otherwise FALSE
 */
static gboolean
gst_dspaudio_check_read (int dev_fd)
{
  struct pollfd pfd;

  pfd.fd = dev_fd;
  pfd.events = POLLIN;

  if (poll (&pfd,1,0)>0) return TRUE;
  else return FALSE;
}


/*
 * gst_dspaudio_read_cmd:
 * @dsp: DSP audio object
 * @timeout_millis: timeout how long a command will be waited
 *
 * This method reads one command reply from DSP. It can also receive
 * DATA_WRITE / DATA_READ command and set the flag accordingly.
 *
 * Returns: DSPWaitStatus
 */
DSPWaitStatus
gst_dspaudio_read_cmd (GstDSPAudio *dsp, gint timeout_millis)
{
  unsigned short int data[3];

  g_return_val_if_fail (dsp, DSP_WAIT_ERROR);

  if (dsp->mode == DSP_MODE_UNINITIALIZED ||
      dsp->mode == DSP_MODE_ERROR ||
      dsp->codec.fd < 0)
      return DSP_WAIT_ERROR;

  dsp->error_status = dsp->error_cmd = 0;

  // Read the command and it's status
  while (TRUE) {

    if (timeout_millis) {
      struct timeval tv;
      fd_set fdset;
      int retval;

      FD_ZERO (&fdset);
      FD_SET (dsp->codec.fd, &fdset);
      tv.tv_sec = timeout_millis / 1000;
      tv.tv_usec = (timeout_millis % 1000) * 1000;

      retval = select (dsp->codec.fd+1, &fdset, NULL, NULL, &tv);
      if (retval < 1) {
        GST_DEBUG ("Read_cmd interrupt. no reply");
        return DSP_WAIT_INTERRUPT;
      }
    }

#ifdef HAVE_DSP_H
    if (read (dsp->codec.fd, data, 2*sizeof (unsigned short int)) < 0) {
      dsp->error_status = -1;
      dsp->error_cmd = DSP_CMD_NONE;
      goto read_cmd_error;
    }
	//GST_INFO ("READ_CMD GOT REPLY: %d, status: %d", data[0], data[1]);

    dsp->error_cmd = data[0];

    if (data[0] == DSP_CMD_STATE) {
      dsp->info.stream_ID = data[1];
      if (read (dsp->codec.fd, &(dsp->info.ds_stream_ID),
         sizeof (AUDIO_STATUS_INFO) - 2 * sizeof (unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_cmd_error;
      }
      dsp->info.num_frames = ((dsp->info.num_frames & 0x0000ffff) << 16) |
                             ((dsp->info.num_frames & 0xffff0000) >> 16);
      break;
    }

    if (data[1] != DSP_OK) {
     GST_WARNING ("READ_CMD got ERROR: cmd = 0x%X, status = 0x%0X",
                  data[0], data[1]);

     dsp->error_status = data[1];

     // This error can never be returned with "normal" command
     if (dsp->error_status == DSP_ERROR_SYNC ||
        dsp->error_status == DSP_ERROR_STREAM)
     {
       dsp->reset_needed = TRUE;
       return DSP_WAIT_RESET;
     }
     goto read_cmd_error;
    }
    if (data[0] == DSP_CMD_DATA_WRITE) {
      if (read (dsp->codec.fd, &(dsp->dsp_buffer_free),
         sizeof (unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_cmd_error;
      }
      dsp->rw_pending++;
      continue;
    }
    else if (data[0] == DSP_CMD_DATA_READ) {
      if (read (dsp->codec.fd, &(dsp->readinfo),
         sizeof (READ_STATUS) - 2*sizeof (unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_cmd_error;
      }
      dsp->read_sent = FALSE;
      dsp->rw_pending++;
      continue;
    } else if (data[0] == DSP_CMD_EOF) {
      dsp->eof_received = TRUE;
    }
#endif
    // We got an "ordinary" command -> leave
    break;
  }
  return DSP_WAIT_OK;

#ifdef HAVE_DSP_H
read_cmd_error:
  dsp->mode = DSP_MODE_ERROR;
  GST_WARNING ("READ_CMD error, cmd: %d, status: %d", 
               dsp->error_cmd , dsp->error_status);
  return DSP_WAIT_ERROR;
#endif
}


/*
 * gst_dspaudio_wait_buffer:
 * @dsp: DSP audio object
 *
 * This method reads messages from DSP. It reads as many
 * messages as available. It waits especially for DATA_WRITE and
 * DATA_READ messages.
 *
 * Returns: DSPWaitStatus
 */
DSPWaitStatus
gst_dspaudio_wait_buffer (GstDSPAudio *dsp)
{
#ifdef HAVE_DSP_H
  DSP_CMD_STATUS incoming;
#endif

  if (dsp->mode == DSP_MODE_ERROR) {
    GST_WARNING ("Entered wait_buffer in error");
    return DSP_WAIT_ERROR;
  }

  // Need to remember error status for plugin when resetting
  if (!dsp->reset_needed) {
    dsp->error_status = dsp->error_cmd = 0;
  }
#ifdef HAVE_DSP_H
  g_mutex_lock (dsp->dsp_mutex);

  while (TRUE) {

    if (dsp->reset_needed) {
      GST_WARNING ("error tolerance - trying to reset playback to recover");
      // Remember error status, some plugins may use this for actions
      gint status = dsp->error_status;
      g_mutex_unlock (dsp->dsp_mutex);
      if (gst_dspaudio_reset_node (dsp) == TRUE) {
        dsp->error_status = status; // recall error status
        dsp->reset_needed = FALSE;
        return DSP_WAIT_OK;
      }
      else {
        dsp->mode = DSP_MODE_ERROR;
        return DSP_WAIT_ERROR;
      }
    }

    if (dsp->rw_pending) {
      g_mutex_unlock (dsp->dsp_mutex);
      return DSP_WAIT_OK;
    }

    FD_SET (dsp->codec.fd, &dsp->read_fdset);
    FD_SET (dsp->pipe_fds[0], &dsp->read_fdset);
    select ((dsp->codec.fd>dsp->pipe_fds[0]?dsp->codec.fd:dsp->pipe_fds[0])+1,
             &dsp->read_fdset, NULL, NULL, NULL);

    if (FD_ISSET (dsp->pipe_fds[0], &dsp->read_fdset)) {
      char tmp;
      dsp->error_status = dsp->error_cmd = 0;
      read (dsp->pipe_fds[0], &tmp, 1);

      if (tmp == PIPE_CMD_NEED_LOCK) {
        GST_LOG ("PIPE_CMD_NEED_LOCK");
        // This unlocks DSP mutex and waits for another (clock query)
        // thread to wake us up again when it has finished using DSP
        g_cond_wait (dsp->dsp_cond, dsp->dsp_mutex);
        GST_LOG ("wait_buffer got mutex back");
        continue;
      }
      else if (tmp == PIPE_CMD_INTERRUPT) {
        // Someone wants us to stop waiting. Most probably this is caused
        // by _unlock() call. Return back to main loop.
        g_mutex_unlock (dsp->dsp_mutex);
        GST_LOG ("PIPE_CMD_INTERRUPT");
        return DSP_WAIT_INTERRUPT;
      }
    }

    // Read the command
    if (read (dsp->codec.fd, &incoming, sizeof (unsigned short int)) < 0) {
      dsp->error_status = -1;
      dsp->error_cmd = DSP_CMD_NONE;
      goto read_buffer_error;
    }

    dsp->error_cmd = incoming.dsp_cmd;

    if (incoming.dsp_cmd == DSP_CMD_STATE) {
      if (read (dsp->codec.fd, &(dsp->info.stream_ID),
         sizeof (AUDIO_STATUS_INFO) - sizeof (unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_buffer_error;
      }
    }
    else {
      // Read the command status value
      if (read (dsp->codec.fd, &(incoming.status),
          sizeof (unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_buffer_error;
      }

      if (incoming.status != DSP_OK) {
        dsp->error_status = incoming.status;

        GST_WARNING ("WAIT_BUFFER: CMD: %d, ERROR: %d",
                     incoming.dsp_cmd, incoming.status);

        if (incoming.status == DSP_ERROR_SYNC ||
           incoming.status == DSP_ERROR_STREAM)
        {
          dsp->reset_needed = TRUE;
          continue;
        }
        goto read_buffer_error;
      }

      // Check for "special" commands
      if (incoming.dsp_cmd == DSP_CMD_DATA_WRITE) {
        if (read (dsp->codec.fd, &(dsp->dsp_buffer_free),
           sizeof (unsigned short int)) < 0)
        {
          dsp->error_status = -1;
          goto read_buffer_error;
        }
        dsp->rw_pending++;
      }
      else if (incoming.dsp_cmd == DSP_CMD_DATA_READ) {
        if (read (dsp->codec.fd, &(dsp->readinfo),
           sizeof (READ_STATUS) - 2*sizeof (unsigned short int)) < 0)
        {
          dsp->error_status = -1;
          goto read_buffer_error;
        }
        dsp->read_sent = FALSE;
        dsp->rw_pending++;
      }
    }

    // Read all possible messages from DSP
    if (gst_dspaudio_check_read (dsp->codec.fd)) {
      continue;
    }

    if (dsp->rw_pending) {
      break;
    }
  }
  g_mutex_unlock (dsp->dsp_mutex);
#endif
  return TRUE;

#ifdef HAVE_DSP_H
read_buffer_error:
  g_mutex_unlock (dsp->dsp_mutex);
  dsp->mode = DSP_MODE_ERROR;
  GST_WARNING ("READ_BUFFER error, cmd: %d, status: %d",
               dsp->error_cmd, dsp->error_status);
  return DSP_WAIT_ERROR;
#endif
}



/*
 * gst_dspaudio_wait_eos:
 * @dsp: DSP audio object
 *
 * Waits until EOS is received from DSP
 */
void
gst_dspaudio_wait_eos (GstDSPAudio *dsp)
{
  g_return_if_fail (dsp);
  g_return_if_fail (dsp->mode != DSP_MODE_ERROR);

  dsp->mode = DSP_MODE_EOS;
  while (TRUE) {
    GST_DEBUG ("Waiting for EOS...");
    if (gst_dspaudio_get_info (dsp) == FALSE) break;
    if (dsp->eof_received) break;
    if (dsp->interrupted) break;

    GST_DEBUG ("Got cmd: %d", dsp->error_cmd);
    usleep (200000);
  }
}



/*
 * gst_dspaudio_reset_node:
 * @dsp: DSP audio object
 *
 * Resets the audio node by sending STOP and PLAY commands.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_reset_node (GstDSPAudio *dsp)
{
  gboolean ret = FALSE;

  GST_DEBUG ("Reset node");
  if (gst_dspaudio_stop (dsp)) {
    GST_DEBUG ("Stop command successful");
    ret = gst_dspaudio_play (dsp);
    GST_DEBUG ("Play command: %d", ret);
  }
  return ret;
}


/*
 * gst_dspaudio_mic_ctl:
 * @dsp: DSP audio object
 * @mic_enable: If TRUE, microphone input is turned on
 *
 * Control microphone input
 *
 * Returns: TRUE of operation was successful
 */
static gboolean
gst_dspaudio_mic_ctl (GstDSPAudio *dsp, gboolean mic_enable)
{
  DBusConnection *connection;
  DBusMessage *message;
  DBusMessage *reply = NULL;
  gchar *method;

  method = mic_enable ? "request" : "release";

  connection = (DBusConnection *) dsp->dbus_connection;
  if (connection != NULL) {
    message = dbus_message_new_method_call (AUDIO_PM_SERVICE,
                                            AUDIO_PM_RECORD_RESOURCE,
                                            RESOURCE_INTERFACE,
                                            method);
    if (message != NULL) {
      if (!mic_enable) {
        dbus_int32_t handle = 0;
        /* FIXME: append args for request () when defined (currently none,
         * but RM spec still undergoing some changes)
         * Also, this assumes that the resource handle is always 0 (true
         * for PM resources, but could also save the handle from request ())
         */
        dbus_message_append_args (message,
                                  DBUS_TYPE_INT32, &handle,
                                  DBUS_TYPE_INVALID);
      }
      reply = dbus_connection_send_with_reply_and_block (connection, message,
                                                         RESOURCE_TIMEOUT,
                                                         NULL);
      /* TODO: check if reply indicates an error (currently doesn't matter
       * except for the debug output, as the return code is ignored anyway)
       * The return parameters might also have to changed a bit still in the
       * RM spec, so we accept anything for now */
      dbus_message_unref (message);
    }
  }
  if (reply == NULL) {
    GST_DEBUG ("Mic power management %s failed", method);
    return FALSE;
  } else {
    dbus_message_unref (reply);
    return TRUE;
  }
}


/*
 * gst_dspaudio_update_dsp_settings:
 * @dsp DSP audio object
 *
 * Sets the volume, panning and mute values to DSP if they
 * are changed since last time.
 */
void
gst_dspaudio_update_dsp_settings (GstDSPAudio *dsp)
{
  GST_DEBUG ("dspaudio_update_dsp_settings");

  // FIXME: Need prop_mutex?
  if (dsp->volume_changed) {
    if (gst_dspaudio_set_volume (dsp)) {
      GST_DEBUG ("Volume set: %d", dsp->volume);
      gst_dspaudio_read_cmd (dsp, 0);
    }
  }
  if (dsp->panning_changed) {
    if (gst_dspaudio_set_panning (dsp)) {
      GST_DEBUG ("Panning set: %d", dsp->panning);
      gst_dspaudio_read_cmd (dsp, 0);
    }
  }
  if (dsp->mute_changed) {
    if (gst_dspaudio_set_mute (dsp)) {
      GST_DEBUG ("Mute set: %d", dsp->mute);
      gst_dspaudio_read_cmd (dsp, 0);
    }
  }
  GST_LOG ("dspaudio_update_dsp_settings ok");
}


/*
 * gst_dspaudio_interrupt_render:
 * @dsp DSP audio object
 *
 * Informs blocking data reading methods _wait_buffer () and _read_cmd ()
 * that someone wants them to stop waiting for data from DSP. This happens
 * e.g. when the pipeline is set to PAUSED.
 */
void
gst_dspaudio_interrupt_render (GstDSPAudio *dsp)
{
  GST_DEBUG ("dspaudio_interrupt_render: dsp->mode = %d", dsp->mode);
  if (dsp->mode == DSP_MODE_PLAYING || dsp->mode == DSP_MODE_EOS) {
    char tmp = PIPE_CMD_INTERRUPT;
    GST_LOG ("Writing interrupt cmd to pipe");
    dsp->interrupted = TRUE;
    write (dsp->pipe_fds[1], &tmp, 1);
    GST_LOG ("Writing to pipe ok");
  }
}


/*
 * gst_dspaudio_flush_cmd_queue:
 * @dsp DSP audio object
 *
 * Flushes internal command queue that delivers commands from other threads
 * to renderer thread.
 */
void
gst_dspaudio_flush_cmd_queue (GstDSPAudio *dsp)
{
  char tmp;
  while (gst_dspaudio_check_read (dsp->pipe_fds[0])) {
    read (dsp->pipe_fds[0], &tmp,1);
    GST_DEBUG ("Flushing internal command pipeline");
  }
}


/*
 * gst_dspaudio_remove_interrupt:
 * @dsp DSP audio object
 *
 * Removes the "interrupted" flag that informs data reading functions that
 * they should interrupt their work and return immediately.
 */
void
gst_dspaudio_remove_interrupt (GstDSPAudio *dsp)
{
  // Remove interrupt flag and flush the "command pipeline"
  dsp->interrupted = FALSE;
  gst_dspaudio_flush_cmd_queue (dsp);
}


/*
 * gst_dspaudio_set_property:
 * @dsp: DSP audio object
 * @prop_id: ID value of the property
 * @value: Pointer that holds the value to be set
 *
 * Set a property into certain value
 *
 * Returns: TRUE if property was known, otherwise FALSE
 */
gboolean
gst_dspaudio_set_property (GstDSPAudio *dsp, guint prop_id, const GValue *value)
{
  gboolean ret = TRUE;

  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  switch (prop_id) {
    case DSPAUDIO_PROP_VOLUME:
    {
      guint tmp = g_value_get_uint (value);
      if (tmp != dsp->volume) {
        g_mutex_lock (dsp->prop_mutex);
        dsp->volume = tmp;
        dsp->volume_f = (gdouble) dsp->volume / DSP_VOLUME_MAX_INT;
        GST_DEBUG ("Setting integer volume property: %d", dsp->volume);
        dsp->volume_changed = TRUE;
        g_mutex_unlock (dsp->prop_mutex);
      }
      break;
    }
    case DSPAUDIO_PROP_VOLUME_FLOAT:
    {
      gdouble tmp = g_value_get_double (value);
      if (tmp != dsp->volume_f) {
        g_mutex_lock (dsp->prop_mutex);
        dsp->volume_f = tmp;
        dsp->volume = (guint) DSP_VOLUME_MAX_INT *
                (dsp->volume_f / DSP_VOLUME_MAX_FLOAT);
        GST_DEBUG ("Setting float volume property: %f", dsp->volume_f);
        dsp->volume_changed = TRUE;
        g_mutex_unlock (dsp->prop_mutex);
      }
      break;
    }
    case DSPAUDIO_PROP_PANNING:
    {
      g_mutex_lock (dsp->prop_mutex);
      dsp->panning = g_value_get_int (value);
      GST_DEBUG ("Setting panning property: %d", dsp->panning);
      dsp->panning_changed = TRUE;
      g_mutex_unlock (dsp->prop_mutex);
      break;
    }
    case DSPAUDIO_PROP_MUTE:
    {
      gboolean tmp = g_value_get_boolean (value);
      if (tmp != dsp->mute) {
        g_mutex_lock (dsp->prop_mutex);
        dsp->mute = tmp;
        GST_DEBUG ("Setting mute property: %d", dsp->mute);
        dsp->mute_changed = TRUE;
        g_mutex_unlock (dsp->prop_mutex);
      }
      break;
    }
    case DSPAUDIO_PROP_PRIORITY:
      dsp->priority = g_value_get_int (value);
      GST_DEBUG ("Setting priority property: %d", dsp->priority);
      break;
    case DSPAUDIO_PROP_DTXMODE:
      dsp->dtxmode = g_value_get_enum (value);
      break;
    default:
      ret = FALSE;
      break;
  }

  gst_dspaudio_borrow_lock (dsp);
  gst_dspaudio_update_dsp_settings (dsp);
  gst_dspaudio_return_lock (dsp);
  return ret;
}


/*
 * gst_dspaudio_get_property:
 * @dsp: DSP audio object
 * @prop_id: ID value of the property
 * @value: Pointer where to set the value
 *
 * Retrieve property value from audio plugin
 *
 * Returns: TRUE if property was known, otherwise FALSE
 */
gboolean
gst_dspaudio_get_property (GstDSPAudio *dsp, guint prop_id, GValue *value)
{
  gboolean ret = TRUE;

  g_return_val_if_fail (dsp, FALSE);

  switch (prop_id) {
    case DSPAUDIO_PROP_VOLUME:
      g_value_set_uint (value, dsp->volume);
      break;
    case DSPAUDIO_PROP_VOLUME_FLOAT:
      g_value_set_double (value, dsp->volume_f);
      break;
    case DSPAUDIO_PROP_PANNING:
      g_value_set_int (value, dsp->panning);
      break;
    case DSPAUDIO_PROP_MUTE:
      g_value_set_boolean (value, dsp->mute);
      break;
    case DSPAUDIO_PROP_PRIORITY:
      g_value_set_int (value, dsp->priority);
      break;
    case DSPAUDIO_PROP_DTXMODE:
      g_value_set_enum (value, dsp->dtxmode);
      break;
    default:
      ret = FALSE;
      break;
  }
  return ret;
}


/*
 * gst_dspaudio_install_properties:
 * @klass: The object where to set the properties
 *
 * Adds default properties that are common to all DSP audio elements
 */
void
gst_dspaudio_install_general_properties (GObjectClass *klass)
{
  g_object_class_install_property (klass, DSPAUDIO_PROP_MUTE,
      g_param_spec_boolean ("mute", "Mute on/off",
          "Mutes or unmutes the stream",
          FALSE, G_PARAM_READWRITE));
}


/*
 * gst_dspaudio_install_src_properties:
 * @klass: The object where to set the properties
 *
 * Add source-element specific properties and general properties
 */
void
gst_dspaudio_install_src_properties (GObjectClass *klass)
{
  // Install general properties first
  gst_dspaudio_install_general_properties (klass);

  g_object_class_install_property (klass, DSPAUDIO_PROP_DTXMODE,
      g_param_spec_enum ("dtx", "DTX mode",
          "Discontinuous transmission mode",
          GST_TYPE_DSPAUDIO_DTX_MODE, GST_DSPAUDIO_DTX_MODE_OFF,
          G_PARAM_READWRITE));
}


/*
 * gst_dspaudio_install_sink_properties:
 * @klass: The object where to set the properties
 *
 * Add sink-element specific properties and general properties
 */
void
gst_dspaudio_install_sink_properties (GObjectClass *klass)
{
  // Install general properties first
  gst_dspaudio_install_general_properties (klass);

  g_object_class_install_property (klass, DSPAUDIO_PROP_VOLUME,
                                   g_param_spec_uint ("volume", "Stream volume level",
                                       "Defines the volume level of the stream (Integer)", 0,
                                       DSP_VOLUME_MAX_INT, DSP_VOLUME_DEFAULT_INT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPAUDIO_PROP_VOLUME_FLOAT,
                                   g_param_spec_double ("fvolume", "Volume",
                                       "Defines the volume level of the stream (Float)", 0.0,
                                       DSP_VOLUME_MAX_FLOAT, DSP_VOLUME_DEFAULT_FLOAT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPAUDIO_PROP_PANNING,
                                   g_param_spec_int ("panning", "Audio balance",
                                       "Defines the balance between left/right channels",
                                       DSP_PANNING_MIN, DSP_PANNING_MAX, DSP_PANNING_DEFAULT,
                                       G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPAUDIO_PROP_PRIORITY,
                                   g_param_spec_int ("priority", "Stream priority",
                                       "Defines the priority value of the stream",
                                       DSP_PRIORITY_MIN, DSP_PRIORITY_MAX, DSP_PRIORITY_DEFAULT,
                                       G_PARAM_READWRITE));
}


/*
 * gst_dspaudio_aep_initialize:
 * @dsp: DSP audio object
 *
 * Initializes DSP AEP node. This needs to be called before
 * any other method when using DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_aep_initialize (GstDSPAudio *dsp)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  if (gst_dspaudio_open_node (&dsp->aep, AEP_DEVICE_NAME) == FALSE) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  GST_DEBUG ("AEP opened");
  return TRUE;
}


/*
 * gst_dspaudio_aep_close:
 * @dsp: DSP audio object
 *
 * Closes AEP node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_aep_close (GstDSPAudio *dsp)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  return gst_dspaudio_close_node (&dsp->aep);
}


/*
 * gst_dspaudio_ed_microphone:
 * @dsp: DSPAudio object
 * @mic_enable: new mic status
 *
 * Enables or disables microphone input
 *
 * Returns: TRUE if operation was successful
 */
gboolean
gst_dspaudio_ed_microphone (GstDSPAudio *dsp, gboolean mic_enable)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  /* This used to access the old sysfs interface which could be written
   * to twice without problems. Is it possible that someone might do just
   * that (could result in a double resource allocation, and mic being
   * left in the powered-on state until the plugin exits, if not also
   * disabled twice) */
  gst_dspaudio_mic_ctl (dsp, mic_enable);

  /* Always return true (should continue anyway if resource manager is
   * not responding. This is to let the plugins work if RM is not in use
   * and mic power is on by default. Might be changed in future?) */
  return TRUE;
}


/*
 * gst_dspaudio_default_voice_frame_handler:
 * @mmap_ptr: Pointer to audio frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * Default handler for voice frames. It just creates a GstBuffer and copies
 * the audio frame there. Some audio formats (e.g. G.729) need some processing
 * of the frame before it can be passed forward and they can override this
 * handler with their own.
 *
 * Returns: New GstBuffer containing the audio frame
 */
static GstBuffer *
gst_dspaudio_default_voice_frame_handler (short int *mmap_ptr, guint fsize)
{
    GstBuffer *newbuf = NULL;

    if (fsize) {
        newbuf = gst_buffer_new_and_alloc (fsize);
        memcpy (GST_BUFFER_DATA (newbuf), mmap_ptr, fsize);
    }
    return newbuf;
}


/*
 * gst_dspaudio_default_sid_frame_handler:
 * @mmap_ptr: Pointer to audio frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * Default handler for SID_UPDATE frames. Converts the DSP frame representation
 * into G.711 CN frame format. This also sets GST_BUFFER_FLAG_LAST flag in
 * in the buffer to mark it as CN buffer.
 *
 * Returns: New GstBuffer containing the CN frame
 */
static GstBuffer *
gst_dspaudio_default_sid_frame_handler (short int *mmap_ptr, guint fsize)
{
  /* Currently the coefficients seem to cause distorted noise, so 
     they are not used. Only the "level" value is sent */

#if 0
    gint output_cn_size = DSP_ILBC_SID_UPDATE_FRAMESIZE / sizeof (short int);
    GstBuffer *newbuf = gst_buffer_new_and_alloc (output_cn_size + 2);
    char *data = GST_BUFFER_DATA (newbuf);
    gint i;

    /* Do transformation into G.711 appendix II A format */
    for (i=0; i < output_cn_size; i++) {
        data[i] = (char) mmap_ptr[i];
    }
#else
    GstBuffer *newbuf = gst_buffer_new_and_alloc (1);
    char *data = GST_BUFFER_DATA (newbuf);
    data[0] = mmap_ptr[0]; /* Copy only the level value */
#endif

    GST_BUFFER_FLAG_SET (newbuf, GST_BUFFER_FLAG_LAST);
    return newbuf;
}


/*
 * gst_dspaudio_clock_change_state:
 * @dsp: DSPAudio object
 * @transition: what state change to perform
 *
 * Informs the clocking system about a state change
 *
 * Returns: GstStateChangeReturn
 */
GstStateChangeReturn
gst_dspaudio_clock_change_state (GstDSPAudio *dsp, GstStateChange transition)
{
  switch (transition) {

  case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
    /* Disable interpolating clock time using system clock *
     * until next clock time is read from dsp */
    dsp->last_dsp_update = 0;
    break;
  case GST_STATE_CHANGE_NULL_TO_READY:
    dsp->clock_time = 0;
    dsp->last_dsp_update = 0;
  default:
    break;
  }

  return GST_STATE_CHANGE_SUCCESS;
}


/*
 * gst_dspaudio_borrow_lock:
 *
 */
void
gst_dspaudio_borrow_lock (GstDSPAudio *dsp)
{
  if (g_mutex_trylock (dsp->dsp_mutex) == FALSE) {
    char tmp = PIPE_CMD_NEED_LOCK;
    write (dsp->pipe_fds[1], &tmp, 1);
    g_mutex_lock (dsp->dsp_mutex);
  }
}


/*
 * gst_dspaudio_return_lock:
 *
 */
void
gst_dspaudio_return_lock (GstDSPAudio *dsp)
{
  gst_dspaudio_flush_cmd_queue (dsp);
  g_cond_broadcast (dsp->dsp_cond);
  g_mutex_unlock (dsp->dsp_mutex);
}


/*
 * gst_dspaudio_get_info:
 * @dsp: DSPAudio object
 *
 * Informs data reading functions that they should free the dsp mutex, since
 * someone wants to read the clock value from DSP. After this the clock
 * value is retrieved with DSP_CMD_STATE call. This needs to be called
 * if g_mutex_trylock (dsp_mutex) fails.
 *
 * Returns: TRUE if the clock time was successfully retrieved.
 */
gboolean
gst_dspaudio_get_info (GstDSPAudio *dsp)
{
  gboolean ret = FALSE;

  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

//   if (!dsp->interrupted) {
    gst_dspaudio_borrow_lock (dsp);
    if (gst_dspaudio_get_info_unlocked (dsp)) {
      if (gst_dspaudio_read_cmd (dsp, 0) == DSP_WAIT_OK) {
        ret = TRUE;
      }
    }
    gst_dspaudio_return_lock (dsp);
//   }
  return ret;
}


/*
 * gst_dspaudio_info_to_time:
 * @dsp: DSPAudio object
 *
 * Convert DSP info structure into GstClockTime.
 *
 * Returns: GstClockTime representing DSP time
 */
GstClockTime
gst_dspaudio_info_to_time (GstDSPAudio *dsp)
{
  guint64 time64 = 0;
#ifdef HAVE_DSP_H
  time64 = (dsp->info.samples_played_high << 16) |
      (dsp->info.samples_played_high >> 16);

  time64 <<= 32;

  time64 |= ((dsp->info.samples_played_low << 16 ) |
      (dsp->info.samples_played_low >> 16 ));
#endif
  return time64 * GST_MSECOND;
}

/*
 * gst_dspaudio_get_dsp_clock:
 * @dsp: DSPAudio object
 * @use_jitter: should the jitter correction be used
 *
 * Implementation to calculate clock based on DSP Clock
 *
 * Returns: Clock time
 */
GstClockTime
gst_dspaudio_get_dsp_clock (GstDSPAudio *dsp, gboolean use_jitter)
{
  GstClockTime systime_abs = 0, ret = dsp->clock_time;
  GTimeVal systime;

  // Using dsp clock only after setting dsp to playing state.
  if (dsp->mode == DSP_MODE_PLAYING || dsp->mode == DSP_MODE_EOS) {

    g_get_current_time (&systime);
    systime_abs = GST_TIMEVAL_TO_TIME (systime);

    //GST_LOG ("DSPAUDIO elapsed time since last update: %lld ms",
    //        ((systime_abs - dsp->last_dsp_update)/GST_MSECOND));

    // This function might get called from multiple threads simultaneously
    // so we need to restrict the access to DSP clock reading
    gboolean got_lock = g_mutex_trylock (dsp->clock_mutex);

    if (dsp->mode == DSP_MODE_PLAYING &&
        systime_abs > dsp->last_dsp_update + (50 * GST_MSECOND) &&
        got_lock)
    {
      // Update clock time from dsp
      // If the time query fails, this will return the "old" DSP time.
      // We will not interpolate it by using system time in this case.
      ret = gst_dspaudio_get_dsp_time (dsp);
      if (ret != GST_CLOCK_TIME_NONE) {
        dsp->last_dsp_update = systime_abs;
      }
      else {
          ret = dsp->clock_time;
      }
    }
    else if (dsp->last_dsp_update) {
      // Update clock time using system clock
      ret = dsp->clock_time + systime_abs - dsp->last_dsp_update;
    }
    if (got_lock) g_mutex_unlock (dsp->clock_mutex);
  }
  GST_LOG ("DSP clock: %"GST_TIME_FORMAT"", GST_TIME_ARGS (ret));
  return ret;
}


/*
 * gst_dspaudio_create_track_list:
 * @dsp: DSPAudio object
 *
 * Creates a mixer track list. GstMixer interface needs this.
 *
 * Returns: GList containing one GstMixerTrack object.
 */
const GList *
gst_dspaudio_create_track_list (GstDSPAudio *dsp)
{
  if (dsp->tracklist == NULL) {
    GstMixerTrack *track;
    track = (GstMixerTrack *) g_object_new (GST_TYPE_MIXER_TRACK, NULL);
    GST_INFO ("created new mixer track %p", track);
    track->flags = GST_MIXER_TRACK_OUTPUT;
    track->num_channels = 2;                // Left and right channel
    track->label = g_strdup ("Master");
    track->min_volume = 0;
    track->max_volume = DSP_VOLUME_MAX_INT;
    dsp->tracklist = g_list_append (dsp->tracklist, track);
  }
  return dsp->tracklist;
}


/*
 * gst_dspaudio_dtmf_start:
 * @dsp: DSP audio object
 * @tone_char: DTMF tone character to be sent (start) or -1 (stop)
 *
 * Send DTMF_START command to DSP
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_dtmf_tone_start (GstDSPAudio *dsp, gint tone_id)
{
  g_return_val_if_fail (dsp, FALSE);

  // FIXME: do we need to _borrow() this?
  g_mutex_lock (dsp->dsp_mutex);
  gboolean ret = gst_dspaudio_dtmf_tone_start_unlocked (dsp, tone_id);
  g_mutex_unlock (dsp->dsp_mutex);
  return ret;
}


/*
 * gst_dspaudio_dtmf_stop:
 * @dsp: DSP audio object
 *
 * Send DTMF_STOP command to DSP
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_dtmf_tone_stop (GstDSPAudio *dsp)
{
  g_return_val_if_fail (dsp, FALSE);

  // FIXME: do we need to _borrow() this?
  g_mutex_lock (dsp->dsp_mutex);
  gboolean ret = gst_dspaudio_dtmf_tone_stop_unlocked (dsp);
  g_mutex_unlock (dsp->dsp_mutex);
  return ret;
}



/*
 * gst_dspaudio_check_upstream_event:
 * @dsp: DSP audio object
 * @event: GstEvent to be checked
 *
 * Check if the event is DTMF event
 *
 * Returns: TRUE if the event was handled, otherwise FALSE.
 */

gboolean
gst_dspaudio_check_upstream_event (GstDSPAudio *dsp, GstEvent *event)
{
  g_return_val_if_fail (dsp, FALSE);
  g_return_val_if_fail (dsp->mode != DSP_MODE_ERROR, FALSE);

  const GstStructure *structure = gst_event_get_structure (event);
  if(structure && gst_structure_has_name (structure, "dtmf-event")) {
    gboolean start = FALSE;
    gint event_type;

    GST_DEBUG ("got DTMF event");
    if (gst_structure_get_int (structure, "type", &event_type) &&
        gst_structure_get_boolean (structure, "start", &start) &&
        event_type == 1)
    {
      guint method = 0;
      if (gst_structure_get_int (structure, "method", &method) && method != 2) {
          GST_DEBUG ("Wrong DTMF method (not in-band)");
          return FALSE;
      }
      g_mutex_lock (dsp->dsp_mutex);
      if(start) {
          gint tone_id = -1;
          if(gst_structure_get_int (structure, "number", &tone_id)) {
              gst_dspaudio_dtmf_tone_start_unlocked (dsp, tone_id);
          }
      }
      else {
          gst_dspaudio_dtmf_tone_stop_unlocked (dsp);
      }
      g_mutex_unlock (dsp->dsp_mutex);
      GST_LOG ("DTMF event handled");
      return TRUE;
    }
  }
  return FALSE;
}

/*
 * gst_dspaudio_round_bsize:
 * @bsize: value to be rounded
 * @min: lower limit of rounding
 * @max: upper limit of rounding
 *
 * Rounds the blocksize to be dividable by 80 (that is 10ms in time)
 * and makes sure that it will be between certain limits
 *
 * Returns: Rounded value
 */
guint
gst_dspaudio_round_bsize(guint bsize, guint min, guint max)
{
  guint multiplier = bsize / 80;
  guint retval = multiplier * 80;
  if(retval < min) return min;
  if(retval > max) return max;
  return retval;
}


/*
 * This function is there so that caps negotiation is done properly
 */
gboolean
gst_dspaudio_activate_push (GstPad * pad, gboolean active)
{
  GstCaps *thiscaps;
  GstCaps *caps = NULL;
  GstCaps *peercaps = NULL;
  gboolean result = FALSE;

  GST_INFO ("activate push\n");

  if (!active)
    return TRUE;

  /* first see what is possible on our source pad */
  thiscaps = gst_pad_get_caps (pad);
  GST_LOG ("caps of src: %" GST_PTR_FORMAT, thiscaps);
  /* nothing or anything is allowed, we're done */
  if (thiscaps == NULL)
    goto no_nego_needed;

  /* get the peer caps */
  peercaps = gst_pad_peer_get_caps (pad);
  GST_LOG ("caps of peer: %" GST_PTR_FORMAT, peercaps);
  if (peercaps) {
    GstCaps *icaps;

    /* get intersection */
    icaps = gst_caps_intersect (thiscaps, peercaps);
    GST_LOG ("intersect: %" GST_PTR_FORMAT, icaps);
    gst_caps_unref (thiscaps);
    gst_caps_unref (peercaps);
    if (icaps) {
      /* take first (and best, since they are sorted) possibility */
      caps = gst_caps_copy_nth (icaps, 0);
      gst_caps_unref (icaps);
    }
  } else {
    /* no peer, work with our own caps then */
    caps = thiscaps;
  }
  if (caps) {
    caps = gst_caps_make_writable (caps);
    gst_caps_truncate (caps);

    /* now fixate */
    if (!gst_caps_is_empty (caps)) {
      gst_pad_fixate_caps (pad, caps);
      GST_LOG ("fixated to: %" GST_PTR_FORMAT, caps);

      if (gst_caps_is_any (caps)) {
        /* hmm, still anything, so element can do anything and
         * nego is not needed */
        result = TRUE;
      } else if (gst_caps_is_fixed (caps)) {
        /* yay, fixed caps, use those then */
        gst_pad_set_caps (pad, caps);
        result = TRUE;
      }
    }
    gst_caps_unref (caps);
  }
  return result;

no_nego_needed:
  {
    GST_INFO ("no negotiation needed");
    if (thiscaps)
      gst_caps_unref (thiscaps);
    return TRUE;
  }
}

