/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <string.h>
#include <time.h>
#include <unistd.h>
#include <gst/gst.h>
#include <sys/time.h>

#include "gstdspamrsink.h"
#include "dspmixer.h"

#define MULTIPLE_PACKETS 10

#define AMR_FPS 50
#define AMR_TPF (GST_SECOND/AMR_FPS)

// If we find more than AVG_LEN_CALC_THRESHOLD frames in calc_length() function,
// we will start using an estimation algorithm instead of accurate length
// 5 minutes seems to be good value :)

#define AVG_LEN_CALC_THRESHOLD (5 * 60 * AMR_FPS)

GST_DEBUG_CATEGORY_STATIC (dspamr_debug);
#define GST_CAT_DEFAULT dspamr_debug

static const short int FrameLength[2][16] =
{
  {13, 14, 16, 18, 20, 21, 27, 32,  6, 1, 1, 1, 1, 1, 1, 1},    // Narrow-band
  {18, 24, 33, 37, 41, 47, 51, 59, 61, 6, 6, 1, 1, 1, 1, 1}     // Wideband
};

static const gchar *devname[2] =
{
  "/dev/dsptask/amrnb",
  "/dev/dsptask/amrwb"
};


static GstStaticPadTemplate dspamrsink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/x-amr-nb-sh; "
                      "audio/x-amr-wb-sh; "
                      "audio/AMR-WB; "
                      "audio/AMR ") );


// Function prototypes

static void gst_dspamrsink_class_init (GstDSPAMRSinkClass * klass);
static void gst_dspamrsink_base_init (gpointer g_class);
static void gst_dspamrsink_init (GstDSPAMRSink * dspamrsink, GstDSPAMRSinkClass *g_class);

static void gst_dspamrsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspamrsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);
static const GstQueryType *gst_dspamrsink_get_query_types (GstElement * element);
static gboolean gst_dspamrsink_query (GstElement * element, GstQuery *query);
static GstClock *gst_dspamrsink_provide_clock (GstElement * elem);
static GstClockTime gst_dspamrsink_get_time (GstClock * clock, gpointer sink);
static gboolean gst_dspamrsink_element_event (GstElement * element, GstEvent *event);
static gboolean gst_dspamrsink_initialize(GstDSPAMRSink *dsp);
static GstStateChangeReturn gst_dspamrsink_change_state(GstElement * element, GstStateChange transition);
static gboolean gst_dspamrsink_paused_to_playing(GstDSPAMRSink *dsp);

// For basesink
static GstFlowReturn gst_dspamrsink_preroll(GstBaseSink *sink, GstBuffer * buffer);
static GstFlowReturn gst_dspamrsink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspamrsink_event (GstBaseSink *sink, GstEvent *event);
static gboolean gst_dspamrsink_setcaps (GstBaseSink * basesink, GstCaps * caps);
static GstStateChangeReturn gst_dspamrsink_async_play(GstBaseSink *sink);
static gboolean gst_dspamrsink_unlock (GstBaseSink *sink);
static gboolean gst_dspamrsink_unlock_stop (GstBaseSink *sink);


// static void gst_dspamrsink_calculate_length (GstDSPAMRSink *dsp);


GST_IMPLEMENT_DSPMIXER_METHODS (GstDSPAMRSink, gst_dspamrsink);


/*
 * gst_dspamrsink_do_init:
 * @type:
 *
 *
 */
static void
gst_dspamrsink_do_init (GType type)
{
  static const GInterfaceInfo iface_info = {
    (GInterfaceInitFunc) gst_dspamrsink_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo mixer_iface_info = {
    (GInterfaceInitFunc) gst_dspamrsink_mixer_interface_init,
    NULL,
    NULL,
  };

  gst_dspaudio_init();
  GST_DEBUG_CATEGORY_INIT (dspamr_debug, "dspamrsink", 0,
                           "DSP AMR NB/WB audio sink");
  g_type_add_interface_static (type, GST_TYPE_IMPLEMENTS_INTERFACE, &iface_info);
  g_type_add_interface_static (type, GST_TYPE_MIXER, &mixer_iface_info);
}

GST_BOILERPLATE_FULL (GstDSPAMRSink, gst_dspamrsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, gst_dspamrsink_do_init);


/*
 * gst_dspamrsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes AMR sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspamrsink_dispose (GObject * object)
{
  GstDSPAMRSink *dsp = (GstDSPAMRSink *) object;

  if(dsp->clock) {
    gst_object_unref(dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }
  g_object_unref(dsp->adapter);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspamrsink_base_init:
 * @g_class: AMR sink class
 *
 * Does the basic initialization of AMR sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspamrsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP AMR Sink",
    "Sink/Audio",
    "AMR NB/WB audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspamrsink_sink_template));
  gst_element_class_set_details (gstelement_class, &details);

}


/*
 * gst_dspamrsink_class_init:
 * @klass: AMR Sink class object
 *
 * Initializes AMR sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspamrsink_class_init (GstDSPAMRSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspamrsink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspamrsink_get_property);
  gobject_class->dispose      = GST_DEBUG_FUNCPTR (gst_dspamrsink_dispose);

  gstelement_class->provide_clock   = GST_DEBUG_FUNCPTR (gst_dspamrsink_provide_clock);
  gstelement_class->change_state    = GST_DEBUG_FUNCPTR (gst_dspamrsink_change_state);
  gstelement_class->get_query_types = GST_DEBUG_FUNCPTR (gst_dspamrsink_get_query_types);
  gstelement_class->query           = GST_DEBUG_FUNCPTR (gst_dspamrsink_query);
  gstelement_class->send_event      = GST_DEBUG_FUNCPTR (gst_dspamrsink_element_event);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_sink_properties (G_OBJECT_CLASS (klass));

  gstbase_sink_class->preroll    = GST_DEBUG_FUNCPTR (gst_dspamrsink_preroll);
  gstbase_sink_class->render     = GST_DEBUG_FUNCPTR (gst_dspamrsink_render);
  gstbase_sink_class->set_caps   = GST_DEBUG_FUNCPTR (gst_dspamrsink_setcaps);
  gstbase_sink_class->event      = GST_DEBUG_FUNCPTR (gst_dspamrsink_event);
  gstbase_sink_class->async_play = GST_DEBUG_FUNCPTR (gst_dspamrsink_async_play);
  gstbase_sink_class->unlock     = GST_DEBUG_FUNCPTR (gst_dspamrsink_unlock);
  gstbase_sink_class->unlock_stop= GST_DEBUG_FUNCPTR (gst_dspamrsink_unlock_stop);
}


/*
 * gst_dspamrsink_init:
 * @dsp: DSP AMR sink object
 * @g_class: AMR sink Class object
 *
 * Initializes the given AMR sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspamrsink_init (GstDSPAMRSink * dspamrsink, GstDSPAMRSinkClass *g_class)
{
  GstBaseSink *bsink = GST_BASE_SINK(dspamrsink);

  dspamrsink->sinkpad = bsink->sinkpad;
  dspamrsink->audio = gst_dspaudio_new();
  dspamrsink->adapter = gst_adapter_new();

  dspamrsink->clock = gst_audio_clock_new ("DSPClock",
                                           (GstAudioClockGetTimeFunc) gst_dspamrsink_get_time,
                                           dspamrsink);
  // DSP sinks do not need syncing
  gst_base_sink_set_sync (GST_BASE_SINK (dspamrsink), FALSE);
}


/*
 * gst_dspamrsink_setcaps:
 * @basesink: GstBaseSink
 * @caps: GstCaps that were offered by the connecting element
 *
 * Returns: TRUE if caps can be accepted, otherwise FALSE
 */
static gboolean
gst_dspamrsink_setcaps(GstBaseSink * basesink, GstCaps * caps)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (basesink);
  GST_DEBUG("gst_dspamrsink_setcaps");

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if (!strncmp(mimetype, "audio/x-amr-wb-sh", 17) ||
      !strncmp(mimetype, "audio/AMR-WB", 12)) {
    dsp->read_mime_header = FALSE;
    dsp->wide = 1;
  }
  else if (!strncmp(mimetype, "audio/x-amr-nb-sh", 17) ||
           !strncmp(mimetype, "audio/AMR", 9)) {
    dsp->read_mime_header = FALSE;
    dsp->wide = 0;
  }
  else {
    GST_WARNING("Unknown caps");
    return FALSE;
  }

  dsp->bandindex = dsp->wide ? 1 : 0;
  return TRUE;
}


/*
 * gst_dspamrsink_fill_input_buffer;
 * @dsp: GstDSPAMRSink to be handled
 *
 * This method fills the MMAP buffer with appropriate amount
 * of data according to the header information.
 *
 * Returns: TRUE if buffer was filled, otherwise FALSE
 */
static gboolean
gst_dspamrsink_fill_input_buffer(GstDSPAMRSink *dsp)
{
  guint frame_index, frame_length;
  const guint8 *data;
  char *outbuf;
  guint avail;

  avail = gst_adapter_available(dsp->adapter);

  while (dsp->packet_index < MULTIPLE_PACKETS) {

    // Read frame header
    if(avail < 1) {
      return FALSE;
    }

    data = gst_adapter_peek(dsp->adapter, 1);

    // Check the sync!
    if( (data[0] & 0x83) != 0 ) {
      gst_adapter_flush(dsp->adapter, 1);
      GST_LOG("Find sync curr_frame = %d", dsp->current_framecount);
      avail--;
      continue;
    }

    frame_index = (data[0] >> 3) & 0x0F;
    frame_length = FrameLength[dsp->bandindex][frame_index];
    if(avail < frame_length+1) {
      return FALSE;
    }

    data = gst_adapter_peek(dsp->adapter, frame_length+1);

    // Check the next frame header. If a header is not found, this
    // is not a real frame - it just looks like a frame header
    if( (data[frame_length] & 0x83) != 0 ) {
      gst_adapter_flush(dsp->adapter, 1);
      //GST_INFO("FIND SYNC curr_frame = %d", dsp->current_framecount);
      avail--;
      continue;
    }

    outbuf = dsp->audio->codec.mmap_buffer +
        (dsp->packet_index * (60-dsp->bandindex)) * sizeof(short int);

    // Normal frame must start with 0x0000
    *outbuf++ = 0x00;
    *outbuf++ = 0x00;

    memcpy(outbuf, data, frame_length);
    dsp->current_bytes += frame_length;
    dsp->current_framecount++;
    avail -= frame_length;
    dsp->packet_index++;
    gst_adapter_flush(dsp->adapter, frame_length);
  }
  return TRUE;
}


/*
 * gst_dspamrsink_perform_seek:
 *
 */
static gboolean
gst_dspamrsink_perform_seek(GstDSPAMRSink *dsp, gint64 seek_pos)
{
  guint64 pos_in_frames = seek_pos / AMR_TPF;
  gint64 pos = -1;
  GstFormat fmt = GST_FORMAT_BYTES;

  if(dsp->total_framecount) {
    pos = dsp->total_length_bytes * pos_in_frames / dsp->total_framecount;
  }
  else if (dsp->current_framecount) {
    pos = pos_in_frames * (dsp->current_bytes / dsp->current_framecount);
  }

  if(pos != -1) {
    dsp->current_framecount = pos_in_frames;
    dsp->current_bytes = pos;

    GST_DEBUG("Seeking position: %lld bytes (%lld secs)", pos, seek_pos/GST_SECOND);

    GstEvent *new_event = gst_event_new_seek(1.0, fmt, GST_SEEK_FLAG_FLUSH,
                                              GST_SEEK_TYPE_SET, pos,
                                              GST_SEEK_TYPE_NONE,
                                              GST_CLOCK_TIME_NONE);

    if(gst_pad_push_event (dsp->sinkpad, new_event)) {
      GST_DEBUG("Seek successful");
      gst_adapter_clear(dsp->adapter);
      dsp->packet_index  = 0;
      return TRUE;
    }
  }
  return FALSE;
}


/*
 * gst_dspamrsink_render:
 * @sink:
 * @buffer:
 *
 */
static GstFlowReturn
gst_dspamrsink_render(GstBaseSink *sink, GstBuffer * buffer)
{
  DSPWaitStatus ret = DSP_WAIT_OK;
  GstFlowReturn retval = GST_FLOW_OK;
  gboolean wret = TRUE;

  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);

  gst_buffer_ref(buffer);
  gst_adapter_push(dsp->adapter, buffer);
  GST_LOG("Got buffer of %d bytes, total %d",
           GST_BUFFER_SIZE(buffer), gst_adapter_available(dsp->adapter));

  // Send initial PLAY command if not done yet
  if(dsp->audio->mode < DSP_MODE_PLAYING) {
    if (gst_dspamrsink_paused_to_playing (dsp) == FALSE) {
      GST_ELEMENT_ERROR (sink, STREAM, TYPE_NOT_FOUND, (NULL),
                         ("gst_dspamrsink_paused_to_playing"));
      GST_WARNING ("Render returning gst_flow_error");
      return GST_FLOW_ERROR;
    }

    GstClockTime latency = gst_base_sink_get_latency (sink);
    if (latency) {
      GST_INFO ("Latency: %"GST_TIME_FORMAT, GST_TIME_ARGS (latency));
      g_usleep (latency / GST_USECOND);
    }
  }

  // Consume the incoming buffer
  while (TRUE) {

    // Wait for DATA_WRITE command
    ret = gst_dspaudio_wait_buffer (dsp->audio);

    if (ret == DSP_WAIT_OK) {
      if (gst_dspamrsink_fill_input_buffer (dsp)) {
        wret = gst_dspaudio_write_data (dsp->audio,
                                        MULTIPLE_PACKETS * (60-dsp->bandindex),
                                        FALSE);
        dsp->packet_index = 0;
        GST_LOG ("Data written to DSP");
      }
      else {
        GST_LOG ("Not enough data. Only %d bytes",
                 gst_adapter_available (dsp->adapter));
        break;
      }
    }
    else if (ret == DSP_WAIT_INTERRUPT) {
      GST_DEBUG ("DSP_WAIT_INTERRUPT");
      retval = gst_base_sink_wait_preroll (sink);
      if (retval != GST_FLOW_OK)
        break;
      continue;
    }

    if (ret == DSP_WAIT_ERROR || wret == FALSE) {
      GST_WARNING ("error occured");
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                        ("error cmd: %d status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status));
      retval = GST_FLOW_ERROR;
      break;
    }
  }

  GST_LOG ("render exit");
  return retval;
}


/*
 * gst_dspamrsink_initialize:
 * @dsp: DSP AMR sink object
 *
 * Initializes the underlying audio stream object and sets up the
 * AMR audio parameters in DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspamrsink_initialize (GstDSPAMRSink *dsp)
{
  guint format,sample_rate,channels;

  // Read the junk out...
  gst_dspaudio_flush (dsp->audio);

  if(!gst_dspaudio_map_channels (1, &channels)) {
    GST_ERROR("Not supported channel number : 1!");
    return FALSE;
  }
  if(!gst_dspaudio_map_samplerate (dsp->wide?16000:8000,&sample_rate)) {
    GST_WARNING ("unsupported sample rate: 16000/8000");
    return FALSE;
  }
  if (!gst_dspaudio_map_format (dsp->audio, GST_DSPAUDIO_CODEC_AMR, G_LITTLE_ENDIAN,
    TRUE, 2, &format, NULL)) {
    GST_ERROR("Cannot determine audio format");
    return FALSE;
  }

  
  if (!gst_dspaudio_set_audioparams (dsp->audio,
    format, sample_rate, channels)) {
    return FALSE;
  }
  
#ifdef HAVE_DSP_H
  unsigned short mode_buf[2];
  mode_buf[0] = DSP_CMD_SET_MODE;
  mode_buf[1] = 1;  // MIME mode

  if (write (dsp->audio->codec.fd, mode_buf, 2*sizeof(short int)) < 0) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  if (gst_dspaudio_read_cmd(dsp->audio, 0) != DSP_WAIT_OK) {
    return FALSE;
  }
#endif
  dsp->audio->mode = DSP_MODE_CONFIGURED;
  GST_DEBUG ("DSP AMR configured");
  return TRUE;
}


/*
 * gst_dspamrsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspamrsink_set_property (GObject * object,
                             guint prop_id,
                             const GValue * value,
                             GParamSpec * pspec)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (object);
  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    GST_WARNING("Trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspamrsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspamrsink_get_property (GObject * object,
                             guint prop_id,
                             GValue * value,
                             GParamSpec * pspec)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (object);
  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    GST_WARNING("Trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspamrsink_get_query_types:
 * @element: GstElement that received this query
 *
 * Returns: a list of supported query types
 */
static const GstQueryType *
gst_dspamrsink_get_query_types (GstElement * element)
{
  static const GstQueryType gst_dspamrsink_query_types[] = {
    GST_QUERY_POSITION,
    GST_QUERY_DURATION,
    0
  };

  return gst_dspamrsink_query_types;
}


/*
 * gst_dspamrsink_query:
 * @element: GstElement that received this query
 * @query: GstQuery object
 *
 * Returns: TRUE, iff the query was successful.
 */
static gboolean
gst_dspamrsink_query (GstElement * element, GstQuery *query)
{
  gboolean res = FALSE;
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (element);
  GstQueryType type = GST_QUERY_TYPE (query);
  GstFormat format;
  gint64 value = 0;

  // First we check if parent class can handle the query (except LATENCY)
  if (type != GST_QUERY_LATENCY) {
    res = GST_ELEMENT_CLASS (parent_class)->query (element, query);
    if (res) {
      GST_DEBUG ("parent class handled query");
      return res;
    }
  }

  if(type == GST_QUERY_POSITION) {
    GST_DEBUG("Position query");

    gst_query_parse_position(query, &format, NULL);

    if (format != GST_FORMAT_TIME)
      return FALSE;

    value = dsp->current_framecount * AMR_TPF;
    gst_query_set_position (query, GST_FORMAT_TIME, value);
    res = TRUE;
  }
  else if(type == GST_QUERY_DURATION) {
    GST_DEBUG("Duration query");

    gst_query_parse_duration(query, &format, NULL);

    if (format != GST_FORMAT_TIME)
      return FALSE;

    if(dsp->total_framecount) {
      value = dsp->total_framecount * AMR_TPF;
      res = TRUE;
    }
    else if(dsp->total_length_bytes && dsp->current_bytes) {
//       value = dsp->avg_framecount * AMR_TPF;
      gdouble rel = (gdouble) dsp->current_bytes / dsp->total_length_bytes;
      if(rel != 0.0) {
        value = (guint64) (dsp->current_framecount * AMR_TPF / rel);
        GST_DEBUG("VBR length: %lld sec", value / GST_SECOND);
        res = TRUE;
      }
    }
    if (res) {
      gst_query_set_duration (query, GST_FORMAT_TIME, value);
    }
  }
  else if (type == GST_QUERY_LATENCY) {
    gboolean live, us_live;
    GstClockTime min_l, max_l;
    GST_DEBUG ("latency query");

    if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
      GST_DEBUG ("we are not yet negotiated, can't report latency yet");
      res = FALSE;
    }
    else if ((res = gst_base_sink_query_latency (GST_BASE_SINK_CAST (element),
              &live, &us_live, &min_l, &max_l)))
    {
      GstClockTime min_latency, max_latency;
      /* upstream is live, adjust the min_latency */
      if (us_live) {
        // we send MULTIPLE_PACKETS frames at once
        min_latency = min_l + MULTIPLE_PACKETS * AMR_TPF;
        max_latency = min_latency + (max_l == -1 ? 0 : max_l);

        GST_DEBUG ("peer min %" GST_TIME_FORMAT ", our min latency: %"
                   GST_TIME_FORMAT, GST_TIME_ARGS (min_l),
                   GST_TIME_ARGS (min_latency));
      } else {
        GST_DEBUG ("live = %d, us_live = %d", live, us_live);
        GST_DEBUG ("peer or we are not live, don't care about latency");
        min_latency = 0;
        max_latency = -1;
      }
      gst_query_set_latency (query, live, min_latency, max_latency);
    }
  }
  GST_LOG ("Query returns %d", res);
  return res;
}


/*
 * gst_dspamrsink_element_event:
 * @element: GstElement that received this event
 * @event: The GstEvent structure itself
 *
 * Returns: TRUE, iff the event was successful
 */
static gboolean
gst_dspamrsink_element_event (GstElement * element, GstEvent * event)
{
  gboolean res = FALSE;

  GstDSPAMRSink *dsp = GST_DSPAMRSINK (element);
  GstEventType type = event
                    ? GST_EVENT_TYPE (event)
                    : GST_EVENT_UNKNOWN;

  GST_DEBUG("Event %p (%d)", event, type);

  // Try if upstream element can handle the event
  gst_event_ref (event);
  res = GST_ELEMENT_CLASS (parent_class)->send_event (element, event);

  if(type == GST_EVENT_SEEK && res == FALSE) {
    gdouble rate;
    GstFormat format;
    GstSeekFlags flags;
    GstSeekType cur_type, stop_type;
    gint64 cur, stop;

    GST_DEBUG("parent failed to SEEK");

    gst_event_parse_seek (event, &rate, &format, &flags,
                          &cur_type, &cur, &stop_type, &stop);

    if(format == GST_FORMAT_TIME) {
      gst_dspamrsink_perform_seek(dsp, cur);
      res = TRUE;
    }
  }
  gst_event_unref(event);
  GST_DEBUG("Event handled");
  return res;
}


/*
 * gst_dspamrsink_event:
 *
 * This method handles events coming from basesink
 */
static gboolean
gst_dspamrsink_event (GstBaseSink *sink, GstEvent * event)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  if(type == GST_EVENT_EOS) {
    GST_DEBUG("EOS event");
    dsp->audio->mode = DSP_MODE_EOS;
  }
  else if(type == GST_EVENT_FLUSH_STOP) {
    GST_DEBUG("Flush stop");
    gst_dspaudio_discont (dsp->audio);
    gst_adapter_clear (dsp->adapter);
    dsp->packet_index = 0;
    GST_DEBUG("flush stop handled");
  }
  else {
    GST_DEBUG("other event");
  }
  return TRUE;
}


/*
 * gst_dspamrsink_check_stream:
 * @dsp: DSP AMR sink object
 *
 * Checks if the incoming stream really is either AMR-NB or
 * AMR-WB format.
 *
 * Returns: TRUE if the content was recognized, otherwise FALSE
 */
static gboolean
gst_dspamrsink_check_stream(GstDSPAMRSink *dsp, const guint8 *data)
{
  gboolean retval = FALSE;

  GST_DEBUG("check_stream: %c%c%c%c%c%c%c%c", data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7]);
  if (memcmp(data, "#!AMR-WB\n", 9) == 0) {
    dsp->wide = TRUE;
    dsp->bandindex = 1;
    retval = TRUE;
  }
  else if (memcmp(data, "#!AMR\n", 6) == 0) {
    dsp->wide = FALSE;
    dsp->bandindex = 0;
    retval = TRUE;
  }
  else if(dsp->read_mime_header == FALSE) {
    // Probably we got the band information from Caps
    retval = TRUE;
  }
  GST_DEBUG("gst_dspamrsink_check_stream: %d", retval);
  return retval;
}


/*
 * gst_dspamrsink_open:
 *
 */
static gboolean
gst_dspamrsink_open(GstDSPAMRSink *dsp, guint bandindex)
{
  if(!gst_dspaudio_open(dsp->audio, devname[dsp->bandindex])) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }
  return TRUE;
}


/*
 * gst_dspamrsink_paused_to_playing:
 *
 */
static gboolean
gst_dspamrsink_paused_to_playing(GstDSPAMRSink *dsp)
{
  GST_INFO("gst_dspamrsink_paused_to_playing");
  gst_dspaudio_play(dsp->audio);

  // Settings might have changed between PAUSE->PLAY transition
  gst_dspaudio_update_dsp_settings(dsp->audio);
  gst_adapter_flush(dsp->adapter, dsp->wide ? 9 : 6);
  return TRUE;
}


/*
 * gst_dspamrsink_query_length:
 * @dsp: GstDSPAMRSink object
 *
 * Retrieves the length of stream, in bytes. Stores the value
 * in dsp->total_length_bytes variable
 */
static void
gst_dspamrsink_query_length (GstDSPAMRSink *dsp)
{
  GstPad *peer;

  peer = gst_pad_get_peer(dsp->sinkpad);
  if(peer) {
    GstFormat fmt = GST_FORMAT_BYTES;
    gboolean ret = gst_pad_query_duration (peer, &fmt, &dsp->total_length_bytes);
    gst_object_unref(peer); // _get_peer() ref'ed it
    if(fmt != GST_FORMAT_BYTES || ret == FALSE) {
      // It gave the length in some other format or query failed
      dsp->total_length_bytes = 0;
    }
  }
  GST_DEBUG("Total file length: %lld bytes", dsp->total_length_bytes);
}


/*
 * gst_dspamrsink_preroll:
 *
 */
static GstFlowReturn
gst_dspamrsink_preroll (GstBaseSink *sink, GstBuffer * buffer)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);
  GST_DEBUG("gst_dspamrsink_preroll");

  if (dsp->audio->mode < DSP_MODE_CONFIGURED) {
    if (!gst_dspamrsink_check_stream (dsp, GST_BUFFER_DATA(buffer))) {
      GST_ELEMENT_ERROR (dsp, STREAM, TYPE_NOT_FOUND, (NULL),
                         ("gst_dspamrsink_preroll"));
      return GST_FLOW_ERROR;
    }
    if (!gst_dspamrsink_open (dsp, dsp->bandindex)) {
      return GST_FLOW_ERROR;
    }
    if (!gst_dspamrsink_initialize (dsp)) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                         ("gst_dspamrsink_initialize"));
      return GST_FLOW_ERROR;
    }
    gst_dspamrsink_query_length (dsp);
    //gst_dspamrsink_calculate_length(dsp);
  }
  GST_DEBUG ("gst_dspamrsink_preroll OK");
  return GST_FLOW_OK;
}


/*
 * gst_dspamrsink_async_play:
 *
 */
static GstStateChangeReturn
gst_dspamrsink_async_play (GstBaseSink *sink)
{
  GstDSPAMRSink *dspamrsink = GST_DSPAMRSINK (sink);
  GST_DEBUG("Async play");

  if (dspamrsink->audio->mode == DSP_MODE_PAUSED) {
    gst_dspaudio_play (dspamrsink->audio);
  }
  GST_DEBUG("sync play success");
  return GST_STATE_CHANGE_SUCCESS;
}


/*
 * gst_dspamrsink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Returns: GST_STATE_CHANGE_SUCCESS if operation was successful.
 *          Otherwise GST_STATE_CHANGE_FAILURE will be returned.
 */
static GstStateChangeReturn
gst_dspamrsink_change_state(GstElement * element, GstStateChange transition)
{
  GstDSPAMRSink *dspamrsink = GST_DSPAMRSINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  GST_DEBUG("Change_state called");

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      GST_DEBUG("GST_STATE_CHANGE_NULL_TO_READY");
      dspamrsink->read_mime_header = TRUE;
      dspamrsink->total_length_bytes = 0;
      dspamrsink->current_framecount = 0;
      dspamrsink->total_framecount = 0;
      dspamrsink->avg_framecount = 0;
      dspamrsink->current_bytes = 0;
      dspamrsink->wide = FALSE;
      dspamrsink->bandindex = 0;
      dspamrsink->packet_index = 0;
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      GST_DEBUG("GST_STATE_CHANGE_READY_TO_PAUSED");
      gst_adapter_clear (dspamrsink->adapter);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      gst_dspamrsink_async_play (GST_BASE_SINK_CAST (element));
      break;

    default:
      break;
  }

  gst_dspaudio_clock_change_state(dspamrsink->audio, transition);
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      GST_DEBUG("GST_STATE_CHANGE_PLAYING_TO_PAUSED2");
      gst_dspaudio_pause (dspamrsink->audio);
      GST_DEBUG("GST_STATE_CHANGE_PLAYING_TO_PAUSED2 OK");
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      GST_DEBUG("GST_STATE_CHANGE_PAUSED_TO_READY");
      gst_dspaudio_stop (dspamrsink->audio);
      gst_adapter_clear(dspamrsink->adapter);
      GST_DEBUG("GST_STATE_CHANGE_PAUSED_TO_READY OK");
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      GST_DEBUG("GST_STATE_CHANGE_READY_TO_NULL");
      gst_dspaudio_close(dspamrsink->audio);
      gst_dspaudio_reset(dspamrsink->audio);
      break;

    default:
      break;
  }

  return ret;
}


/*
 * gst_dspamrsink_provide_clock:
 *
 */
static GstClock *
gst_dspamrsink_provide_clock (GstElement * elem)
{
  GST_DEBUG ("gst_dspamrsink_provide_clock");
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 * gst_dspamrsink_get_time:
 *
 */
static GstClockTime
gst_dspamrsink_get_time (GstClock * clock, gpointer sink)
{
  GST_LOG ("get time");
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);
  return gst_dspaudio_get_dsp_clock (dsp->audio, FALSE);
}


/*
 * gst_dspamrsink_unlock:
 * @sink: GstBaseSink
 *
 * Interrupts the blocking :render() function.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspamrsink_unlock (GstBaseSink *sink)
{
  GST_DEBUG ("unlock");
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);
  gst_dspaudio_interrupt_render (dsp->audio);
  return TRUE;
}


/*
 * gst_dspamrsink_unlock_stop:
 * @sink: GstBaseSink
 *
 * Removes :render() unlocking.
 *
 * Returns: TRUE
 */
static gboolean
gst_dspamrsink_unlock_stop (GstBaseSink *sink)
{
  GST_DEBUG ("unlock_stop");
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);
  gst_dspaudio_remove_interrupt (dsp->audio);
  return TRUE;
}


#define AMR_LEN_CALC_BLOCKSIZE (4*1024)

/* For now this can wait until we upgrade the plugin to work
   in both pull and push mode. */

/*
static void
gst_dspamrsink_calculate_length(GstDSPAMRSink *dsp)
{
  guint frame_header, frame_index, frame_length;
  guint64 blockindex, startpos, pos, currlen = 0;
  GstBuffer *buf;
  guint8 *data;

  guint64 blocksize = AMR_LEN_CALC_BLOCKSIZE;
  GstFormat fmt = GST_FORMAT_BYTES;

#ifdef DEBUG
  struct timeval tv1, tv2, tv3;
  gettimeofday(&tv1, NULL);
#endif

// If we can not use pull_range, we can not calculate length
  if(! gst_pad_check_pull_range (dsp->sinkpad) ) {
    GST_INFO("AMR: pull_range not supported");
    return;
  }

  pos = dsp->wide ? 9 : 6;
  gst_pad_query_position(GST_PAD_PEER(dsp->sinkpad), &fmt, &startpos);
  GST_INFO("CURRENT POS: %lld", startpos);

  while(TRUE) {

    // Can we read one more byte?
    if( (pos+AMR_LEN_CALC_BLOCKSIZE) >= dsp->total_length_bytes ) {
      blocksize = dsp->total_length_bytes - pos;
    }

    if(!blocksize) break;

    if(gst_pad_pull_range(dsp->sinkpad, pos, blocksize, &buf) != GST_FLOW_OK) {
      return;
    }
    data = GST_BUFFER_DATA(buf);
    blockindex = 0;

    // Loop the data block through
    do {
      frame_header = data[blockindex];
      blockindex++;
      pos++;

      if( (frame_header & 0x83) != 0 ) {
        continue;
      }
      frame_index = (frame_header >> 3) & 0x0F;
      frame_length = FrameLength[dsp->bandindex][frame_index]; //no. of bytes to be read

      pos += frame_length-1;
      blockindex += frame_length-1;
      dsp->total_framecount++;
      currlen += frame_length;
    }
    while(blockindex < blocksize);

    gst_buffer_unref(buf);

    if(dsp->total_framecount > AVG_LEN_CALC_THRESHOLD) {
      dsp->avg_framecount = dsp->total_length_bytes / (currlen / dsp->total_framecount);
      dsp->total_framecount = 0;
      GST_INFO("Too many frames");
      break;
    }
  }
  GST_INFO("AMR: TOTAL FRAMES: %lld, AVG = %lld", dsp->total_framecount, dsp->avg_framecount);

  GstEvent *new_event = gst_event_new_seek(1.0, GST_FORMAT_BYTES,
                                           GST_SEEK_FLAG_NONE,
                                           GST_SEEK_TYPE_SET, startpos,
                                           GST_SEEK_TYPE_NONE,
                                           GST_CLOCK_TIME_NONE);

  gst_pad_push_event (dsp->sinkpad, new_event);

#ifdef DEBUG
  gettimeofday(&tv2, NULL);
  timersub(&tv2, &tv1, &tv3);
  printf("AMR length calculation take %d sec %d usec", (int) tv3.tv_sec, (int) tv3.tv_usec);
#endif
}
*/


/*
 * plugin_init:
 * @plugin: GstPlugin to be initialized
 *
 * This initializes the plugin when it is loaded by GStreamer
 *
 * Returns: Boolean value, TRUE if initialization was successful
 */
static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dspamrsink", GST_RANK_PRIMARY,
                               GST_TYPE_DSPAMRSINK);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
                   GST_VERSION_MINOR,
                   "dspamr",
                   "DSP AMR NB/WB audio sink",
                   plugin_init,
                   VERSION,
                   "LGPL",
                   "dspaudio",
                   "http://www.nokia.com/");

