/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 * Author: Onne Gorter <onne.gorter@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include "notification.h"
#include "notification-notify.h"

typedef struct
{
  EdsNotifyContext *context;

  gint handle;

  gchar *account;
  gchar *name;
  gchar *address;
  gchar *message;

  guint is_accepted : 1;
} AuthData;

static gboolean
auth_request (gpointer data)
{
  AuthData *auth_data = data;

  g_debug ("context[%p] -> account:%s, name:%s, address:%s, message:%s",
           auth_data->context,
           auth_data->account,
           auth_data->name,
           auth_data->address,
           auth_data->message);

  return FALSE;
}

static gboolean
auth_response (gpointer data)
{
  AuthData *auth_data = data;

  g_debug ("context[%p] -> account:%s, address:%s, message:%s, accepted:%s",
           auth_data->context,
           auth_data->account,
           auth_data->address,
           auth_data->message,
           auth_data->is_accepted ? "true" : "false");

  g_signal_emit_by_name (auth_data->context, "reply",
                         auth_data->handle,
                         auth_data->is_accepted);

  return FALSE;
}

static void
auth_clean_up (gpointer data)
{
  AuthData *auth_data = data;

  g_free (auth_data->account);
  g_free (auth_data->name);
  g_free (auth_data->address);
  g_free (auth_data->message);

  g_free (auth_data);
}

static gint
eds_notify_context_notify_auth_request (EdsNotifyContext *context,
                                       gint              handle,
                                       const gchar      *account,
                                       const gchar      *name,
                                       const gchar      *address,
                                       const gchar      *message)
{
  AuthData *auth_data;
  gint res;

  auth_data = g_new (AuthData, 1);
  auth_data->context = context;
  auth_data->handle = handle;
  auth_data->account = g_strdup (account);
  auth_data->name = g_strdup (name);
  auth_data->address = g_strdup (address);
  auth_data->message = g_strdup (message);
  auth_data->is_accepted = FALSE;

  res = g_idle_add_full (G_PRIORITY_DEFAULT_IDLE,
                         auth_request,
                         auth_data,
                         auth_clean_up);

  return res;
}

static gint
eds_notify_context_notify_auth_response (EdsNotifyContext *context,
                                        gint              handle,
                                        const gchar      *account,
                                        const gchar      *address,
                                        const gchar      *message,
                                        gboolean          is_accepted)
{
  AuthData *auth_data;
  gint res;

  auth_data = g_new (AuthData, 1);
  auth_data->context = context;
  auth_data->handle = handle;
  auth_data->account = g_strdup (account);
  auth_data->address = g_strdup (address);
  auth_data->message = g_strdup (message);
  auth_data->is_accepted = is_accepted;

  res = g_idle_add_full (G_PRIORITY_DEFAULT_IDLE,
                         auth_response,
                         auth_data,
                         auth_clean_up);
  
  return res;
}

static void
eds_notify_context_notify_cancel_operations (EdsNotifyContext *context,
                                            GArray           *ids)
{
  gint i;

  for (i = 0; i < ids->len; i++)
    {
      g_source_remove (g_array_index (ids, guint, i));
    }
}

static void
eds_notify_context_iface_init (EdsNotifyContextIface *iface)
{
  iface->auth_request = eds_notify_context_notify_auth_request;
  iface->auth_response = eds_notify_context_notify_auth_response;
  iface->cancel_operations = eds_notify_context_notify_cancel_operations;
}

G_DEFINE_TYPE_WITH_CODE (EdsNotifyContextNotify,
                         eds_notify_context_notify,
                         G_TYPE_OBJECT,
                         G_IMPLEMENT_INTERFACE (EDS_TYPE_NOTIFY_CONTEXT,
                                                eds_notify_context_iface_init));

static void
eds_notify_context_notify_class_init (EdsNotifyContextNotifyClass *klass)
{
}

static void
eds_notify_context_notify_init (EdsNotifyContextNotify *notify)
{
}

EdsNotifyContext *
eds_sync_get_notify_context (void)
{
  return g_object_new (EDS_TYPE_NOTIFY_CONTEXT_NOTIFY, NULL);
}
