/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 * Author: Onne Gorter <onne.gorter@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <glib-object.h>

#include <dbus/dbus-glib-bindings.h>
#include <libtelepathy/tp-conn.h>

#include "eds-sync.h"
#include "telepathy.h"
#include "util.h"

#include "eds-sync-private.h"
#include "eds-sync-glue.h"
#include "eds-sync-marshal.h"

GQuark
eds_sync_error_quark (void)
{
  return g_quark_from_static_string ("eds-sync-error");
}

G_DEFINE_TYPE (EdsSync, eds_sync, G_TYPE_OBJECT);

enum {
	AVATAR_UPDATED,
	LAST_SIGNAL
};

static guint32 signals[LAST_SIGNAL] = { 0, };

/* forward declaration */
typedef struct _MapData MapData;

static MapData *map_data_new  (void);
static void     map_data_free (MapData *data);

static void
eds_sync_dispose (GObject *gobject)
{
  EdsSync *sync = EDS_SYNC (gobject);
  EdsSyncPrivate *priv = sync->priv;

  dbus_g_connection_flush (priv->connection);

  G_OBJECT_CLASS (eds_sync_parent_class)->dispose (gobject);
}

static void
eds_sync_finalize (GObject *gobject)
{
  EdsSync *sync = EDS_SYNC (gobject);
  EdsSyncPrivate *priv = sync->priv;

  if (priv->address_to_contact_hash)
    {
      g_hash_table_destroy (priv->address_to_contact_hash);
      priv->address_to_contact_hash = NULL;
    }

  G_OBJECT_CLASS (eds_sync_parent_class)->finalize (gobject);
}

static void
eds_sync_class_init (EdsSyncClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->dispose = eds_sync_dispose;
  gobject_class->finalize = eds_sync_finalize;

  g_type_class_add_private (klass, sizeof (EdsSyncPrivate));

  signals[AVATAR_UPDATED] =
    g_signal_new ("avatar-updated",
                  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_NO_RECURSE,
                  G_STRUCT_OFFSET (EdsSyncClass, avatar_updated),
                  NULL, NULL,
                  eds_sync_marshal_VOID__STRING_STRING,
                  G_TYPE_NONE,
                  2,
                  G_TYPE_STRING, G_TYPE_STRING);

  dbus_g_object_type_install_info (EDS_TYPE_SYNC,
                                   &dbus_glib_eds_sync_object_info);
}

static void
eds_sync_init (EdsSync *sync)
{
  EdsSyncPrivate *priv;

  GError *error = NULL;

  sync->priv = priv =
    G_TYPE_INSTANCE_GET_PRIVATE ((sync), EDS_TYPE_SYNC, EdsSyncPrivate);

  g_assert (sync->priv);

  priv->chan_count = 0;
  priv->timeout_id = 0;

  priv->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (!priv->connection) {
    g_error ("Failed to open connection to bus: %s", _ERROR_MSG (error));
  }

  /* This has table has to be here as it is for cross-connection merging */
  priv->address_to_contact_hash =
    g_hash_table_new_full (g_str_hash, g_str_equal,
                           g_free,
                           (GDestroyNotify) map_data_free);

  dbus_g_connection_register_g_object (priv->connection,
                                       "/org/freedesktop/Telepathy/ChannelHandler",
                                       G_OBJECT (sync));
}

EdsSync*
eds_sync_new (void)
{
  GError *error = NULL;
  EdsSync *sync;
  DBusGProxy *bus_proxy;
  guint32 request_name_ret;

  sync = g_object_new (EDS_TYPE_SYNC, NULL);

  /* Request the name so we only are running once */
  bus_proxy = dbus_g_proxy_new_for_name (sync->priv->connection,
                                         DBUS_SERVICE_DBUS, DBUS_PATH_DBUS,
                                         DBUS_INTERFACE_DBUS);

  if (!org_freedesktop_DBus_request_name (bus_proxy, "com.nokia.eds-sync", 0, &request_name_ret, &error))
    g_error ("failed to get request name: %s", _ERROR_MSG (error));

  g_object_unref (bus_proxy);

  if (request_name_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
    g_error ("another eds-sync is already running, exiting");
    g_object_unref (sync);
    return NULL;
  }

  queue_avatar_cache_expire (NULL, NULL);

  return sync;
}

gboolean
eds_sync_handle_channel (EdsSync      *sync,
                         const gchar  *bus_name,
                         const gchar  *object_path,
                         const gchar  *channel_type,
                         const gchar  *channel,
                         guint         handle_type,
                         guint         handle,
                         GError      **error)
{
  EdsSyncPrivate *priv;
  gboolean ret = TRUE;
  TpConn *conn;
  char *name;
  GError *internal_error;
  guint status;
  
  g_return_val_if_fail (EDS_IS_SYNC (sync), FALSE);
  g_return_val_if_fail (bus_name != NULL, FALSE);
  g_return_val_if_fail (object_path != NULL, FALSE);
  g_return_val_if_fail (channel_type != NULL, FALSE);

  priv = sync->priv;

  if (strcmp (channel_type, TP_IFACE_CHANNEL_TYPE_CONTACT_LIST) != 0 ||
      handle_type != TP_CONN_HANDLE_TYPE_LIST)
    return TRUE;

  conn = tp_conn_new_without_connect (priv->connection, bus_name, object_path, &status, error);
  if (!conn) {
    v_debug (G_STRLOC ": cannot create TpConn");
    
    if (error && (*error == NULL)) {
      g_set_error (error,
                   EDS_SYNC_ERROR, EDS_SYNC_ERROR_TELEPATHY,
                   "Cannot connect to Telepathy on bus `%s' at `%s'",
                   bus_name,
                   object_path);
    }
    
    return TRUE;
  }

  internal_error = NULL;
  if (!eds_tp_conn_inspect_handle (DBUS_G_PROXY (conn),
                                   handle_type, handle,
                                   &name, &internal_error)) {
    v_debug ("cannot inspect handle %d: %s",
             handle,
             _ERROR_MSG (internal_error));

    g_set_error (error,
                 EDS_SYNC_ERROR, EDS_SYNC_ERROR_TELEPATHY,
                 "Cannot inspect Telepathy handle %d (type: %d): %s",
                 handle,
                 handle_type,
                 _ERROR_MSG (internal_error));
    
    g_clear_error (&internal_error);

    return TRUE;
  }

  if (g_ascii_strcasecmp (name, "subscribe") == 0) {
    ret = eds_sync_telepathy_new (sync, conn,
                                  bus_name, object_path,
                                  channel_type, channel,
                                  handle_type, handle, error);
    if (ret) {
      priv->chan_count++;

      v_debug ("New telepathy connection (name:%s, channel:%s)",
               name,
               channel);

      if (priv->timeout_id) {
        g_source_remove (priv->timeout_id);
        priv->timeout_id = 0;
      }
    }
  }
  else
    g_object_unref (conn);

  g_free (name);

  return ret;
}

static gboolean
on_timeout (gpointer data)
{
  EdsSync *sync = EDS_SYNC (data);

  g_assert (sync);

  sync->priv->timeout_id = 0;

  g_object_unref (sync);

  return FALSE;
}

/* Quit when there are no more channels after 5 seconds */
#define QUIT_TIMEOUT (5*1000)

/**
 * Called by a channel handler when they close, so that eds-sync can quit when
 * it isn't being used.
 */
void
eds_sync_channel_closed (EdsSync *sync)
{
  g_return_if_fail (EDS_IS_SYNC (sync));

  sync->priv->chan_count--;

  if (sync->priv->chan_count < 1) {
    if (debug_flags == 0) {
      /* Only when not debugging, do we quit on timeout */
      sync->priv->timeout_id = g_timeout_add (QUIT_TIMEOUT, on_timeout, sync);
    } else {
      g_debug("channel count < 1, but staying around for debugging");
    }
  }
}

/**
 * Called whenever an avatar is stored on disk so that concerned applications
 * know to refresh the images
 */
void
eds_sync_signal_avatar_updated (EdsSync    *sync,
				const char *account,
				const char *token)
{
  g_signal_emit (sync, signals[AVATAR_UPDATED], 0, account, token);
}

/*
 * Account-to-Contact global map
 */

struct _MapData
{
  GSList *uids;
};

gboolean
map_data_has_uid (MapData *data, const gchar *uid)
{
  GSList *l;

  for (l = data->uids; l; l = l->next)
    {
      if (strcmp (uid, l->data) == 0)
        return TRUE;
    }

  return FALSE;
}

void
map_data_add_uid (MapData *data, const gchar *uid)
{
  if (map_data_has_uid (data, uid))
    return;

  data->uids = g_slist_prepend (data->uids, g_strdup (uid));
}

void
map_data_remove_uid (MapData *data, const gchar *uid)
{
  GSList *l;

  for (l = data->uids; l; l = l->next)
    {
      gchar *u = l->data;

      if (strcmp (uid, u) == 0)
        {
          data->uids = g_slist_remove_link (data->uids, l);
          
          g_free (u);
          g_slist_free1 (l);

          return;
        }
    }
}

void
map_data_free (MapData *data)
{
  if (G_LIKELY (data))
    {
      g_slist_foreach (data->uids, (GFunc) g_free, NULL);
      g_slist_free (data->uids);

      g_free (data);
    }
}

MapData *
map_data_new (void)
{
  return g_new0 (MapData, 1);
}

void
eds_sync_register_account (EdsSync *sync,
                           const gchar *vcard_field,
                           const gchar *address,
                           const gchar *uid)
{
  EdsSyncPrivate *priv;
  gchar *full;
  MapData *data;

  g_return_if_fail (EDS_IS_SYNC (sync));
  g_return_if_fail (vcard_field != NULL);
  g_return_if_fail (address != NULL);
  g_return_if_fail (uid != NULL);

  priv = sync->priv;

  full = g_strjoin (":", vcard_field, address, NULL);

  if (!eds_sync_has_account (sync, vcard_field, address))
    {
      data = map_data_new ();
      map_data_add_uid (data, uid);

      g_hash_table_replace (priv->address_to_contact_hash,
                            g_strdup (full), data);
    }
  else
    {
      data = g_hash_table_lookup (priv->address_to_contact_hash, full);
      g_assert (data != NULL);

      map_data_add_uid (data, uid); /* safe, will not add dupes */
    }
  
  g_free (full);
}

gboolean
eds_sync_has_account (EdsSync     *sync,
                      const gchar *vcard_field,
                      const gchar *address)
{
  EdsSyncPrivate *priv;
  gchar *full = NULL;
  gboolean res = FALSE;

  g_return_val_if_fail (EDS_IS_SYNC (sync), FALSE);
  g_return_val_if_fail (vcard_field != NULL, FALSE);
  g_return_val_if_fail (address != NULL, FALSE);

  priv = sync->priv;

  if (!priv->address_to_contact_hash)
    return FALSE;

  full = g_strjoin (":", vcard_field, address, NULL);
  res = (g_hash_table_lookup (priv->address_to_contact_hash, full) != NULL);
  g_free (full);

  return res;
}

G_CONST_RETURN gchar *
eds_sync_get_uid_for_account (EdsSync     *sync,
                              const gchar *vcard_field,
                              const gchar *address)
{
  EdsSyncPrivate *priv;
  gchar *full;
  MapData *map_data;

  g_return_val_if_fail (EDS_IS_SYNC (sync), NULL);
  g_return_val_if_fail (vcard_field != NULL, NULL);
  g_return_val_if_fail (address != NULL, NULL);

  priv = sync->priv;

  if (!eds_sync_has_account (sync, vcard_field, address))
    return NULL;

  full = g_strjoin (":", vcard_field, address, NULL);
  map_data = g_hash_table_lookup (priv->address_to_contact_hash, full);
  g_free (full);

  g_assert (map_data->uids);

  /* return the UID we added last */
  return map_data->uids->data;
}

static gboolean
remove_contact_uid (gpointer key,
                    gpointer value,
                    gpointer data)
{
  MapData *map_data = value;
  gchar *uid = data;

  if (map_data_has_uid (map_data, uid))
    {
      map_data_remove_uid (map_data, uid);

      /* if this was the last UID assigned to the
       * account, expire the whole map entry
       */
      if (!map_data->uids)
        return TRUE;
    }
  
  return FALSE;
}

void
eds_sync_remove_uid (EdsSync     *sync,
                     const gchar *uid)
{
  EdsSyncPrivate *priv;

  g_return_if_fail (EDS_IS_SYNC (sync));
  g_return_if_fail (uid != NULL);

  priv = sync->priv;

  g_hash_table_foreach_remove (priv->address_to_contact_hash,
                               remove_contact_uid,
                               (char *) uid);

}
