/* 
 * CyberLinkC AV API for Control Points
 *
 * cdidllite.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#include <string.h>

#include <cybergarage/contentdirectory/cdidllite.h>
#include <cybergarage/avdebug.h>

#include <cybergarage/util/cstring.h>

/**
 * Check the validity of a DIDL-Lite nodelist and return the direct list
 * of children for the DIDL-Lite root node
 *
 * @param node A DIDL-Lite root node
 * @param strict If TRUE, check that xmlns, xmlns:dc and xmlns:upnp match
 *               exactly the ones in the CDS specification
 *
 * @return The direct list of children if valid; otherwise NULL
 */
CgXmlNodeList* cg_upnp_av_cds_didllite_getchildren(CgXmlNodeList* nodelist,
						   BOOL strict)
{
	CgXmlNode* node = NULL;
	CgXmlNodeList* children = NULL;

	DEBUG("BEGIN: cg_upnp_av_cds_didllite_getchildren\n");
	
	if (nodelist == NULL)
	{
		DEBUG("Invalid parameters!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_getchildren\n");
		return NULL;
	}

	/* Get the first item from the node list */
	node = cg_upnp_av_cds_didllite_nodelist_gets(nodelist);
	if (node == NULL)
	{
		DEBUG("Empty nodelist!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_getchildren\n");
		return NULL;
	}
	
	/* Check the node is a <DIDL-Lite> */
	if (cg_upnp_av_cds_didllite_node_isname(node, DIDL_LITE_NAME) == TRUE)
	{
		/* Check that the node has children */
		children = cg_upnp_av_cds_didllite_node_getchildnodelist(node);
		if (children == NULL)
		{
			DEBUG("No children for <DIDL-Lite> node!\n");
		}
		else
		{
			/* Check xmlns, xmlns:dc and xmlns:upnp */
			if (cg_strstr(cg_upnp_av_cds_didllite_node_getattribute(
					node, DIDL_LITE_XMLNS), 
			    DIDL_LITE_XMLNS_URL) != 0)
			{
				DEBUG("Invalid value for xmlns!\n");
				DEBUG("xmlns: %s\n",
					cg_upnp_av_cds_didllite_node_getattribute(
						node, DIDL_LITE_XMLNS));
				if (strict)
					children = NULL;
			}
			else if (cg_strstr(
					cg_upnp_av_cds_didllite_node_getattribute(
					node, DIDL_LITE_XMLNS_DC), 
				 DIDL_LITE_XMLNS_DC_URL) != 0)
			{
				DEBUG("Invalid value for xmlns:dc!\n");
				DEBUG("xmlns:dc: %s\n",
					cg_upnp_av_cds_didllite_node_getattribute(
						node, DIDL_LITE_XMLNS_DC));
				if (strict)
					children = NULL;
			}
			else if (cg_strstr(
					cg_upnp_av_cds_didllite_node_getattribute(
					node, DIDL_LITE_XMLNS_UPNP),
				 DIDL_LITE_XMLNS_UPNP_URL) != 0)
			{
				DEBUG("Invalid value for xmlns:upnp!\n");
				DEBUG("xmlns:upnp: %s\n",
					cg_upnp_av_cds_didllite_node_getattribute(
						node, DIDL_LITE_XMLNS_UPNP));
				if (strict)
					children = NULL;
			}
		}
	}
	else
	{
		DEBUG("This is not a valid DIDL-Lite document!\n");
		children = NULL;
	}

	DEBUG("END: cg_upnp_av_cds_didllite_getchildren\n");
	
	return children;
}

/**
 * Extract the protocol parameter from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the protocol or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getprotocol(CgXmlNode* resNode)
{
	char* protocol = NULL;
	char* protocolInfo = NULL;
	CgXmlAttribute* attr = NULL;
	int tail = 0;
	int protocolInfoLen = 0;

	DEBUG("BEGIN: cg_upnp_av_cds_didllite_res_getprotocol\n");

	/* Check that we have an XML node */
	if (resNode == NULL)
	{
		DEBUG("resNode is NULL!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getprotocol\n");
		return NULL;
	}
	
	/* Check that this is a <res> node */
	if (cg_strcmp(cg_xml_node_getname(resNode), DIDL_LITE_RES) != 0)
	{
		DEBUG("This is not a <res> node!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getprotocol\n");
		return NULL;
	}

	/* Get the XML attribute */
	attr = cg_xml_node_getattribute(resNode, DIDL_LITE_RES_PROTOCOLINFO);
	if (attr == NULL)
	{
		DEBUG("Node doesn't have a \"protocolInfo\" attribute!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getprotocol\n");
		return NULL;
	}
	else
	{
		protocolInfo = cg_xml_attribute_getvalue(attr);
		protocolInfoLen = cg_strlen(protocolInfo);
	}
	
	/* Find the next delimiter ':' character */
	for (tail = 0; tail < protocolInfoLen; tail++)
	{
		if (protocolInfo[tail] == ':')
		{
			tail--; /* Don't take the colon */
			break;
		}
	}
	
	/* Copy the protocol to a new buffer */
	if (tail > 0 && tail < protocolInfoLen)
	{
		protocol = (char*) malloc(tail + 2);
		strncpy(protocol, protocolInfo, tail + 1);
		protocol[tail + 1] = '\0';
		
		DEBUG("Protocol: [%s]\n", protocol);
	}
	else
	{
		DEBUG("Protocol not found!\n");
		protocol = NULL;
	}
	
	DEBUG("END: cg_upnp_av_cds_didllite_res_getprotocol\n");

	return protocol;
}

/**
 * Extract the network parameter from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the network or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getnetwork(CgXmlNode* resNode)
{
	char* network = NULL;
	char* protocolInfo = NULL;
	CgXmlAttribute* attr = NULL;
	int head = 0;
	int tail = 0;
	int protocolInfoLen = 0;

	DEBUG("BEGIN: cg_upnp_av_cds_didllite_res_getnetwork\n");

	/* Check that we have an XML node */
	if (resNode == NULL)
	{
		DEBUG("resNode is NULL!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getnetwork\n");
		return NULL;
	}
	
	/* Check that this is a <res> node */
	if (cg_strcmp(cg_xml_node_getname(resNode), DIDL_LITE_RES) != 0)
	{
		DEBUG("This is not a <res> node!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getnetwork\n");
		return NULL;
	}

	/* Get the XML attribute */
	attr = cg_xml_node_getattribute(resNode, DIDL_LITE_RES_PROTOCOLINFO);
	if (attr == NULL)
	{
		DEBUG("Node doesn't have a \"protocolInfo\" attribute!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getnetwork\n");
		return NULL;
	}
	else
	{
		protocolInfo = cg_xml_attribute_getvalue(attr);
		protocolInfoLen = cg_strlen(protocolInfo);
	}
	
	/* Find the first delimiter ':' character */
	for (head = 0; head < protocolInfoLen; head++)
	{
		if (protocolInfo[head] == ':')
		{
			head++; /* Don't take the colon */
			break;
		}
	}

	/* Find the next delimiter ':' character */
	for (tail = head; tail < protocolInfoLen; tail++)
	{
		if (protocolInfo[tail] == ':')
		{
			tail--; /* Don't take the colon */
			break;
		}
	}

	
	/* Copy the protocol to a new buffer */
	if (head > 0 && head < protocolInfoLen &&
	    tail > 0 && tail < protocolInfoLen)
	{
		network = (char*) malloc(tail - head + 2);
		strncpy(network, protocolInfo + head, tail - head + 1);
		network[tail - head + 1] = '\0';
		
		DEBUG("Network: [%s]\n", network);
	}
	else
	{
		DEBUG("Network not found!\n");
		network = NULL;
	}
	
	DEBUG("END: cg_upnp_av_cds_didllite_res_getnetwork\n");

	return network;
}

/**
 * Extract the mime type parameter from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the mime type or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getmimetype(CgXmlNode* resNode)
{
	char* mimetype = NULL;
	char* protocolInfo = NULL;
	CgXmlAttribute* attr = NULL;
	int head = 0;
	int tail = 0;
	int protocolInfoLen = 0;
	int count = 0;
	
	DEBUG("BEGIN: cg_upnp_av_cds_didllite_res_getmimetype\n");

	/* Check that we have an XML node */
	if (resNode == NULL)
	{
		DEBUG("resNode is NULL!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getmimetype\n");
		return NULL;
	}
	
	/* Check that this is a <res> node */
	if (cg_strcmp(cg_xml_node_getname(resNode), DIDL_LITE_RES) != 0)
	{
		DEBUG("This is not a <res> node!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getmimetype\n");
		return NULL;
	}

	/* Get the XML attribute */
	attr = cg_xml_node_getattribute(resNode, DIDL_LITE_RES_PROTOCOLINFO);
	if (attr == NULL)
	{
		DEBUG("Node doesn't have a \"protocolInfo\" attribute!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getmimetype\n");
		return NULL;
	}
	else
	{
		protocolInfo = cg_xml_attribute_getvalue(attr);
		protocolInfoLen = cg_strlen(protocolInfo);
	}
	
	/* Skip the first two delimiter ':' characters */
	for (head = 0; head < protocolInfoLen; head++)
	{
		if (protocolInfo[head] == ':')
		{
			if (count == 1)
			{
				head++; /* Don't take the colon */
				break;
			}
			else
			{
				count++;
			}
		}
	}

	/* Find the next delimiter ':' character */
	for (tail = head; tail < protocolInfoLen; tail++)
	{
		if (protocolInfo[tail] == ':')
		{
			tail--; /* Don't take the colon */
			break;
		}
	}

	
	/* Copy the protocol to a new buffer */
	if (head > 0 && head < protocolInfoLen &&
	    tail > 0 && tail < protocolInfoLen)
	{
		mimetype = (char*) malloc(tail - head + 2);
		strncpy(mimetype, protocolInfo + head, tail - head + 1);
		mimetype[tail - head + 1] = '\0';
		
		DEBUG("MIME type: [%s]\n", mimetype);
	}
	else
	{
		DEBUG("MIME type not found!\n");
		mimetype = NULL;
	}
	
	DEBUG("END: cg_upnp_av_cds_didllite_res_getmimetype\n");

	return mimetype;
}

/**
 * Extract the additional info parameter from the protocolInfo field in a <res> node
 *
 * @param resNode A CgXmlNode containing a <res> element
 *
 * @return Newly created char* containing the additional info or NULL if failed
 */
char* cg_upnp_av_cds_didllite_res_getadditionalinfo(CgXmlNode* resNode)
{
	char* info = NULL;
	char* protocolInfo = NULL;
	CgXmlAttribute* attr = NULL;
	int head = 0;
	int tail = 0;
	int protocolInfoLen = 0;
	int count = 0;
	
	DEBUG("BEGIN: cg_upnp_av_cds_didllite_res_getadditionalinfo\n");

	/* Check that we have an XML node */
	if (resNode == NULL)
	{
		DEBUG("resNode is NULL!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getadditionalinfo\n");
		return NULL;
	}
	
	/* Check that this is a <res> node */
	if (cg_strcmp(cg_xml_node_getname(resNode), DIDL_LITE_RES) != 0)
	{
		DEBUG("This is not a <res> node!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getadditionalinfo\n");
		return NULL;
	}

	/* Get the XML attribute */
	attr = cg_xml_node_getattribute(resNode, DIDL_LITE_RES_PROTOCOLINFO);
	if (attr == NULL)
	{
		DEBUG("Node doesn't have a \"protocolInfo\" attribute!\n");
		DEBUG("END: cg_upnp_av_cds_didllite_res_getadditionalinfo\n");
		return NULL;
	}
	else
	{
		protocolInfo = cg_xml_attribute_getvalue(attr);
		protocolInfoLen = cg_strlen(protocolInfo);
	}
	
	/* Skip the first three delimiter ':' characters */
	for (head = 0; head < protocolInfoLen; head++)
	{
		if (protocolInfo[head] == ':')
		{
			if (count == 2)
			{
				head++; /* Don't take the colon */
				break;
			}
			else
			{
				count++;
			}
		}
	}

	/* Take everything that's left in the string */
	tail = protocolInfoLen - 1;

	/* Copy the protocol to a new buffer */
	if (head > 0 && head < protocolInfoLen &&
	    tail > 0 && tail < protocolInfoLen)
	{
		info = (char*) malloc(tail - head + 2);
		strncpy(info, protocolInfo + head, tail - head + 1);
		info[tail - head + 1] = '\0';
		
		DEBUG("Additional Info: [%s]\n", info);
	}
	else
	{
		DEBUG("Additional Info not found!\n");
		info = NULL;
	}

	DEBUG("END: cg_upnp_av_cds_didllite_res_getadditionalinfo\n");

	return info;
}
