/*
 * UPnP Browser for Maemo
 *
 * settings.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#include <libintl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include "settings.h"
#include "interface.h"
#include "now_playing.h"

extern UPnPBrowserWidgets* widgets;

#define KEY_BASE "/nokia/extras/upnpbrowser"
#define KEY_TRANSPORT_UPDATE_INTERVAL "/transport_update_interval"
#define KEY_UPDATE_SERVERS "/update_servers"
#define KEY_RESOLVE_TITLES "/resolve_titles"
#define KEY_SEND_META_DATA "/send_meta_data"

/**
 * Load settings
 */
gboolean settings_load(UPnPBrowserSettings* settings)
{
	gchar* key = NULL;
	GError* err = NULL;
	GConfClient* gconf_client = NULL;
	
	g_return_val_if_fail(settings != NULL, FALSE);
	
	gconf_client = gconf_client_get_default();
	g_return_val_if_fail(gconf_client != NULL, FALSE);
	
	/* Transport update interval */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_TRANSPORT_UPDATE_INTERVAL) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_TRANSPORT_UPDATE_INTERVAL);
	settings->transportUpdateInterval = 
		gconf_client_get_int(gconf_client, key, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_get_int: %s\n", err->message);
		return FALSE;
	}

	/* Update servers on startup */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_UPDATE_SERVERS) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_UPDATE_SERVERS);
	settings->updateServers = 
		gconf_client_get_bool(gconf_client, key, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_get_bool: %s\n", err->message);
		return FALSE;
	}

	/* Resolve titles */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_RESOLVE_TITLES) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_RESOLVE_TITLES);
	settings->resolveTitles = 
		gconf_client_get_bool(gconf_client, key, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_get_bool: %s\n", err->message);
		return FALSE;
	}

	/* Send Meta Data */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_SEND_META_DATA) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_SEND_META_DATA);
	settings->sendMetaData = 
		gconf_client_get_bool(gconf_client, key, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_get_bool: %s\n", err->message);
		return FALSE;
	}
	
	return TRUE;
}

/**
 * Save settings
 */
gboolean settings_save(UPnPBrowserSettings* settings)
{
	gchar* key = NULL;
	GConfClient* gconf_client = NULL;
	GError* err = NULL;
	
	g_return_val_if_fail(settings != NULL, FALSE);

	gconf_client = gconf_client_get_default();
	g_return_val_if_fail(gconf_client != NULL, FALSE);

	/* Add GConf node if absent */
	gconf_client_add_dir(gconf_client, KEY_BASE,
			     GCONF_CLIENT_PRELOAD_NONE, &err);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_add_dir: %s\n", err->message);
		return FALSE;
	}

	/* Transport update interval */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_TRANSPORT_UPDATE_INTERVAL) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_TRANSPORT_UPDATE_INTERVAL);
	gconf_client_set_int(gconf_client, key,
			     settings->transportUpdateInterval, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_set_int: %s\n", err->message);
		return FALSE;
	}
	
	/* Update servers on startup */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_UPDATE_SERVERS) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_UPDATE_SERVERS);
	gconf_client_set_bool(gconf_client, key,
			      settings->updateServers, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_set_bool: %s\n", err->message);
		return FALSE;
	}

	/* Resolve titles */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_RESOLVE_TITLES) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_RESOLVE_TITLES);	
	gconf_client_set_bool(gconf_client, key, 
			      settings->resolveTitles, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_set_bool: %s\n", err->message);
		return FALSE;
	}

	/* Send Meta Data */
	key = g_new0(gchar,
		     strlen(KEY_BASE) + strlen(KEY_SEND_META_DATA) + 1);
	sprintf(key, "%s%s", KEY_BASE, KEY_SEND_META_DATA);
	gconf_client_set_bool(gconf_client, key, 
			      settings->sendMetaData, &err);
	g_free(key);

	if (err != NULL)
	{
		fprintf (stderr, "gconf_client_set_bool: %s\n", err->message);
		return FALSE;
	}
	
	return TRUE;
}

/**
 * Edit settings with a dialog
 */
gboolean settings_edit_dialog(UPnPBrowserSettings* settings)
{
	GtkWidget* dialog = NULL;
	UPnPBrowserSettings* cache = NULL;
	
	cache = g_new0(UPnPBrowserSettings, 1);
	g_return_val_if_fail(cache != NULL, FALSE);

	memcpy(cache, settings, sizeof(UPnPBrowserSettings));
	
	dialog = create_settings_dialog(cache);
	if (dialog != NULL)
	{
		gtk_widget_show_all(dialog);

		if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
		{
			memcpy(settings, cache, sizeof(UPnPBrowserSettings));

			reset_now_playing_update_timeout(
				cache->transportUpdateInterval);

		}
	}
	
	g_free(cache);
	gtk_widget_destroy(dialog);
	
	return TRUE;
}

/**
 * Create the settings dialog
 */
GtkWidget* create_settings_dialog(UPnPBrowserSettings* cache)
{
	GtkWidget *settings_dialog;
	GtkWidget *dialog_vbox1;
	GtkWidget *vbox1;
	GtkWidget *hbox1;
	GtkWidget *label1;
	GtkObject *transport_update_spin_adj;
	GtkWidget *transport_update_spin;
	GtkWidget *update_servers;
	GtkWidget *resolve_titles;
	GtkWidget *send_meta_data;
	GtkWidget *dialog_action_area1;
	GtkWidget *cancel_button;
	GtkWidget *ok_button;

	g_return_val_if_fail(cache != NULL, NULL);
	
	settings_dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(settings_dialog),
			     _("UPnP Browser Settings"));
	gtk_window_set_type_hint(GTK_WINDOW (settings_dialog),
				 GDK_WINDOW_TYPE_HINT_DIALOG);

	dialog_vbox1 = GTK_DIALOG(settings_dialog)->vbox;
	gtk_widget_show(dialog_vbox1);

	vbox1 = gtk_vbox_new(TRUE, 10);
	gtk_widget_show(vbox1);
	gtk_box_pack_start(GTK_BOX(dialog_vbox1), vbox1, TRUE, TRUE, 0);

	hbox1 = gtk_hbox_new(FALSE, 10);
	gtk_widget_show(hbox1);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox1, TRUE, TRUE, 0);

	label1 = gtk_label_new(_("Transport update interval"));	
	gtk_widget_show(label1);
	gtk_box_pack_start(GTK_BOX(hbox1), label1, FALSE, FALSE, 0);

	transport_update_spin_adj = gtk_adjustment_new(1, 0, 60, 1, 10, 10);
	transport_update_spin = 
		gtk_spin_button_new(GTK_ADJUSTMENT(transport_update_spin_adj),
				    1, 0);
	gtk_widget_show(transport_update_spin);
	gtk_box_pack_start(GTK_BOX(hbox1), transport_update_spin,
			   TRUE, TRUE, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(transport_update_spin), 
				    TRUE);
				    
	update_servers = gtk_check_button_new_with_mnemonic(
			_("Update media servers/renderers on startup"));
	gtk_widget_show(update_servers);
	gtk_box_pack_start(GTK_BOX(vbox1), update_servers,
			   FALSE, FALSE, 0);

	resolve_titles = gtk_check_button_new_with_mnemonic(
			_("Resolve media titles from media servers"));
	gtk_widget_show(resolve_titles);
	gtk_box_pack_start(GTK_BOX(vbox1), resolve_titles,
			   FALSE, FALSE, 0);
	gtk_widget_set_sensitive(resolve_titles, FALSE);

	send_meta_data = gtk_check_button_new_with_mnemonic(
			_("Send meta data to media renderers"));
	gtk_widget_show(send_meta_data);
	gtk_box_pack_start(GTK_BOX(vbox1), send_meta_data,
			   FALSE, FALSE, 0);

	dialog_action_area1 = GTK_DIALOG(settings_dialog)->action_area;
	gtk_widget_show(dialog_action_area1);
	gtk_button_box_set_layout(GTK_BUTTON_BOX (dialog_action_area1),
				  GTK_BUTTONBOX_END);

	ok_button = gtk_button_new_with_label("OK");
	gtk_widget_show(ok_button);
	gtk_dialog_add_action_widget(GTK_DIALOG (settings_dialog), 
				     ok_button, GTK_RESPONSE_OK);
	GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);

	cancel_button = gtk_button_new_with_label("Cancel");
	gtk_widget_show(cancel_button);
	gtk_dialog_add_action_widget(GTK_DIALOG(settings_dialog),
				     cancel_button, GTK_RESPONSE_CANCEL);
	GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);

	g_object_set(G_OBJECT(cancel_button), "has-default", TRUE);

	/* Transport update interval spin */
	g_signal_connect(G_OBJECT(transport_update_spin), "value-changed",
			 GTK_SIGNAL_FUNC(transport_update_interval_changed),
			 cache);

	/* Update servers */
	g_signal_connect(G_OBJECT(update_servers), "toggled",
			 GTK_SIGNAL_FUNC(update_servers_toggled),
			 cache);

	/* Resolve titles */
	g_signal_connect(G_OBJECT(resolve_titles), "toggled",
			 GTK_SIGNAL_FUNC(resolve_titles_toggled),
			 cache);

	/* Send Meta Data*/
	g_signal_connect(G_OBJECT(send_meta_data), "toggled",
			 GTK_SIGNAL_FUNC(send_meta_data_toggled),
			 cache);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(update_servers), 
				     cache->updateServers);

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(transport_update_spin), 
				  (double) cache->transportUpdateInterval);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(resolve_titles),
				     cache->resolveTitles);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(send_meta_data),
				     cache->sendMetaData);

	return settings_dialog;
}

void transport_update_interval_changed(GtkSpinButton *spinbutton,
				       gpointer user_data)
{
	UPnPBrowserSettings* cache = (UPnPBrowserSettings*) user_data;
	
	g_return_if_fail(spinbutton != NULL);
	g_return_if_fail(cache != NULL);
	
	cache->transportUpdateInterval = 
		gtk_spin_button_get_value_as_int(spinbutton);
	
}

void update_servers_toggled(GtkToggleButton *togglebutton,
			   gpointer user_data)
{
	UPnPBrowserSettings* cache = (UPnPBrowserSettings*) user_data;

	g_return_if_fail(togglebutton != NULL);
	g_return_if_fail(cache != NULL);
	
	cache->updateServers = 
		gtk_toggle_button_get_active(togglebutton);
}

void resolve_titles_toggled(GtkToggleButton *togglebutton,
			   gpointer user_data)
{
	UPnPBrowserSettings* cache = (UPnPBrowserSettings*) user_data;
	
	g_return_if_fail(togglebutton != NULL);
	g_return_if_fail(cache != NULL);
	
	cache->resolveTitles = 
		gtk_toggle_button_get_active(togglebutton);
}

void send_meta_data_toggled(GtkToggleButton *togglebutton,
			    gpointer user_data)
{
	UPnPBrowserSettings* cache = (UPnPBrowserSettings*) user_data;
	
	g_return_if_fail(togglebutton != NULL);
	g_return_if_fail(cache != NULL);
	
	cache->sendMetaData = 
		gtk_toggle_button_get_active(togglebutton);
}
