/*
 * xresponse - Interaction latency tester,
 *
 * Written by Ross Burton & Matthew Allum  
 *              <info@openedhand.com> 
 *
 * Copyright (C) 2005 Nokia
 *
 * Licensed under the GPL v2 or greater.
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/time.h>

#include <wchar.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>
#include <X11/extensions/Xdamage.h>

#include "kbd.h"

/* 
 * defs
 */

#define streq(a,b)      (strcmp(a,b) == 0)
#define CMD_STRING_MAXLEN 256
typedef struct Rectangle { int x,y,width,height; } Rectangle;

/*
 * Global variables
 */

static FILE      *LogFile = NULL;       /* The file to output the log output too */
static int        DamageEventNum;       /* Damage Ext Event ID */
static Atom       AtomTimestamp;        /* Atom for getting server time */
static int        DamageWaitSecs = 5;   /* Max time to collect damamge */
static Rectangle  InterestedDamageRect; /* Damage rect to monitor */
static Time       LastEventTime;      /* When last last event was started */
static unsigned long DefaultDelay = 100; /* Default delay for key synthesis */

enum { /* for 'dragging' */
  XR_BUTTON_STATE_NONE,
  XR_BUTTON_STATE_PRESS,
  XR_BUTTON_STATE_RELEASE
};

static int 
handle_xerror(Display *dpy, XErrorEvent *e)
{
  /* Really only here for debugging, for gdb backtrace */
  char msg[255];
  XGetErrorText(dpy, e->error_code, msg, sizeof msg);
  fprintf(stderr, "X error (%#lx): %s (opcode: %i)\n",
	  e->resourceid, msg, e->request_code);

  exit(1);
}


/** 
 * Perform simple logging with timestamp and diff from last log
 */
static void
log_action(Time time, int is_stamp, const char *format, ...)
{
  va_list     ap;
  char       *tmp = NULL;
  static int  displayed_header;

  va_start(ap,format);
  vasprintf(&tmp, format, ap);
  va_end(ap);

  if (!displayed_header) 		/* Header */
    {
      fprintf(LogFile, "\n"
	      " Server Time : Diff    : Info\n"
	      "-----------------------------\n");
      displayed_header = 1;
    }

  if (is_stamp)
    {
      fprintf(LogFile, "%s\n", tmp);
    }
  else
    {
      fprintf(LogFile, "%10lums : %5lums : %s",
	      time,
	      (LastEventTime > 0 && time > 0) ? time - LastEventTime : 0,
	      tmp);
    }

  if (tmp) free(tmp);
}

/**
 * Get the current timestamp from the X server.
 */
static Time 
get_server_time(Display *dpy) 
{
  XChangeProperty (dpy, DefaultRootWindow (dpy), 
		   AtomTimestamp, AtomTimestamp, 8, 
		   PropModeReplace, "a", 1);
  for (;;) 
    {
      XEvent xevent;

      XMaskEvent (dpy, PropertyChangeMask, &xevent);
      if (xevent.xproperty.atom == AtomTimestamp)
	return xevent.xproperty.time;
    }
}

/**  
 * Get an X event with a timeout ( in secs ). The timeout is
 * updated for the number of secs left.
 */
static Bool
get_xevent_timed(Display        *dpy, 
		 XEvent         *event_return, 
		 struct timeval *tv)      /* in seconds  */
{

  if (tv == NULL)
    {
      XNextEvent(dpy, event_return);
      return True;
    }

  XFlush(dpy);

  if (XPending(dpy) == 0) 
    {
      int    fd = ConnectionNumber(dpy);
      fd_set readset;

      FD_ZERO(&readset);
      FD_SET(fd, &readset);

      if (select(fd+1, &readset, NULL, NULL, tv) == 0) 
	return False;
      else 
	{
	  XNextEvent(dpy, event_return);

	  /* *timeout = tv.tv_sec; */ /* XXX Linux only ? */

	  return True;
	}

    } else {
      XNextEvent(dpy, event_return);
      return True;
    }
}

/** 
 * Set up Display connection, required extensions and req other X bits
 */
static Display*
setup_display(char *dpy_name) 
{
  Display *dpy;
  Damage   damage;
  int      unused;

  if ((dpy = XOpenDisplay(dpy_name)) == NULL)
    {
      fprintf (stderr, "Unable to connect to DISPLAY.\n");
      return NULL;
    }

  /* Check the extensions we need are available */

  if (!XTestQueryExtension (dpy, &unused, &unused, &unused, &unused)) {
    fprintf (stderr, "No XTest extension found\n");
    return NULL;
  }

  if (!XDamageQueryExtension (dpy, &DamageEventNum, &unused)) {
    fprintf (stderr, "No DAMAGE extension found\n");
    return NULL;
  }

  /* Set up our interested rect */
  InterestedDamageRect.x      = 0;
  InterestedDamageRect.y      = 0;
  InterestedDamageRect.width  = DisplayWidth(dpy, DefaultScreen(dpy));
  InterestedDamageRect.height = DisplayHeight(dpy, DefaultScreen(dpy));

  XSetErrorHandler(handle_xerror); 

  XSynchronize(dpy, True);

  /* Needed for get_server_time */
  AtomTimestamp = XInternAtom (dpy, "_X_LATENCY_TIMESTAMP", False);  
  XSelectInput(dpy, DefaultRootWindow(dpy), PropertyChangeMask);

  /* XXX Return/global this ? */
  damage = XDamageCreate (dpy, 
			  DefaultRootWindow(dpy), 
			  XDamageReportBoundingBox);
  return dpy;
}


/**
 * Eat all Damage events in the X event queue.
 */
static void 
eat_damage(Display *dpy) 
{
  while (XPending(dpy)) 
    {
      XEvent              xev;
      XDamageNotifyEvent *dev;

      XNextEvent(dpy, &xev);

      if (xev.type == DamageEventNum + XDamageNotify) 
	{
	  dev = (XDamageNotifyEvent*)&xev;
	  XDamageSubtract(dpy, dev->damage, None, None);
	}
    }
}

/** 
 * 'Fakes' a mouse click, returning time sent.
 */
static Time
fake_event(Display *dpy, int x, int y)
{
  Time start;

  XTestFakeMotionEvent(dpy, DefaultScreen(dpy), x, y, CurrentTime);

  /* Eat up any damage caused by above pointer move */
  eat_damage(dpy);

  start = get_server_time(dpy);

  /* 'click' mouse */
  XTestFakeButtonEvent(dpy, Button1, True, CurrentTime);
  XTestFakeButtonEvent(dpy, Button1, False, CurrentTime);

  return start;
}

static Time
drag_event(Display *dpy, int x, int y, int button_state)
{
  Time start;

  start = get_server_time(dpy);

  XTestFakeMotionEvent(dpy, DefaultScreen(dpy), x, y, CurrentTime);

  if (button_state == XR_BUTTON_STATE_PRESS)
    {
      eat_damage(dpy); 	/* ignore damage from first drag */
      XTestFakeButtonEvent(dpy, Button1, True, CurrentTime);
    }

  if (button_state == XR_BUTTON_STATE_RELEASE)
    XTestFakeButtonEvent(dpy, Button1, False, CurrentTime);

  return start;
}

/** 
 * Waits for a damage 'response' to above click / keypress(es)
 */
static Bool
wait_response(Display *dpy)
{
  XEvent e;
  struct timeval tv; 
  struct timeval *timeout = NULL;

  if (DamageWaitSecs)
    {
      tv.tv_sec = DamageWaitSecs;
      tv.tv_usec = 0;
      timeout = &tv;
    }

  while (get_xevent_timed(dpy, &e, timeout))
    {
      if (e.type == DamageEventNum + XDamageNotify) 
	{
	  XDamageNotifyEvent *dev = (XDamageNotifyEvent*)&e;
	  
	  if (dev->area.x + dev->area.width >= InterestedDamageRect.x
	      && dev->area.x <= (InterestedDamageRect.x +
				 InterestedDamageRect.width)
	      && dev->area.y + dev->area.height >= InterestedDamageRect.y
	      && dev->area.y <= (InterestedDamageRect.y + 
				 InterestedDamageRect.height))
	    {
	      log_action(dev->timestamp, 0, "Got damage event %dx%d+%d+%d\n",
			 dev->area.width, dev->area.height, 
			 dev->area.x, dev->area.y);
	    }
	  
	  XDamageSubtract(dpy, dev->damage, None, None);
	} 
      else 
	{
	  fprintf(stderr, "Got unwanted event type %d\n", e.type);
	}

      fflush(LogFile);
    }

  return True;
}

void
usage(char *progname)
{
  fprintf(stderr, "%s: usage, %s <-o|--logfile output> [commands..]\n" 
	          "Commands are any combination/order of;\n"
	          "-c|--click <XxY>                Send click and await damage response\n" 
	          "-d|--drag <XxY,XxY,XxY,XxY..>   Simulate mouse drag and collect damage\n"
	          "-k|--key <keysym[,delay]>         Simulate pressing and releasing a key\n"
	          "                                Delay is in milliseconds.\n"
                  "                                If not specified, default of %lu ms is used\n"
	          "-m|--monitor <WIDTHxHEIGHT+X+Y> Watch area for damage ( default fullscreen )\n"
	          "-w|--wait <seconds>             Max time to wait for damage, set to 0 to\n"
	          "                                monitor for ever.\n"
	          "                                ( default 5 secs)\n"
	          "-s|--stamp <string>             Write 'string' to log file\n\n"
	          "-t|--type <string>              Simulate typing a string\n"
	          "-i|--inspect                    Just display damage events\n"
	          "-v|--verbose                    Output response to all command line options \n\n",
	  progname, progname, DefaultDelay);
  exit(1);
}

/* Code below this comment has been copied from xresponse / vte.c and
 * modified minimally as required to co-operate with xresponse. */

/* All key events need to go through here because it handles the lookup of
 * keysyms like Num_Lock, etc.  Thing should be something that represents a
 * single key on the keyboard, like KP_PLUS or Num_Lock or just A */
static KeyCode 
thing_to_keycode(Display *dpy, char *thing ) 
{
  KeyCode kc;
  KeySym ks;
  
  ks = XStringToKeysym(thing);
  if (ks == NoSymbol){
    fprintf(stderr, "Unable to resolve keysym for '%s'\n", thing);
    return(thing_to_keycode(dpy, "space" ));
  }

  kc = XKeysymToKeycode(dpy, ks);
  return(kc);
}

/* Simulate pressed key(s) to generate thing character
 * Only characters where the KeySym corresponds to the Unicode
 * character code and KeySym < MAX_KEYSYM are supported,
 * except the special character 'Tab'. */
static Time send_string( Display *dpy, char *thing_in ) 
{
  char *wrap_key = NULL;
  int i = 0;

  KeyCode keycode;
  KeySym keysym;
  Time start;

  wchar_t thing[ CMD_STRING_MAXLEN ];
  wchar_t wc_singlechar_str[2];
  wmemset( thing, L'\0', CMD_STRING_MAXLEN );
  mbstowcs( thing, thing_in, CMD_STRING_MAXLEN );
  wc_singlechar_str[ 1 ] = L'\0';

  eat_damage(dpy);
  start = get_server_time(dpy);

  while( ( thing[ i ] != L'\0' ) && ( i < CMD_STRING_MAXLEN ) ) {

    wc_singlechar_str[ 0 ] = thing[ i ];

    /* keysym = wchar value */

    keysym = wc_singlechar_str[ 0 ];

    if( keysym >= MAX_KEYSYM ) 
      {
	fprintf( stderr, "Special character '%ls' is currently not supported.\n", thing );
      }else{
      /* Keyboard modifier and KeyCode lookup */
      wrap_key = key_modifiers[ keysym_to_modifier_map[ keysym ] ];
      keycode = keysym_to_keycode_map[keysym];

      if( wrap_key != NULL )
	XTestFakeKeyEvent( dpy, thing_to_keycode( dpy, wrap_key ), True, CurrentTime );
      XTestFakeKeyEvent( dpy, keycode, True, CurrentTime );
      XTestFakeKeyEvent( dpy, keycode, False, CurrentTime );

      if( wrap_key != NULL )
	XTestFakeKeyEvent( dpy, thing_to_keycode( dpy, wrap_key ), False, CurrentTime );

      /* Not flushing after every key like we need to, thanks
       * thorsten@staerk.de */
      XFlush( dpy );
    }
    
    i++;

  }
  return start;
}

/* Load keycodes and modifiers of current keyboard mapping into arrays,
 * this is needed by the send_string function */
void 
load_keycodes( Display *dpy ) 
{
  int min_keycode,max_keycode,keysyms_per_keycode,keycode_index,wrap_key_index,num_modifiers;
  char *str;
  KeySym *keysyms, keysym;
  KeyCode keycode;

  XDisplayKeycodes( dpy, &min_keycode, &max_keycode);
  keysyms = XGetKeyboardMapping( dpy,
    (KeyCode)min_keycode, max_keycode + 1 - min_keycode,
    &keysyms_per_keycode );

  for( keysym=0; keysym<MAX_KEYSYM; keysym++ ) {
    keysym_to_modifier_map[keysym]=-1;
    keysym_to_keycode_map[keysym]=0;
  }

  if( keysyms_per_keycode < NUM_KEY_MODIFIERS ) {
    num_modifiers = keysyms_per_keycode;
  } else {
    num_modifiers = NUM_KEY_MODIFIERS;
  }

  for( keycode_index = 0; keycode_index < ( max_keycode + 1 - min_keycode ); keycode_index++ ) {
    for( wrap_key_index = 0; wrap_key_index < num_modifiers; wrap_key_index++ ) {
      str = XKeysymToString( keysyms[ keycode_index * keysyms_per_keycode +
				      wrap_key_index ] );
      if( str != NULL ) {
        keysym = XStringToKeysym( str );
        keycode = XKeysymToKeycode( dpy, keysym );
        //printf("i=%d (keysym %lld), j=%d (keycode %d): %s\n",keycode_index,(long long int)keysym,wrap_key_index,keycode,str);

        if( ( keysym < MAX_KEYSYM ) && ( keysym_to_modifier_map[ keysym ] == -1 ) ) {
          keysym_to_modifier_map[ keysym ] = wrap_key_index;
          keysym_to_keycode_map[ keysym ] = keycode;
        }
      }
    }
  }
  XFree(keysyms);
}


static Time 
send_key(Display *dpy, char *thing, unsigned long delay) 
{
  Time start;

  /* It seems that we need to eat the initial 800x480 - sized damage
     event here, which has earlier timestamp than anything caused
     by the keypress/-release pair, where does it come from? And does
     this make us miss any other events? */

  eat_damage(dpy);
  start = get_server_time(dpy);
  XTestFakeKeyEvent(dpy, thing_to_keycode( dpy, thing ), True, CurrentTime);
  XTestFakeKeyEvent(dpy, thing_to_keycode( dpy, thing ), False, delay);
  return start;
}

/* Code copy from xresponse / vte.c ends */


int 
main(int argc, char **argv) 
{
  Display *dpy;
  int      cnt, x, y, i = 0, verbose = 0;

  if (argc == 1)
    usage(argv[0]);

  if ((dpy = setup_display(getenv("DISPLAY"))) == NULL)
    exit(1);

  load_keycodes(dpy);

  if (streq(argv[1],"-o") || streq(argv[1],"--logfile"))
    {
      i++;

      if (++i > argc) usage (argv[0]);

      if ((LogFile = fopen(argv[i], "w")) == NULL)
	fprintf(stderr, "Failed to create logfile '%s'\n", argv[i]);
    }

  if (LogFile == NULL) 
    LogFile = stdout;

  while (++i < argc)
    {
      if (streq(argv[i],"-v") || streq(argv[i],"--verbose"))
        {
	   verbose = 1;
	   continue;
	}

      if (streq("-c", argv[i]) || streq("--click", argv[i])) 
	{
	  if (++i>=argc) usage (argv[0]);
	  
	  cnt = sscanf(argv[i], "%ux%u", &x, &y);
	  if (cnt != 2) 
	    {
	      fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
	      usage(argv[0]);
	    }
	  
	  /* Send the event */
	  LastEventTime = fake_event(dpy, x, y);
	  log_action(LastEventTime, 0, "Clicked %ix%i\n", x, y);
	  
	  /* .. and wait for the damage response */
	  wait_response(dpy);
	  
	  continue;
	}

      if (streq("-s", argv[i]) || streq("--stamp", argv[i])) 
	{
	  if (++i>=argc) usage (argv[0]);
	  log_action(0, 1, argv[i]);
	  continue;
	}

      
      if (streq("-m", argv[i]) || streq("--monitor", argv[i])) 
	{
	  if (++i>=argc) usage (argv[0]);
	  
	  if ((cnt = sscanf(argv[i], "%ux%u+%u+%u", 
			    &InterestedDamageRect.width,
			    &InterestedDamageRect.height,
			    &InterestedDamageRect.x,
			    &InterestedDamageRect.y)) != 4)
	    {
	      fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
	      usage(argv[0]);
	    }

	  if (verbose)
	      printf("Set monitor rect to %ix%i+%i+%i\n",
		     InterestedDamageRect.width,InterestedDamageRect.height,
		     InterestedDamageRect.x,InterestedDamageRect.y);

	  continue;
	}

      if (streq("-w", argv[i]) || streq("--wait", argv[i])) 
	{
	  if (++i>=argc) usage (argv[0]);
	  
	  if ((DamageWaitSecs = atoi(argv[i])) < 0)
	    {
	      fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
	      usage(argv[0]);
	    }
	  if (verbose)
	    log_action(0, 0, "Set event timout to %isecs\n", DamageWaitSecs);

	  continue;
	}

      if (streq("-d", argv[i]) || streq("--drag", argv[i])) 
	{
	  Time drag_time;
	  char *s = NULL, *p = NULL;
	  int first_drag = 1, button_state = XR_BUTTON_STATE_PRESS;
	  
	  if (++i>=argc) usage (argv[0]);

	  s = p = argv[i];

	  while (1)
	    {
	      if (*p == ',' || *p == '\0')
		{
		  Bool end = False;

		  if (*p == '\0')
		    {
		      if (button_state == XR_BUTTON_STATE_PRESS)
			{
			  fprintf(stderr, 
				  "*** Need at least 2 drag points!\n");
			  usage(argv[0]);
			}

		      /* last passed point so make sure button released */
		      button_state = XR_BUTTON_STATE_RELEASE;
		      end = True;
		    }
		  else *p = '\0';

		  cnt = sscanf(s, "%ux%u", &x, &y);
		  if (cnt != 2) 
		    {
		      fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
		      usage(argv[0]);
		    }

		  /* Send the event */
		  drag_time = drag_event(dpy, x, y, button_state);
		  if (first_drag)
		    {
		      LastEventTime = drag_time;
		      first_drag = 0;
		    }
		  log_action(drag_time, 0, "Dragged to %ix%i\n", x, y);

		  /* Make sure button state set to none after first point */
		  button_state = XR_BUTTON_STATE_NONE;

		  if (end)
		    break;

		  s = p+1;
		}
	      p++;
	    }

	  /* .. and wait for the damage response */
	  wait_response(dpy);

	  continue;
	}

      if (streq("-k", argv[i]) || streq("--key", argv[i]))
	{
	  char *key = NULL;
	  char separator;
	  unsigned long delay = 0;

	  if (++i>=argc) usage (argv[0]);
	  cnt = sscanf(argv[i], "%a[^,]%c%lu", &key, &separator, &delay);
	  if (cnt == 1)
	    {
	      log_action(0, 0, "Using default delay between press/release\n",
			 delay);
	      delay = DefaultDelay;
	    }
	  else if (cnt != 3 || separator != ',')
	    {
	      fprintf(stderr, "cnt: %d\n", cnt);
	      fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
	      if (key != NULL)
		free(key);
	      usage(argv[0]);
	    }

	  LastEventTime = send_key(dpy, key, delay);

	  log_action(LastEventTime, 0,
		     "Simulating keypress/-release pair (keycode '%s')\n",
		     key);
	  free(key);
	  /* .. and wait for the damage response */
	  wait_response(dpy);

	  continue;
	}
      
      if (streq("-t", argv[i]) || streq("--type", argv[i]))
	{
	  if (++i>=argc) usage (argv[0]);
	  LastEventTime = send_string(dpy, argv[i]);
	  
	  log_action(LastEventTime, 0, "Simulated keys for '%s'\n", argv[i]);

	  /* .. and wait for the damage response */
	  wait_response(dpy);
	  
	  continue;
	}

      if (streq("-i", argv[i]) || streq("--inspect", argv[i])) 
	{
	  if (verbose)
	    log_action(0, 0, "Just displaying damage events until timeout\n");
	  wait_response(dpy);
	  continue;
	}

      fprintf(stderr, "*** Dont understand  %s\n", argv[i]);
      usage(argv[0]);
    }

  /* Clean Up */

  XCloseDisplay(dpy);
  fclose(LogFile);

  return 0;
}
