/**
  @file auth-handler.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <getopt.h>
#include <glib.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#define AUTH_PATH "/my/auth/handler"

/** Startup config options */
static struct {
    char *object;
    char *root;
} cfg;

typedef struct {
    char    *object;
    char    *bda;
    char    *name;
    char    *type;
    int      length;
    time_t   time;
    char    *filename;
} Transfer;

/* For getopt */
static struct option const long_options[] = {
    {"help",     no_argument,       0, 'h'},
    {"version",  no_argument,       0, 'V'},
    {"root",     required_argument, 0, 'r'},
    {"object",   required_argument, 0, 'o'},
    {NULL, 0, NULL, 0}
};

static GSList *transfers = NULL;

static char *program_name;

static DBusConnection *connection = NULL;
static GMainLoop      *event_loop = NULL;

/* Print usage information and exit with status */
static void usage(int status)
{
    printf("%s - BT PIN Handler test application %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help             Display this help and exit\n"
     "-V, --version          Show version info and exit\n"
     "-o, --object           Object path of auth handler\n"
     "-r, --root             Default root directory\n"
     "\n", program_name);

    exit(status);
}

/* Process commandline options. Returns index of first
 * non-option argument */
static int decode_switches(int argc, char *argv[])
{
    int c;

    /* Set defaults */
    memset(&cfg, 0, sizeof(cfg));
    cfg.object = g_strdup(AUTH_PATH);

    while ((c = getopt_long(argc, argv,
                    "h"   /* help    */
                    "V"   /* version */
                    "o:"  /* object  */
                    "r:"  /* root    */
                    ,long_options, (int *) 0)) != EOF) {
        switch (c) {
            case 'V':
                printf("GW SDP Application %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'o':
                g_free(cfg.object);
                cfg.object = g_strdup(optarg);
                break;

            case 'r':
                g_free(cfg.root);
                cfg.root = g_strdup(optarg);
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

static DBusHandlerResult signal_filter(DBusConnection *conn, DBusMessage *msg, void *data)
{
    Transfer *transfer = NULL;
    dbus_bool_t success = FALSE;
    GSList *l;

    for (l = transfers; l != NULL; l = l->next) {
        Transfer *t = l->data;
        if (dbus_message_has_path(msg, t->object)) {
            transfer = t;
            break;
        }
    }

    if (!transfer) {
        printf("No matching transfer for %s\n", dbus_message_get_path(msg));
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (dbus_message_is_signal(msg, "com.nokia.ObexServer.Transfer", "Started")) {
        dbus_message_get_args(msg, NULL,
                DBUS_TYPE_BOOLEAN, &success,
                DBUS_TYPE_INVALID);
        if (!success) {
            printf("Starting transfer failed\n");
            transfers = g_slist_remove(transfers, transfer);
        }
        else
            printf("Transfer started\n");
    }
    else if (dbus_message_is_signal(msg, "com.nokia.ObexServer.Transfer", "Completed")) {
        dbus_message_get_args(msg, NULL,
                DBUS_TYPE_BOOLEAN, &success,
                DBUS_TYPE_INVALID);
        if (!success)
            printf("\nTransfer failed\n");
        else
            printf("\nTransfer completed\n");
        transfers = g_slist_remove(transfers, transfer);
    }
    else if (dbus_message_is_signal(msg, "com.nokia.ObexServer.Transfer", "Progress")) {
        int current, final;
        dbus_message_get_args(msg, NULL,
                DBUS_TYPE_INT32, &current,
                DBUS_TYPE_INT32, &final,
                DBUS_TYPE_INVALID);
        printf("Transfer progress: %d/%d                     \r", current, final);
    }
    else {
        printf("Got unrecognized signal\n");
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static gboolean send_request(const gchar *method)
{
    DBusMessage *request, *reply;

    request = dbus_message_new_method_call("com.nokia.ObexServer",
                                           "/com/nokia/ObexServer",
                                           "com.nokia.ObexServer",
                                           method);
    dbus_message_append_args(request,
            DBUS_TYPE_STRING, &cfg.object,
            DBUS_TYPE_INVALID);

    reply = dbus_connection_send_with_reply_and_block(connection, request,
                                                      -1, NULL);
    dbus_connection_flush(connection);

    dbus_message_unref(request);

    if (reply == NULL) {
        fprintf(stderr, "sending failed (got NULL reply)\n");
        return FALSE;
    }

    if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR) {
        fprintf(stderr, "got error D-BUS reply\n");
        dbus_message_unref(reply);
        return FALSE;
    }

    dbus_message_unref(reply);

    return TRUE;
}

static void cleanup(int signal)
{
    g_main_loop_quit(event_loop);
}

static char *ask_filename(void)
{
    char filename[PATH_MAX], *nl;

    printf("\nEnter filename: ");
    if (!fgets(filename, sizeof(filename), stdin))
        return NULL;

    if (strlen(filename) < 2)
        return NULL;

    nl = strchr(filename, '\n');
    if (nl)
        *nl = '\0';

    return g_strdup(filename);
}

/* Handler for D-Bus requests */
static DBusHandlerResult authreq_handler(DBusConnection *connection,
                                         DBusMessage    *message,
                                         void           *user_data)
{
    DBusMessage *reply;
    Transfer *xfer;
    const char *path, *bda, *name, *type;
    int length;
    time_t modtime;

    printf("Got auth request\n");

    dbus_message_get_args(message, NULL,
                          DBUS_TYPE_STRING, &path,
                          DBUS_TYPE_STRING, &bda,
                          DBUS_TYPE_STRING, &name,
                          DBUS_TYPE_STRING, &type,
                          DBUS_TYPE_INT32, &length,
                          DBUS_TYPE_INT32, &modtime,
                          DBUS_TYPE_INVALID);

    printf("Incoming transfer:\n"
            "\tTransfer path: %s\n"
            "\tRemote address: %s\n"
            "\tObject name: %s\n"
            "\tObject type: %s\n"
            "\tObject length: %d\n"
            "\tObject time: %s\n",
            path, bda, name, type, length, ctime(&modtime));

    xfer = g_new0(Transfer, 1);

    if (cfg.root)
        xfer->filename = g_strdup_printf("%s/%s", cfg.root, name);
    else
        xfer->filename = ask_filename();

    if (!xfer->filename) {
        g_free(xfer);
        reply = dbus_message_new_error(message, "com.nokia.ObexServer.Error.Refused", "Refused");
    }
    else {
        xfer->object = g_strdup(path);
        xfer->bda = g_strdup(bda);
        xfer->name = g_strdup(name);
        xfer->type = g_strdup(type);
        xfer->length = length;
        xfer->time = modtime;

        transfers = g_slist_append(transfers, xfer);

        reply = dbus_message_new_method_return(message);
        dbus_message_append_args(reply,
                DBUS_TYPE_STRING, &xfer->filename,
                DBUS_TYPE_INVALID);
    }

    dbus_connection_send(connection, reply, NULL);
    dbus_connection_flush(connection);

    printf("Reply sent.\n");

    dbus_message_unref(reply);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusObjectPathVTable test_vtable = {
    .message_function    = authreq_handler,
    .unregister_function = NULL
};

int main(int argc, char *argv[])
{
    gint i;

    program_name = argv[0];
    i = decode_switches(argc, argv);

    event_loop = g_main_loop_new(NULL, TRUE);

    connection = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    if (connection == NULL) {
        fprintf(stderr, "D-BUS connection failed\n");
        exit(EXIT_FAILURE);
    }

    if (!dbus_connection_register_object_path(connection,
                                              AUTH_PATH,
                                              &test_vtable,
                                              NULL)) {
        fprintf(stderr, "Registering object path \"%s\" failed\n",
                AUTH_PATH);
        exit(EXIT_FAILURE);
    }

    dbus_connection_setup_with_g_main(connection, NULL);

    printf("Registering..."); fflush(stdout);
    if (!send_request("RegisterHandler")) {
        fprintf(stderr, "Registering handler failed\n");        
        exit(EXIT_FAILURE);
    }
    printf("done.\n");

    dbus_bus_add_match(connection, "interface=com.nokia.ObexServer.Transfer", NULL);
    dbus_connection_add_filter(connection, signal_filter, NULL, NULL);

    signal(SIGTERM, cleanup);
    signal(SIGINT, cleanup);

    printf("Entering mainloop\n");
    g_main_loop_run(event_loop);

    printf("Unregistering..."); fflush(stdout);
    if (!send_request("UnregisterHandler")) {
        fprintf(stderr, "Registering handler failed\n");        
    }
    printf("done.\n");

    exit(EXIT_SUCCESS);
}
