/**
 * Clinkc-av control point test tool
 *
 * Copyright (c) 2006 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "cptest.h"
#include "avttest.h"
#include "cdstest.h"
#include "rcstest.h"

#define NUM_TEST_CASES 3

#define TEST_CASE_AVT "AV Transport Service Test"
#define TEST_CASE_CDS "Content Directory Service Test"
#define TEST_CASE_RCS "Rendering Control Service Test"

static AVCPTestCase tc[NUM_TEST_CASES] = 
{
	{TEST_CASE_CDS, 0, cp_test_cds},
	{TEST_CASE_AVT, 0, cp_test_avt},
	{TEST_CASE_RCS, 0, cp_test_rcs}
};

int main(int argc, char** argv)
{
	int i = 0;
	int num_devices = 0;
	int passed = 0;
	int num_test_cases = NUM_RCS_TEST_CASES + NUM_CDS_TEST_CASES + NUM_AVT_TEST_CASES;
	CgUpnpControlPoint* cp = NULL;
	TEST_CASE_FUNCTION func = NULL;
		
	cp = init_upnp_controlpoint();
	if (cp == NULL)
	{
		return -1;
	}

	printf("Searching for root devices\n");

	/* Do a search first */
	cg_upnp_controlpoint_search(cp, CG_UPNP_DEVICE_UPNP_ROOTDEVICE);

	/* Wait for MX time to pass */
	cg_wait(CG_UPNP_CONTROLPOINT_SSDP_DEFAULT_SEARCH_MX * 1000);

	/* Check how many devices are present */
	num_devices = cg_upnp_controlpoint_getndevices(cp);
	if (num_devices == 0)
	{
		printf("Unable to continue: no UPnP devices found\n");
		printf("END: cp_test_cds_browse\n");
		
		return FALSE;
	}
	else
	{
		printf("Found %d UPnP devices\n", num_devices);
	}
	
	/* Run thru the test cases */
	for (i = 0; i < NUM_TEST_CASES; i++)
	{
		func = tc[i].func;
		tc[i].result = func(cp);
	}
	
	printf("\n********************************* TEST RESULTS *********************************\n");

	passed += cp_test_cds_print_statistics();
	passed += cp_test_avt_print_statistics();
	passed += cp_test_rcs_print_statistics();
	
	printf("\n***************************** Overall test results *****************************\n");
	
	/* Print test case results */
	for (i = 0; i < NUM_TEST_CASES; i++)
	{
		printf("%s: %s\n", tc[i].name,
			(tc[i].result != 0) ? "PASS" : "FAIL");
	}

	printf("\nOverall pass rate: %d of %d (%.0f%%)\n\n",
		passed, num_test_cases,
		ceil(((float) passed / (float) num_test_cases) * (float) 100));
	
	destroy_upnp_controlpoint(cp);
	
	return 0;
}

/**
 * Create a control point and start it.
 */
CgUpnpControlPoint* init_upnp_controlpoint(void)
{
	CgUpnpControlPoint* cp = NULL;
	
        /* Create the cybergarage control point */
        cp = cg_upnp_controlpoint_new();

        /* Start the control point */
        if (cg_upnp_controlpoint_start(cp) == FALSE)
        {
                printf("Unable to start control point\n");
                cg_upnp_controlpoint_delete(cp);
		cp = NULL;
        }
        else
        {
                printf("Control point started\n");
        }
	
	return cp;
}

/**
 * Stop a control point and destroy it.
 */
int destroy_upnp_controlpoint(CgUpnpControlPoint* cp)
{
	cg_upnp_controlpoint_stop(cp);
	printf("Control point stopped\n");

        cg_upnp_controlpoint_delete(cp);
        printf("Control point destroyed\n");

        return 0;
}

/**
 * Find a device with a given service
 *
 * @param cp A control point instance
 * @param service_name The name of the service to search for from cp's devices
 *
 * @return A newly-created string containing the UDN of the device that has
 *         the given service or; NULL if a suitable device was not found
 */
char* cp_test_get_device_udn_with_service(CgUpnpControlPoint* cp,
					  char* service_name)
{
	char* udn = NULL;
	CgUpnpDevice* device = NULL;
	CgUpnpService* service = NULL;

	if (cp == NULL || service_name == NULL)
	{
		return NULL;
	}
	
	cg_upnp_controlpoint_lock(cp);

	for (device = cg_upnp_controlpoint_getdevices(cp); device != NULL;
	     device = cg_upnp_device_next(device))
	{
		service = cg_upnp_device_getservicebytype(device, service_name);
		if (service)
		{
			udn = cg_strdup(cg_upnp_device_getudn(device));
			printf("Testing with: %s (%s)\n",
				cg_upnp_device_getfriendlyname(device), udn);
			break;
		}
	}

	if (service == NULL)
	{
		printf("Unable to find a device with service: %s\n",
			service_name);
	}

	cg_upnp_controlpoint_unlock(cp);
	
	return udn;
}

/**
 * Find a test case by its name from an array of test cases
 *
 * @param tc_array The array of test cases
 * @param size Number of test cases in the array
 * @param name The name of the test case to search for
 *
 * @return The index of the test case struct or -1 if it wasn't found
 */
int cp_test_find_test_case(AVCPTestCase* tc_array, int size, char* name)
{
	int i = 0;
	
	for (i = 0; i < size; i++)
	{
		if (strcmp(tc_array[i].name, name) == 0)
		{
			return i;
		}
	}
	
	return -1;
}
