/* 
 * CyberLinkC AV API for Control Points
 *
 * cavcontrolpoint.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#include <stdarg.h>
#include <cybergarage/avdebug.h>

#include <cybergarage/util/clist.h>
#include <cybergarage/upnp/caction.h>
#include <cybergarage/upnp/cdevice.h>
#include <cybergarage/upnp/cservice.h>
#include <cybergarage/upnp/ccontrolpoint.h>
#include <cybergarage/avcontrol/cavcontrolpoint.h>
#include <cybergarage/upnp/control/ccontrol.h>

/*****************************************************************************
 * AV Control point
 *****************************************************************************/

/**
 * Invoke an action on the upnp device
 *
 * @param upnpDevice The UPnP device
 * @param actionName The name of the action to invoke
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param numArgs The number of action-specific arguments
 * @param ... A list of action-specific arguments
 *
 * @return An integer containing the UPnP status code
 *
 * The list of action-specific arguments is a comma-separated list, 
 * that is organised as pairs of char* ARG_NAME and CgString* ARG_VALUE.
 * ARG_NAME contains the name of the argument, while ARG_VALUE contains
 * the value of the argument. You don't need to put anything inside OUT
 * arguments, as they are not sent along with the action; their contents
 * are set when the action post returns with OUT values that were received
 * from the peer.
 */
int cg_upnp_av_control_invokeaction(CgUpnpService* upnpService,
				    char* actionName,
				    char** errorString,
				    ...)
{
	va_list va;
	CgUpnpAction* action = NULL;
	int result = 0;

	DEBUG("BEGIN: cg_upnp_av_control_invokeaction\n");

	if (upnpService == NULL || actionName == NULL)
	{
		DEBUG("Invalid parameters!\n");
		DEBUG("END: cg_upnp_av_control_invokeaction\n");
		return CG_UPNP_STATUS_INVALID_ARGS;
	}
	
	/* Get the action specified in actionName argument */
	action = cg_upnp_service_getactionbyname(upnpService, actionName);
	if (action == NULL)
	{		
		return CG_UPNP_STATUS_INVALID_ACTION;
	}
	
	/* Set the IN arguments */
	va_start(va, errorString);
	result = cg_upnp_av_control_parseactionarguments(action, TRUE, &va);
	va_end(va);

	if (!cg_upnp_av_control_iserrorcodesuccessful(result))
	{
		DEBUG("Action's IN argument parsing failed\n");
		DEBUG("END: cg_upnp_av_control_invokeaction\n");
		return result;
	}
	
	/* Invoke the action */
	if (cg_upnp_action_post(action) == TRUE)
	{
		/* Get the OUT arguments */
		va_start(va, errorString);
		result = cg_upnp_av_control_parseactionarguments(action, 
								 FALSE,
								 &va);
		va_end(va);
				
		if (!cg_upnp_av_control_iserrorcodesuccessful(result))
		{
			DEBUG("Action's OUT argument parsing failed\n");
			DEBUG("END: cg_upnp_av_control_invokeaction\n");
			return result;
		}
		else
		{
			DEBUG("Action [%s] post successful\n", actionName);
		}
	}
	else
	{
		DEBUG("Action [%s] post failed:\n",
			actionName);
		DEBUG("-->Error: [%s]\n",
			cg_upnp_action_getstatusdescription(action));
			
		if (errorString)
		{
			*errorString = 
			cg_strdup(cg_upnp_action_getstatusdescription(action));
		}
	}

	/* Return the resulting action status code */
	result = cg_upnp_action_getstatuscode(action);

	DEBUG("END: cg_upnp_av_control_invokeaction\n");

	return result;
}

/**
 * A helper function to set or get dynamic action arguments to/from an action.
 *
 * @param action The UPnP action
 * @param inDirection Use TRUE for IN arguments, FALSE for OUT arguments
 * @param va The dynamic argument list, followed by a single NULL
 *
 * @return upnp error code
 */
int cg_upnp_av_control_parseactionarguments(CgUpnpAction* action, 
					    BOOL inDirection,
					    va_list* va)
{
	CgUpnpArgument* argument = NULL;
	char* param = NULL;
	char** value = NULL;

	DEBUG("BEGIN: cg_upnp_av_control_parseactionarguments\n");

	if (action == NULL)
	{
		DEBUG("Invalid parameters!\n");
		DEBUG("END: cg_upnp_av_control_parseactionarguments\n");
		return CG_UPNP_STATUS_INVALID_ARGS;
	}
	
	/* Set/Get action parameters */
	while ((param = va_arg(*va, char*)) != NULL)
	{
		value = va_arg(*va, char**);
		
		/* Get the argument by its name */
		argument = cg_upnp_action_getargumentbyname(action, param);
		
		/* Set either IN or OUT arguments */
		if (argument != NULL)
		{
			if (cg_upnp_argument_isindirection(argument)
				== TRUE && inDirection == TRUE)
			{
				if (*value)
				{
					cg_upnp_argument_setvalue(argument, *value);
				}
				else
				{
					cg_upnp_argument_setvalue(argument, "");
				}

				DEBUG("IN argument: [%s]\n", param);
				DEBUG("IN value: [%s]\n", *value);
			}
			else if (cg_upnp_argument_isindirection(argument) 
				== FALSE && inDirection == FALSE)
			{
				if (cg_upnp_argument_getvalue(argument))
				{
					*value = cg_strdup(cg_upnp_argument_getvalue(argument));
				}

				DEBUG("OUT argument: [%s]\n", param);
				DEBUG("OUT value: [%s]\n", *value);
			}
		}
		else
		{
			DEBUG("No such argument [%s] for:\n", param);
			DEBUG("-->action [%s]\n", cg_upnp_action_getname(action));
		}
	}

	DEBUG("END: cg_upnp_av_control_parseactionarguments\n");
	
	return 0;
}

/**
 * Check, whether a UPnP error/status code is actually a success.
 *
 * @param errorCode The error/status code
 *
 * @return TRUE if the code is not an error; otherwise FALSE
 */
BOOL cg_upnp_av_control_iserrorcodesuccessful(int errorCode)
{
	if (errorCode > 400 && errorCode < 900)
	{
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}
