/*
 * UPnP Browser for Maemo
 *
 * upnp.h
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#ifndef UPNP_H
#define UPNP_H

#include <gtk/gtk.h>
#include <cybergarage/upnp/ccontrolpoint.h>

/**
 * THE Control Point
 */
CgUpnpControlPoint* controlPoint;

/*****************************************************************************
 * Functions
 *****************************************************************************/

/**
 * Create a control point and start it.
 */
int init_upnp_controlpoint();

/**
 * Stop a control point and destroy it.
 */
int destroy_upnp_controlpoint();

/**
 * Get CDS contents from given depth in device and put them under the given 
 * node in the tree model.
 *
 * @param model The model that is to contain the device structure
 * @param parent Parent device tree node
 * @param udn UDN of the device, whose CDS services to add
 * @param parent_id ID of the container parent
 * 
 * @return TRUE if successful; otherwise false
 */
BOOL get_upnp_content_directory(GtkTreeStore* model,
				GtkTreeIter* parent, 
			        gchar* udn,
				gchar *parent_id);

/**
 * Put content from CDS under the given node in the tree model.
 *
 * @param model The model that is to contain the CDS structure
 * @param parent Parent service tree node
 * @param nodelist XML representation of the contents
 * @param udn The UDN of the device
 * 
 * @return TRUE if successful; otherwise false
 */
BOOL parse_didllite_nodelist_to_model(GtkTreeStore* model,
					GtkTreeIter* parent,
					CgXmlNodeList* nodelist,
					gchar* udn);

/**
 * Parse the children of one DIDL node to the given tree item
 *
 * @param model The tree model to update
 * @param iter A TreeIter iterator pointing to a tree node
 * @param node The DIDL node, whose children to parse
 * 
 * @return TRUE if successful; otherwise false
 */
BOOL parse_didllite_node_to_model(GtkTreeStore* model,
				  GtkTreeIter* iter,
				  CgXmlNode* node);

/**
 * Update the text displayed in the "Now Playing" area
 * from a remote renderer device
 *
 * @param buffer The GtkTextBuffer* to update
 * @param renderer_udn The UDN of the renderer device
 *
 */
void update_remote_now_playing_text(GtkTextBuffer* buffer,
				     gchar* renderer_udn);

/**
 * Insert remote transport state to text buffer
 *
 * @param buffer The text buffer to insert state to
 * @param text_iter Insert point in the buffer
 * @param state UPnP transport state string
 */
void insert_remote_transport_state(GtkTextBuffer* buffer,
				   GtkTextIter* text_iter,
				   char* state);

/**
 * Fetch the track name from a CDS service using the resource's URI
 *
 * @param uri The URI of the resource
 * @param name The "friendly" name of the resource pointed by uri
 */
gboolean get_track_name_from_cds(gchar* uri, gchar** name);

/**
 * Start playing media in a remote renderer device
 *
 * @param udn The UDN of the UPnP renderer device
 * @param uri The URI of the selected media
 * @param media_class The media class
 * @param objectID The object ID of the media to be played
 * @param srv_udn UDN of the content server (for metadata extraction)
 */
gboolean play_remote_media(gchar* udn, gchar* uri,
			   gchar* media_class,
			   gchar* objectID, gchar* srv_udn);

gboolean get_meta_data(CgUpnpDevice* device, gchar* id, gchar** metaData);

/**
 * Stop playing remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean stop_remote_media(gchar* udn);

/**
 * Pause playing remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean pause_remote_media(gchar* udn);

/**
 * Check, whether the remote server is muted
 *
 * @param udn The UDN of the UPnP renderer device
 * @param mute The renderer's mute status
 * @return TRUE if successful; otherwise FALSE
 */
gboolean get_remote_mute(gchar* udn, gboolean* mute);

/**
 * Un/Mute a remote media renderer
 *
 * @param udn The UDN of the UPnP renderer device
 * @param mute TRUE to mute; FALSE to unmute the device
 */
gboolean mute_remote_media(gchar* udn, gboolean mute);

/**
 * Skip to the next remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean next_remote_media(gchar* udn);

/**
 * Skip to the previous remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean previous_remote_media(gchar* udn);

/**
 * Increase volume on a remote media renderer
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean increase_remote_volume(gchar* udn);

/**
 * Decrease volume on a remote media renderer
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean decrease_remote_volume(gchar* udn);

/**
 * Get remote renderer presets
 *
 * @param udn The UDN of the UPnP renderer device
 * @param presets A comma-separated-value of available presets
 */
gboolean get_remote_renderer_presets(gchar* udn, gchar** presets);

/**
 * Seek remote media to the given position (seconds)
 *
 * @param udn The renderer device's UDN
 * @param seekto The position to seek to
 */
void seek_remote_media(gchar* udn, long seekto);

/**
 * Check, whether a UPnP device supports a certain action
 *
 * @param udn The UDN of the device to check
 * @param serviceType Type of the service to look for from the device
 * @param actionName Name of the action to look for from the service
 */
gboolean device_hasaction(gchar* udn,
			  gchar* serviceType,
			  gchar* actionName);

/**
 * Device listener callback that is called by the clinkc stack whenever
 * a device leaves or enters the network.
 *
 * @param udn The UDN of the device entering/leaving the network
 * @param status The device's new status
 */
void device_listener(char* udn, CgUpnpDeviceStatus status);

/**
 * Remove a device from CDS or renderer models
 *
 * @param udn The UDN of the device to remove
 */
void remove_device(gchar* udn);

/**
 * Add a device either to CDS servers or renderers
 *
 * @param udn The UDN of the device to add
 */
void add_device(gchar* udn);

/**
 * Add a device to the list of content directory servers.
 *
 * @param dev The UPnP device to add
 * @param model The model to manipulate
 */
void add_cds_device(gchar* name, gchar* udn, GtkTreeStore* store);

/**
 * Add the device to the list of renderers.
 *
 * @param dev The device to add
 * @param store The GtkListStore to manipulate
 */
void add_renderer_device(gchar* name, gchar* udn, GtkListStore* store);

#endif /* UPNP_H */
