/* 
 * CyberLinkC AV API for Control Points
 *
 * ccdscontrolpoint.h
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#ifndef CG_UPNP_AV_CCDSCONTROLPOINT_H
#define CG_UPNP_AV_CCDSCONTROLPOINT_H

#include <cybergarage/contentdirectory/ccontentdirectory.h>
#include <cybergarage/contentdirectory/ccdsactionarguments.h>
#include <cybergarage/upnp/cdevice.h>

/*****************************************************************************
 * Content Directory Service action invocations
 *****************************************************************************/

/**
 * Invoke an action on the upnp device
 *
 * @param upnpDevice The UPnP device
 * @param actionName The name of the action to invoke
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param numArgs The number of action-specific arguments
 * @param ... A list of action-specific arguments
 *
 * @return An integer containing the UPnP status code
 *
 * The list of action-specific arguments is a comma-separated list, 
 * that is organised as pairs of char* ARG_NAME and char** ARG_VALUE.
 *
 * ARG_NAME contains the name of the argument, while ARG_VALUE contains
 * the value of the argument. You don't need to put anything inside OUT
 * arguments, as they are not sent along with the action; their contents
 * are set when the action post returns with OUT values that were received
 * from the peer. Finally, put a single NULL to end the argument list.
 */
int cg_upnp_av_cds_control_invokeaction(CgUpnpDevice* upnpDevice,
					char* actionName,
					char** errorString,
					...);

/**
 * Parse an XML document into a CgXmlNodeList.
 *
 * @param xmldoc The raw XML document to parse
 * @param nodelist A pre-allocated XML node list to save the parsed nodes to
 * @return TRUE if successful; otherwise FALSE
 */
BOOL cg_upnp_av_cds_create_cg_xml(char* xmldoc, CgXmlNodeList* nodelist);

/*****************************************************************************
 * Capabilities
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to get the 
 * device's CDS (if any) search capabilities.
 *
 * @param upnDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param searchCaps The device's search capabilities (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_getsearchcaps(upnpDevice, \
					     errorString, \
					     searchCaps) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_GETSEARCHCAPABILITIES, \
		errorString, \
		CG_UPNP_CDS_ARG_SEARCHCAPS, \
		searchCaps, \
		NULL)
			
/**
 * Invoke an action on an AV device to get the
 * device's CDS (if any) sort capabilities.
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param sortCaps The device's sort capabilities (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_getsortcaps(upnpDevice, \
					   errorString, \
					   sortCaps) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_GETSORTCAPABILITIES, \
		errorString, \
		CG_UPNP_CDS_ARG_SORTCAPS, \
		sortCaps, \
		NULL)

/*****************************************************************************
 * System Update ID
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to get the
 * device's CDS (if any) system update ID.
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param id The device's update ID (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_getsystemupdateid(upnpDevice, \
						 errorString, \
						 id) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_GETSYSTEMUPDATEID, \
		errorString, \
		CG_UPNP_CDS_ARG_ID, \
		id, \
		NULL)


/*****************************************************************************
 * Browse
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to get the contents of an object.
 * See \ref ccdsactionarguments.h for some common values.
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param objectID The object ID to browse (IN)
 * @param browseFlag CG_UPNP_AV_CDS_BROWSE_FLAG_METADATA or
                     CG_UPNP_AV_CDS_BROWSE_FLAG_DIRECTCHILDREN (IN)
 * @param filter Browse filter (IN)
 * @param startingIndex Return results starting from the given index (IN)
 * @param requestedCount Maximum number of results to return (IN)
 * @param sortCriteria Sort results according to the given criteria (IN)
 * @param result The raw DIDL-Lite result from the device (OUT)
 * @param numberReturned Number of objects returned (OUT)
 * @param totalMatches Total number of objects in the container (OUT)
 * @param updateID Indicates, whether there have been changes in the container (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_browse(upnpDevice, \
				      errorString, \
				      objectID, \
				      browseFlag, \
				      filter, \
				      startingIndex, \
				      requestedCount, \
				      sortCriteria, \
				      result, \
				      numberReturned, \
				      totalMatches, \
				      updateID) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_BROWSE, \
		errorString, \
		CG_UPNP_CDS_ARG_OBJECTID, \
		objectID, \
		CG_UPNP_CDS_ARG_BROWSEFLAG, \
		browseFlag, \
		CG_UPNP_CDS_ARG_FILTER, \
		filter, \
		CG_UPNP_CDS_ARG_STARTINGINDEX, \
		startingIndex, \
		CG_UPNP_CDS_ARG_REQUESTEDCOUNT, \
		requestedCount, \
		CG_UPNP_CDS_ARG_SORTCRITERIA, \
		sortCriteria, \
		CG_UPNP_CDS_ARG_RESULT, \
		result, \
		CG_UPNP_CDS_ARG_NUMBERRETURNED, \
		numberReturned, \
		CG_UPNP_CDS_ARG_TOTALMATCHES, \
		totalMatches, \
		CG_UPNP_CDS_ARG_UPDATEID, \
		updateID, \
		NULL)

/*****************************************************************************
 * Search
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to search the contents for objects
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param containerID The container to start searching from (IN)
 * @param searchCriteria The search pattern (IN)
 * @param filter Browse filter (IN)
 * @param startingIndex Return results starting from the given index (IN)
 * @param requestedCount Number of results to return; use "0" to get everything (IN)
 * @param sortCriteria Sort results according to the given criteria (IN)
 * @param result The raw XML result from the \ref upnpDevice (OUT)
 * @param numberReturned Number of results returned (numberReturned <= totalMatches) (OUT)
 * @param totalMatches Total number of matches (numberReturned <= totalMatches) (OUT)
 * @param updateID The device's update ID (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_search(upnpDevice, \
				      errorString, \
				      containerID, \
				      searchCriteria, \
				      filter, \
				      startingIndex, \
				      requestedCount, \
				      sortCriteria, \
				      result, \
				      numberReturned, \
				      totalMatches, \
				      updateID) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_SEARCH, \
		errorString, \
		CG_UPNP_CDS_ARG_CONTAINERID, \
		containerID, \
		CG_UPNP_CDS_ARG_SEARCHCRITERIA, \
		searchCriteria, \
		CG_UPNP_CDS_ARG_FILTER, \
		filter, \
		CG_UPNP_CDS_ARG_STARTINGINDEX, \
		startingIndex, \
		CG_UPNP_CDS_ARG_REQUESTEDCOUNT, \
		requestedCount, \
		CG_UPNP_CDS_ARG_SORTCRITERIA, \
		sortCriteria, \
		CG_UPNP_CDS_ARG_RESULT, \
		result, \
		CG_UPNP_CDS_ARG_NUMBERRETURNED, \
		numberReturned, \
		CG_UPNP_CDS_ARG_TOTALMATCHES, \
		totalMatches, \
		CG_UPNP_CDS_ARG_UPDATEID, \
		updateID, \
		NULL)

/*****************************************************************************
 * Object manipulation
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to create an object to the server
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param containerID The container, under which to create the object (IN)
 * @param elements Initial object properties as a DIDL-Lite document (IN)
 * @param objectID The server-assigned object ID of the created object (OUT)
 * @param result The created object's <item>, <container> or <desc> hierarchy (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_createobject(upnpDevice, \
					    errorString, \
					    containerID, \
					    elements, \
					    objectID, \
					    result) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_CREATEOBJECT, \
		errorString, \
		CG_UPNP_CDS_ARG_CONTAINERID, \
		containerID, \
		CG_UPNP_CDS_ARG_ELEMENTS, \
		elements, \
		CG_UPNP_CDS_ARG_OBJECTID, \
		objectID, \
		CG_UPNP_CDS_ARG_RESULT, \
		result, \
		NULL)

/**
 * Invoke an action on a UPnP device to destroy an object from the server
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param objectID The ID of the object to destroy (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_destroyobject(upnpDevice, \
					     errorString, \
					     objectID) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_DESTROYOBJECT, \
		errorString, \
		CG_UPNP_CDS_ARG_OBJECTID, \
		objectID, \
		NULL)

/**
 * Invoke an action on a UPnP device to modify, insert or
 * delete an object's metadata
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param objectID The ID of the object to update (IN)
 * @param currentTagValue (IN)
 * @param newTagValue (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_updateobject(upnpDevice, \
					    errorString, \
					    objectID, \
					    currentTagValue, \
					    newTagValue) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_UPDATEOBJECT, \
		errorString, \
		CG_UPNP_CDS_ARG_OBJECTID, \
		objectID, \
		CG_UPNP_CDS_ARG_CURRENTTAGVALUE, \
		currentTagValue, \
		CG_UPNP_CDS_ARG_NEWTAGVALUE, \
		newTagValue, \
		NULL)

/*****************************************************************************
 * Object import/export
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to import a resource to a server
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param sourceURI The URI of the file to import (IN)
 * @param destinationURI The destination URI for the file (OUT)
 * @param transferURI A server-assigned ID for the data transfer (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_importresource(upnpDevice, \
					      errorString, \
					      sourceURI, \
					      destinationURI, \
					      transferID) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_IMPORTRESOURCE, \
		errorString, \
		CG_UPNP_CDS_ARG_SOURCEURI, \
		sourceURI, \
		CG_UPNP_CDS_ARG_DESTINATIONURI, \
		destinationURI, \
		CG_UPNP_CDS_ARG_TRANSFERID, \
		transferID, \
		NULL)

/**
 * Invoke an action on a UPnP device to export a resource from a server
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param sourceURI The URI of the file to export from the server (IN)
 * @param destinationURI The destination URI for the file (IN)
 * @param transferID A server-assigned ID for the data transfer (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_exportresource(upnpDevice, \
					      errorString, \
					      sourceURI, \
					      destinationURI, \
					      transferID) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_EXPORTRESOURCE, \
		errorString, \
		CG_UPNP_CDS_ARG_SOURCEURI, \
		sourceURI, \
		CG_UPNP_CDS_ARG_DESTINATIONURI, \
		destinationURI, \
		CG_UPNP_CDS_ARG_TRANSFERID, \
		transferID, \
		NULL)

/*****************************************************************************
 * Transfer status
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to stop a file transfer (import/export)
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param transferID An ID from a call to import/export resource (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_stoptransferresource(upnpDevice, \
						    errorString, \
						    transferID) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_STOPTRANSFERRESOURCE, \
		errorString, \
		CG_UPNP_CDS_ARG_TRANSFERID, \
		transferID, \
		NULL)

/**
 * Invoke an action on a UPnP device to get the progress of a file transfer
 * (import/export)
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param transferID An ID from a call to import/export resource (IN)
 * @param transferStatus Current transfer status (OUT)
 * @param transferLength Current transfer length (OUT)
 * @param transferTotal The total length of the transfer (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_gettransferprogress(upnpDevice, \
						   errorString, \
						   transferID, \
						   transferStatus, \
						   transferLength, \
						   transferTotal) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_GETTRANSFERPROGRESS, \
		errorString, \
		CG_UPNP_CDS_ARG_TRANSFERID, \
		transferID, \
		CG_UPNP_CDS_ARG_TRANSFERSTATUS, \
		transferStatus, \
		CG_UPNP_CDS_ARG_TRANSFERLENGTH, \
		transferLength, \
		CG_UPNP_CDS_ARG_TRANSFERTOTAL, \
		transferTotal, \
		NULL)

/*****************************************************************************
 * Resource deletion
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to delete a resource from an object
 *
 * NOTE: This does not delete an object. If you need to do that, use 
 * cg_upnp_av_cds_control_destroyobject()
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param resourceURI The <res> element to delete (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_deleteresource(upnpDevice, \
					      errorString, \
					      resourceURI) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_DELETERESOURCE, \
		errorString, \
		CG_UPNP_CDS_ARG_RESOURCEURI, \
		resourceURI, \
		NULL)

/*****************************************************************************
 * Reference creation
 *****************************************************************************/

/**
 * Invoke an action on a UPnP device to create a reference for an object
 * 
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param containerID (IN)
 * @param objectID The object ID to create a reference to (IN)
 * @param newID The new object ID of the reference (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_cds_control_createreference(upnpDevice, \
					       errorString, \
					       containerID, \
					       objectID, \
					       newID) \
	\
	cg_upnp_av_cds_control_invokeaction(upnpDevice, \
		CG_UPNP_CDS_ACTION_CREATEREFERENCE, \
		errorString, \
		CG_UPNP_CDS_ARG_CONTAINERID, \
		containerID, \
		CG_UPNP_CDS_ARG_OBJECTID, \
		objectID, \
		CG_UPNP_CDS_ARG_NEWID, \
		newID, \
		NULL)

#endif /* CG_UPNP_AV_CCDSCONTROLPOINT_H */
