/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Leonid Zolotarev <leonid.zolotarev@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#include <glib-object.h>

#ifndef __GWEBCOMMON_H__
#define __GWEBCOMMON_H__

G_BEGIN_DECLS

/** @mainpage EAL - Web Engine Abstraction Layer
  *
  * @section intro_sec Introduction
  *
  * @par
  *
  * EAL - the Web Engine Abstraction Layer Library provides an abstract interface
  * to the Web Engine Component.
  * It gives the possibility to develop applications with WEB Engine support
  * without any actual dependency to any particular WEB Engine.
  *
  * @par
  *
  * EAL was developed as a part of the
  * <a href="http://www.nokia.com/770/">Nokia 770</a> Browser Framework.
  * The library is based on GObject interface model and depends just on
  * <a href="http://developer.gnome.org/doc/API/2.0/glib/">GLib</a>.
  *
  * @par
  *
  * For more information refer to the Doxygen generated documentation
  * or directly to the header and source files.
  *
  * @section usage_sec Making applications with EAL
  *
  * @par
  *
  * The following items describe different aspects of the EAL Library
  * based application development.
  *
  * @subsection app_implementation 1. Implementation Layer
  *
  * The EAL Library contains the interface classes needed to provide an
  * implementation independent interface to the WEB engine functionality.
  * A browser application that uses the EAL interface should also
  * be linked to an "implementation library". The "Implementation library"
  * contains the actual functionality for the EAL Library interface classes.
  * The browser application should never use any of the engine specific
  * functions from the "implementation library".
  * It should completely rely upon the EAL Library interface.
  *
  * @par
  *
  * In this approach the module dependences are as follows:
  *
  * @dot
  *
  * digraph AppLibs {
  *     node [shape=record, fontname=Helvetica, fontsize=10];
  *     edge [arrowhead=open];
  *
  *     app     [label="Application"]
  *     imp_eal [label="EAL Implementation Library"]
  *     eal     [label="EAL Library"]
  *
  *     app -> imp_eal -> eal
  * }
  *
  * @enddot
  *
  * @par
  *
  * and logical dependences are:
  *
  * @dot
  *
  * digraph LogicalLibs {
  *     node [shape=record, fontname=Helvetica, fontsize=10];
  *
  *     app     [color=blue, label="Application"]
  *     imp_eal [color=blue, label="EAL Implementation Library"]
  *     eal     [label="interface\n\nEAL Library"]
  *
  *     app -> eal [arrowhead=open, style=dashed];
  *     imp_eal -> eal [style=dashed];
  * }
  *
  * @enddot
  *
  * @par
  *
  * The EAL library itself is useless without the implementation layer.
  * In order to use it in the browser application we need to implement
  * this layer using on some existing web engine. For example,
  * if we want to use the Mozilla Engine as an implementation of our engine
  * interface we need to create a wrapper for it. This wrapper or
  * "implementation library" will embed the Mozilla Engine functions and map them to
  * the relevant functions of the EAL Library.
  *
  * @par
  *
  * This figure below will describes the module dependences:
  *
  * @dot
  *
  * digraph MozAppLibs {
  *     node [shape=record, fontname=Helvetica, fontsize=10];
  *     edge [arrowhead=open]
  *
  *     app         [label="Application"]
  *     mozilla_eal [label="Mozilla EAL Library"]
  *     mozilla     [label="Mozilla Engine Libraries"]
  *     eal         [label="EAL Library"]
  *
  *     app -> mozilla_eal -> eal
  *     mozilla_eal -> mozilla
  * }
  *
  * @enddot
  *
  * @par
  *
  * and this logical dependences are:
  *
  * @dot
  *
  * digraph MozLogicalLibs {
  *     node [shape=record, fontname=Helvetica, fontsize=10];
  *
  *     app         [color=blue, label="Application"]
  *     mozilla_eal [color=blue, label="Mozilla EAL Library"]
  *     mozilla     [color=blue, label="Mozilla Engine Libraries"]
  *     eal         [label="interface\n\nEAL Library"]
  *
  *     app -> eal [arrowhead=open, style=dashed];
  *     mozilla_eal -> eal [style=dashed];
  *     mozilla_eal -> mozilla [arrowhead=open, style=dashed];
  * }
  *
  * @enddot
  *
  * @par
  *
  * See also <a href="http://www.mozilla.org/unix/gtk-embedding.html">
  * Gtk Mozilla Embedding Widget</a>.
  *
  * @subsection app_functions 2. Functions
  *
  * Every "implementation library" should implement g_web_new() function.
  * This function is used to instantiate a structure containing a pool of functions.
  * These functions implement the actual engine support. When application needs to
  * perform some engine specific operation it calls an EAL Library function with
  * the pointer to the object created with g_web_new() function.
  * The EAL Library function just calls the actual engine implementation function
  * from the function pool delivered with this object. This implementation
  * function wraps engine specific internals and hides them
  * from the rest of the application.
  *
  * @par
  *
  * Below, there is a call diagram for our Mozilla Engine example:
  *
  * @dot
  *
  * digraph MozAppFuncs {
  *     node [shape=record, fontname=Helvetica, fontsize=10]
  *     edge [arrowhead=open]
  *     compound=true;
  *     nodesep=1.0;
  *
  *     subgraph cluster_Libs {
  *         label="Libraries", fontname=Helvetica, fontsize=12
  *
  *         app         [label="Application"]
  *         eal         [label="EAL Library"]
  *         mozilla_eal [label="Mozilla EAL Library"]
  *         mozilla     [label="Mozilla Engine Libraries"]
  *
  *         app -> eal -> mozilla_eal -> mozilla
  *     }
  *
  *     subgraph cluster_Calls {
  *         label="Function calls", fontname=Helvetica, fontsize=12
  *
  *         c_app         [label="g_web_engine_load_url"]
  *         c_eal         [label="g_mozilla_engine_load_url"]
  *         c_mozilla_eal [label="gtk_mozilla_load_url"]
  *
  *         c_app -> c_eal -> c_mozilla_eal
  *     }
  *
  *     subgraph cluster_EalCalls {
  *         label="EAL Library", fontname=Helvetica, fontsize=12
  *
  *         e_pool [label="Pull function pointer: load_url"]
  *         e_func [label="g_mozilla_engine_load_url"]
  *
  *         e_pool -> e_func
  *     }
  *
  *     c_app:ne -> e_pool:sw [ lhead=cluster_EalCalls ];
  * }
  *
  * @enddot
  *
  * @subsection app_development 3. Application Development
  *
  * A typical EAL based application should also include some activation code for
  * the WEB Engine layer.
  *
  * @par
  *
  * The list below gives main steps to enable EAL support in your application:
  *
  * -# Include the gweb.h header file.
  * -# Instantiate the global WEB object with the g_web_new() function.
  * -# Perform some global initializations.
  *    For example, set the user agent string with the g_web_set_ua_string() function.
  * -# Get global implementation WEB object with g_web_get_global() function.  
  * -# Connect some global signals to the global implementation WEB object.
  *    For example, set the modal dialog callback with the 'g_signal_connect' function.
  * -# Create a new WEB Engine object with the g_web_new_web_engine_window() function.
  * -# Get a WEB Engine widget object from the WEB Engine object with
  *    the g_web_engine_get_engine_widget() function.
  * -# Connect some engine widget signals.
  *    For example, set URL changed callback with 'g_signal_connect' function.
  * -# Set the engine widget into the application window.
  * -# Finally, load some URL with the g_web_engine_load_url() function.
  *
  * @subsection app_linking 4. Linking
  *
  * The application should be linked to at least the EAL Library and
  * the Engine EAL Library.
  *
  * @par
  *
  * In case of the Mozilla engine 'Engine EAL Library' is the
  * 'Mozilla EAL Library'.
  *
  * @subsection app_deps 5. Application Dependencies
  *
  * The typical EAL based application dependencies are given below:
  *
  * @dot
  *
  * digraph AllMozAppLibs {
  *     node [shape=record, fontname=Helvetica, fontsize=10];
  *     edge [arrowhead=open]
  *
  *     app         [label="Application"]
  *     mozilla_eal [label="Mozilla EAL Library"]
  *     gtk         [label="GTK"]
  *     system      [label="System Libraries", style=dashed]
  *     eal         [label="EAL Library"]
  *     mozilla     [label="Mozilla Engine Libraries"]
  *     glib        [label="GLib"]
  *
  *     app -> mozilla_eal
  *     app -> gtk
  *     app -> system
  *
  *     mozilla_eal -> eal -> glib
  *     mozilla_eal -> mozilla
  * }
  *
  * @enddot
  *
  * @subsection app_headers 6. Headers
  *
  * The following headers are available for use in EAL based applications:
  *
  * @par
  *
  * -# gweb.h - the GWebIface definition.
  * -# gwebcommon.h - describes common enumerations.
  * -# gwebengine.h - the GWebEngineIface definition.
  * -# gwebhistorylist.h - the GWebHistoryListIface definition.
  * -# gwebpluginlist.h - the GWebPluginListIface definition.
  * -# gwebtransferitem.h - the GWebTransferItemIface definition.
  * -# gwebtransferitemlist.h - the GWebTransferItemListIface.
  * -# gwebwidget.h - contains the engine widget callback prototypes.
  *
  * @par
  *
  * The "Implementation library" headers must not be used in
  * an EAL based application directly.
  *
  * @subsection app_language 7. Language
  *
  * Object-oriented C based on
  * <a href="http://developer.gnome.org/doc/API/2.0/gobject/">
  * the GObject</a>
  * and
  * <a href="http://developer.gnome.org/doc/API/2.0/gobject/gtype-non-instantiable-classed.html">
  * GTypeInterface</a> concepts.
  *
  * @section interface_sec Class Interfaces
  *
  * The EAL Library introduces following class interfaces:
  *
  * @subsection iface_gweb 1. GWebIface
  *
  * GWebIface is an application wide object interface.
  * It is used for setting the web engine properties and performing the web engine related
  * operations globally in the application.
  *
  * @subsection iface_gwebengine 2. GWebEngineIface
  *
  * GWebEngineIface is the web engine object interface.
  * In the
  * <a href="http://wikipedia.org/wiki/Single_document_interface">
  * SDI</a> application architecture there is one engine object per
  * application. In the
  * <a href="http://wikipedia.org/wiki/Multiple_document_interface">
  * MDI</a> application architecture
  * there is one engine object per page or child window.
  *
  * @par
  *
  * Every web engine object has a corresponding rendering widget.
  * The application should implement callbacks for the web engine widget to be able
  * to handle page loading events and so on.
  * See the file gwebwidget.h for details.
  *
  * @par
  *
  * The web engine object itself is used for performing operations applied
  * to this particular rendering widget.
  *
  * @subsection iface_gwebhistorylist 3. GWebHistoryListIface
  *
  * GWebHistoryListIface - a list of browsing history objects per engine widget.
  * Used for implementing the "back/forward" functionality.
  *
  * @subsection iface_gwebpluginlist 4. GWebPluginListIface
  *
  * GWebPluginListIface - a list of installed plugins.
  * This list is an application wide object.
  *
  * @subsection iface_gwebtransferitem 5. GWebTransferItemIface
  *
  * GWebTransferItemIface represents a download item.
  * The application should set object specific callbacks to be able
  * to control downloading.
  *
  * @subsection iface_gwebtransferitemlist 6. GWebTransferItemListIface
  *
  * GWebTransferItemListIface - a list of download items.
  * This list is an application wide object.
  *
  * @section future_sec Future development
  *
  * In the future library interface classes can be decomposed into smaller
  * instances dedicated to some particular engine related task clusters.
  *
  */


/** GWebStatus.
  * Enumerates status types.
  */
typedef enum
{
	G_WEB_STATUS_OK          =  0,
	G_WEB_ERR                = -1,
	G_WEB_ERR_NO_MEMORY      = -2,
	G_WEB_ERR_NULL_POINTER   = -3,
	G_WEB_ERR_OUT_OF_RANGE   = -4,
	G_WEB_ERR_NO_ACCESS      = -5,
	G_WEB_ERR_FILE_NOT_FOUND = -7,
	G_WEB_ERR_NO_DISK        = -8,
	G_WEB_ERR_NOT_SUPPORTED  = -9
} GWebStatus;

/** GWebEngineEncoding.
  * Enumerates encoding types.
  */
typedef enum
{
	G_WEBENGINE_ENCODING_AUTOMATIC,
	G_WEBENGINE_ENCODING_UTF8,
	G_WEBENGINE_ENCODING_UTF16,
	G_WEBENGINE_ENCODING_USASCII,
	G_WEBENGINE_ENCODING_ISO8859_1,
	G_WEBENGINE_ENCODING_ISO8859_2,
	G_WEBENGINE_ENCODING_ISO8859_3,
	G_WEBENGINE_ENCODING_ISO8859_4,
	G_WEBENGINE_ENCODING_ISO8859_5,
	G_WEBENGINE_ENCODING_ISO8859_6,
	G_WEBENGINE_ENCODING_ISO8859_7,
	G_WEBENGINE_ENCODING_ISO8859_8,
	G_WEBENGINE_ENCODING_ISO8859_9,
	G_WEBENGINE_ENCODING_ISO8859_10,
	G_WEBENGINE_ENCODING_ISO8859_11,
	G_WEBENGINE_ENCODING_ISO8859_13,
	G_WEBENGINE_ENCODING_ISO8859_14,
	G_WEBENGINE_ENCODING_ISO8859_15,
	G_WEBENGINE_ENCODING_KOI8_R,
	G_WEBENGINE_ENCODING_KOI8_U,
	G_WEBENGINE_ENCODING_CP_1250,
	G_WEBENGINE_ENCODING_CP_1251,
	G_WEBENGINE_ENCODING_CP_1252,
	G_WEBENGINE_ENCODING_CP_1253,
	G_WEBENGINE_ENCODING_CP_1254,
	G_WEBENGINE_ENCODING_CP_1255,
	G_WEBENGINE_ENCODING_CP_1256,
	G_WEBENGINE_ENCODING_CP_1257,
	G_WEBENGINE_ENCODING_CP_1258,
	G_WEBENGINE_ENCODING_SHIFTJIS,
	G_WEBENGINE_ENCODING_ISO2022_JP,
	G_WEBENGINE_ENCODING_EUC_JP,
	G_WEBENGINE_ENCODING_GB2312,
	G_WEBENGINE_ENCODING_VISCII,
	G_WEBENGINE_ENCODING_EUC_KR,
	G_WEBENGINE_ENCODING_HZ_GB2312,
	G_WEBENGINE_ENCODING_GBK,
	G_WEBENGINE_ENCODING_GB_18030,
	G_WEBENGINE_ENCODING_BIG5,
	G_WEBENGINE_ENCODING_ISO_2022_CN,
	G_WEBENGINE_ENCODING_EUC_TW,
	G_WEBENGINE_ENCODING_WS2,
	G_WEBENGINE_ENCODING_IBM866,
	G_WEBENGINE_ENCODING_TCVN,
	G_WEBENGINE_ENCODING_VPS,
	G_WEBENGINE_ENCODING_JAPANESE_AUTOMATIC,
	G_WEBENGINE_ENCODING_CHINESE_AUTOMATIC
} GWebEngineEncoding;

/** GWebEngineLayoutMode.
  * Enumerates layout modes.
  */
typedef enum
{
	G_WEBENGINE_LAYOUT_NORMAL,
	G_WEBENGINE_LAYOUT_SSR,
	G_WEBENGINE_LAYOUT_MSR
} GWebEngineLayoutMode;

/** GWebEngineImagePolicy.
  * Enumerates image policy types.
  */
typedef enum
{
	G_WEBENGINE_POLICY_NO_IMAGES = 1,
	G_WEBENGINE_POLICY_LOADED_IMAGES,
	G_WEBENGINE_POLICY_ALL_IMAGES
} GWebEngineImagePolicy;


/** GWebWandAction.
  * Enumerates password storage action types.
  */
typedef enum { 
	G_WEB_PASSWORD_STORE = 0,
	G_WEB_PASSWORD_STORE_ENTIRE_SERVER, 
	G_WEB_PASSWORD_DONT_STORE, 
	G_WEB_PASSWORD_NEVER_STORE_ON_THIS_PAGE, 
	G_WEB_PASSWORD_NEVER_STORE_ON_ENTIRE_SERVER 
} GWebPasswordAction;

/** GWebEngineSaveAction.
  * Enumerates save actions.
  */
typedef enum
{
	G_WEBENGINE_SAVE_NO_INLINES,
	G_WEBENGINE_SAVE_ALL_INLINES,
	G_WEBENGINE_SAVE_FRAMES
} GWebEngineSaveAction;

/** GWebCookieAction.
  * Enumerates cookie actions.
  */
typedef enum
{
	G_WEB_COOKIE_NO_ACTION = 0,
	G_WEB_COOKIE_ACCEPT_AND_UPDATES = 1,
	G_WEB_COOKIE_ACCEPT_AND_SHOW_UPDATES = 2,
	G_WEB_COOKIE_ACCEPT_BUT_DISCARD = 4,
	G_WEB_COOKIE_ACCEPT_AND_SAVE = 8,
	G_WEB_COOKIE_ACCEPT_ALL_FROM_SERVER = 16,
	G_WEB_COOKIE_REFUSE_ALL_FROM_SERVER = 32,
	G_WEB_COOKIE_ACCEPT_ALL_FROM_DOMAIN = 64,
	G_WEB_COOKIE_REFUSE_ALL_FROM_DOMAIN = 128,
	G_WEB_COOKIE_3P_ACCEPT_ALL_FROM_SERVER = 256,
	G_WEB_COOKIE_3P_REFUSE_ALL_FROM_SERVER = 512,
	G_WEB_COOKIE_3P_ACCEPT_ALL_FROM_DOMAIN = 1024,
	G_WEB_COOKIE_3P_REFUSE_ALL_FROM_DOMAIN = 2048
} GWebCookieAction;

/** GWebCertificateAction.
  * Enumerates certificate actions.
  */
typedef enum
{
	G_WEB_CERT_ACTION_OK      = 0x0001, /**< OK */
	G_WEB_CERT_ACTION_CANCEL  = 0x0002, /**< Cancel */
	G_WEB_CERT_ACTION_ACCEPT  = 0x0004, /**< Accept */
	G_WEB_CERT_ACTION_INSTALL = 0x0008, /**< Install */
	G_WEB_CERT_ACTION_DELETE  = 0x0010  /**< Delete */
} GWebCertificateAction;

/** GWebCertificateReaction.
  * Enumerates certificate reactions.
  */
typedef enum
{
	G_WEB_CERT_REACTION_NOTHING, /**< No reaction needed. */
	G_WEB_CERT_REACTION_CLOSE,   /**< Close the dialog. */
	G_WEB_CERT_REACTION_REMOVE   /**< Remove the certificate from the dialog GUI. (The action caused the GWebCertificate to be deleted) */
} GWebCertificateReaction;

/** GWebCertificateMessage.
  * Enumerates certificate messages.
  */
typedef enum
{
  G_WEB_CERT_MESS_NOSTATE       = 0x0000,	/**< dummy state */
  G_WEB_CERT_MESS_UNTRUSTED	= 0x0001,	/**< untrusted */ 
  G_WEB_CERT_MESS_WRONGNAME	= 0x0002,       /**< invalid or mismatched name */
  G_WEB_CERT_MESS_EXPIRED	= 0x0004,       /**< expired */ 
  G_WEB_CERT_MESS_PREVALID	= 0x0008,       /**< not valid yet*/
  G_WEB_CERT_MESS_ALGORITHM	= 0x0010,       /**< wrong algorithm */
  G_WEB_CERT_MESS_VERSION	= 0x0020,       /**< wrong version */
  G_WEB_CERT_MESS_UNKNOWN	= 0x0040,       /**< unknown error */
  G_WEB_CERT_MESS_VALID  	= 0x0080        /**< valid */
} GWebCertificateMessage;

/** GWebDialogButton.
  * Enumerates dialog button types.
  */
typedef enum
{
	G_WEB_DIALOG_NO_BUTTON,
	G_WEB_DIALOG_BUTTON_OK,
	G_WEB_DIALOG_BUTTON_CANCEL,
	G_WEB_DIALOG_BUTTON_YES,
	G_WEB_DIALOG_BUTTON_NO,
	G_WEB_DIALOG_BUTTON_ABORT,
	G_WEB_DIALOG_BUTTON_RETRY,
	G_WEB_DIALOG_BUTTON_IGNORE
} GWebDialogButton;

/** GWebDialogMessageType.
  * Enumerates dialog message types.
  */
typedef enum
{
	G_WEB_DIALOG_MESSAGE_TYPE_INFO,
	G_WEB_DIALOG_MESSAGE_TYPE_WARNING,
	G_WEB_DIALOG_MESSAGE_TYPE_ERROR
} GWebDialogMessageType;

/** GWebPluginEvent.
  * Enumerates events to be sent to the plugin.
  */
typedef enum
{
	G_WEB_PLUGINS_SHOW = 1,
	G_WEB_PLUGINS_HIDE = 2,
	G_WEB_PLUGINS_RESUME = 3,
	G_WEB_PLUGINS_SUSPEND = 4,
	G_WEB_PLUGINS_LOW_MEMORY = 5,
	G_WEB_PLUGINS_NORMAL_MEMORY = 6,
	G_WEB_PLUGINS_LAST = 7
} GWebPluginEvent;

/** Create global Web Engine object.
  * @return Web Engine object
  * @note This function is not implemented in EAL library.
  * It should be implemented in engine specific EAL "implementation" library.
  */
GObject* g_web_new (void);

/** Set Web Engine specific environment.
  * @note This function is not implemented in EAL library.
  * It should be implemented in engine specific EAL "implementation" library.
  */
void g_web_set_env (void);

G_END_DECLS

#endif /* __GWEBCOMMON_H__ */
