/*
** Author: Pr Emanuelsson <pell@lysator.liu.se>
** Hacked by: Peter Eriksson <pen@lysator.liu.se>
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/file.h>

#define IN_LIBIDENT_SRC
#include "ident.h"

#include <arpa/inet.h>


static char *id_strdup __P1(char *, str)
{
    char *cp;

    cp = (char *) malloc(strlen(str)+1);
    if (cp == NULL)
    {
#ifdef DEBUG
	perror("libident: malloc");
#endif
        return NULL;
    }

    strcpy(cp, str);

    return cp;
}


static char *id_strtok __P3(char *, cp,
		      char *, cs,
		      char *, dc)
{
    static char *bp = 0;
    
    if (cp)
	bp = cp;
    
    /*
    ** No delimitor cs - return whole buffer and point at end
    */
    if (!cs)
    {
	while (*bp)
	    bp++;
	return cs;
    }
    
    /*
    ** Skip leading spaces
    */
    while (isspace(*bp))
	bp++;
    
    /*
    ** No token found?
    */
    if (!*bp)
	return 0;
    
    cp = bp;
    while (*bp && !strchr(cs, *bp))
	bp++;
    
    /*
    ** Remove trailing spaces
    */
    *dc = *bp;
    for (dc = bp-1; dc > cp && isspace(*dc); dc--)
	;
    *++dc = '\0';
    
    bp++;
    
    return cp;
}


static ident_t *id_open __P((	__STRUCT_IN_ADDR_P laddr,
			__STRUCT_IN_ADDR_P faddr,
			__STRUCT_TIMEVAL_P timeout))
{
    ident_t *id;
    int res, tmperrno;
    struct sockaddr_in sin_laddr, sin_faddr;
    fd_set rs, ws, es;
    int on = 1;
    
    if ((id = (ident_t *) malloc(sizeof(*id))) == 0)
	return 0;
    
    if ((id->fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
	free(id);
	return 0;
    }
    
    if (timeout)
    {
	if ((res = fcntl(id->fd, F_GETFL, 0)) < 0)
	    goto ERROR;

#ifndef VMS
	if (fcntl(id->fd, F_SETFL, res | FNDELAY) < 0)
	    goto ERROR;
#endif
    }

    (void) setsockopt(id->fd, SOL_SOCKET, SO_REUSEADDR, (void *) &on, sizeof(on));
    
    id->buf[0] = '\0';
    
    bzero((char *)&sin_laddr, sizeof(sin_laddr));
    sin_laddr.sin_family = AF_INET;
    sin_laddr.sin_addr = *laddr;
    sin_laddr.sin_port = 0;
    
    if (bind(id->fd, (struct sockaddr *) &sin_laddr, sizeof(sin_laddr)) < 0)
    {
#ifdef DEBUG
	perror("libident: bind");
#endif
	goto ERROR;
    }
    
    bzero((char *)&sin_faddr, sizeof(sin_faddr));
    sin_faddr.sin_family = AF_INET;
    sin_faddr.sin_addr = *faddr;
    sin_faddr.sin_port = htons(IDPORT);

    errno = 0;
    res = connect(id->fd, (struct sockaddr *) &sin_faddr, sizeof(sin_faddr));
    if (res < 0 && errno != EINPROGRESS)
    {
#ifdef DEBUG
	perror("libident: connect");
#endif
	goto ERROR;
    }

    if (timeout)
    {
	FD_ZERO(&rs);
	FD_ZERO(&ws);
	FD_ZERO(&es);
	
	FD_SET(id->fd, &rs);
	FD_SET(id->fd, &ws);
	FD_SET(id->fd, &es);

#ifdef __hpux
	if ((res = select(FD_SETSIZE, (int *) &rs, (int *) &ws, (int *) &es, timeout)) < 0)
#else
	if ((res = select(FD_SETSIZE, &rs, &ws, &es, timeout)) < 0)
#endif
	{
#ifdef DEBUG
	    perror("libident: select");
#endif
	    goto ERROR;
	}
	
	if (res == 0)
	{
	    errno = ETIMEDOUT;
	    goto ERROR;
	}
	
	if (FD_ISSET(id->fd, &es))
	    goto ERROR;
	
	if (!FD_ISSET(id->fd, &rs) && !FD_ISSET(id->fd, &ws))
	    goto ERROR;
    }
    
    return id;
    
  ERROR:
    tmperrno = errno;		/* Save, so close() won't erase it */
    close(id->fd);
    free(id);
    errno = tmperrno;
    return 0;
}


static int id_close __P1(ident_t *, id)
{
    int res;
  
    res = close(id->fd);
    free(id);
    
    return res;
}



static int id_parse __P((  ident_t *id,               
                    __STRUCT_TIMEVAL_P timeout,
                    int *lport,                
                    int *fport,                
                    char **identifier,         
                    char **opsys,              
                    char **charset))           
{
    char c, *cp, *tmp_charset;
    fd_set rs;
    int pos, res=0, lp, fp;
    
    errno = 0;
    
    tmp_charset = 0;
    
    if (!id)
	return -1;
    if (lport)
	*lport = 0;
    if (fport)
	*fport = 0;
    if (identifier)
	*identifier = 0;
    if (opsys)
	*opsys = 0;
    if (charset)
	*charset = 0;
    
    pos = strlen(id->buf);
    
    if (timeout)
    {
	FD_ZERO(&rs);
	FD_SET(id->fd, &rs);

#ifdef __hpux
	if ((res = select(FD_SETSIZE, (int *) &rs, (int *)0, (int *)0, timeout)) < 0)
#else
	if ((res = select(FD_SETSIZE, &rs, (fd_set *)0, (fd_set *)0, timeout)) < 0)
#endif
	    return -1;
	
	if (res == 0)
	{
	    errno = ETIMEDOUT;
	    return -1;
	}
    }
    
    /* Every octal value is allowed except 0, \n and \r */
    while (pos < sizeof(id->buf) &&
	   (res = read(id->fd, id->buf + pos, 1)) == 1 &&
	   id->buf[pos] != '\n' && id->buf[pos] != '\r')
	pos++;
    
    if (res < 0)
	return -1;
    
    if (res == 0)
    {
	errno = ENOTCONN;
	return -1;
    }
    
    if (id->buf[pos] != '\n' && id->buf[pos] != '\r')
	return 0;		/* Not properly terminated string */
    
    id->buf[pos++] = '\0';
    
    /*
    ** Get first field (<lport> , <fport>)
    */
    cp = id_strtok(id->buf, ":", &c);
    if (!cp)
	return -2;
    
    if (sscanf(cp, " %d , %d", &lp, &fp) != 2)
    {
	if (identifier)
	{
	    *identifier = id_strdup(cp);
	    if (*identifier == NULL)
	        return -4;
	}
	return -2;
    }
    
    if (lport)
	*lport = lp;
    if (fport)
	*fport = fp;
    
    /*
    ** Get second field (USERID or ERROR)
    */
    cp = id_strtok((char *)0, ":", &c);
    if (!cp)
	return -2;
    
    if (strcmp(cp, "ERROR") == 0)
    {
	cp = id_strtok((char *)0, "\n\r", &c);
	if (!cp)
	    return -2;
	
	if (identifier)
	{
	    *identifier = id_strdup(cp);
	    if (*identifier == NULL)
	        return -4;
	}
	
	return 2;
    }
    else if (strcmp(cp, "USERID") == 0)
    {
	/*
	** Get first subfield of third field <opsys>
	*/
	cp = id_strtok((char *) 0, ",:", &c);
	if (!cp)
	    return -2;
	
	if (opsys)
	{
	    *opsys = id_strdup(cp);
	    if (*opsys == NULL)
	        return -4;
	}
	
	/*
	** We have a second subfield (<charset>)
	*/
	if (c == ',')
	{
	    cp = id_strtok((char *)0, ":", &c);
	    if (!cp)
		return -2;
	    
	    tmp_charset = cp;
	    if (charset)
	    {
		*charset = id_strdup(cp);
		if (*charset == NULL)
		    return -4;
	    }
	    
	    /*
	    ** We have even more subfields - ignore them
	    */
	    if (c == ',')
		id_strtok((char *)0, ":", &c);
	}
	
	if (tmp_charset && strcmp(tmp_charset, "OCTET") == 0)
	    cp = id_strtok((char *)0, (char *)0, &c);
	else
	    cp = id_strtok((char *)0, "\n\r", &c);
	
	if (identifier && cp)
	{
	    *identifier = id_strdup(cp);
	    if (*identifier == NULL)
	        return -4;
	}
	return 1;
    }
    else
    {
	if (identifier)
	{
	    *identifier = id_strdup(cp);
	    if (*identifier == NULL)
	        return -4;
	}
	return -3;
    }
}



static int    id_query __P((	ident_t *id,                 
			int lport,                   
			int fport,                   
			__STRUCT_TIMEVAL_P timeout))
{
#ifdef SIGRETURNTYPE
    SIGRETURNTYPE (*old_sig)();
#else
    void (*old_sig) __P((int));
#endif
    int res;
    char buf[80];
    fd_set ws;
    
    sprintf(buf, "%d , %d\r\n", lport, fport);
    
    if (timeout)
    {
	FD_ZERO(&ws);
	FD_SET(id->fd, &ws);

#ifdef __hpux
	if ((res = select(FD_SETSIZE, (int *)0, (int *)&ws, (int *)0, timeout)) < 0)
#else
	if ((res = select(FD_SETSIZE, (fd_set *)0, &ws, (fd_set *)0, timeout)) < 0)
#endif
	    return -1;
	
	if (res == 0)
	{
	    errno = ETIMEDOUT;
	    return -1;
	}
    }

    old_sig = signal(SIGPIPE, SIG_IGN);
    
    res = write(id->fd, buf, strlen(buf));
    
    signal(SIGPIPE, old_sig);
    
    return res;
}


static IDENT *ident_query __P5(struct in_addr *, laddr,
			struct in_addr *, raddr,
			int, lport,
			int, rport,
			int, timeout)
{
    int res;
    ident_t *id;
    struct timeval timout;
    IDENT *ident=0;

    
    timout.tv_sec = timeout;
    timout.tv_usec = 0;
    
    if (timeout)
	id = id_open( laddr, raddr, &timout);
    else
	id = id_open( laddr, raddr, (struct timeval *)0);
    
    if (!id)
    {
	errno = EINVAL;
	return 0;
    }
  
    if (timeout)
	res = id_query(id, rport, lport, &timout);
    else
	res = id_query(id, rport, lport, (struct timeval *) 0);
    
    if (res < 0)
    {
	id_close(id);
	return 0;
    }
    
    ident = (IDENT *) malloc(sizeof(IDENT));
    if (!ident) {
	id_close(id);
	return 0;
    }
    
    if (timeout)
	res = id_parse(id, &timout,
		       &ident->lport,
		       &ident->fport,
		       &ident->identifier,
		       &ident->opsys,
		       &ident->charset);
    else
	res = id_parse(id, (struct timeval *) 0,
		       &ident->lport,
		       &ident->fport,
		       &ident->identifier,
		       &ident->opsys,
		       &ident->charset);
    
    if (res != 1)
    {
	free(ident);
	id_close(id);
	return 0;
    }
    
    id_close(id);
    return ident;			/* At last! */
}


/* Do a complete ident query and return result */

static IDENT *ident_lookup __P2(int, fd,
			 int, timeout)
{
    struct sockaddr_in localaddr, remoteaddr;
    int len;
    
    len = sizeof(remoteaddr);
    if (getpeername(fd, (struct sockaddr*) &remoteaddr, &len) < 0)
	return 0;
    
    len = sizeof(localaddr);
    if (getsockname(fd, (struct sockaddr *) &localaddr, &len) < 0)
	return 0;

    return ident_query( &localaddr.sin_addr, &remoteaddr.sin_addr,
		       ntohs(localaddr.sin_port), ntohs(remoteaddr.sin_port),
		       timeout);
}


static void ident_free __P1(IDENT *, id)
{
    if (!id)
	return;
    if (id->identifier)
	free(id->identifier);
    if (id->opsys)
	free(id->opsys);
    if (id->charset)
	free(id->charset);
    free(id);
}


char *ident_id __P2(int, fd,
		    int, timeout)
{
    IDENT *ident;
    char *id=0;
    
    ident = ident_lookup(fd, timeout);
    if (ident && ident->identifier && *ident->identifier)
    {
	id = id_strdup(ident->identifier);
	if (id == NULL)
	    return NULL;
    }

    ident_free(ident);
    return id;
}
