/**
    @file callbacks.c

    Callbacks for the user interface.

    Copyright (c) 2004-2006 Nokia Corporation.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#include <string.h>
#include <stdlib.h>
#include <math.h>

#include <gtk/gtk.h>
#include <libosso.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-banner.h>
#include <gdk/gdkkeysyms.h>
#include <ossoemailinterface.h>
#include <conbtdialogs-dbus.h>
#include <dbus/dbus-glib.h>

#include <comapp_sendvia.h>

#include "callbacks.h"
#include "interface.h"
#include "utility.h"
#include "pdfviewer.h"
#include "constant.h"
#include "state_save.h"
#include "debug.h"
#include "main.h"


#define NO_RESET_STATE_FILE

#define ACTIVE_AREA_WIDTH 72
#define SCREEN_WIDTH 800
#define MOVE_THRESHOLD 25

/*
 * Action callbacks for document menu
 */

void
on_document_open(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;
    
    /* Don't show dialog if application is rendering or saving */
    if( !pdf_viewer_is_rendering() &&
    	app_ui_data->app_data->state != PDF_VIEWER_STATE_SAVING ) {
    	
        ui_run_file_chooser(app_ui_data, GTK_FILE_CHOOSER_ACTION_OPEN);
    } else {
    	ui_show_banner( NULL, _("pdfv_ib_menu_not_available") );
    }
}

void
on_document_save(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;
    
    if( app_ui_data->app_data->state != PDF_VIEWER_STATE_LOADED ) {
    	ui_show_banner( NULL, _("pdfv_ib_menu_not_available") );
    } else {
	    ui_run_file_chooser(app_ui_data, GTK_FILE_CHOOSER_ACTION_SAVE);
	}
}


void
on_document_send_bt(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;
    GnomeVFSURI *uri = pdf_viewer_get_uri();
    gchar *file = NULL;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if (!uri)
        return;

    file = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);
    comapp_sendvia_bluetooth(file, GTK_WIDGET(app_ui_data->app_view));

    if (file)
        g_free(file);
}

void
on_document_send(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;
    GnomeVFSURI *uri = pdf_viewer_get_uri();
    gchar *file = NULL;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if (!uri)
        return;

    file = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);
    comapp_sendvia_email(GTK_WIDGET(app_ui_data->app_view),
                         app_ui_data->app_data->comapp_system->osso, file);

}

void
on_document_details(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    ui_run_details_dialog(app_ui_data);
}

/*
 * Action callbacks for page menu
 */

void
on_page_previous(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if( !pdf_viewer_is_rendering() ) {
        pdf_viewer_navigate(DOC_NAVI_PREVIOUS);
        ui_update_current_page(app_ui_data);
    }
}


void
on_page_next(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if( !pdf_viewer_is_rendering() ) {
    	
        pdf_viewer_navigate(DOC_NAVI_NEXT);
        ui_update_current_page(app_ui_data);
    }
}


void
on_page_first(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if ( !pdf_viewer_is_rendering() ) {
        pdf_viewer_navigate(DOC_NAVI_FIRST);
        ui_update_current_page(app_ui_data);
    }
}


void
on_page_last(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if ( !pdf_viewer_is_rendering() ) {
        pdf_viewer_navigate(DOC_NAVI_LAST);
        ui_update_current_page(app_ui_data);
    }
}


void
on_page_switch_to(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if( !pdf_viewer_is_rendering() ) {
        ui_run_switch_page_dialog(app_ui_data);
    }
}


/*
 * Action callbacks for screen menu
 */

void
on_screen_full_screen(GtkAction * action, gpointer user_data)
{
    gboolean active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
    ui_set_fullscreen((AppUIData *) user_data, active);
}


void
on_screen_zoom_in(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);

    if( !pdf_viewer_is_rendering() ) {
        pdf_viewer_zoom(DOC_ZOOM_IN);
    }
}


void
on_screen_zoom_out(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);

    if( !pdf_viewer_is_rendering() ) {
        pdf_viewer_zoom(DOC_ZOOM_OUT);
    }
}


void
on_screen_zoom_page(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);

    if( !pdf_viewer_is_rendering() ) {
        pdf_viewer_zoom(DOC_ZOOM_PAGE);
    }
}


void
on_screen_zoom_width(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);

    if( !pdf_viewer_is_rendering() ) {
        pdf_viewer_zoom(DOC_ZOOM_WIDTH);
    }
}


void
on_screen_show_images(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    gboolean active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

    g_return_if_fail(app_ui_data != NULL);

    if (active != PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES))
    {

        gboolean images;

        PDF_FLAGS_TOGGLE(app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);

        images = PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);

        ui_set_images_mode(app_ui_data, images);

    }

    if (pdf_viewer_get_num_pages() > 0)
        pdf_viewer_toggle_images();
}


/*
 * Action callbacks for tools menu
 */

void
on_tools_help(GtkAction * action, gpointer user_data)
{
    ui_run_help((AppUIData *) user_data, PDFV_HELP_MAINVIEW);
}


/*
 * Other action callbacks
 */

void
on_recent_item_cb(ComappRecent * recent, gchar * fileuri, gpointer data)
{
    AppUIData *app_ui = (AppUIData *) data;

    ui_open_document(app_ui, fileuri, NULL);
}

void
on_close(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    TDB("On close\n");
    app_ui_data->close_called = TRUE;
    g_return_if_fail(app_ui_data != NULL);

    /* destroy possible sources */
    if (app_ui_data->arrow_put_id)
    {
        g_source_remove(app_ui_data->arrow_put_id);
        app_ui_data->arrow_put_id = 0;
    }
    if (app_ui_data->arrow_hide_id)
    {
        g_source_remove(app_ui_data->arrow_hide_id);
        app_ui_data->arrow_hide_id = 0;
    }
    if (app_ui_data->arrow_right_id)
    {
        g_source_remove(app_ui_data->arrow_right_id);
        app_ui_data->arrow_right_id = 0;
    }
    if (app_ui_data->arrow_left_id)
    {
        g_source_remove(app_ui_data->arrow_left_id);
        app_ui_data->arrow_left_id = 0;
    }

    /* destroy state information */
    destroy_state_data(app_ui_data->app_data);
    /* Close application */
    gtk_main_quit();
}


void
on_switch_to_page_help(GtkAction * action, gpointer user_data)
{
    ui_run_help((AppUIData *) user_data, PDFV_HELP_SWITCH_TO_PAGE);
}


/** Action callback for undefined actions.
  * This function is invoked by all undefined menu and toolbar interactions.
  *	
  * @param action Corresponding action
  * @param user_data AppUIData
  * @return void
  */
void
on_undefined_action(GtkAction * action, gpointer user_data)
{
    g_debug("Unhandled action \"%s\"", gtk_action_get_name(action));
}


static gboolean
key_press_disabled(AppUIData * app_ui_data, GdkEventKey * event)
{
    /* 
     * rendering a document or no document is opened disabling HW keys,
     * except if it's SELECT HW button. */
    if( pdf_viewer_is_rendering() ) {
        return TRUE;             // rendering
    }

	/*
    if (pdf_viewer_get_uri() != NULL)
        return (0);             // all keys enabled

    if (event->keyval == GDK_Return)
        return (0);
    if (event->keyval == GDK_Escape)
        return (0);
    */

    return FALSE;                 // uri == null
}

/* Callback for keypresses */
gboolean
key_press(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
    AppUIData *app_ui_data;
    GtkAdjustment *adj = NULL;
    GtkWidget *w = NULL;

    app_ui_data = (AppUIData *) data;
    app_ui_data->key_pressed = event->keyval;
    
    switch (event->keyval)
    {
        /* F6 (or SELECT when file open) = toggle full screen mode */
        case GDK_KP_Enter:
        case GDK_Return:      
			if( !PDF_FLAGS_IS_SET(app_ui_data->flags,
				PDF_FLAGS_SELECT_KEY_ALLOWED ) ||
				pdf_viewer_get_num_pages() == 0 ) {
							
				break;
			}
			PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);			
        case GDK_F6:
            ui_toggle_fullscreen(app_ui_data,
	            !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_FULLSCREEN));
            return TRUE;
        case GDK_F4:           // we only return for menu key
        	return FALSE;
		case GDK_Escape:
            return FALSE;
    }

    if( key_press_disabled(app_ui_data, event) ) {
        return TRUE;
    }

    switch (event->keyval)
    {

		/* scrolling UP or LEFT is almost the same, just different
         * GtkAdjustments */
        /* UP = scroll up 20% */
        case GDK_KP_Up:
        case GDK_Up:
            adj = gtk_layout_get_vadjustment(GTK_LAYOUT(app_ui_data->layout));

            /* LEFT = scroll left 20% */
        case GDK_KP_Left:
        case GDK_Left:
            
            if (!adj) {

				/* Ignore when no document loaded */
				if( app_ui_data->app_data->state == PDF_VIEWER_STATE_EMPTY ) {
					ui_show_banner(GTK_WIDGET(app_ui_data->app_view),
						_("pdfv_ib_menu_not_available") );
            		return TRUE;
            	}

                adj = gtk_layout_get_hadjustment(
                	GTK_LAYOUT(app_ui_data->layout));
            }

            /* If scroll is far left/top or page fits to screen then try to
               move to previous page */
            if ( adj->upper <= adj->page_size || adj->value == 0.0 ) {
            	w = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					"/ToolBar/pdfv_me_menu_page_previous");

				if ((w != NULL) && (GTK_WIDGET_IS_SENSITIVE(w))) {
					on_page_previous(NULL, data);
				} else {
					if (pdf_viewer_get_current_page() == 1) {
                            ui_show_banner(GTK_WIDGET(app_ui_data->app_view),
                                           _("pdfv_ib_first_page_reached"));
                    }
                }

                return TRUE;
            }            
            
            /* scroll the page by 20% */            
            if (adj->lower <= adj->value - 0.2 * adj->page_size)
                gtk_adjustment_set_value(adj,
                                         adj->value - 0.2 * adj->page_size);
            else
                gtk_adjustment_set_value(adj, adj->lower);

            return TRUE;

            /* scrolling DOWN or RIGHT is almost the same, just different
             * GtkAdjustments */
            /* DOWN = scroll down 20% */
        case GDK_KP_Down:
        case GDK_Down:
            adj = gtk_layout_get_vadjustment(GTK_LAYOUT(app_ui_data->layout));

            /* RIGHT = scroll to the right 20% */
        case GDK_KP_Right:
        case GDK_Right:

			if( !adj ) {
	            adj = gtk_layout_get_hadjustment(GTK_LAYOUT(app_ui_data->layout));

				/* Ignore when no document loaded */
				if( app_ui_data->app_data->state == PDF_VIEWER_STATE_EMPTY ) {
					ui_show_banner(GTK_WIDGET(app_ui_data->app_view),
						_("pdfv_ib_menu_not_available") );
	           		return TRUE;
	           	}
	           	
	        }
            
	        if( adj->page_size < adj->upper &&
	           	adj->value < ( adj->upper - adj->page_size ) ) {
		           
				/* scroll the page by 20% */
	    	    if (adj->value + adj->page_size * 0.2 <=
	    	    	adj->upper - adj->page_size)
						gtk_adjustment_set_value(adj,
							adj->value + adj->page_size * 0.2);
	    	    else
					gtk_adjustment_set_value(adj, adj->upper - adj->page_size);
	    	    
			/* Move more space to move, to next page */             
			} else {
	           	on_page_next(NULL, data);
			}

            return TRUE;

            /* display infoprint when maximum or minimum zoom level has been
             * reached. See Bugzilla 18393 */

        /* F8 = zoom out */
        case GDK_F8:
	        w = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				"/ToolBar/pdfv_me_menu_screen_zoom_out");
			if ((w != NULL) && (GTK_WIDGET_IS_SENSITIVE(w))) {
				pdf_viewer_zoom(DOC_ZOOM_OUT);
			} else if (pdf_viewer_get_zoom_percent() <= 50) {
				ui_show_banner(GTK_WIDGET(app_ui_data->app_view),
					_("pdfv_ib_minimum_zoom"));
			}
			return TRUE;

        /* F7 = zoom in */
        case GDK_F7:
	        w = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
 	           "/ToolBar/pdfv_me_menu_screen_zoom_in");
            if ((w != NULL) && (GTK_WIDGET_IS_SENSITIVE(w))) {
				pdf_viewer_zoom(DOC_ZOOM_IN);
			} else if (pdf_viewer_get_zoom_percent() >= 400) {
	            ui_show_banner(GTK_WIDGET(app_ui_data->app_view),
	                _("pdfv_ib_maximum_zoom"));
			}
			return TRUE;

            /* 
             * RETURN/SELECT = if no document present Open file dialog else
             * full screen mode 
        case GDK_KP_Enter:
        case GDK_Return:
	        if( !PDF_FLAGS_IS_SET(app_ui_data->flags,
	        	PDF_FLAGS_SELECT_KEY_ALLOWED ) ) {
	        	break;
	        }
            PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);
            
            if( (pdf_viewer_get_num_pages() != 0) ||
                !(GTK_WIDGET_IS_SENSITIVE( gtk_ui_manager_get_widget
                  (app_ui_data->ui_manager,
                  	"/MenuBar/pdfv_me_main_menu_document/"
                  	"pdfv_me_menu_document_open")))) {

                ui_toggle_fullscreen(app_ui_data,
                	!PDF_FLAGS_IS_SET( app_ui_data->flags,
						PDF_FLAGS_FULLSCREEN));
                  	
            }
            
            return TRUE;
			*/

        default:
            break;
    }

    return FALSE;
}


/* 
 * Navigation keys Arrow keys GDK_Left, GDK_Right, GDK_Up, GDK_Down Cancel
 * (Escape) Esc GDK_Escape Menu key F4 GDK_F4 Home F5 GDK_F5 Fullscreen F6
 * GDK_F6 Plus (Zoom in) F7 GDK_F7 Minus (Zoom out) F8 GDK_F8 */


/* non-repeating key handlers */
gboolean
key_release(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
    AppUIData *app_ui_data = (AppUIData *) data;
    guint key_pressed_was;

    g_return_val_if_fail(app_ui_data != NULL, FALSE);
    
    key_pressed_was = app_ui_data->key_pressed;
    app_ui_data->key_pressed = 0;
       
    switch (event->keyval)
    {
        case GDK_KP_Enter:
        case GDK_Return:
                    
            if( ( key_pressed_was == GDK_KP_Enter ||
            	key_pressed_was == GDK_Return ) &&
            	(pdf_viewer_get_num_pages() == 0) &&
                (GTK_WIDGET_IS_SENSITIVE
                 (gtk_ui_manager_get_widget
                  (app_ui_data->ui_manager,
                   "/MenuBar/pdfv_me_main_menu_document/"
                   "pdfv_me_menu_document_open"))))
            {
                ui_run_file_chooser(app_ui_data,
                                    GTK_FILE_CHOOSER_ACTION_OPEN);
            }
            
        case GDK_F6:                    
            
            PDF_FLAGS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);            
            return TRUE;
            
        case GDK_Escape:
            if( key_pressed_was == GDK_Escape &&
            	PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_FULLSCREEN )) {
                ui_toggle_fullscreen( app_ui_data, FALSE );
                return FALSE;
            }
            break;            
    }

    if (key_press_disabled(app_ui_data, event))
        return TRUE;

	return FALSE;

}


void
on_screen_scroll(GtkAdjustment * adjustment, gpointer user_data)
{
    AppData *app_data = get_app_data();
    ui_hide_arrows_if_exists(app_data->app_ui_data, FALSE);
    pdf_viewer_scroller_changed((PDFScroll) GPOINTER_TO_UINT(user_data));
}


void
on_screen_tap_and_hold_event(GtkWidget * widget, gpointer data)
{
    gtk_grab_remove(widget);

    gtk_menu_popup(GTK_MENU(data), NULL, NULL, NULL, NULL, 0,
                   gtk_get_current_event_time());
}


gboolean
on_screen_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    g_return_val_if_fail(app_ui_data != NULL, TRUE);

    if (app_ui_data->app_data->state <= PDF_VIEWER_STATE_LOADING &&
        event->type == GDK_BUTTON_PRESS)
        return TRUE;

    return FALSE;
}


gboolean
on_screen_press(GtkWidget * widget, GdkEventButton * event,
                gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    g_return_val_if_fail(app_ui_data != NULL, TRUE);
    
    /* Initialize values for potential panning */
    app_ui_data->press_lastx = app_ui_data->lastx = (gint) event->x;
    app_ui_data->press_lasty = app_ui_data->lasty = (gint) event->y;

    /* We are not panning yet */
    PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_PANNING);
    PDF_FLAGS_SET(app_ui_data->flags, PDF_FLAGS_BUTTON_DOWN);

    gtk_grab_add(widget);

    return FALSE;
}


gboolean
on_screen_release(GtkWidget * widget, GdkEventButton * event,
                  gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    guint press_pos = 0, scrollbar_pos = 0;
	gint move = 0; /* -1 prev, 1 next */

    g_return_val_if_fail(app_ui_data != NULL, TRUE);
           
    scrollbar_pos =
        (guint) gtk_range_get_value(GTK_RANGE(app_ui_data->hscroll));
    press_pos = (guint) event->x;

    /* Stop panning */
    PDF_FLAGS_UNSET(app_ui_data->flags,
                    PDF_FLAGS_PANNING | PDF_FLAGS_BUTTON_DOWN);

    gtk_grab_remove(widget);
      
    /* Check for FS and tap, then identify the action */
    if( PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_FULLSCREEN ) ) {
    
    	/* If empty or loading always give banner */
    	if( app_ui_data->app_data->state == PDF_VIEWER_STATE_EMPTY ||
    		app_ui_data->app_data->state == PDF_VIEWER_STATE_LOADING ) {
	    	ui_show_banner( NULL, _("pdfv_ib_menu_not_available") );
	    	return FALSE;    	
    	
    	} else if(
    		( abs(app_ui_data->press_lastx - event->x) < MOVE_THRESHOLD ) &&
			( abs(app_ui_data->press_lasty - event->y) < MOVE_THRESHOLD ) ) {
		
			if (press_pos >= (scrollbar_pos + SCREEN_WIDTH - ACTIVE_AREA_WIDTH)) {
        		move = 1;
        	} else if( press_pos <= (scrollbar_pos + ACTIVE_AREA_WIDTH) ) {
        		move = -1;
        	}
        }
    }
    
    /* We can quit if we don't have an action */
    if( move == 0 ) {
    	return FALSE;
    }
    
    /* Check if it's ok to run action */
    if( pdf_viewer_is_rendering() || 
    	!PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED) ) {
    	ui_show_banner( NULL, _("pdfv_ib_menu_not_available") );
    	
	/* Do the actions */
    } else if( move == 1 ) {
	    on_page_next(NULL, user_data);
	} else if( move == -1 ) {
		on_page_previous(NULL, user_data);
	}

    return FALSE;
}


gboolean
on_screen_motion(GtkWidget * widget, GdkEventMotion * event,
                 gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    GtkAdjustment *adj;
    gdouble value;
    gint x, y;
    gint dx, dy;
    g_return_val_if_fail(app_ui_data != NULL, TRUE);

    /* ignore event if button is not pressed, or page is rendering */
    /* TODO could add test to ensure that document is loaded */
    if (!PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_BUTTON_DOWN)
        || pdf_viewer_is_rendering() ) {
        return TRUE;
    }

    if (event->is_hint)
    {
        /* hint event, retrieve pointer location */
        gdk_window_get_pointer(event->window, &x, &y, NULL);
    }
    else
    {
        /* use motion event coordinates */
        x = (gint) event->x;
        y = (gint) event->y;
    }

    /* calculate delta values */
    dx = x - app_ui_data->lastx;
    dy = y - app_ui_data->lasty;

    if (!PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_PANNING))
    {
        /* not panning yet, check if threshold exceeded */
        if (ABS(dx) > PANNING_THRESHOLD || ABS(dy) > PANNING_THRESHOLD)
        {
            /* start panning */
            PDF_FLAGS_SET(app_ui_data->flags, PDF_FLAGS_PANNING);

            /* don't move just yet */
            app_ui_data->lastx = x;
            app_ui_data->lasty = y;
        }
    }
    else
    {
        /* panning */

        /* retrieve and set vertical adjustment, ensure that range is not
         * exceeded */
        adj = gtk_layout_get_vadjustment(GTK_LAYOUT(app_ui_data->layout));
        value = adj->value - (gdouble) dy;
        if (value < adj->lower)
            value = adj->lower;
        else if (value > (adj->upper - adj->page_size))
            value = adj->upper - adj->page_size;
        gtk_adjustment_set_value(adj, value);

        /* retrieve and set horizontal adjustment */
        adj = gtk_layout_get_hadjustment(GTK_LAYOUT(app_ui_data->layout));
        value = adj->value - (gdouble) dx;
        if (value < adj->lower)
            value = adj->lower;
        else if (value > (adj->upper - adj->page_size))
            value = adj->upper - adj->page_size;
        gtk_adjustment_set_value(adj, value);

        /* NOTE on_screen_scroll handler is invoked automatically when
         * gtk_adjustment_set_value is called */
    }

    return FALSE;
}


/**
 
	Handles top/untop event.

	@param widget Pointer to window widget
	@param data Pointer to AppUIData
*/

void
top_changed(GObject * self, GParamSpec * property_param, gpointer data)
{
    HildonProgram *program = HILDON_PROGRAM(self);
    if (hildon_program_get_is_topmost(program))
    {
        AppUIData *app_ui_data = (AppUIData *) data;

        /* topmost status acquired */
        g_return_if_fail(app_ui_data != NULL);
        g_return_if_fail(app_ui_data->app_data != NULL);
        // g_return_if_fail(app_ui_data->app_data->osso_context != NULL);

        /* destory the state file */
        // destroy_state_data(app_ui_data->app_data);

        PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_BACKGROUNDED);

        hildon_program_set_can_hibernate(app_ui_data->app, FALSE);

    }
    else
    {
        /* topmost status lost, save state */

        AppUIData *app_ui_data = (AppUIData *) data;

        if (app_ui_data->app_data == NULL ||
            app_ui_data->app_data->comapp_system->osso == NULL)
        {
            return;
        }

        /* Save application's state */
        save_app_state(app_ui_data->app_data);

        /* Key repeat flags need to be reset on focus out */
        PDF_FLAGS_SET(app_ui_data->flags,
                      PDF_FLAGS_SELECT_KEY_ALLOWED | PDF_FLAGS_BACKGROUNDED);

        hildon_program_set_can_hibernate(app_ui_data->app, TRUE);

    }
}



gboolean
on_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    gboolean retval = FALSE;
    g_return_val_if_fail(app_ui_data != NULL, FALSE);
    g_return_val_if_fail(app_ui_data ->app_data != NULL, FALSE);

    TDB("%s\n", __FUNCTION__);

    if (app_ui_data->app_data->state == PDF_VIEWER_STATE_LOADING ) {
        retval = TRUE;
    } else {
        app_ui_data->close_called = TRUE;
        g_return_val_if_fail(app_ui_data != NULL, FALSE);

        destroy_state_data(app_ui_data->app_data);
        retval = TRUE;
        gtk_main_quit();
    }
    return retval;
}

typedef struct {
    HildonNumberEditor *hildonnumbereditor;
    AppUIData *app_ui_data;
} SNumberError;


gboolean
show_number_editor_error(gpointer data)
{
    SNumberError *number = (SNumberError *) data;
    guint min = PDF_PAGE_INIT + 1,
        max = pdf_viewer_get_num_pages();

    gchar *info = NULL;
    GDK_THR_ENTER;
    if (!G_IS_OBJECT(number->hildonnumbereditor))
    {
        GDK_THR_LEAVE;
        return FALSE;
    }

    if ((info =
         g_strdup_printf(SUPPRESS_FORMAT_WARNING(_("pdfv_error_pagenumber")),
                         min, max)) == NULL)
    {
        GDK_THR_LEAVE;
        return FALSE;
    }

    hildon_banner_show_information(GTK_WIDGET(number->app_ui_data->app_view),
                                   NULL, info);
    g_free(info);
    g_object_set_data(G_OBJECT(number->hildonnumbereditor), "idle-id",
                      (gpointer) 0);
    GDK_THR_LEAVE;
    return FALSE;
}

gboolean
on_number_editor_error(HildonNumberEditor * hildonnumbereditor,
                       gint arg1, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    SNumberError *error;

    g_return_val_if_fail(app_ui_data != NULL, FALSE);

    if (g_object_get_data(G_OBJECT(hildonnumbereditor), "idle-id"))
        return TRUE;

    error = g_new(SNumberError, 1);
    error->app_ui_data = app_ui_data;
    error->hildonnumbereditor = hildonnumbereditor;
    g_object_set_data(G_OBJECT(hildonnumbereditor), "idle-id",
                      (gpointer) g_idle_add_full(G_PRIORITY_DEFAULT_IDLE,
                                                 show_number_editor_error,
                                                 error,
                                                 (GDestroyNotify) g_free));

    return TRUE;
}

/**
	Handles on_file_changed event

	@param handle Gnome vfs monitor handle
	@param monitor_uri uri to monitor
	@param info_uri, not used
	@param event_type the type of the file event
	@param user_data pointer to AppData as gpointer
*/
void
on_file_changed(GnomeVFSMonitorHandle * handle,
                const gchar * monitor_uri,
                const gchar * info_uri,
                GnomeVFSMonitorEventType event_type, gpointer user_data)
{
    AppData *app_data = (AppData *) user_data;
    gchar *loaded_uri = NULL;
    GtkWidget *widget = NULL;
    g_return_if_fail(app_data != NULL);

    switch (event_type)
    {
        case GNOME_VFS_MONITOR_EVENT_DELETED:

            /* check if this document is currently loaded */
            if (app_data->state == PDF_VIEWER_STATE_LOADED)
            {
                loaded_uri = gnome_vfs_uri_to_string(pdf_viewer_get_uri(),
                                                     GNOME_VFS_URI_HIDE_NONE);

                if (strcmp(monitor_uri, loaded_uri) == 0)
                {
                    /* Dim 'save as' item */
                    widget =
                        gtk_ui_manager_get_widget(app_data->app_ui_data->
                                                  ui_manager,
                                                  "/MenuBar/pdfv_me_main_menu_document/"
                                                  "pdfv_me_menu_document_save");
                    gtk_widget_set_sensitive(widget, FALSE);
                    /* as per spec, a deleted file which is open in an
                     * application should be closed */
                    pdf_viewer_unload();
                }

                g_free(loaded_uri);
            }

            comapp_recent_delete_item(COMAPP_RECENT
                                      (app_data->app_ui_data->recent_menu),
                                      monitor_uri);
            OSSO_LOG_DEBUG("file has been deleted! info: %s, monitor: %s\n",
                           info_uri, monitor_uri);

            break;
        case GNOME_VFS_MONITOR_EVENT_CREATED:
        case GNOME_VFS_MONITOR_EVENT_CHANGED:
            OSSO_LOG_DEBUG("file has been changed! info: %s, monitor: %s\n",
                           info_uri, monitor_uri);
            break;
        default:
            OSSO_LOG_DEBUG("file has been something! info: %s, monitor: %s\n",
                           info_uri, monitor_uri);
            break;
    }
}
