
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "gps.h"

/* 
   Probably you need to run this module test inside scratchbox.

   ${CC:=gcc} -I. -g view_last_saved_report.c -lgps

*/


extern int gps_get_last_saved_report(struct gps_data_t *gpsdata, char *buf, int buflen);


static void
dump_report (struct gps_data_t *gps_data)
{
	printf("Dumping last saved report\n");
	printf("Fields set - %d (0x%x)\n", gps_data->set, gps_data->set);
	printf("Time set - %f\n", gps_data->fix.time);
	printf("Latitude - %f\n", gps_data->fix.latitude);
	printf("Longitude - %f\n", gps_data->fix.longitude);
	printf("Altitude - %f\n", gps_data->fix.altitude);
	printf("Track - %f\n", gps_data->fix.track);
	printf("Speed - %f\n", gps_data->fix.speed);
	printf("Climb - %f\n", gps_data->fix.climb);

	printf("Satellites in view - %d\n", gps_data->satellites);
	printf("Satellites in use - %d\n", gps_data->satellites_used);
}


void data_dump(struct gps_data_t *collect, time_t now)
{
    char *status_values[] = {"NO_FIX", "FIX", "DGPS_FIX"};
    char *mode_values[] = {"", "NO_FIX", "MODE_2D", "MODE_3D"};

    if (collect->set & ONLINE_SET)
	printf("online: %lf\n", collect->online);
    if (collect->set & LATLON_SET)
	printf("P: lat/lon: %lf %lf\n", collect->fix.latitude, collect->fix.longitude);
    if (collect->set & ALTITUDE_SET)
	printf("A: altitude: %lf  U: climb: %lf\n", 
	       collect->fix.altitude, collect->fix.climb);
    if (collect->set & TRACK_SET)
	    if (!isnan(collect->fix.track))
		    printf("T: track: %lf  V: speed: %lf\n", 
			   collect->fix.track, collect->fix.speed);
    if (collect->set & STATUS_SET)
	printf("S: status: %d (%s)\n", 
	       collect->status, status_values[collect->status]);
    if (collect->fix.mode & MODE_SET)
	printf("M: mode: %d (%s)\n", 
	   collect->fix.mode, mode_values[collect->fix.mode]);
    if (collect->fix.mode & (HDOP_SET | VDOP_SET | PDOP_SET))
	printf("Q: satellites %d, pdop=%lf, hdop=%lf, vdop=%lf\n",
	   collect->satellites_used, 
	   collect->pdop, collect->hdop, collect->vdop);

    if (collect->set & SATELLITE_SET) {
	int i;

	printf("Y: satellites in view: %d\n", collect->satellites);
	for (i = 0; i < collect->satellites; i++) {
	    printf("    %2.2d: %2.2d %3.3d %3.3d %c\n", collect->PRN[i], collect->elevation[i], collect->azimuth[i], collect->ss[i], collect->used[i]? 'Y' : 'N');
	}
	printf("Y: satellites used : %d\n", collect->satellites_used);
    }
    if (collect->set & DEVICE_SET)
	printf("Device is %s\n", collect->gps_device);
    if (collect->set & DEVICEID_SET)
	printf("GPSD ID is %s\n", collect->gps_id);
    if (collect->set & DEVICELIST_SET) {
	int i;
	printf("%d devices:\n", collect->ndevices);
	for (i = 0; i < collect->ndevices; i++) {
	    printf("%d: %s\n", collect->ndevices, collect->devicelist[i]);
	}
    }
	
}



int last_report_test(char *file, struct gps_report_t *last_report)
{
	if (last_report) {
		struct stat fst = {0};
		int fd, fsize = sizeof(struct gps_report_t);
		int ret, st;

		fd = open(file, O_RDONLY);
		if (fd < 0) {
			printf("Cannot open %s [%s,%d]\n", file, strerror(errno), errno);
			return -1;
		}

		if (fstat(fd, &fst) < 0) {
			perror("fstat");
			st = -1;
			goto ERR;
		}

		if (fst.st_size == 0) {
			/* gpsd is running, use the gpsd interface to get the current fix */
			st = 1;
			goto ERR;
		}

		if ((int)fst.st_size != fsize) {
			printf("File %s size mismatch (was %d, should be %d)\n", file, (int)fst.st_size, fsize);
			st = -1;
			goto ERR;
		}

		ret = read(fd, (void *)last_report, fsize);
		if (ret != fsize) {
			printf("File %s read error, bytes read was %d, should be %d\n", file, ret, fsize);
			perror("read");
			st = -1;
			goto ERR;
		}

		close(fd);
		goto OK;
		
	ERR:
		close(fd);
		return st;

	} else {
		return -1;
	}

OK:
	return 0;
}


#define DUMP(flag)				\
	if ((set) & (flag))			\
		printf("%s ", #flag)

void dump_set(gps_mask_t set)
{
	DUMP(ONLINE_SET);
	DUMP(TIME_SET);
	DUMP(TIMERR_SET);
	DUMP(LATLON_SET);
	DUMP(ALTITUDE_SET);
	DUMP(SPEED_SET);
	DUMP(TRACK_SET);
	DUMP(CLIMB_SET);
	DUMP(STATUS_SET);
	DUMP(MODE_SET);
	DUMP(HDOP_SET);
	DUMP(VDOP_SET);
	DUMP(PDOP_SET);
	DUMP(TDOP_SET);
	DUMP(GDOP_SET);
	DUMP(DOP_SET);
	DUMP(HERR_SET);
	DUMP(VERR_SET);
	DUMP(PERR_SET);
	DUMP(SATELLITE_SET);
	DUMP(PSEUDORANGE_SET);
	DUMP(USED_SET);
	DUMP(SPEEDERR_SET);
	DUMP(TRACKERR_SET);
	DUMP(CLIMBERR_SET);
	DUMP(DEVICE_SET);
	DUMP(DEVICELIST_SET);
	DUMP(DEVICEID_SET);
	DUMP(ERROR_SET);
	DUMP(CYCLE_START_SET);
	DUMP(RTCM_SET);
	DUMP(FIX_SET);

	printf("\n");
}


int main(int argc, char **argv)
{
	int st = 0, i;
	struct gps_data_t gpsdata;
	struct gps_report_t last_report = {0};
	char *file;

	if (argc>1) {
		file = argv[1];
		if (!last_report_test(file, &last_report)) {
			printf("Current set = 0x%x\nAccumulated set = 0x%x\n",
			       last_report.gpsdata.set, last_report.set);
			printf("Current = "); dump_set(last_report.gpsdata.set);
			printf("Accumulated = "); dump_set(last_report.set);

			printf("------\nWith current set:\n");
			memcpy(&gpsdata, &last_report.gpsdata, sizeof(gpsdata));
			dump_report(&gpsdata);
			printf("----\n");
			data_dump(&gpsdata, time(0));
			printf("\n");
			printf("------\nWith accumulated set:\n");
			memcpy(&gpsdata, &last_report.gpsdata, sizeof(gpsdata));
			gpsdata.set = last_report.set;
			dump_report(&gpsdata);
			printf("----\n");
			data_dump(&gpsdata, time(0));

		} else
			printf("last report test failed\n");
	} else {
		memset(&gpsdata, 0, sizeof(gpsdata));
		if (!gps_get_last_saved_report(&gpsdata, NULL, -1)) {
			dump_report(&gpsdata);
			printf("----\n");
			data_dump(&gpsdata, time(0));
		} else
			printf("Cannot get last report\n");
	}
}

