/**
 * Copyright (C) 2007 Nokia Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */


#define DBUS_API_SUBJECT_TO_CHANGE

#include <libmodest-dbus-client.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <string.h>
#include "libosso.h"
#include <log-functions.h>
#include "osso-log.h"
#include "ossoemailinterface-modest.h"
#include "libmodest-dbus-client.h"

/**
   This function will send a top D-Bus message to the email
   application with a list of gnomeVFS URI's as argutments.
 
   @param osso The library context as returned by #osso_initialize.
   @param list A GSList of strings -- URI's in GnomeVFS format.
   
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
*/
osso_return_t osso_email_files_email_modest (osso_context_t * osso, GSList * list)
{
	gchar *args = NULL;
	gboolean ret = FALSE;

	d_log(LOG_D, "send as files to email  application \n");

	if ((osso == NULL) || (list == NULL)) {
		return OSSO_INVALID;
	}

	ret = libmodest_dbus_client_compose_mail (osso,
					       NULL, /*to*/
					       NULL, /*cc*/ 
					       NULL, /*bcc*/
					       NULL, /*subject*/
					       NULL, /*body*/
					       list  /*attachments*/);
	g_free(args);

	if(ret) {
		return OSSO_OK;
	} else {
		return OSSO_ERROR;
	}
}

osso_return_t osso_email_url_email_modest (osso_context_t * osso, const gchar * url)
{
	gboolean ret = FALSE;
	d_log(LOG_D, "send url link to email application \n");


	if ((osso == NULL) || (url == NULL)) {
		return OSSO_INVALID;
	}

	ret = libmodest_dbus_client_compose_mail (osso, NULL /* to */, 
	NULL /* cc */, NULL /* bcc */, NULL /* subject */, url /* body */, 
	NULL /* attachments */);
	printf ("%s: ret = %d\n", __FUNCTION__, ret);
	
	if(ret) {
		return OSSO_OK;
	} else {
		return OSSO_ERROR;
	}
}


/**
   This function will cause the email application to be topped
   with an empty email to the specified recipients opened. This is
   implemented with the top D-BUS message to the email
   application.
 
   @param osso The library context as returned by #osso_initialize.
   @param to A GSList of mailto URI strings.
 
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
*/

osso_return_t osso_email_emailto_email_modest (osso_context_t * osso, GSList * to)
{
	GSList *next = NULL;
	gchar *args = NULL;
	gchar *tmp = NULL;
	gboolean ret = FALSE;

	d_log(LOG_D, "send mailto parameters to email application \n");

	if ((osso == NULL) || (to == NULL) || (to->data == NULL)) {
		return OSSO_INVALID;
	}
/*
	if (osso_get_dbus_connection(osso) == NULL) {
		DLOG_OPEN("libosso");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return OSSO_INVALID;
	}
*/
	if (strncmp(to->data, "mailto:", strlen("mailto:")) != 0) {
		d_log(LOG_D, "Input doesn't start with 'mailto' string \n");
		return OSSO_INVALID;
	}
/*
	args = g_strconcat("--to=", (gchar *) (to->data), NULL);
*/
	args = g_strdup((gchar *) (to->data));
	
	if (args == NULL) {
		return OSSO_ERROR;
	}

	for (next = g_slist_next(to); next != NULL; next = g_slist_next(next)) {
		tmp = g_strconcat(args, ",", (gchar *) (next->data), NULL);
		g_free(args);
		args = tmp;
		if (args == NULL) {
			return OSSO_ERROR;
		}
	}

	ret = libmodest_dbus_client_mail_to (osso, args);

	g_free(args);
	
	if(ret) {
		return OSSO_OK;
	} else {
		return OSSO_ERROR;
	}
}


/**
 *  This function is called to get the email folder list to other applications
 *  (like Global search etc) 
 *
 *  @param osso The library context as returned by #osso_initialize.
 *  @return GSList of folder names (gchar*)or NULL in case of error. The items 
 *  should be freed with g_free() and the list should be freed with g_slist_free().
*/
GSList *osso_email_get_folder_list_modest (osso_context_t * ctx)
{
	GSList *result = NULL;
	GList *iter = NULL;
	
	GList *modest_folders = NULL;
	gboolean ret = libmodest_dbus_client_get_folders (ctx,
					    &modest_folders);

	if (ret == FALSE) {
		osso_log (LOG_INFO, "Error during osso_email_get_folder_list");
		return NULL;
	}

	/*
	printf ("DEBUG: %s: GList=%p\n", __FUNCTION__, modest_folders);
	printf ("DEBUG: %s: GList length=%d\n", __FUNCTION__, g_list_length (modest_folders));
	*/

	for (iter = modest_folders; iter; iter = iter->next) {
		ModestFolderResult *item = (ModestFolderResult *) iter->data;
		if (item) {
			const gchar *folder_name = item->folder_name;
	
			if (folder_name) {
				/* printf ("DEBUG: %s: folder_name=%s\n", __FUNCTION__, folder_name); */
				result = g_slist_append (result, g_strdup (folder_name));
			}
			else {
				g_warning ("%s: ModestFolderResult::folder_name was NULL\n", __FUNCTION__);
			}
		}
	}
	
	modest_folder_result_list_free (modest_folders);

	return result;
}

