/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gmozillaengine.h"
#include "gmozillamarshal.h"
#include "gmozillaweb.h"
#include "gtkmozembed.h"
#include "gtkmozembed_common.h"
#include "gmozillatransferitem.h"
#include "common.h"
#include "gmozillaconnectivity.h"
#include "gmozillahistorylist.h"
#include "gmozillasupport.h"

#include <sys/time.h>
#include <time.h>
#include <string.h>

#ifdef USE_OPENSSL
#include <openssl/x509.h>
#endif

#define SINGLE_LINE_WIDGET_NAME "sinleline"
#ifdef USE_HILDON2007
#include <hildon/hildon-helper.h>
#else
#ifdef USE_HILDON
#ifndef USE_OS2006
#include <hildon-widgets/hildon-finger.h>
#endif
#endif
#endif

#define NODE_SCROLL_TIMEOUT 200

static const gchar *localhosts[] = {
    "localhost",
    "127.0.0.1",
    "127.0.1",
    "127.1",
    "[::1]",
    NULL
};

// global download var.
gpointer global_last_gtk_moz_embed_download_object = NULL;
struct timeval current_time_press, former_time_press, result; //double click emulation timers

GWebEngineEncoding g_mozilla_engine_get_encoding (GtkMozEmbed *);
static void g_mozilla_engine_notify_memory (GMozillaEngine *self);

// class and instance initialization
enum
{
    TITLE_CHANGED,                /** < When the title is changed */
    URL_CHANGED,                  /** < When the url is changed */
    LOADING_PROGRESS,             /** < Loagind progress signal */
    START_LOADING,                /** < Start loading signal */
    FINISHED_LOADING,             /** < Finished loading signal */
    ENCODING,                     /** < Encoding signal */
    DOWNLOAD_REQUEST,             /** < Download request signal */
    AUTHENTIFICATION_REQUIRED,    /** < When an authentification is required */
    ALERT,                        /** < When an alert is required */
    CONFIRM,                      /** < When a confirm dialog is required */
    PROMPT,                       /** < When a prompt dialog is required */
    SELECT,                       /** < When a select dialog is required */
    MAILTO,                       /** < When a mail signal is required */
    UNKNOWN_PROTOCOL,             /** < Unknown protocol signal is required */
    SECURITY_MODE_CHANGED,        /** < When security mode is changed */
    WINDOW_REQUEST,               /** < When a new window is required */
    CLOSE_WINDOW,                 /** < Close window signal */
    RAISE_REQUEST,                /** < Raise request signal */
    LOWER_REQUEST,                /** < Lower request signal */
    UPLOAD_DIALOG,                /** < When an upload is required */
    FILE_UPLOAD_STARTED,          /** < When an upload is started */
    FILE_UPLOAD_PROGRESS,         /** < When an upload is in progress */
    FILE_UPLOAD_FINISHED,         /** < when an upload is finished */
    WEBENGINE_POPUP_MENU,         /** < When a pop up window is required */
    EXTERNAL_ACTION,              /** < When an external action is required */
    MODAL_DIALOG,                 /** < When a modal dialog is required */
    FRAME_STACKING_CHANGED,       /** < When a focus frame is changed */
    MODAL_CERTIFICATE_DIALOG,     /** < When a certificate dialog is required */
    ASK_COOKIE,                   /** < When an ask cookie dialog is required */
    CERTIFICATE_DIALOG,           /** < When a certificate dialog is required */
    CERTIFICATE_PASSWORD_DIALOG,  /** < When a certificate dialog requires a password */
    CERTIFICATE_DETAILS,          /** < When a certificate details dialog is required */
    HISTORY_ADDED,                /** < When a url is added into history */
    SELECT_MATCH_SIGNAL,
    ON_SUBMIT_SIGNAL,
    ICON_SIGNAL,
    RSS_SIGNAL,
    MOZILLA_ENGINE_LAST_SIGNAL	  /** < Last signal used */
};

guint g_mozilla_engine_signals[MOZILLA_ENGINE_LAST_SIGNAL] = { 0 };

// class stuff
static void g_mozilla_engine_class_init (GMozillaEngineClass *);
static void g_mozilla_engine_destroy (GMozillaEngine *);
static void load_finished_cb (GtkMozEmbed *embed, GMozillaEngine *self);
static gboolean g_mozilla_engine_update_context_data (GMozillaEngine *self, GtkMozEmbed *embed, gpointer event);
static gboolean g_mozilla_engine_signals_disconnect(GMozillaEngine *self);
static void g_mozilla_unregister_im_context(GMozillaEngine *self);
static GtkIMContext* register_im_context(GMozillaEngine *self);
static void stop_cursor_animation_handler (GMozillaEngine *self);
static gboolean vkb_handle_action(GMozillaEngine *self, guint moz_embed_ctx, GdkEventKey * event);

//**
static gboolean content_set_cursor(GMozillaEngine *self, GdkCursorType type);

static void user_iteraction_happens(GMozillaEngine *self)
{
   self->user_iteraction_happens = TRUE;
}

static void set_extension_mode(GMozillaEngine *self, GdkExtensionMode mode)
{
    GtkWidget * child = gtk_bin_get_child(GTK_BIN(self->engine));
    if (child) {
        gtk_widget_set_extension_events (child, mode);
	self->extended_im_mode = mode;
    }
}

static gboolean
protocol_is_local(const gchar *uri)
{
    if (!*uri)
        return FALSE;

    return '/'==uri[0]
           || g_str_has_prefix(uri, "file:")
           || g_str_has_prefix(uri, "data:")
           || g_str_has_prefix(uri, "about:")
           || g_str_has_prefix(uri, "upnp:")
           || g_str_has_prefix(uri, "obex:");
}

/* You can replace this mess with gnome_vfs_uri_get_host_name
 * if you can live with the gnome-vfs dependency */
static gchar *
uri_get_host(const gchar *uri)
{
    const gchar *beg = NULL, *tmp = NULL;
    gint len;

    if (NULL == uri)
        return NULL;

    if ((beg = strstr(uri, "://")) == NULL)
        beg = uri;
    else
        beg += 3;

    /* authority/path => authority */
    if ((tmp = strchr(beg, '/')) == NULL)
        len = strlen(beg);
    else
        len = tmp - beg;

    /* authority ('user:pass@host:port') => 'host:port' */
    if ((tmp = g_strrstr_len(beg, len, "@")) == NULL)
        tmp = beg;
    else {
        tmp++;
        len -= (tmp - beg);
        beg = tmp;
    }

    /* 'host:port' => 'host' */
    if ((tmp = g_strrstr_len(beg, len, ":")) != NULL) {
        /* make sure we're not inside bracketed ipv6 address */
        if (g_strrstr_len(beg, len, "]") < tmp)
          len = tmp - beg;
    }

    return g_strndup(beg, len);
}

static gboolean
host_is_local(const gchar *uri)
{
    const gchar *elem = NULL;
    gchar *host = NULL;
    gint i = 0, len;

    if (!*uri)
        return FALSE;

    host = uri_get_host(uri);

    for (elem = localhosts[0]; elem != NULL; elem = localhosts[++i]) {
        len = strlen(elem);
        if (strncasecmp(elem, host, len) == 0) {
            g_free(host);
            return TRUE;
        }
    }

    g_free(host);
    return FALSE;
}

static gboolean
send_progress_signal(GMozillaEngine *self, guint state, const gchar *url, gint cur, gint max)
{
    switch (state) {
      case LOADING_PROGRESS:
      {
//        if (self->progress_bar_state != 0)
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[LOADING_PROGRESS] , 0, cur, max);
        self->progress_bar_state = 1;
        break;
      }
      case START_LOADING:
      {
        if (self->progress_bar_state != 0) {
          if (url)
            g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[START_LOADING] , 0, url);
        }
        self->progress_bar_state = 0;
        break;
      }
      case FINISHED_LOADING:
      {
        if (self->progress_bar_state != -1) {
          if (self->progress_bar_state == 0)
             g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[LOADING_PROGRESS] , 0, 98, 99);
          g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[FINISHED_LOADING], 0, self->last_net_status, self->page_size, FALSE);
        }
        self->progress_bar_state = -1;
        break;
      }
      default:
        break;
    }
    return TRUE;
}

static void 
move_focus_to_engine(GMozillaEngine *self)
{
    TRACE_LOG();
    if (self && GTK_IS_MOZ_EMBED(self->engine))
    {
        GtkWidget *child = gtk_bin_get_child(GTK_BIN(self->engine));
        if (child)
        {
            gtk_widget_grab_focus(GTK_WIDGET(child));
        }
    }
}

static void
g_mozilla_engine_load_url (GMozillaEngine *self, const gchar *url)
{
    TRACE_LOG();
    if (self && GTK_IS_MOZ_EMBED(self->engine) && url)
    {
        user_iteraction_happens(self);
        move_focus_to_engine(self);
        EAL_IF_GFREE(self->will_open_location);
        self->page_size = 0;
        self->will_open_location = g_strdup(url);
        send_progress_signal(self, START_LOADING, url, 0, 0);
#ifdef USE_CONIC
        if (!g_mozilla_connectivity_is_connected() && !protocol_is_local(url) && !host_is_local(url)) {
            if (g_mozilla_connectivity_connect(self, FALSE)) {
                return;
            }
        }
#endif
        ULOG_INFO_F("Engine: %p: LoadUrl: '%s'\n", self->engine, url);
        gtk_moz_embed_load_url(GTK_MOZ_EMBED(self->engine), url);
        content_set_cursor(self, GDK_WATCH);
    }
}

static GWebStatus
g_mozilla_engine_load_follow_link (GMozillaEngine *self, const gchar *url)
{
    TRACE_LOG();
    /* MTBD | Not used */
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_engine_load_url_with_css (GMozillaEngine *self, const gchar *url, const gchar *cssfile)
{
    TRACE_LOG();
    /* MTBD | Not Used */
    return G_WEB_STATUS_OK;
}

static void
g_mozilla_engine_load_url_with_js (GMozillaEngine *self, gboolean  enable)
{
    TRACE_LOG();
    /* MTBD | Not Used But can be usefull in future ;) */
    return;
}

static void
g_mozilla_engine_load_home_page (GMozillaEngine *self)
{
    TRACE_LOG();
    gchar * home_page = NULL;
    if (gtk_moz_embed_common_get_pref(G_TYPE_STRING, G_MOZILLA_PREF_HOME_PAGE_LOCATION, &home_page))
        if (home_page)
            g_mozilla_engine_load_url (self, home_page);
    g_mozilla_engine_notify_memory(self);
    return;
}

static void
g_mozilla_engine_stop_load (GMozillaEngine *self)
{
    TRACE_LOG();
    g_mozilla_connectivity_check_buggy();
    gtk_moz_embed_stop_load(GTK_MOZ_EMBED(self->engine));
    load_finished_cb(GTK_MOZ_EMBED(self->engine), self);
    return;
}

static void
g_mozilla_engine_reload (GMozillaEngine *self)
{
    TRACE_LOG();
    /*
       GTK_MOZ_EMBED_FLAG_RELOADNORMAL = 0;
       GTK_MOZ_EMBED_FLAG_RELOADBYPASSCACHE = nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE;
       GTK_MOZ_EMBED_FLAG_RELOADBYPASSPROXY = nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY;
       GTK_MOZ_EMBED_FLAG_RELOADBYPASSPROXYANDCACHE = (nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY | nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE);
       GTK_MOZ_EMBED_FLAG_RELOADCHARSETCHANGE = nsIWebNavigation::LOAD_FLAGS_CHARSET_CHANGE;
       */
    user_iteraction_happens(self);
    gtk_moz_embed_reload(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
    move_focus_to_engine(self);
    return;
}

static void
g_mozilla_engine_go_back (GMozillaEngine *self, guint nsteps)
{
    TRACE_LOG();
    user_iteraction_happens(self);
    if(nsteps == 1) {
        gtk_moz_embed_go_back(GTK_MOZ_EMBED(self->engine));
    } else {
        gint cur_index = gtk_moz_embed_get_shistory_index (GTK_MOZ_EMBED(self->engine));
        if(cur_index >= 0)
        {
            gint index = cur_index - nsteps;
            gtk_moz_embed_shistory_goto_index(GTK_MOZ_EMBED(self->engine), index);
        }
    }
    move_focus_to_engine(self);
    return;
}

static void
g_mozilla_engine_go_forward (GMozillaEngine *self, guint nsteps)
{
    TRACE_LOG();
    user_iteraction_happens(self);
    if(nsteps == 1) {
        gtk_moz_embed_go_forward(GTK_MOZ_EMBED(self->engine));
    } else {
        gint cur_index = gtk_moz_embed_get_shistory_index (GTK_MOZ_EMBED(self->engine));
        if(cur_index >= 0)
        {
            gint index = cur_index + nsteps;
            gtk_moz_embed_shistory_goto_index(GTK_MOZ_EMBED(self->engine), index);
        }
    }
    move_focus_to_engine(self);
    return;
}

static gboolean
g_mozilla_engine_can_go_back (GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    return gtk_moz_embed_can_go_back(GTK_MOZ_EMBED(self->engine));
}

static gboolean
g_mozilla_engine_can_go_forward (GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    return gtk_moz_embed_can_go_forward(GTK_MOZ_EMBED(self->engine));
}

static gint glevel;
static gpointer gcontext;

static gboolean
zoom_to_thread (void *data)
{

  GMozillaEngine *self = G_MOZILLA_ENGINE(data);
#ifndef MOZ_CAIRO_GFX
  gtk_moz_embed_common_observe ("@mozilla.org/extensions/imagezoom;1",
                                gcontext, "set_images_scaling",
                                glevel == 100 ? NULL : (gunichar*)"Do");
#endif
  gtk_moz_embed_set_zoom_level (GTK_MOZ_EMBED (self->engine), glevel,
                                gcontext);
#ifndef MOZ_CAIRO_GFX
  gtk_moz_embed_common_observe ("@mozilla.org/extensions/imagezoom;1",
                                self->ctx_node, "scrool_to_last_node", NULL);
#endif
  if (self->optimized_view)
    gtk_moz_embed_common_observe ("@mozilla.org/extensions/ftv;1", NULL,
                                  "optimized_view", NULL);
  return FALSE;
}

static gboolean
g_mozilla_engine_set_zoom_level_wrap(GMozillaEngine *self, gint level, gpointer context)
{
    TRACE_LOG();

    glevel = level;
    gcontext = context;
    //Hack to display INFO BANNER before doing zooming
    g_timeout_add(200, zoom_to_thread, self);

    return TRUE;
}

static void
g_mozilla_engine_set_zoom_level (GMozillaEngine *self, gint level)
{
    TRACE_LOG();
    if (level != self->current_zoom) {
        g_mozilla_engine_set_zoom_level_wrap(self, level, NULL);
    }
    self->current_zoom = level;
    return;
}

static GWebStatus
g_mozilla_engine_set_zoom_level_around_point (GMozillaEngine *self, guint x, guint y, gint level)
{
    TRACE_LOG();
    if (!g_mozilla_engine_set_zoom_level_wrap(self, level, self->ctx_node))
        return G_WEB_ERR;
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_engine_compare_frames_zoom_level (GMozillaEngine *self, gboolean *equal)
{
    TRACE_LOG();
    //    gint zoom_level;
    //    gboolean tequal = TRUE;
    //  if (gtk_moz_embed_get_zoom_level(GTK_MOZ_EMBED(self->engine), &zoom_level, NULL, &tequal))
    *equal = TRUE;
    //  else
    //    return G_WEB_ERR;
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_engine_get_image_properties (GMozillaEngine *self, const gchar *url, const gchar **file_type, guint *file_size, guint *width, guint *height)
{
    TRACE_LOG("Url: '%s'", url);
    gtk_moz_embed_get_doc_info(GTK_MOZ_EMBED(self->engine), self->ctx_node, -1, NULL, &url, file_type, file_size, width, height);
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_engine_get_frame_properties (GMozillaEngine *self, const gchar **file_type, guint *file_size, GWebEngineEncoding *file_encoding, guint *security, gboolean *automatic_encoding)
{
    TRACE_LOG();
    /* TBD */
    *automatic_encoding = TRUE;

    gchar *url = self->ctx_docurl;
    if (self->doc_index == -1)
       url = gtk_moz_embed_get_location(GTK_MOZ_EMBED(self->engine));

    if (security)
        *security = self->sec_mode;

    gtk_moz_embed_get_doc_info(GTK_MOZ_EMBED(self->engine), NULL, -1, NULL, (const gchar**)&url, file_type, file_size, NULL, NULL);

    if (self->doc_index == -1)
      g_free(url);
    
    if (file_size)
        if (*file_size == 0)
            *file_size = self->page_size;
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_engine_load_image (GMozillaEngine *self, const gchar* url)
{
    TRACE_LOG();
    if (!gtk_moz_embed_load_image(GTK_MOZ_EMBED(self->engine), url))
        return G_WEB_ERR;
    
    return G_WEB_STATUS_OK;
}

static gint
g_mozilla_engine_get_back_window_history_list (GMozillaEngine *self,
                                               GObject **history)
{
    TRACE_LOG();
    *history = G_OBJECT(g_object_new(g_mozilla_history_list_get_type(), NULL));
    G_MOZILLA_HISTORY_LIST(*history)->isSHistory = TRUE;
    G_MOZILLA_HISTORY_LIST(*history)->num_items = gtk_moz_embed_get_shistory_list(GTK_MOZ_EMBED(self->engine),
                                (GtkMozHistoryItem**)&G_MOZILLA_HISTORY_LIST(*history)->items,
                                GTK_MOZ_EMBED_BACK_SHISTORY);
    return G_MOZILLA_HISTORY_LIST(*history)->num_items;
}

static gint
g_mozilla_engine_get_forward_window_history_list (GMozillaEngine *self,
                                                  GObject **history)
{
    TRACE_LOG();
    *history = G_OBJECT(g_object_new(g_mozilla_history_list_get_type(), NULL));
    G_MOZILLA_HISTORY_LIST(*history)->isSHistory = TRUE;
    G_MOZILLA_HISTORY_LIST(*history)->num_items = gtk_moz_embed_get_shistory_list(GTK_MOZ_EMBED(self->engine),
                                (GtkMozHistoryItem**)&G_MOZILLA_HISTORY_LIST(*history)->items,
                                GTK_MOZ_EMBED_FORWARD_SHISTORY);
    return G_MOZILLA_HISTORY_LIST(*history)->num_items;
}

static guint
g_mozilla_engine_get_zoom_level (GMozillaEngine *self)
{
    TRACE_LOG();
    gint zoom = 100;
    if (gtk_moz_embed_get_zoom_level(GTK_MOZ_EMBED(self->engine), &zoom, NULL))
        self->current_zoom = zoom;
    return self->current_zoom;
}

static void
g_mozilla_engine_zoom_by_steps (GMozillaEngine *self, guint relative_zoom, guint* legal_zoom_value)
{
    TRACE_LOG();
    g_mozilla_engine_set_zoom_level_wrap(self, relative_zoom, NULL);
    return;
}

static void
g_mozilla_engine_new_zoom_by_steps (GMozillaEngine *self, gboolean up, guint* legal_zoom_value)
{
    TRACE_LOG();
    //gtk_moz_embed_zoom_by_steps(GTK_MOZ_EMBED(self->engine), up, legal_zoom_value);
    return;
}

static void
g_mozilla_engine_set_frame_zoom_level (GMozillaEngine *self, guint level)
{
    TRACE_LOG();
    g_mozilla_engine_set_zoom_level_wrap(self, level, self->ctx_node);
    return;
}

static guint
g_mozilla_engine_get_frame_zoom_level (GMozillaEngine *self)
{
    TRACE_LOG();
    gint zoom = 100;
    if (gtk_moz_embed_get_zoom_level(GTK_MOZ_EMBED(self->engine), &zoom, self->ctx_node))
        return zoom;
    return 100;
}

static void
g_mozilla_engine_set_layout_mode (GMozillaEngine *self, GWebEngineLayoutMode mode)
{
    TRACE_LOG();
    gboolean value = FALSE;
    switch (mode)
    {
    case G_WEBENGINE_LAYOUT_NORMAL:
        value = FALSE;
        break;
    case G_WEBENGINE_LAYOUT_SSR:
        value = TRUE;
        break;
    case G_WEBENGINE_LAYOUT_MSR:
        value = TRUE;
        break;
    default:
        value = FALSE;
    }

    if (gtk_moz_embed_common_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SSR_ENABLED, &value))
        gtk_moz_embed_common_save_prefs();

    if(self->optimized_view == value)
        return;

    self->optimized_view = value;
 
    gpointer *webbrowser = NULL;
    gtk_moz_embed_get_nsIWebBrowser(self->engine, &webbrowser);
    if(value)
    {
        gtk_moz_embed_common_observe("@mozilla.org/extensions/ftv;1", webbrowser, "set_optimized_view", NULL);
        if(self->is_loading == FALSE)
            gtk_moz_embed_common_observe("@mozilla.org/extensions/ftv;1", webbrowser, "optimized_view", NULL);
    }
    else gtk_moz_embed_common_observe("@mozilla.org/extensions/ftv;1", webbrowser, "unset_optimized_view", NULL);

}

static GWebEngineLayoutMode
g_mozilla_engine_get_layout_mode (GMozillaEngine *self)
{
    TRACE_LOG();
    if (gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SSR_ENABLED, &self->optimized_view))
        if (self->optimized_view)
            return G_WEBENGINE_LAYOUT_MSR;

    return G_WEBENGINE_LAYOUT_NORMAL;
}

static void
g_mozilla_engine_set_load_images(GMozillaEngine *self, GWebEngineImagePolicy level)
{
    TRACE_LOG();
    gint moz_level = 1;
    switch (level)
    {
    case G_WEBENGINE_POLICY_ALL_IMAGES:
        moz_level = 1;
        break;
    case G_WEBENGINE_POLICY_NO_IMAGES:
        moz_level = 2;
        break;
    case G_WEBENGINE_POLICY_LOADED_IMAGES:
        moz_level = 4;
        break;
    }
    gtk_moz_embed_common_set_image_policy(moz_level);
}

static GWebEngineImagePolicy
g_mozilla_engine_get_load_images (GMozillaEngine *self)
{
    TRACE_LOG();
    gint show_images = 1;
    if (!gtk_moz_embed_common_get_pref(G_TYPE_INT, G_MOZILLA_PREF_SHOW_IMAGES, &show_images))
        return G_WEBENGINE_POLICY_ALL_IMAGES;

    switch (show_images)
    {
    case 1:
        return G_WEBENGINE_POLICY_ALL_IMAGES;
    case 2:
        return G_WEBENGINE_POLICY_NO_IMAGES;
    case 4:
        return G_WEBENGINE_POLICY_LOADED_IMAGES;
    default:
        return G_WEBENGINE_POLICY_ALL_IMAGES;
    }
}

static void
g_mozilla_engine_set_frame_stacking (GMozillaEngine *self, gboolean enabled)
{
    TRACE_LOG();
    if (gtk_moz_embed_common_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_MERGE_FRAMES, &enabled))
        gtk_moz_embed_common_save_prefs();
    g_mozilla_engine_update_context_data(self, GTK_MOZ_EMBED(self->engine), NULL);
    char test[2][16] = {
        "d\0i\0s\0a\0b\0l\0e\0\0",
        "e\0n\0a\0b\0l\0e\0\0",
    };
    gtk_moz_embed_common_observe("@browser.garage.maemo.org/microb/mergeframes;1", self->ctx_node, "microb-mergeframes", (gunichar *)test[enabled]);
    return;
}

static gboolean
g_mozilla_engine_get_frame_stacking (GMozillaEngine *self)
{
    TRACE_LOG();
    gboolean fr_stacking = FALSE;
    gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_MERGE_FRAMES, &fr_stacking);


    return fr_stacking;
}

static GWebStatus
g_mozilla_engine_save_web_page (GMozillaEngine *self, GWebEngineSaveAction setting, const gchar *file_name_with_path)
{
    TRACE_LOG();
    gchar * url = gtk_moz_embed_get_location(GTK_MOZ_EMBED(self->engine));
    GWebStatus status = G_WEB_ERR;
    //FIXME gtk_moz_embed_save_target
    if (gtk_moz_embed_save_target (GTK_MOZ_EMBED(self->engine), url, (gchar*)file_name_with_path, setting,NULL))
        status = G_WEB_STATUS_OK;
    g_free(url);
    return status;
}

static GWebStatus
g_mozilla_engine_save_current_frame (GMozillaEngine *self, GWebEngineSaveAction setting, const gchar *file_name_with_path)
{
    TRACE_LOG();
    gchar * url = gtk_moz_embed_get_location(GTK_MOZ_EMBED(self->engine));
    GWebStatus status = G_WEB_ERR;
    if (gtk_moz_embed_save_target (GTK_MOZ_EMBED(self->engine), url, (gchar*)file_name_with_path, setting, self->ctx_node))
        status = G_WEB_STATUS_OK;
    g_free(url);
    return status;
}

static const gchar*
g_mozilla_engine_get_frame_title (GMozillaEngine *self, gint frame_number)
{
    TRACE_LOG();
    gchar * frame_title = NULL;
    // bug 46234 - solved
    gtk_moz_embed_get_doc_info(GTK_MOZ_EMBED(self->engine), self->ctx_node, -1,
                               (const gchar **)&frame_title, NULL, NULL, NULL, NULL, NULL);
    return frame_title;
}

static gint
g_mozilla_engine_get_frame_number (GMozillaEngine *self)
{
    TRACE_LOG();
    gint frame_number = 0;
    frame_number = self->doc_index;
    /*IMP
      frame_number = gtk_moz_embed_get_frame_number(GTK_MOZ_EMBED(self->engine), NULL[Title], NULL[Location]);
      */
    return frame_number;
}

static GWebStatus
g_mozilla_engine_save_web_page_image (GMozillaEngine *self, const gchar *url, const gchar* path)
{
    TRACE_LOG();
    //FIXME gtk_moz_embed_save_target
    if (!gtk_moz_embed_save_target (GTK_MOZ_EMBED(self->engine), (gchar*)url, (gchar*)path, 2, NULL))
        return G_WEB_ERR;
    return G_WEB_STATUS_OK;
}

GWebEngineEncoding
g_mozilla_engine_get_encoding (GtkMozEmbed *embed)
{
    gchar *encoding;

    encoding = gtk_moz_embed_get_encoding (embed, -1);
    if (!encoding)
        return 0;

    if (g_ascii_strcasecmp(encoding, "UTF-8") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_UTF8; };
    if (g_ascii_strcasecmp(encoding, "UTF-16BE") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_UTF16; };
    if (g_ascii_strcasecmp(encoding, "ISO-8859-1") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_ISO8859_1; };
    if (g_ascii_strcasecmp(encoding, "ISO-8859-15") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_ISO8859_15; };
    if (g_ascii_strcasecmp(encoding, "ISO-8859-2") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_ISO8859_2; };
    if (g_ascii_strcasecmp(encoding, "ISO-8859-7") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_ISO8859_7; };
    if (g_ascii_strcasecmp(encoding, "ISO-8859-9") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_ISO8859_9; };
    if (g_ascii_strcasecmp(encoding, "windows-1250") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_CP_1250; };
    if (g_ascii_strcasecmp(encoding, "windows-1251") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_CP_1251; };
    if (g_ascii_strcasecmp(encoding, "windows-1253") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_CP_1253; };
    if (g_ascii_strcasecmp(encoding, "windows-1254") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_CP_1254; };
    if (g_ascii_strcasecmp(encoding, "KOI8-R") == 0) { g_free(encoding); return G_WEBENGINE_ENCODING_KOI8_R; };

    /* we should never get here, but if we do we raise an error */
    return 0;
}

static void
g_mozilla_engine_set_encoding (GMozillaEngine *self, GWebEngineEncoding encoding, gint frame_number)
{
    TRACE_LOG();
    gchar * encoding_text = NULL;
    switch (encoding)
    {
    case G_WEBENGINE_ENCODING_UTF8:
        encoding_text = "UTF-8";
        break;
    case G_WEBENGINE_ENCODING_UTF16:
        encoding_text = "UTF-16BE";
        break;
    case G_WEBENGINE_ENCODING_ISO8859_1:
        encoding_text = "ISO-8859-1";
        break;
    case G_WEBENGINE_ENCODING_ISO8859_15:
        encoding_text = "ISO-8859-15";
        break;
    case G_WEBENGINE_ENCODING_ISO8859_2:
        encoding_text = "ISO-8859-2";
        break;
    case G_WEBENGINE_ENCODING_ISO8859_7:
        encoding_text = "ISO-8859-7";
        break;
    case G_WEBENGINE_ENCODING_ISO8859_9:
        encoding_text = "ISO-8859-9";
        break;
    case G_WEBENGINE_ENCODING_CP_1250:
        encoding_text = "windows-1250";
        break;
    case G_WEBENGINE_ENCODING_CP_1251:
        encoding_text = "windows-1251";
        break;
    case G_WEBENGINE_ENCODING_CP_1253:
        encoding_text = "windows-1253";
        break;
    case G_WEBENGINE_ENCODING_CP_1254:
        encoding_text = "windows-1254";
        break;
    case G_WEBENGINE_ENCODING_KOI8_R:
        encoding_text = "KOI8-R";
        break;
    default:
        encoding_text = gtk_moz_embed_get_encoding (GTK_MOZ_EMBED(self->engine), frame_number);
        break;
    }
    gtk_moz_embed_set_encoding(GTK_MOZ_EMBED(self->engine), encoding_text, frame_number);
    g_mozilla_engine_reload (self);
}

static void
g_mozilla_engine_set_language_encoding (GMozillaEngine *self, GWebEngineEncoding encoding)
{
    g_mozilla_engine_set_encoding (self, encoding, -1);
}

static void
g_mozilla_engine_set_frame_encoding (GMozillaEngine *self, GWebEngineEncoding encoding)
{
    g_mozilla_engine_set_encoding (self, encoding, self->doc_index); //0|FRAME_NUMBER
}

static gboolean
g_mozilla_engine_find_text (GMozillaEngine *self, const gchar *string, gboolean reverse, gboolean whole_word, gboolean case_sensitive, gboolean restart)
{
    TRACE_LOG();
    gboolean found = FALSE;
    found = gtk_moz_embed_find_text (GTK_MOZ_EMBED(self->engine), string, reverse, whole_word,
                                     case_sensitive, restart, -1);

    return found;
}

static void
g_mozilla_engine_authenticate (GMozillaEngine *self, gboolean ok, const gchar *username, const gchar *password)
{
    TRACE_LOG();

    if (ok && self)
    {
        EAL_IF_GFREE(self->username);
        EAL_IF_GFREE(self->password);

        if (username)
            self->username = g_strdup(username);
        if (password)
            self->password = g_strdup(password);
        self->accept = TRUE;
    }
    return;
}

static void
g_mozilla_engine_select_all_text (GMozillaEngine *self)
{
    TRACE_LOG();
    if (!gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_SELECT_ALL, -1))
        ULOG_WARN_F("Failed to do GTK_MOZ_EMBED_SELECT_ALL");
    return;
}

static gboolean
g_mozilla_engine_can_selecttext (GMozillaEngine *self)
{
    TRACE_LOG();
    if (!gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_CAN_SELECT, -1))
        return FALSE;
    return TRUE;
}

static GWebStatus
g_mozilla_engine_cut (GMozillaEngine *self)
{
    TRACE_LOG();
    if (gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_CUT, -1))
        return G_WEB_STATUS_OK;
    return G_WEB_ERR;
}

static GWebStatus
g_mozilla_engine_copy (GMozillaEngine *self)
{
    TRACE_LOG();
    if (gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_COPY, -1))
        return G_WEB_STATUS_OK;
    return G_WEB_ERR;
}

static GWebStatus
g_mozilla_engine_paste (GMozillaEngine *self)
{
    TRACE_LOG();
    if (gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_PASTE, -1))
        return G_WEB_STATUS_OK;
    return G_WEB_ERR;
}

static gboolean
g_mozilla_engine_can_cut (GMozillaEngine *self)
{
    TRACE_LOG();
    if (!gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_CAN_CUT, -1) || ( self->ctx_type == G_WEBENGINE_POPUPMENU_NOTYPE ))
        return FALSE;
    return TRUE;
}

static gboolean
g_mozilla_engine_can_paste (GMozillaEngine *self)
{
    TRACE_LOG();
    if (!gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_CAN_PASTE, -1))
        return FALSE;
    return TRUE;
}

static gboolean
g_mozilla_engine_can_copy (GMozillaEngine *self)
{
    TRACE_LOG();
    //Disabled copy option in password fields.
    if (!gtk_moz_embed_clipboard(GTK_MOZ_EMBED(self->engine), GTK_MOZ_EMBED_CAN_COPY, -1) || self->moz_ctx_type & GTK_MOZ_EMBED_CTX_IPASSWORD)
        return FALSE;
    return TRUE;
}

static gchar*
g_mozilla_engine_get_favicon_name (GMozillaEngine *self, const gchar *favicon_folder)
{
    TRACE_LOG();
    /* TBD | It is really usefull stuff for us? I think favicon_signal is enough */
    /*
       If yes, then we should save favicon for current url into favicon_folder, and provide path for it
       */
    return NULL;
}

static GObject*
g_mozilla_engine_get_engine_widget (GMozillaEngine *self)
{
    TRACE_LOG();
    return self->engine;
}

static GObject*
g_mozilla_engine_get_engine_notifier (GMozillaEngine *self)
{
    TRACE_LOG("Notifier: %p", self);
    return G_OBJECT(self);
}

static void
g_mozilla_engine_set_open_link_mode (GMozillaEngine *self, gboolean in_new_window)
{
    TRACE_LOG();
    /* MTBD | Not used*/
    return;
}

static GWebStatus
g_mozilla_engine_set_true_zoom_mode (GMozillaEngine *self, gboolean status)
{
    TRACE_LOG();
    /* MTBD | Don't know what it's mean :( */
    return G_WEB_STATUS_OK;
}

static void
g_mozilla_engine_notify_memory (GMozillaEngine *self)
{
    gtk_moz_embed_common_observe(NULL, NULL, "memory-pressure", (gunichar *)"h\0e\0a\0p\0-\0m\0i\0n\0i\0m\0i\0z\0e\0\0");
    malloc_trim(0);
}

static void
g_mozilla_engine_notify_plugins (GMozillaEngine *self, gint event)
{
    TRACE_LOG();
    if (self && self->engine)
      gtk_moz_embed_notify_plugins(GTK_MOZ_EMBED(self->engine), event);
    if (G_WEB_PLUGINS_LOW_MEMORY != event)
      return;
    g_mozilla_engine_notify_memory(self);
    return;
}

static void
g_mozilla_engine_preload_current_skin (GMozillaEngine *self)
{
    TRACE_LOG();
    /* MTBD | Not Used */
    return;
}

static void
g_mozilla_engine_invoke_selectmatch (GMozillaEngine *self)
{
    TRACE_LOG();
    //GObject *info=NULL;
    //gint retval=0;
    //    g_signal_emit_by_name(g_mozilla_get_current_web(), "select-match", &info, &retval);
    //if (retval >= 0) gtk_mozilla_login_select (retval);
}

GObject *
g_mozilla_engine_get_server_certificate (GMozillaEngine *self)
{
    gpointer severCert = NULL;
    gboolean status = FALSE;
    gint len = 0;

    status = gtk_moz_embed_get_server_cert(GTK_MOZ_EMBED(self->engine), &severCert, NULL);
    if(!status || severCert == NULL)
        return NULL;

    unsigned char *p = gtk_moz_embed_common_nsx509_to_raw(severCert, &len);
    gpointer certPtr = NULL;
#ifndef NEW_OPENSSL
    certPtr = d2i_X509 (NULL, (unsigned char **)&p, len);
#else
    certPtr = d2i_X509 (NULL, (unsigned char **) &p, len);
#endif

    return certPtr;
}

static void
g_mozilla_engine_reset (GMozillaEngine *self)
{
    TRACE_LOG();
    g_mozilla_engine_signals_disconnect(self);
    EAL_IF_GFREE_FUNC(self->engine, gtk_widget_destroy);
/* I think it also should be destroyed
    self->ctx_node = NULL;
    self->last_event = NULL;
    self->ctx_url = NULL;
    self->ctx_objurl = NULL;
    self->ctx_docurl = NULL;
    self->will_open_location = NULL;
*/
    return;
}

static void
g_mozilla_engine_interface_init (GWebEngineIface *iface)
{
    TRACE_LOG();
    iface->destroy                         = (void (*) (GWebEngine*))
        g_mozilla_engine_destroy;
    iface->reset                           = (void (*) (GWebEngine*))
        g_mozilla_engine_reset;
    iface->load_url                        = (void (*) (GWebEngine*, const gchar*))
        g_mozilla_engine_load_url;
    iface->load_follow_link                = (GWebStatus (*) (GWebEngine*, const gchar*))
        g_mozilla_engine_load_follow_link;
    iface->load_url_with_css               = (GWebStatus (*) (GWebEngine*, const gchar*, const gchar*))
        g_mozilla_engine_load_url_with_css;
    iface->load_url_with_js                = (void (*) (GWebEngine*, gboolean))
        g_mozilla_engine_load_url_with_js;
    iface->load_home_page                  = (void (*) (GWebEngine*))
        g_mozilla_engine_load_home_page;
    iface->stop_load                       = (void (*) (GWebEngine*))
        g_mozilla_engine_stop_load;
    iface->reload                          = (void (*) (GWebEngine*))
        g_mozilla_engine_reload;
    iface->go_back                         = (void (*) (GWebEngine*, guint))
        g_mozilla_engine_go_back;
    iface->go_forward                      = (void (*) (GWebEngine*, guint))
        g_mozilla_engine_go_forward;
    iface->can_go_back                     = (gboolean (*) (GWebEngine*))
        g_mozilla_engine_can_go_back;
    iface->can_go_forward                  = (gboolean (*) (GWebEngine*))
        g_mozilla_engine_can_go_forward;
    iface->set_zoom_level                  = (void (*) (GWebEngine*, gint ))
        g_mozilla_engine_set_zoom_level;
    iface->zoom_by_steps                    = (void (*) (GWebEngine*, guint, guint *))
        g_mozilla_engine_zoom_by_steps;
    iface->new_zoom_by_steps                = (void (*) (GWebEngine*, gboolean, guint *))
        g_mozilla_engine_new_zoom_by_steps;
    iface->set_zoom_level_around_point     = (GWebStatus (*) (GWebEngine*, guint, guint, gint))
        g_mozilla_engine_set_zoom_level_around_point;
    iface->compare_frames_zoom_level       = (GWebStatus (*) (GWebEngine*, gboolean*))
        g_mozilla_engine_compare_frames_zoom_level;
    iface->get_image_properties            = (GWebStatus (*) (GWebEngine*, const gchar*, gchar**, guint*, guint*, guint*))
        g_mozilla_engine_get_image_properties;
    iface->get_frame_properties            = (GWebStatus (*) (GWebEngine*, gchar**, guint*, GWebEngineEncoding*, guint*, gboolean*))
        g_mozilla_engine_get_frame_properties;
    iface->load_image                      = (GWebStatus (*) (GWebEngine*, const gchar*))
        g_mozilla_engine_load_image;
    iface->get_back_window_history_list    = (gint (*) (GWebEngine*, GWebHistoryList **))
        g_mozilla_engine_get_back_window_history_list;
//    iface->get_forward_window_history_list = (gint (*) (GWebEngine*, GObject**))
    iface->get_forward_window_history_list = (gint (*) (GWebEngine*, GWebHistoryList **))
        g_mozilla_engine_get_forward_window_history_list;
    iface->get_zoom_level                  = (guint (*) (GWebEngine*))
        g_mozilla_engine_get_zoom_level;
    iface->set_frame_zoom_level            = (void (*) (GWebEngine*, guint))
        g_mozilla_engine_set_frame_zoom_level;
    iface->get_frame_zoom_level            = (guint (*) (GWebEngine*))
        g_mozilla_engine_get_frame_zoom_level;
    iface->set_layout_mode                 = (void (*) (GWebEngine*, GWebEngineLayoutMode))
        g_mozilla_engine_set_layout_mode;
    iface->get_layout_mode                 = (GWebEngineLayoutMode (*) (GWebEngine*))
        g_mozilla_engine_get_layout_mode;
    iface->set_load_images                 = (void (*) (GWebEngine*, GWebEngineImagePolicy))
        g_mozilla_engine_set_load_images;
    iface->get_load_images                 = (GWebEngineImagePolicy (*) (GWebEngine *self))
        g_mozilla_engine_get_load_images;
    iface->set_frame_stacking              = (void (*) (GWebEngine*, gboolean))
        g_mozilla_engine_set_frame_stacking;
    iface->get_frame_stacking              = (gboolean (*) (GWebEngine*))
        g_mozilla_engine_get_frame_stacking;
    iface->save_web_page                   = (GWebStatus (*) (GWebEngine*, GWebEngineSaveAction, const gchar*))
        g_mozilla_engine_save_web_page;
    iface->save_current_frame              = (GWebStatus (*) (GWebEngine*, GWebEngineSaveAction, const gchar*))
        g_mozilla_engine_save_current_frame;
    iface->get_frame_title                 = (const gchar* (*) (GWebEngine*, gint))
        g_mozilla_engine_get_frame_title;
    iface->get_frame_number                = (gint (*) (GWebEngine*))
        g_mozilla_engine_get_frame_number;
    iface->save_web_page_image             = (GWebStatus (*) (GWebEngine*, const gchar*, const gchar*))
        g_mozilla_engine_save_web_page_image;
    iface->select_all_text                 = (void (*) (GWebEngine*))
        g_mozilla_engine_select_all_text;
    iface->set_language_encoding           = (void (*) (GWebEngine*, GWebEngineEncoding))
        g_mozilla_engine_set_language_encoding;
    iface->set_frame_encoding              = (void (*) (GWebEngine*, GWebEngineEncoding))
        g_mozilla_engine_set_frame_encoding;
    iface->find_text                       = (gboolean (*) (GWebEngine*, const gchar*, gboolean, gboolean, gboolean, gboolean))
        g_mozilla_engine_find_text;
    iface->authenticate                    = (void (*) (GWebEngine*, gboolean, const gchar*, const gchar*))
        g_mozilla_engine_authenticate;
    iface->cut                             = (GWebStatus (*) (GWebEngine*))
        g_mozilla_engine_cut;
    iface->copy                            = (GWebStatus (*) (GWebEngine*))
        g_mozilla_engine_copy;
    iface->paste                           = (GWebStatus (*) (GWebEngine*))
        g_mozilla_engine_paste;
    iface->can_cut                         = (gboolean (*) (GWebEngine*))
        g_mozilla_engine_can_cut;
    iface->can_paste                       = (gboolean (*) (GWebEngine*))
        g_mozilla_engine_can_paste;
    iface->can_copy                        = (gboolean (*) (GWebEngine*))
        g_mozilla_engine_can_copy;
    iface->can_selecttext                  = (gboolean (*) (GWebEngine*))
        g_mozilla_engine_can_selecttext;
    iface->get_favicon_name                = (gchar* (*) (GWebEngine*, const gchar*))
        g_mozilla_engine_get_favicon_name;
    iface->get_engine_widget               = (GObject* (*) (GWebEngine*))
        g_mozilla_engine_get_engine_widget;
    iface->get_engine_notifier             = (GObject* (*) (GWebEngine*))
        g_mozilla_engine_get_engine_notifier;
    iface->set_open_link_mode              = (void (*) (GWebEngine*, gboolean))
        g_mozilla_engine_set_open_link_mode;
    iface->set_true_zoom_mode              = (GWebStatus (*) (GWebEngine*, gboolean))
        g_mozilla_engine_set_true_zoom_mode;
    iface->notify_plugins                  = (void (*) (GWebEngine*, gint))
        g_mozilla_engine_notify_plugins;
    iface->preload_current_skin            = (void (*) (GWebEngine*))
        g_mozilla_engine_preload_current_skin;
    iface->invoke_selectmatch              = (void (*) (GWebEngine*))
        g_mozilla_engine_invoke_selectmatch;
    iface->get_server_certificate          = (GObject* (*) (GWebEngine*))
        g_mozilla_engine_get_server_certificate;
}

static void
g_mozilla_engine_instance_init (GTypeInstance *instance, gpointer  g_class)
{
    TRACE_LOG();
    G_MOZILLA_ENGINE(instance)->engine = NULL;
}

GType
g_mozilla_engine_get_type (void)
{
    static GType type = 0;
    if (type == 0)
    {
        static const GTypeInfo info =
        {
            sizeof (GMozillaEngineClass),
            NULL,   /* base_init */
            NULL,   /* base_finalize */
            (GClassInitFunc) g_mozilla_engine_class_init, /* class_init */
            NULL,   /* class_finalize */
            NULL,   /* class_data */
            sizeof (GMozillaEngine),
            0,      /* n_preallocs */
            g_mozilla_engine_instance_init    /* instance_init */
        };
        static const GInterfaceInfo iface_info =
        {
            (GInterfaceInitFunc) g_mozilla_engine_interface_init,
            NULL,   /* interface_finalize */
            NULL    /* interface_data */
        };
        type = g_type_register_static (G_TYPE_OBJECT,
                                       "GMozillaEngine",
                                       &info, 0);
        g_type_add_interface_static (type,
                                     G_TYPE_WEB_ENGINE,
                                     &iface_info);
    }
    return type;
}
/* a macro for creating a new object of our type */
#define GET_MOZILLA_ENGINE_NEW (g_object_new(g_mozilla_engine_get_type(), NULL))

static void
g_mozilla_engine_class_init(GMozillaEngineClass *klass)
{
    TRACE_LOG();
    // set up UI specific signals
    g_mozilla_engine_signals[URL_CHANGED]
        = g_signal_new (G_WEBWIDGET_SIGNAL_URL_CHANGED,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, url_changed),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__STRING,
                        G_TYPE_NONE, 1, G_TYPE_STRING);

    g_mozilla_engine_signals[TITLE_CHANGED]
        = g_signal_new (G_WEBWIDGET_SIGNAL_TITLE,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, title_changed),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__STRING,
                        G_TYPE_NONE, 1, G_TYPE_STRING);

    g_mozilla_engine_signals[LOADING_PROGRESS]
        = g_signal_new (G_WEBWIDGET_SIGNAL_LOADING_PROGRESS,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, progress),
                        NULL, NULL,
                        g_mozeal_marshal_VOID__INT_INT,
                        G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_INT);

    g_mozilla_engine_signals[START_LOADING]
        = g_signal_new (G_WEBWIDGET_SIGNAL_START_LOADING,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, start_loading),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__STRING,
                        G_TYPE_NONE, 1, G_TYPE_STRING);

    g_mozilla_engine_signals[FINISHED_LOADING]
        = g_signal_new (G_WEBWIDGET_SIGNAL_FINISHED_LOADING,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, finished_loading),
                        NULL, NULL,
                        g_mozeal_marshal_VOID__INT_INT_BOOL,
                        G_TYPE_NONE, 3, G_TYPE_INT, G_TYPE_INT, G_TYPE_BOOLEAN);

    g_mozilla_engine_signals[WINDOW_REQUEST]
        = g_signal_new (G_WEBWIDGET_SIGNAL_WINDOW_REQUEST,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, window_request),
                        NULL, NULL,
                        g_mozeal_marshal_VOID__POINTER_STRING_BOOLEAN_BOOLEAN,
                        G_TYPE_NONE,
                        4,
                        G_TYPE_POINTER,
                        G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);

    g_mozilla_engine_signals[ENCODING]
        = g_signal_new (G_WEBWIDGET_SIGNAL_ENCODING,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, encoding),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1,
                        G_TYPE_UINT);

    g_mozilla_engine_signals[AUTHENTIFICATION_REQUIRED]
        = g_signal_new (G_WEBWIDGET_SIGNAL_AUTHENTIFICATION_REQUIRED,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass,
                                         authentification_required), NULL, NULL,
                        g_mozeal_marshal_VOID__INT_STRING_STRING,
                        G_TYPE_NONE, 3, G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING);

    g_mozilla_engine_signals[CONFIRM]
        = g_signal_new (G_WEBWIDGET_SIGNAL_CONFIRM,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, confirm),
                        NULL, NULL,
                        g_mozeal_marshal_BOOLEAN__STRING,
                        G_TYPE_BOOLEAN, 1, G_TYPE_STRING);

    g_mozilla_engine_signals[PROMPT]
        = g_signal_new (G_WEBWIDGET_SIGNAL_PROMPT,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, prompt),
                        NULL, NULL,
                        g_mozeal_marshal_STRING__STRING_STRING,
                        G_TYPE_STRING, 2, G_TYPE_STRING, G_TYPE_STRING);

    g_mozilla_engine_signals[SELECT]
        = g_signal_new (G_WEBWIDGET_SIGNAL_SELECT,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, select),
                        NULL, NULL,
                        g_mozeal_marshal_BOOLEAN__STRING_STRING_POINTER_POINTER,
                        G_TYPE_BOOLEAN, 4, G_TYPE_STRING, G_TYPE_STRING,
                        G_TYPE_POINTER, G_TYPE_POINTER);

    g_mozilla_engine_signals[DOWNLOAD_REQUEST]
        = g_signal_new (G_WEBWIDGET_SIGNAL_DOWNLOAD_REQUEST,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, download_request),
                        NULL, NULL,
                        g_mozeal_marshal_VOID__STRING_STRING_STRING_INT_INT,
                        G_TYPE_NONE, 5,
                        G_TYPE_STRING | G_SIGNAL_TYPE_STATIC_SCOPE,
                        G_TYPE_STRING | G_SIGNAL_TYPE_STATIC_SCOPE,
                        G_TYPE_STRING | G_SIGNAL_TYPE_STATIC_SCOPE, G_TYPE_INT,
                        G_TYPE_INT);

    g_mozilla_engine_signals[UPLOAD_DIALOG]
        = g_signal_new (G_WEBWIDGET_SIGNAL_UPLOAD_DIALOG,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, upload_dialog),
                        NULL, NULL,
                        g_mozeal_marshal_BOOL__STRING_STRING_POINTER,
                        G_TYPE_BOOLEAN, 3, G_TYPE_STRING, G_TYPE_STRING,
                        G_TYPE_POINTER);

    g_mozilla_engine_signals[FILE_UPLOAD_PROGRESS]
        = g_signal_new (G_WEBWIDGET_SIGNAL_FILE_UPLOAD_PROGRESS,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, file_upload_progress),
                        NULL, NULL,
                        g_mozeal_marshal_VOID__INT_STRING,
                        G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_STRING);

    g_mozilla_engine_signals[WEBENGINE_POPUP_MENU]
        = g_signal_new (G_WEBWIDGET_SIGNAL_WEBENGINE_POPUP_MENU,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, webengine_popup_menu),
                        NULL, NULL,
                        g_mozeal_marshal_VOID__UINT_INT_INT_STRING_STRING_STRING_STRING,
                        G_TYPE_NONE, 7, G_TYPE_UINT, G_TYPE_INT, G_TYPE_INT,
                        G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                        G_TYPE_STRING);

    g_mozilla_engine_signals[MODAL_CERTIFICATE_DIALOG]
        = g_signal_new (G_WEBWIDGET_SIGNAL_MODAL_CERTIFICATE_DIALOG,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass,
                                         modal_certificate_dialog), NULL, NULL,
                        g_mozeal_marshal_INT__STRING_INT_INT_INT_INT_INT,
                        G_TYPE_INT, 6, G_TYPE_STRING, G_TYPE_INT, G_TYPE_INT,
                        G_TYPE_INT, G_TYPE_INT, G_TYPE_INT);

    g_mozilla_engine_signals[ALERT]
        = g_signal_new (G_WEBWIDGET_SIGNAL_ALERT,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, alert),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1,
                        G_TYPE_STRING);

    g_mozilla_engine_signals[MAILTO]
        = g_signal_new (G_WEBWIDGET_SIGNAL_MAILTO,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, mailto),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1,
                        G_TYPE_STRING);

    g_mozilla_engine_signals[MODAL_DIALOG]
        = g_signal_new (G_WEBWIDGET_SIGNAL_MODAL_DIALOG,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, modal_dialog),
                        NULL, NULL,
                        g_mozeal_marshal_INT__STRING_STRING_INT_INT_INT_INT,
                        G_TYPE_INT,
                        6, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT,
                        G_TYPE_INT, G_TYPE_INT, G_TYPE_INT);

    g_mozilla_engine_signals[UNKNOWN_PROTOCOL]
        = g_signal_new (G_WEBWIDGET_SIGNAL_UNKNOWN_PROTOCOL,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, unknown_protocol),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1,
                        G_TYPE_STRING);

    g_mozilla_engine_signals[RAISE_REQUEST]
        = g_signal_new (G_WEBWIDGET_SIGNAL_RAISE_REQUEST,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, raise_request),
                        NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

    g_mozilla_engine_signals[LOWER_REQUEST]
        = g_signal_new (G_WEBWIDGET_SIGNAL_LOWER_REQUEST,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, lower_request),
                        NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

    g_mozilla_engine_signals[FILE_UPLOAD_STARTED]
        = g_signal_new (G_WEBWIDGET_SIGNAL_FILE_UPLOAD_STARTED,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, file_upload_started),
                        NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

    g_mozilla_engine_signals[FILE_UPLOAD_FINISHED]
        = g_signal_new (G_WEBWIDGET_SIGNAL_FILE_UPLOAD_FINISHED,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, file_upload_finished),
                        NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

    g_mozilla_engine_signals[EXTERNAL_ACTION]
        = g_signal_new (G_WEBWIDGET_SIGNAL_EXTERNAL_ACTION,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, external_action),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1,
                        G_TYPE_STRING);

    g_mozilla_engine_signals[FRAME_STACKING_CHANGED]
        = g_signal_new (G_WEBWIDGET_SIGNAL_FRAME_STACKING_CHANGED,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass,
                                         frame_stacking_changed), NULL, NULL,
                        g_cclosure_marshal_VOID__BOOLEAN, G_TYPE_NONE, 1,
                        G_TYPE_BOOLEAN);


    g_mozilla_engine_signals[SECURITY_MODE_CHANGED]
        = g_signal_new (G_WEBWIDGET_SIGNAL_SECURITY_MODE_CHANGED,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_LAST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, security_mode_changed),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1,
                        G_TYPE_UINT);

    g_mozilla_engine_signals[CLOSE_WINDOW]
        = g_signal_new (G_WEBWIDGET_SIGNAL_CLOSE_WINDOW,
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, close_window),
                        NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE,
                        0);

    g_mozilla_engine_signals[ICON_SIGNAL]
        = g_signal_new ("icon",
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, icon),
                        NULL, NULL,
                        g_cclosure_marshal_VOID__POINTER,
                        G_TYPE_NONE, 1, G_TYPE_POINTER);

    g_mozilla_engine_signals[RSS_SIGNAL]
        = g_signal_new ("rss-request",
                        G_TYPE_MOZILLA_ENGINE,
                        G_SIGNAL_RUN_FIRST,
                        G_STRUCT_OFFSET (GMozillaEngineClass, rss_request),
                        NULL, NULL,
                        g_mozeal_marshal_VOID__STRING_STRING,
                        G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);
}

static void
g_mozilla_reset_context_info(GMozillaEngine *self)
{
    EAL_IF_GFREE(self->ctx_url);
    EAL_IF_GFREE(self->ctx_objurl);
    EAL_IF_GFREE(self->ctx_docurl);

    self->ctx_node = NULL;
    self->last_event = NULL;
}

static void
g_mozilla_engine_destroy (GMozillaEngine *self)
{
    TRACE_LOG();
    g_mozilla_engine_reset (self);
    g_mozilla_reset_context_info(self);
    g_mozilla_unregister_im_context(self);
    return;
}

GObject*
g_mozilla_engine_new (void)
{
    TRACE_LOG();
    return g_mozilla_engine_new_with_context (0);
}

GObject*
g_mozilla_engine_new_with_context (gpointer context)
{
    TRACE_LOG();
    return g_mozilla_engine_new_with_context_and_tracks (context, FALSE, FALSE);
}

static gboolean
content_set_cursor(GMozillaEngine *self, GdkCursorType type)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    if(!type)
        return FALSE;

    GdkCursor* cursor = NULL;
    GMozillaWeb *global = G_MOZILLA_WEB(self->global);
    switch(type)
    {
        case GDK_LEFT_PTR:
	    if(global->default_cursor == NULL)
	      global->default_cursor = gdk_cursor_new (GDK_LEFT_PTR);
	    if(global->default_cursor)  
	      cursor = global->default_cursor;   
	    break;
	default:
	    cursor = gdk_cursor_new (type);
    }

    if(cursor && self->engine)
    {
        GtkWidget * child = gtk_bin_get_child((GtkBin *)self->engine);
        if (child)
          gdk_window_set_cursor (GDK_WINDOW(GTK_WIDGET(child)->window), cursor);
        if(cursor != global->default_cursor)  
          gdk_cursor_unref(cursor);
        return TRUE;
    }
    return FALSE;
}

static void
location_changed_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    gchar * location = gtk_moz_embed_get_location(embed);
    if (!location)
        return;
    if(!strcmp(location, "") || !strcmp(location, "data:text/html,")) {
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[URL_CHANGED] , 0, "about:blank");
    } else {
        g_signal_emit_by_name(G_OBJECT(self->global), "global-history-item-added", location);
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[URL_CHANGED] , 0, location);
    }

    g_free(location);
}

static void
title_changed_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    gchar * title = gtk_moz_embed_get_title(embed);
    if (!title)
        return;
    // bug 49447.- solved
    ULOG_DEBUG_F("Embed:%p, Title: '%s'", embed, title);
    if(!strcmp(title, ""))
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[TITLE_CHANGED] , 0, "(Untitled)");
    else
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[TITLE_CHANGED] , 0, title);

    g_free(title);
}

static void
load_started_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    self->is_loading = TRUE;
    EAL_IF_GFREE(self->last_location);

    // part of bug 42663 solution. see more at prompt_auth_cb.
    self->last_location = gtk_moz_embed_get_location(embed);

    self->last_net_status = 0;
    if (!self->will_open_location)
        self->will_open_location = g_strdup(self->last_location);
    send_progress_signal(self, START_LOADING, self->will_open_location, 0, 0);
    content_set_cursor(self, GDK_WATCH);

#ifdef USE_CONIC
    if (!g_mozilla_connectivity_is_connected() && !protocol_is_local(self->will_open_location) && !host_is_local(self->will_open_location)) {
        if (g_mozilla_connectivity_connect(self, FALSE)) {
            return;
        }
    }
#endif
}

static void
load_finished_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    //ULOG_DEBUG_F("last_net_status: %i", self->last_net_status);
    self->is_loading = FALSE;

    GWebEngineEncoding encoding = g_mozilla_engine_get_encoding (embed);
    // (gint finish_status, gint page_size, gboolean has_frames)
    if (self->progress_bar_state != -1)
        location_changed_cb(embed, self);
    send_progress_signal(self, FINISHED_LOADING, NULL, 0, 0);
    content_set_cursor(self, GDK_LEFT_PTR);
    g_signal_emit_by_name (G_OBJECT(self), G_WEBWIDGET_SIGNAL_ENCODING, encoding);
    if (self->optimized_view)
    {
        gtk_moz_embed_common_observe("@mozilla.org/extensions/ftv;1", NULL, "optimized_view", NULL);
    }
    register_im_context(self);
}

static void
progress_change_cb (GtkMozEmbed *embed, gint cur,
                    gint max, GMozillaEngine *self)
{
    TRACE_LOG("cur: %i, max: %i", cur, max);
    if (max > 0)
      self->page_size = max;
    else if (cur > 0)
      self->page_size = cur;
}

static void
progress_change_all_cb (GtkMozEmbed *embed, const char *uri,
                        gint cur, gint max, GMozillaEngine *self)
{
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    TRACE_LOG("Uri: %s, cur: %i, max: %i, is_loading: %i, net_stat: %i", uri, cur, max, self->is_loading, self->last_net_status);
    if (max > cur && self->is_loading) {
        send_progress_signal(self, LOADING_PROGRESS, NULL, cur, max);
        content_set_cursor(self, GDK_WATCH);
    } else {
        if (max >= cur) {
            send_progress_signal(self, FINISHED_LOADING, NULL, 0, 0);
            if (self->last_net_status != 0 && max > cur)
                self->last_net_status = 0;
            content_set_cursor(self, GDK_LEFT_PTR);
        }
    }
}

static void
net_state_change_cb (GtkMozEmbed *embed, gint flags,
                     guint status, GMozillaEngine *self)
{
    TRACE_LOG();
}

static void
net_state_change_all_cb (GtkMozEmbed *embed, const char *uri,
                         gint flags, guint status, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
//    ULOG_DEBUG_F("uri: '%s', flags: %x, status: %x", uri, flags, status);
    // Network stuff
    switch (status)
    {
        case GTK_MOZ_EMBED_STATUS_FAILED_DNS:
            ULOG_DEBUG_F("GTK_MOZ_EMBED_STATUS_FAILED_DNS:");
        case GTK_MOZ_EMBED_STATUS_FAILED_CONNECT:
            ULOG_DEBUG_F("GTK_MOZ_EMBED_STATUS_FAILED_CONNECT:");
        case GTK_MOZ_EMBED_STATUS_PROXY_FAILED:
        {
            ULOG_DEBUG_F("GTK_MOZ_EMBED_STATUS_PROXY_FAILED:");
#ifdef USE_CONIC
            EAL_IF_GFREE(self->will_open_location);
            self->will_open_location = g_strdup(uri);
            if (g_mozilla_connectivity_is_connected()) {
                g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[FINISHED_LOADING], 0, self->last_net_status, self->page_size, FALSE);
                content_set_cursor(self, GDK_LEFT_PTR);
            } else {
                send_progress_signal(self, FINISHED_LOADING, NULL, 0, 0);
                send_progress_signal(self, START_LOADING, uri, 0, 0);
                content_set_cursor(self, GDK_WATCH);

                if (!g_mozilla_connectivity_connect(self, FALSE)) {
                    send_progress_signal(self, FINISHED_LOADING, NULL, 0, 0);
                    content_set_cursor(self, GDK_LEFT_PTR);
                }
            }
#else
            g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_UNKNOWN_PROTOCOL,
                                   self->ctx_url ? self->ctx_url : self->will_open_location);
#endif
            break;
        }
    }
}

static void
link_message_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
}

static void
js_status_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
}

static void
new_window_cb (GtkMozEmbed *embed, GtkMozEmbed **newEmbed,
               guint chromemask, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    //CHROMEMASK IS MISSED// Fixme
    g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[WINDOW_REQUEST] , 0, newEmbed, "", FALSE, FALSE);
}

/*
   static void
   new_window_orphan_cb (GtkMozEmbedSingle * embed,
   GtkMozEmbed ** retval, guint chromemask, gpointer data)
   {
   TRACE_LOG();
   g_print ("new_window_orphan_cb\n");
   }
   */

static void
visibility_cb (GtkMozEmbed *embed, gboolean visibility, GMozillaEngine *self)
{
    TRACE_LOG();
}

static void
destroy_brsr_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[CLOSE_WINDOW] , 0);
}

static gint
open_uri_cb (GtkMozEmbed *embed, const char *uri, GMozillaEngine *self)
{
    g_return_val_if_fail(uri, FALSE);
    g_return_val_if_fail(self, FALSE);
    TRACE_LOG("open_uri_cb %s\n", uri);

    EAL_IF_GFREE(self->will_open_location);
    self->will_open_location = g_strdup(uri);
    
    gboolean pr_local = protocol_is_local (self->will_open_location);
    self->user_iteraction_happens = !(!g_mozilla_connectivity_is_connected () && pr_local && !self->user_iteraction_happens);

    if (!g_mozilla_connectivity_is_connected() && !pr_local && !host_is_local(self->will_open_location))
        if (g_mozilla_connectivity_connect(self, FALSE))
            return TRUE;
    
    g_mozilla_engine_update_context_data(self, embed, NULL);
    // don't interrupt anything
    return FALSE;
}

static void
update_im_context_mode(GMozillaEngine *self)
{
    TRACE_LOG("gtkIMContext:%p, multi:%i, pass:%i\n", self->gtkIMContext, (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_MULTILINE), (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_IPASSWORD));
#ifdef USE_HILDON2007
    if (!self->gtkIMContext) return;
    HildonGtkInputMode mode;
    g_object_get (G_OBJECT (self->gtkIMContext), "hildon-input-mode", &mode, NULL);
    if (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_MULTILINE)
        mode |= HILDON_GTK_INPUT_MODE_MULTILINE;
    else
        mode &= ~HILDON_GTK_INPUT_MODE_MULTILINE;
    if (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_IPASSWORD)
        mode |= HILDON_GTK_INPUT_MODE_INVISIBLE;
    else
        mode &= ~HILDON_GTK_INPUT_MODE_INVISIBLE;
    g_object_set (G_OBJECT (self->gtkIMContext), "hildon-input-mode", mode, NULL);
#endif
}

static gboolean
gdk_key_press_cb (GtkWidget *embed, GdkEventKey *gdkevent,
                              GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(self, FALSE);
    g_return_val_if_fail(register_im_context(self), FALSE);

    gint type = 7;
    gtk_moz_embed_common_get_pref (G_TYPE_INT, "accessibility.tabfocus", &type);

    if (self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_INPUT_TEXT
        || self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_IPASSWORD)
    {
        if (gdkevent->keyval == GDK_Tab && type != 1) {
            type = 1;
            gtk_moz_embed_common_set_pref (G_TYPE_INT, "accessibility.tabfocus", &type);
        }
        gtk_im_context_focus_in((GtkIMContext *)self->gtkIMContext);
        if (gtk_im_context_filter_keypress((GtkIMContext *)self->gtkIMContext, (GdkEventKey *)gdkevent))
        {
            return TRUE;
        }

        if(gdkevent->keyval == GDK_Left || gdkevent->keyval == GDK_Right)
            gtk_im_context_reset ((GtkIMContext *)self->gtkIMContext);
    }

    if (type == 1) {
        type = 7;
        gtk_moz_embed_common_set_pref (G_TYPE_INT, "accessibility.tabfocus", &type);
    }
    return FALSE;
}

static gboolean
gdk_key_release_cb (GtkWidget *embed, GdkEventKey *gdkevent,
                              GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(self, FALSE);
    g_return_val_if_fail((GtkIMContext *)self->gtkIMContext, FALSE);

    if (self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_INPUT_TEXT
        || self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_IPASSWORD)
    {
        gtk_im_context_focus_in((GtkIMContext *)self->gtkIMContext);
        if (gtk_im_context_filter_keypress((GtkIMContext *)self->gtkIMContext, (GdkEventKey *)gdkevent))
        {
            return TRUE;
        }
        if(gdkevent->keyval == GDK_Left || gdkevent->keyval == GDK_Right
           || gdkevent->keyval == GDK_Down || gdkevent->keyval == GDK_Up) {
            vkb_handle_action(self, self->moz_ctx_type, gdkevent);
            gtk_im_context_reset ((GtkIMContext *)self->gtkIMContext);
        }
    }
    return FALSE;
}

static gboolean
gdk_press_event_cb (GtkWidget *embed, GdkEventButton *event,
                              GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(self, FALSE);
    g_return_val_if_fail((GtkIMContext *)self->gtkIMContext, FALSE);
    gtk_im_context_reset ((GtkIMContext *)self->gtkIMContext);
    self->moz_ctx_type = 0;
    gtk_im_context_focus_in((GtkIMContext *)self->gtkIMContext);
#ifdef USE_HILDON
    hildon_gtk_im_context_filter_event((GtkIMContext *)self->gtkIMContext, (GdkEvent *)event);
#endif
    return FALSE;
}

static gboolean
gdk_release_event_cb (GtkWidget *embed, GdkEventButton *event,
                              GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(self, FALSE);
    gboolean is_finger = FALSE;

    gdouble pressure = 0;
//    gdk_event_get_axis ((GdkEvent*)event, GDK_AXIS_PRESSURE, &pressure);

#ifdef USE_HILDON2007
    is_finger = hildon_helper_event_button_is_finger(event);
#else
#ifdef USE_HILDON
#ifndef USE_OS2006
    is_finger = hildon_button_event_is_finger (event);
#endif
#endif
#endif
    ULOG_DEBUG_F("FKB Activate: pressure:%g, is_finger:%s, is_input:%s\n", pressure, is_finger?"TRUE":"FALSE",
                 (self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_INPUT_TEXT || self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_IPASSWORD)?"TRUE":"FALSE");

    if(is_finger)
    if (self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_INPUT_TEXT
        || self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_IPASSWORD)
    {
        ULOG_DEBUG_F("FKB Activate: pressure:%g, is_finger:%s, is_input:%s\n", pressure, is_finger?"TRUE":"FALSE",
                     (self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_INPUT_TEXT || self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_IPASSWORD)?"TRUE":"FALSE");
        gtk_im_context_focus_in((GtkIMContext *)self->gtkIMContext);

        update_im_context_mode(self);

        //Hack to open thumb vkb
        event->button = 2;
#ifdef USE_HILDON
        if (hildon_gtk_im_context_filter_event((GtkIMContext *)self->gtkIMContext, (GdkEvent *)event))
        {
            return TRUE;
        }
        return TRUE; // hildon_gtk_im_context_filter_event by some reasons always return FALSE and we cannot handle FKB properly...
#endif
    }
    return FALSE;
}

static void
size_to_cb (GtkMozEmbed *embed, gint width, gint height,
            GMozillaEngine *self)
{
    TRACE_LOG("*** size_to_cb %d %d", width, height);
}

static guint
convert_ctx_mozembed_to_eal(guint moz_embed_ctx)
{
    TRACE_LOG();
    guint eal_ctx = GTK_MOZ_EMBED_CTX_NONE;
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_MULTILINE)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_MULTILINE:");
        eal_ctx |= G_WEBENGINE_POPUPMENU_DOCUMENT;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_NONE)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_NONE: ");
        eal_ctx = G_WEBENGINE_POPUPMENU_NOTYPE;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_XUL)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_XUL:");
        eal_ctx = G_WEBENGINE_POPUPMENU_NOTYPE;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_SIDEBAR)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_SIDEBAR:");
        eal_ctx = G_WEBENGINE_POPUPMENU_NOTYPE;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_DOCUMENT)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_DOCUMENT:");
        eal_ctx |= G_WEBENGINE_POPUPMENU_DOCUMENT;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_LINK)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_LINK:");
        eal_ctx |= G_WEBENGINE_POPUPMENU_LINK;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_IMAGE)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_IMAGE:");
        eal_ctx |= G_WEBENGINE_POPUPMENU_IMAGE;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_IFRAME)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_IFRAME:");
        eal_ctx |= G_WEBENGINE_POPUPMENU_IFRAME;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_INPUT_TEXT)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_INPUT_TEXT:");
        eal_ctx |= G_WEBENGINE_POPUPMENU_DOCUMENT;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_IPASSWORD)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_IPASSWORD:");
        eal_ctx = G_WEBENGINE_POPUPMENU_NOTYPE;
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_RICHEDIT)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_RICHEDIT:");
    }
    if (moz_embed_ctx  & GTK_MOZ_EMBED_CTX_EMAIL)
    {
        ULOG_INFO_F("CTXTYPE: GTK_MOZ_EMBED_CTX_EMAIL:");
        eal_ctx |= G_WEBENGINE_POPUPMENU_MAILTO;
        //	G_WEBENGINE_POPUPMENU_UNKNOWN_PROTOCOL;
    }
    return eal_ctx;
}

gboolean
g_moz_eal_im_context_has_selection(GtkIMContext *context, GMozillaEngine *self)
{
    TRACE_LOG();
    if ((self->ctx_type != G_WEBENGINE_POPUPMENU_NOTYPE) & ( self->ctx_type != GTK_MOZ_EMBED_CTX_IPASSWORD )) {

        const gchar * selection = gtk_moz_embed_get_selection(GTK_MOZ_EMBED(self->engine));
        if (selection) {
            if (!strcmp(selection, ""))
                return FALSE;
            else
                return TRUE;
        }
        else
            return FALSE;
    } else
        return FALSE;
}

static void
g_moz_eal_im_context_clipboard_operation(GtkIMContext *context, int operation, GMozillaEngine *self)
{
    TRACE_LOG();
#ifdef USE_HILDON
    switch(operation)
    {
    case GTK_IM_CONTEXT_CLIPBOARD_OP_COPY:
        g_mozilla_engine_copy(self);
        break;
    case GTK_IM_CONTEXT_CLIPBOARD_OP_CUT:
        g_mozilla_engine_cut(self);
        break;
    case GTK_IM_CONTEXT_CLIPBOARD_OP_PASTE:
        g_mozilla_engine_paste(self);
        break;
    }
#endif
}

gboolean
g_moz_eal_im_context_commit(GtkIMContext *ctx, const gchar *str, GMozillaEngine *self)
{
    g_return_val_if_fail(str, FALSE);
    TRACE_LOG();
    gtk_moz_embed_insert_text(GTK_MOZ_EMBED(self->engine), str, self->ctx_node,
                              self->im_surr_enable?self->im_surr_offset:0,
                              self->im_surr_enable?self->im_surr_num:0);
    self->im_surr_enable = FALSE;
    return TRUE;
}

void
g_moz_eal_preedit_changed_cb(GtkIMContext *aContext, GMozillaEngine *self)
{
    TRACE_LOG();
    ULOG_DEBUG_F();
}

gboolean
g_moz_eal_im_context_delete_surrounding(GtkIMContext *aContext, gint arg1, gint arg2, GMozillaEngine *self)
{
    TRACE_LOG();
    self->im_surr_offset = arg1;
    self->im_surr_num = arg2;
    self->im_surr_enable = TRUE;
    return TRUE;
}

gboolean
g_moz_eal_im_context_retrieve_surrounding(GtkIMContext *aContext, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    
    gchar *string = NULL;
    gint cursor_pos = 0;

    gtk_moz_embed_get_text(GTK_MOZ_EMBED(self->engine), (gchar**)&string, &cursor_pos);

    if(string != NULL)
    gtk_im_context_set_surrounding (aContext,
                                    string,
                                    strlen(string),
                                    cursor_pos);
    return TRUE;
}

static void
g_mozilla_unregister_im_context(GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    if (!self->gtkIMContext)
        return;
    if (self->im_signal_cb[IM_SIG_HAS_SELECTION])
        g_signal_handler_disconnect(self->gtkIMContext,
                                    self->im_signal_cb[IM_SIG_HAS_SELECTION]);
    if (self->im_signal_cb[IM_SIG_CLIPBOARD_OPERATION])
        g_signal_handler_disconnect(self->gtkIMContext,
                                    self->im_signal_cb[IM_SIG_CLIPBOARD_OPERATION]);
    if (self->im_signal_cb[IM_SIG_COMMIT])
        g_signal_handler_disconnect(self->gtkIMContext,
                                    self->im_signal_cb[IM_SIG_COMMIT]);

    if (self->im_signal_cb[IM_SIG_DELETE_SURROUNDING])
        g_signal_handler_disconnect(self->gtkIMContext,
                                    self->im_signal_cb[IM_SIG_DELETE_SURROUNDING]);
    if (self->im_signal_cb[IM_SIG_RETRIEVE_SURROUNDING])
        g_signal_handler_disconnect(self->gtkIMContext,
                                    self->im_signal_cb[IM_SIG_RETRIEVE_SURROUNDING]);

/*
    if (self->im_signal_cb[IM_SIG_PREEDIT_CHANGED])
        g_signal_handler_disconnect(self->gtkIMContext,
                                    self->im_signal_cb[IM_SIG_PREEDIT_CHANGED]);
    if (self->im_signal_cb[IM_SIG_PREEDIT_END])
       g_signal_handler_disconnect(self->gtkIMContext,
                                   self->im_signal_cb[IM_SIG_PREEDIT_END]);
    if (self->im_signal_cb[IM_SIG_PREEDIT_START])
        g_signal_handler_disconnect(self->gtkIMContext,
                                    self->im_signal_cb[IM_SIG_PREEDIT_START]);
*/
    gtk_im_context_reset(self->gtkIMContext);
    self->gtkIMContext = NULL;
}

static GtkIMContext*
register_im_context(GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), NULL);
    if (self->gtkIMContext)
        return self->gtkIMContext;

    GtkIMContext * im_ctx = NULL;
    im_ctx = gtk_im_multicontext_new ();

    gtk_im_context_set_client_window (im_ctx, GDK_WINDOW(GTK_WIDGET(self->engine)->window));
    self->gtkIMContext = im_ctx;

#ifdef USE_HILDON
    HildonGtkInputMode mode;
    g_object_get (G_OBJECT(im_ctx), "hildon-input-mode", &mode, NULL);
    mode ^= HILDON_GTK_INPUT_MODE_AUTOCAP;
    g_object_set(G_OBJECT(im_ctx), "hildon-input-mode", mode, NULL);
#endif

    self->im_signal_cb[IM_SIG_HAS_SELECTION] =
        g_signal_connect(G_OBJECT(im_ctx), "has_selection", G_CALLBACK(g_moz_eal_im_context_has_selection), self);
    self->im_signal_cb[IM_SIG_CLIPBOARD_OPERATION] =
        g_signal_connect(G_OBJECT(im_ctx), "clipboard_operation", G_CALLBACK(g_moz_eal_im_context_clipboard_operation), self);
    self->im_signal_cb[IM_SIG_COMMIT] =
        g_signal_connect(G_OBJECT(im_ctx), "commit", G_CALLBACK(g_moz_eal_im_context_commit), self);

    self->im_signal_cb[IM_SIG_DELETE_SURROUNDING] =
        g_signal_connect(G_OBJECT(im_ctx), "delete-surrounding", G_CALLBACK(g_moz_eal_im_context_delete_surrounding), self);
    self->im_signal_cb[IM_SIG_RETRIEVE_SURROUNDING] =
        g_signal_connect(G_OBJECT(im_ctx), "retrieve-surrounding", G_CALLBACK(g_moz_eal_im_context_retrieve_surrounding), self);

/*
    self->im_signal_cb[IM_SIG_PREEDIT_CHANGED] =
        g_signal_connect(G_OBJECT(im_ctx), "preedit_changed", G_CALLBACK(g_moz_eal_im_context_preedit_changed_cb), self);
    self->im_signal_cb[IM_SIG_PREEDIT_END] =
        g_signal_connect(G_OBJECT(im_ctx), "preedit-end", G_CALLBACK(g_moz_eal_im_context_preedit_end), self);
    self->im_signal_cb[IM_SIG_PREEDIT_START] =
        g_signal_connect(G_OBJECT(im_ctx), "preedit-start", G_CALLBACK(g_moz_eal_im_context_preedit_start), self);
*/

    GtkWidget * child = gtk_bin_get_child(GTK_BIN(self->engine));
    if (child) {
        g_signal_connect(G_OBJECT(child), "key_press_event",
                           G_CALLBACK(gdk_key_press_cb), self);
        g_signal_connect(G_OBJECT(child), "key_release_event",
                           G_CALLBACK(gdk_key_release_cb), self);
        g_signal_connect(G_OBJECT(child), "button_press_event",
                           G_CALLBACK(gdk_press_event_cb), self);
        g_signal_connect(G_OBJECT(child), "button_release_event",
                           G_CALLBACK(gdk_release_event_cb), self);
//        gtk_widget_set_extension_events (child, GDK_EXTENSION_EVENTS_ALL);
        gtk_widget_set_extension_events (child, GDK_EXTENSION_EVENTS_NONE);
    }

    return self->gtkIMContext;
}

static gboolean scroll_node(GMozillaEngine *self)
{
  gtk_moz_embed_insert_text(GTK_MOZ_EMBED(self->engine), NULL, self->ctx_node, 0, 0);
  EAL_IF_GFREE_FUNC(self->scroll_to_node_timer_id, g_source_remove);
  return FALSE;
}

static gboolean
vkb_handle_action(GMozillaEngine *self, guint moz_embed_ctx, GdkEventKey * event)
{
    TRACE_LOG();
#ifdef USE_HILDON
    GtkIMContext *im = NULL;
    im = register_im_context(self);
    gboolean show_vkb = TRUE;
    
    if (event && (event->keyval != GDK_Return))
        show_vkb = FALSE;

    if (moz_embed_ctx & GTK_MOZ_EMBED_CTX_INPUT_TEXT)
    {
        if(show_vkb) 
        {
            update_im_context_mode(self);

/*          We dont want to put million hacks to enable and disable CAPS mode, like it is done in GTK_ENTRY....
            if (moz_embed_ctx & GTK_MOZ_EMBED_CTX_MULTILINE)
                mode |= HILDON_GTK_INPUT_MODE_AUTOCAP;
            else
                mode &= ~HILDON_GTK_INPUT_MODE_AUTOCAP;
*/
            guint32 t = gtk_moz_embed_get_chrome_mask(GTK_MOZ_EMBED(self->engine));
            t ^=GTK_MOZ_EMBED_FLAG_WINDOWRESIZEON;
            gtk_moz_embed_set_chrome_mask(GTK_MOZ_EMBED(self->engine), t);
            hildon_gtk_im_context_show (im);
            EAL_IF_GFREE_FUNC(self->scroll_to_node_timer_id, g_source_remove);
            self->scroll_to_node_timer_id = g_timeout_add(NODE_SCROLL_TIMEOUT, scroll_node, self);
        }
    } else {
        hildon_gtk_im_context_hide (im);
        gtk_moz_embed_set_chrome_mask(GTK_MOZ_EMBED(self->engine),gtk_moz_embed_get_chrome_mask(GTK_MOZ_EMBED(self->engine)));
    }
#endif
    TRACE_LOG();
    return TRUE;
}

static gboolean g_mozilla_engine_update_context_data (GMozillaEngine *self, GtkMozEmbed *embed, gpointer event)
{
    TRACE_LOG();
    self->x = 0;
    self->y = 0;

    EAL_IF_GFREE(self->ctx_url);
    EAL_IF_GFREE(self->ctx_objurl);
    EAL_IF_GFREE(self->ctx_docurl);

    if (event)
        self->last_event = event;

    gboolean multi = (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_MULTILINE);
    gboolean input = (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_INPUT_TEXT);
    gboolean widget_hacked = !multi && input;
    self->moz_ctx_type = 0;
    if (embed && event)
    self->moz_ctx_type =
        gtk_moz_embed_get_context_info(embed, event, &self->ctx_node,
                                       &self->x, &self->y, &self->doc_index,
                                       (const gchar**)&self->ctx_url, (const gchar**)&self->ctx_objurl,
                                       (const gchar**)&self->ctx_docurl);
    multi = (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_MULTILINE);
    input = (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_INPUT_TEXT);
    gboolean widget_need_hack = !multi && input;
    if (widget_hacked && !widget_need_hack) {
#ifdef USE_HILDON2007
        g_object_steal_data (self->engine, SINGLE_LINE_WIDGET_NAME);
#else
        gtk_widget_set_name (self->engine, "gtkmozembed");
#endif
    } else  if (!widget_hacked && widget_need_hack) {
#ifdef USE_HILDON2007
        g_object_set_data (self->engine, SINGLE_LINE_WIDGET_NAME, SINGLE_LINE_WIDGET_NAME);
#else
        /* Please kill me ASAP */
        char name[] = { 'O','p','e','r','a',' ','e','n','t','r','y', '\0'};
        gtk_widget_set_name (self->engine, &name);
#endif
    }
    self->ctx_type = convert_ctx_mozembed_to_eal(self->moz_ctx_type);
    update_im_context_mode(self);
    return TRUE;
}

static gboolean g_mozilla_engine_show_menu (GMozillaEngine *self)
{
    TRACE_LOG();

    // if context sensitive context has no type, so there is no need to invoke it.
    if (self->ctx_type != G_WEBENGINE_POPUPMENU_NOTYPE)
    {
        const gchar * selection = gtk_moz_embed_get_selection(GTK_MOZ_EMBED(self->engine));
        if (selection)
        {
            if (0 != strcmp(selection, ""))
            {
                ULOG_DEBUG_F("CTXTYPE: SELECTION EXIST = %s\n", selection);
                self->ctx_type |= G_WEBENGINE_POPUPMENU_TEXTSELECTION;
            } else  {
                selection = NULL;
                if (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_INPUT_TEXT)
                    return FALSE;
            }
        }

        // NOTE: Hack for UI
        const gchar *frame_url = NULL;
        if (self->doc_index == -1)
            frame_url = ""; //UI hack
        else
            frame_url = self->ctx_docurl;
        //printf("ctx_url:%s, ctx_objurl:%s, frame_url:%s, index:%i\n", self->ctx_url, self->ctx_objurl, frame_url, self->doc_index);
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[WEBENGINE_POPUP_MENU], 0,
                      self->ctx_type , self->x, self->y,
                      selection, self->ctx_url, self->ctx_objurl, frame_url); //FIXME XXX MEMLEAK
        return TRUE;
    }
    return FALSE;
}

// Cursor animation stuff
static guint animation_timer_id = 0;

static gboolean gtk_mozilla_tap_and_hold_timeout (AnimeCursor *a_cursor)
{
    TRACE_LOG();
    if (!a_cursor || !a_cursor->anime_iter)
      return FALSE;
    GdkPixbuf *pic;
    GdkCursor *cursor;
    GTimeVal time;

    g_get_current_time (&time);

    pic = gdk_pixbuf_animation_iter_get_pixbuf (GDK_PIXBUF_ANIMATION_ITER(a_cursor->anime_iter));
    pic = gdk_pixbuf_copy (pic);

    gint x = gdk_pixbuf_get_width(pic) / 2;
    gint y = gdk_pixbuf_get_height(pic) / 2;

    cursor = gdk_cursor_new_from_pixbuf (gdk_display_get_default (), pic, x, y);

    g_object_unref (pic);

    if (!cursor)
        return FALSE;

    GMozillaEngine *moz_engine = a_cursor->link_to_engine;
    if(moz_engine) {
        GtkWidget * child = GTK_WIDGET(gtk_bin_get_child((GtkBin *)moz_engine->engine));
        if (child)
          gdk_window_set_cursor (GDK_WINDOW(child->window), cursor);
    }

    gdk_cursor_unref (cursor);

    gdk_pixbuf_animation_iter_advance (a_cursor->anime_iter, &time);

    return TRUE;
}

void
start_cursor_animation_handler (GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail (G_MOZILLA_WEB(self->global)->anime_cursor);
    GtkIconTheme *theme = NULL;
    GtkIconInfo *info = NULL;
    GError *error = NULL;
    const gchar *filename = NULL;
    GTimeVal time;
    AnimeCursor *a_cursor = G_MOZILLA_WEB(self->global)->anime_cursor;
    if(!a_cursor->anime)
    {
        theme = gtk_icon_theme_get_default ();
        if (!theme)
        {
            g_warning ("Unable to find icon theme");
            return;
        }
        info = gtk_icon_theme_lookup_icon (theme, "qgn_indi_tap_hold_a", GTK_ICON_SIZE_BUTTON,
                                           GTK_ICON_LOOKUP_NO_SVG);
        if (!info)
        {
            g_warning ("Unable to find icon info");
            return;
        }

        filename = gtk_icon_info_get_filename (info);

        if (!filename)
        {
            gtk_icon_info_free (info);
            g_warning ("Unable to find tap and hold icon filename");
            return;
        }

        a_cursor->anime = gdk_pixbuf_animation_new_from_file (filename, &error);
        gtk_icon_info_free (info);
        info = NULL;
        if (error)
        {
            g_warning ("Unable to create tap and hold animation: %s", error->message);
            a_cursor->anime = NULL;
            g_error_free (error);
            return;
        }
    }

    g_get_current_time (&time);
    a_cursor->anime_iter = gdk_pixbuf_animation_get_iter (a_cursor->anime, &time);
    gint anime_interval = gdk_pixbuf_animation_iter_get_delay_time (a_cursor->anime_iter);

    a_cursor->link_to_engine = self;

    animation_timer_id = g_timeout_add (anime_interval, (GSourceFunc)
                                        gtk_mozilla_tap_and_hold_timeout, a_cursor);
}

static void
stop_cursor_animation_handler (GMozillaEngine *self)
{
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    if (!animation_timer_id)
        return;
    TRACE_LOG();
    GMozillaWeb *global = G_MOZILLA_WEB(self->global);
    AnimeCursor *a_cursor = global->anime_cursor;

    content_set_cursor(self, GDK_LEFT_PTR);    

    if (a_cursor->anime_iter)
    {
        g_object_unref(a_cursor->anime_iter);
        a_cursor->anime_iter = NULL;
    }

    g_source_remove(animation_timer_id);
}

// End Cursor animation stuff

static gint
dom_key_down_cb (GtkMozEmbed *embed, void *event,
                 GMozillaEngine *self)
{
    TRACE_LOG();
    register_im_context(self);
    user_iteraction_happens(self);
    return FALSE;
}

static gint
dom_key_press_cb (GtkMozEmbed *embed, void *event,
                  GMozillaEngine *self)
{
    TRACE_LOG();
    register_im_context(self);
    return FALSE;
}

static gint
dom_key_up_cb (GtkMozEmbed *embed, void *event,
               GMozillaEngine *self)
{
    TRACE_LOG();
    g_mozilla_engine_update_context_data(self, embed, event);
    vkb_handle_action(self, self->moz_ctx_type, event);
    return FALSE;
}

static
gint dom_mouse_down_cb (GtkMozEmbed *embed, void *event,
                        GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    user_iteraction_happens(self);
	/* XXX FROM 450_hildonizing_defaultinput_method.diff, why it was there?
    if (hildon_gtk_im_context_filter_event(im, (GdkEvent*)aEvent))
    return;

	Also: where it should be?
	   if (NS_SUCCEEDED(GetFlags(&flags)) &&
+      // tablet: IME needed for password fields
+      flags & (nsIPlaintextEditor::eEditorReadonlyMask |
+                nsIPlaintextEditor::eEditorDisabledMask))
+     *aState = nsIContent::IME_STATUS_DISABLE;

	*/
    register_im_context(self);
    g_mozilla_engine_update_context_data(self, embed, event);
    if ((self->moz_ctx_type & GTK_MOZ_EMBED_CTX_INPUT_TEXT
        || self->moz_ctx_type & GTK_MOZ_EMBED_CTX_IPASSWORD)
        && !(self->moz_ctx_type & GTK_MOZ_EMBED_CTX_RICHEDIT)
       )
    {
        set_extension_mode(self, GDK_EXTENSION_EVENTS_ALL);
    }else {
        set_extension_mode(self, GDK_EXTENSION_EVENTS_NONE);
    }

    if (self->moz_ctx_type & GTK_MOZ_EMBED_CTX_DOCUMENT)
        start_cursor_animation_handler(self);
    gettimeofday (&current_time_press, NULL);
    g_mozilla_support_timeval_subtract (&result, &current_time_press, &former_time_press);
    former_time_press = current_time_press;
    if (result.tv_usec <= 350000 && !result.tv_sec)
    {
        stop_cursor_animation_handler(self);
        return FALSE;
    }
    else
    {
        return TRUE;
    }
}

static gboolean vkb_up_timeout_cb(GMozillaEngine *self)
{
    TRACE_LOG();
    vkb_handle_action(self, self->moz_ctx_type, NULL);
    EAL_IF_GFREE_FUNC (self->vkb_up_timeout, g_source_remove);
    return FALSE;
}

static gint dom_mouse_up_cb (GtkMozEmbed *embed, void *event,
                             GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    stop_cursor_animation_handler(self);
    set_extension_mode(self, GDK_EXTENSION_EVENTS_NONE);
    // Fixed most of problems with iteractions after IM visiting
    if (!self->isFkbShowed)
        g_mozilla_engine_update_context_data(self, embed, event);
    if (!self->vkb_up_timeout)
        self->vkb_up_timeout = g_timeout_add (NODE_SCROLL_TIMEOUT, vkb_up_timeout_cb, self);
    return FALSE;
}

#ifdef UNUSED
static gint dom_mouse_move_cb (GtkMozEmbed *embed, void *event,
                               GMozillaEngine *self)
{
    TRACE_LOG();
    return FALSE;
}
#endif

static gint dom_mouse_scroll_cb (GtkMozEmbed *embed, void *event,
                                 GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    stop_cursor_animation_handler(self);
    if (self->moz_ctx_type  & GTK_MOZ_EMBED_CTX_DOCUMENT)
    {
        return FALSE;
    }

    TRACE_LOG("Scrolling canceled");
    return TRUE;
}

static gint dom_mouse_long_press_cb (GtkMozEmbed *embed, void *event,
                                     GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    stop_cursor_animation_handler(self);
    user_iteraction_happens(self);
    g_mozilla_engine_show_menu(self);
    set_extension_mode(self, GDK_EXTENSION_EVENTS_NONE);
    return FALSE;
}


static gint dom_focus_cb (GtkMozEmbed *embed, void *event,
                          GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    g_mozilla_engine_update_context_data(self, embed, event);
    //vkb_handle_action(self, self->moz_ctx_type);
    return FALSE;
}


static gint dom_mouse_click_cb (GtkMozEmbed *embed, void *event,
                                GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    stop_cursor_animation_handler(self);

    // Fixed most of problems with iteractions after IM visiting
    //if (!self->isFkbShowed)
    //    g_mozilla_engine_update_context_data(self, embed, event);
    //vkb_handle_action(self, self->moz_ctx_type, NULL);
    return FALSE;
}

static gint dom_mouse_dbl_click_cb (GtkMozEmbed *embed, void *event,
                                    GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    stop_cursor_animation_handler(self);
    return FALSE;
}

static gint dom_mouse_over_cb (GtkMozEmbed *embed, void *event,
                               GMozillaEngine *self)
{
    //    TRACE_LOG();
    return FALSE;
}

static gint dom_mouse_out_cb (GtkMozEmbed *embed, void *event,
                              GMozillaEngine *self)
{
    //    TRACE_LOG();
    if (self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_INPUT_TEXT
        || self->moz_ctx_type &  GTK_MOZ_EMBED_CTX_IPASSWORD)
    {
        set_extension_mode(self, GDK_EXTENSION_EVENTS_ALL);
    }
    else set_extension_mode(self, GDK_EXTENSION_EVENTS_NONE);
    return FALSE;
}

static gint dom_activate_cb (GtkMozEmbed *embed, void *event,
                             GMozillaEngine *self)
{
    TRACE_LOG();
    return FALSE;
}

static gint dom_focus_in_cb (GtkMozEmbed *embed, void *event,
                             GMozillaEngine *self)
{
    TRACE_LOG();
    return FALSE;
}

static gint dom_focus_out_cb (GtkMozEmbed *embed, void *event,
                              GMozillaEngine *self)
{
    TRACE_LOG();
    return FALSE;
}

static gint dom_window_blur_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[LOWER_REQUEST] , 0);
    return TRUE;
}

static gint dom_window_focus_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[RAISE_REQUEST] , 0); //UI buggy and not ready for non direct widget
    return TRUE;
}

static void dom_content_blocked_cb (GtkMozEmbed *embed, const char *uri, const char *host, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    GtkMozEmbed *newEmbed = NULL;
    g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[WINDOW_REQUEST] , 0, &newEmbed, "", FALSE, TRUE);
    if(newEmbed)
        g_warning ("Popup functionality's broken, created unnecessary object!\n");
    return TRUE;
}


static void
download_request_cb (GtkMozEmbed *embed, const gchar *server, const gchar *file_name,
                     const gchar *file_type, glong file_size, gulong mode, GMozillaEngine *self)
{

    TRACE_LOG("embed:%p, server:%s, fname:%s, ftype:%s, fsize:%i\n", embed, server, file_name, file_type, file_size);
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    global_last_gtk_moz_embed_download_object = gtk_moz_embed_download_get_latest_object ();
    if (!global_last_gtk_moz_embed_download_object) return ;

    // tablet browser ui requires 0 for unknow_file_size cases.
    if (file_size == -1)
        file_size = (glong) 0;

    g_signal_emit_by_name (G_OBJECT(self), G_WEBWIDGET_SIGNAL_DOWNLOAD_REQUEST, server, file_name, file_type, file_size, mode);

    return ;
}

static gboolean
upload_dialog_cb (GtkMozEmbed *embed, const gchar *path, const gchar *filter, gchar **file_name_with_path, GMozillaEngine *self)
{

    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);

    gboolean response = TRUE;
    g_signal_emit_by_name (G_OBJECT(self), G_WEBWIDGET_SIGNAL_UPLOAD_DIALOG, getenv("HOME"), "", file_name_with_path, &response, NULL);

    return response;
}

static void
file_upload_started_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);

    g_signal_emit_by_name (G_OBJECT(self), G_WEBWIDGET_SIGNAL_FILE_UPLOAD_STARTED);

    return ;
}

static void
file_upload_finished_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);

    g_signal_emit_by_name (G_OBJECT(self), G_WEBWIDGET_SIGNAL_FILE_UPLOAD_FINISHED);

    return ;
}



gint
alert_cb (GtkMozEmbed *embed, const gchar *dialog_title, const gchar *alert_msg, GMozillaEngine *self)
{

    TRACE_LOG ();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    ULOG_DEBUG_F ("dialog_title: '%s', alert_msg: '%s'", dialog_title, alert_msg);
    if (alert_msg && dialog_title && (g_ascii_strcasecmp(dialog_title, "Unknown Protocol") == 0))
    {
        g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_UNKNOWN_PROTOCOL, self->ctx_url ? self->ctx_url : self->will_open_location);
        load_finished_cb (GTK_MOZ_EMBED(self->engine), self);
        return TRUE;
    }
    if(alert_msg && dialog_title && g_strstr_len(dialog_title, strlen(dialog_title), "JavaScript"))
    {
        gchar *message= g_strdup_printf("jalert:%s", alert_msg);
        g_signal_emit_by_name (G_OBJECT(self), G_WEBWIDGET_SIGNAL_ALERT, message);
        g_free(message);
    } else if (alert_msg)
        g_signal_emit_by_name (G_OBJECT(self), G_WEBWIDGET_SIGNAL_ALERT, alert_msg);
    if (!self->is_loading)
    {
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[FINISHED_LOADING], 0, self->last_net_status, self->page_size, FALSE);
        content_set_cursor(self, GDK_LEFT_PTR);
    }
    // Hack to disable mouse_down fake event
    stop_cursor_animation_handler(self);
    return FALSE;
}

static gboolean
prompt_cb (GtkMozEmbed *embed, const gchar *dialog_title, const gchar *dialog_msg, gchar **default_msg,
           const gchar *check_msg, gboolean *check_value, GMozillaEngine *self)
{

    TRACE_LOG ();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);

    gchar *response = NULL;
    g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_PROMPT, dialog_msg, *default_msg, &response, NULL);

    if (*default_msg && response)
    {
        EAL_GFREE(*default_msg);
        *default_msg = g_strdup_printf ("%s", response);
        return TRUE;
    }
    return FALSE;
}

static gboolean
confirm_cb (GMozillaEngine *embed, const gchar *dialog_title, const gchar *dialog_msg, GMozillaEngine *self)
{

    TRACE_LOG ();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);

    gboolean ret_val;
    g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_CONFIRM, dialog_msg, &ret_val, NULL);

    return ret_val;
}

static gint
confirm_ex_cb (GMozillaEngine *embed,
               const char *title, const char *text, guint bt_flags,
               const char *button1, const char *button2, const char *button3,
               const char *check_msg, gboolean *check_val,
               GMozillaEngine *self)
{

    TRACE_LOG ();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), 1);
    gboolean ret_val;
    gchar *ntext = (gchar*)text;
    gint cur = 0, i = 0;
    gboolean prev_space = FALSE, isspace = FALSE;
    for (i = 0; i != strlen(text); i++) {
        isspace = g_ascii_isspace(text[i]);
        if (!isspace || !prev_space)
            ntext[cur++] = text[i];
        prev_space = isspace;
    }
    ntext[cur] = 0;
    g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_CONFIRM, ntext, &ret_val, NULL);
    return ret_val ? 0 : 1;
}

static gboolean
prompt_auth_cb (GtkMozEmbed * embed, const gchar * dialog_title,
                const gchar * dialog_msg, gchar ** user_name,
                gchar ** user_passwd, const gchar * check_msg,
                gboolean * check_value, GMozillaEngine * self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    GList *logins = NULL;

    EAL_IF_GFREE(self->username);
    EAL_IF_GFREE(self->password);
    self->accept = FALSE;

    gint logins_count = gtk_moz_embed_common_get_logins (dialog_title, &logins);

    GMozillaWeb *global = G_MOZILLA_WEB (self->global);
    global->logins_list = NULL;
    if (logins_count)
        global->logins_list = logins;

    // FIXME: please, that is an workaround for bug 42663, so remove it as soon as it gets solved in a another way.
    gint authentification_type;

    gchar *temp_current_location = gtk_moz_embed_get_location (embed);

    if (self->last_location && !strcmp (self->last_location, temp_current_location))
        authentification_type = G_WEBENGINE_AUTHENTICATION_WRONG;
    else authentification_type = 0;

    EAL_IF_GFREE(temp_current_location);

    g_signal_emit_by_name (G_OBJECT (self),
                           G_WEBWIDGET_SIGNAL_AUTHENTIFICATION_REQUIRED, authentification_type,
                           dialog_title, dialog_msg);

    GList *ptr = g_list_first (logins);
    while (ptr)
    {
        GtkMozLogin *login = (GtkMozLogin *) ptr->data;
        EAL_IF_GFREE(login->user);
        EAL_IF_GFREE(login->pass);
        EAL_IF_GFREE(login->host);
        ptr = ptr->next;
    }

    if (self->accept)
    {
        if (self->username && *self->username)
        {
            if (user_name) {
                if (*user_name)
                    g_free(*user_name);
                *user_name = g_strdup (self->username);
                EAL_GFREE(self->username);
            }
        }
        if (self->password  && *self->password)
        {
            if (user_passwd) {
                if (*user_passwd)
                    g_free(*user_passwd);
                *user_passwd = g_strdup (self->password);
                EAL_GFREE(self->password);
            }
        }
        self->accept = FALSE;
        gboolean rremember;
        if (!gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_REMEMBER_PASSWORDS, &rremember))
            rremember = FALSE;
        *check_value = rremember;
        return TRUE;
    }
    return FALSE;
    // workaround for bug 46012
}

static gboolean
select_cb (GtkMozEmbed * embed, const gchar * dialog_title,
           const gchar * dialog_text, gpointer dialog_options,
           gpointer dialog_choice , GMozillaEngine * self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_MOZILLA_ENGINE(self), FALSE);
    gboolean ret_val;
    g_signal_emit_by_name(G_OBJECT(self), G_WEBWIDGET_SIGNAL_SELECT, dialog_title, dialog_text, dialog_options, dialog_choice, &ret_val, NULL);
    return ret_val;
}

static void
destroy_cb (GtkMozEmbed *embed, GMozillaEngine *self)
{
    TRACE_LOG();
    self->engine = NULL;
}

static void
rss_request_cb (GtkMozEmbed *embed, gchar *url, gchar *title, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_RSS_REQUEST, url, title);
    if (self->progress_bar_state == -1)
        g_signal_emit(G_OBJECT(self), g_mozilla_engine_signals[FINISHED_LOADING], 0, self->last_net_status, self->page_size, FALSE);
    return;
}

static void
icon_changed_cb (GtkMozEmbed *embed,GdkPixbuf* aPixBuf, GMozillaEngine *self )
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    // foward signal to UI ??
#ifdef MOZEAL_LOGGING
    ULOG_DEBUG_F("icon_changed_cb is called in new EAL \n");
    if(aPixBuf == NULL)
        ULOG_DEBUG_F("PixBuf is NULL \n");
    else
        ULOG_DEBUG_F("PixBuf is NOT NULL \n");
    ULOG_DEBUG_F("now send icon signal to UI \n");
#endif
#ifndef USE_OS2006
    g_signal_emit_by_name (G_OBJECT (self), "icon", aPixBuf);
#endif
}

static void
security_mode_changed_cb (GtkMozEmbed *embed, gpointer request, guint state, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    GtkMozEmbedSecurityMode mode;

    mode = gtk_moz_embed_common_get_security_mode (state);
    switch (mode)
    {
    case GTK_MOZ_EMBED_NO_SECURITY:
        self->sec_mode = G_WEBENGINE_NO_SECURITY;
        break;
    case GTK_MOZ_EMBED_HIGH_SECURITY:
        self->sec_mode = G_WEBENGINE_HIGH_SECURITY;
        break;
    case GTK_MOZ_EMBED_MEDIUM_SECURITY:
        self->sec_mode = G_WEBENGINE_MEDIUM_SECURITY;
        break;
    case GTK_MOZ_EMBED_LOW_SECURITY:
        self->sec_mode = G_WEBENGINE_LOW_SECURITY;
        break;
    default:
        self->sec_mode = G_WEBENGINE_UNKNOWN_SECURITY;
        break;
    }

    g_signal_emit_by_name(G_OBJECT(self), G_WEBWIDGET_SIGNAL_SECURITY_MODE_CHANGED, self->sec_mode);

}

static void
mailto_cb(GtkMozEmbed *embed, gchar *url, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_MAILTO, url);
    gtk_moz_embed_stop_load(GTK_MOZ_EMBED(self->engine));
    load_finished_cb(GTK_MOZ_EMBED(self->engine), self);
    return;
}

static void
network_error_cb(GtkMozEmbed *embed, const gint code, const gchar *status, const gchar *url, GMozillaEngine *self)
{
    TRACE_LOG();
    g_return_if_fail(G_MOZILLA_ENGINE(self));
    //    ULOG_DEBUG_F("embed: %p, code: %i, status: '%s', url: '%s'", embed, code, status, url);

    gchar *newURL = NULL;
    gchar **splitURL = NULL;

    splitURL = g_strsplit(url, "//", 2);
    newURL = g_strdup(splitURL[1]);

    //    g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_UNKNOWN_PROTOCOL, newURL);
    self->last_net_status = G_WEBENGINE_LOADING_FINISH_UNKNOWN;
    //    G_WEBENGINE_LOADING_FINISH_COULDNT_CONNECT,
    g_free(splitURL);
    g_free (newURL);

    return ;
}

static void
g_mozilla_engine_array_insert_signal(GMozillaEngine *self, guint signal)
{
    g_array_insert_val(self->signals_array, self->signals_array_len, signal);
    self->signals_array_len++;
}

static gboolean
g_mozilla_engine_signals_connect(GMozillaEngine *self)
{
    TRACE_LOG();
    GObject * engine = G_OBJECT(self->engine);
    self->signals_array =
        g_array_new (FALSE, FALSE, sizeof (gint));

    guint val;
    val = g_signal_connect(G_OBJECT(engine), "location",
                           G_CALLBACK(location_changed_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hook up the title change to update the window title
    val = g_signal_connect(G_OBJECT(engine), "title",
                           G_CALLBACK(title_changed_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hook up the start and stop signals
    val = g_signal_connect(G_OBJECT(engine), "net_start",
                           G_CALLBACK(load_started_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "net_stop",
                           G_CALLBACK(load_finished_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hookup to changes in progress
    val = g_signal_connect(G_OBJECT(engine), "progress",
                           G_CALLBACK(progress_change_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "progress_all",
                           G_CALLBACK(progress_change_all_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hook up to the change in network status
    val = g_signal_connect(G_OBJECT(engine), "net_state",
                           G_CALLBACK(net_state_change_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "net_state_all",
                           G_CALLBACK(net_state_change_all_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hookup to changes in over-link message
    val = g_signal_connect(G_OBJECT(engine), "link_message",
                           G_CALLBACK(link_message_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hookup to changes in js status message
    val = g_signal_connect(G_OBJECT(engine), "js_status",
                           G_CALLBACK(js_status_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hookup to see whenever a new window is requested
    val = g_signal_connect(G_OBJECT(engine), "new_window",
                           G_CALLBACK(new_window_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hookup to any requested visibility changes
    val = g_signal_connect(G_OBJECT(engine), "visibility",
                           G_CALLBACK(visibility_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hookup to the signal that says that the browser requested to be
    // destroyed
    val = g_signal_connect(G_OBJECT(engine), "destroy_browser",
                           G_CALLBACK(destroy_brsr_cb), self);

    // hookup to the signal that is called when someone clicks on a link
    // to load a new uri
    val = g_signal_connect(G_OBJECT(engine), "open_uri",
                           G_CALLBACK(open_uri_cb), self);
    // this signal is emitted when there's a request to change the
    // containing browser window to a certain height, like with width
    // and height args for a window.open in javascript
    val = g_signal_connect(G_OBJECT(engine), "size_to",
                           G_CALLBACK(size_to_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // key event signals
    val = g_signal_connect(G_OBJECT(engine), "dom_key_down",
                           G_CALLBACK(dom_key_down_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_key_press",
                           G_CALLBACK(dom_key_press_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_key_up",
                           G_CALLBACK(dom_key_up_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_down",
                           G_CALLBACK(dom_mouse_down_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_up",
                           G_CALLBACK(dom_mouse_up_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);
#ifdef UNUSED
    g_signal_connect(G_OBJECT(engine), "dom_mouse_move",
                G_CALLBACK(dom_mouse_move_cb), self);
#endif
    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_scroll",
                           G_CALLBACK(dom_mouse_scroll_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_long_press",
                           G_CALLBACK(dom_mouse_long_press_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_focus",
                           G_CALLBACK(dom_focus_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_click",
                           G_CALLBACK(dom_mouse_click_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_dbl_click",
                           G_CALLBACK(dom_mouse_dbl_click_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_over",
                           G_CALLBACK(dom_mouse_over_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_mouse_out",
                           G_CALLBACK(dom_mouse_out_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_activate",
                           G_CALLBACK(dom_activate_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_focus_in",
                           G_CALLBACK(dom_focus_in_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_focus_out",
                           G_CALLBACK(dom_focus_out_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "download_request",
                           G_CALLBACK(download_request_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "upload_dialog",
                           G_CALLBACK(upload_dialog_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "file_upload_started",
                           G_CALLBACK(file_upload_started_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "file_upload_finished",
                           G_CALLBACK(file_upload_finished_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "alert",
                           G_CALLBACK(alert_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "prompt",
                           G_CALLBACK(prompt_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "confirm",
                           G_CALLBACK(confirm_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "confirm_ex",
                           G_CALLBACK(confirm_ex_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "prompt_auth",
                           G_CALLBACK(prompt_auth_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "select",
                           G_CALLBACK(select_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "mailto",
                           G_CALLBACK(mailto_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    // hookup to when the window is destroyed
    val = g_signal_connect(G_OBJECT(engine), "destroy",
                           G_CALLBACK(destroy_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "security_change",
                           G_CALLBACK(security_mode_changed_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "icon_changed",
                           G_CALLBACK(icon_changed_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "network_error",
                           G_CALLBACK(network_error_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "rss_request",
                           G_CALLBACK(rss_request_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_window_focus",
                           G_CALLBACK(dom_window_focus_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_window_blur",
                           G_CALLBACK(dom_window_blur_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);

    val = g_signal_connect(G_OBJECT(engine), "dom_content_blocked",
                           G_CALLBACK(dom_content_blocked_cb), self);
    g_mozilla_engine_array_insert_signal(self, val);


    // set the chrome type so it's stored in the object
    //gtk_signal_connect(G_OBJECT(self->single), "new_window_orphan",
    //                       GTK_SIGNAL_FUNC(new_window_orphan_cb), self);
    //gtk_moz_embed_common_observe("@mozilla.org/extensions/imagezoom;1", NULL, "set_images_scaling", NULL);

    return TRUE;
}

static gboolean
g_mozilla_engine_signals_disconnect(GMozillaEngine *self)
{
    TRACE_LOG();
    gint i = 0;
    guint signal;
    for(i = 0; i < self->signals_array_len; i++)
    {
        signal = g_array_index(self->signals_array, gint, i);
        g_signal_handler_disconnect(G_OBJECT(self->engine), signal);
    }
    if (self->signals_array)
    {
        g_array_free(self->signals_array, TRUE);
        self->signals_array = NULL;
    }
    self->signals_array_len = 0;
    return TRUE;
}

GObject*
g_mozilla_engine_new_with_context_and_tracks (gpointer context,
                                              gboolean leave_tracks, gboolean without_history)
{
    TRACE_LOG();
    GObject* instance = G_OBJECT(GET_MOZILLA_ENGINE_NEW);

    GMozillaEngine *self = G_MOZILLA_ENGINE(instance);

    self->ctx_type = G_WEBENGINE_POPUPMENU_NOTYPE;
    self->moz_ctx_type = 0;

    if (context)
    {
        self->engine = G_OBJECT(gtk_moz_embed_new()); //CHROMEMASK IS MISSED
        GtkMozEmbed **newEmbed = (GtkMozEmbed **)context;
        *newEmbed = GTK_MOZ_EMBED(self->engine);
    }
    else if(leave_tracks && !without_history)
    {
        self->engine = G_OBJECT(gtk_moz_embed_new());
    }
    else if(leave_tracks && without_history)
    {
        self->engine = G_OBJECT(gtk_moz_embed_new());
    }
    else
    {
        self->engine = G_OBJECT(gtk_moz_embed_new());
    }
    g_object_set_data(G_OBJECT(self->engine), "eal_ctx_type", &self->moz_ctx_type);
    gtk_moz_embed_load_url(GTK_MOZ_EMBED(self->engine), "data:text/html,");

    self->signals_array = NULL;
    self->signals_array_len = 0;
    g_mozilla_engine_signals_connect(self);
    self->doc_index = -1;

    self->ctx_node = NULL;
    self->isFkbShowed = FALSE;
    self->last_event = NULL;
    self->ctx_url = NULL;
    self->ctx_objurl = NULL;
    self->ctx_docurl = NULL;
    self->last_net_status = 0;
    self->sec_mode = G_WEBENGINE_NO_SECURITY;
    self->gtkIMContext = NULL;
    self->current_zoom = 100;
    self->optimized_view = FALSE;
    self->will_open_location = NULL;
    self->waiting_connection = FALSE;
    self->im_surr_offset = 0;
    self->im_surr_num = 0;
    self->im_surr_enable = FALSE;
    self->user_iteraction_happens = TRUE;
    self->progress_bar_state = -1;
//    content_set_cursor(self, GDK_LEFT_PTR);
    self->extended_im_mode = GDK_EXTENSION_EVENTS_NONE;
    self->page_size = 0;
    self->scroll_to_node_timer_id = 0;
    self->vkb_up_timeout = 0;

    return instance;
}

