/**
    @file images.c

    Maemo-Blocks picture routines.

  This file is part of Maemo Blocks

  Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
  http://www.indt.org/maemo
 
  This software is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License
  as published by the Free Software Foundation; either version 2.1 of
  the License, or (at your option) any later version.
 
  This software is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
 
  You should have received a copy of the GNU Lesser General Public
  License along with this software; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  02110-1301 USA
*/

#include <SDL.h>
#include <SDL/SDL_image.h>
#include <ft2build.h>
#include "texts.h"
#include "images.h"
#include "game.h"

/* function prototypes */
static SDL_Surface *load_image(char *filename, int transparent);
static inline void draw_piece(int x, int y, int color);

/** The loaded images */
SDL_Surface *screen, *pieces, *background, *numbers, *loose;

/** 
  Load image in video memory 
  @param filename The filename
  @param transparent Transparenty flag
  @return The loaded surface
*/
static SDL_Surface *
load_image(char *filename, int transparent)
{
    char filepath[256];
    SDL_Surface *img, *img2;

    // Load file image
    sprintf(filepath, "%s/%s", PIXMAPSPREFIX, filename);
#ifndef USE_SDL_IMAGE
    img = SDL_LoadBMP(filepath);
#else
    /* if (strstr(filename,".bmp")==filename[strlen(filename)-4]) {
     * img=SDL_LoadBMP(filepath); } else { */
    img = IMG_Load(filepath);
    // }
#endif
    if (!img)
    {
        fprintf(stderr, "File not found: %s\n", filepath);
        return 0;
    }

    // Create hardware surface
    img2 =
        SDL_CreateRGBSurface(SDL_HWSURFACE |
                             (transparent ? SDL_SRCCOLORKEY : 0), img->w,
                             img->h, 16, 0xF800, 0x7E0, 0x1F, 0);
    if (!img2)
    {
        SDL_FreeSurface(img);
        fprintf(stderr, "Error creating surface!\n");
        return 0;
    }

    // Set color key
    if (transparent)
    {
        SDL_SetColorKey(img2, SDL_SRCCOLORKEY, 0xF81F);
    }
    SDL_SetAlpha(img2, 0, 0);

    // Copy surface
    SDL_BlitSurface(img, NULL, img2, NULL);
    SDL_FreeSurface(img);

    return img2;
}

/**
  Load images
  @return 0 on failure
*/
int
load_images(void)
{

    // Blocks
    pieces = load_image("pieces.bmp", 0);
    if (!pieces)
        return 0;

    // Numbers
    numbers = load_image("numbers.bmp", 1);
    if (!numbers)
        return 0;

    // Backround
    background = load_image("board.jpg", 0);
    if (!background)
        return 0;


    // Loose
    loose = load_image("dialog_loose.png", 0);
    if (Print_game_over(loose))
    {
        printf("Error while printig game over string\n");
        return 0;
    }
    if (!loose)
        return 0;

    return 1;
}

/**
  Free memory
*/
void
free_images(void)
{
    SDL_FreeSurface(loose);
    loose = NULL;
    SDL_FreeSurface(background);
    background = NULL;
    SDL_FreeSurface(numbers);
    numbers = NULL;
    SDL_FreeSurface(pieces);
    pieces = NULL;
}

/**
  Draw piece
*/
static inline void
draw_piece(int x, int y, int color)
{
    SDL_Rect src, dest;

    src.y = 0;
    src.x = color * BLOCK_WIDTH;
    src.w = src.h = BLOCK_HEIGHT;

    dest.y = x;
    dest.x = 800 - y - BLOCK_WIDTH;
    SDL_BlitSurface(pieces, &src, screen, &dest);
}

/**
  Draw block
*/
void
draw_block(int x, int y, int block, int frame)
{
    int i, j, mask = 1 << 15;

    for (i = 0; i < 4; i++)
        for (j = 0; j < 4; j++)
        {
            if (blocks[block][frame] & mask)
                draw_piece(x + j * BLOCK_HEIGHT, y + i * BLOCK_WIDTH,
                           block + 1);
            mask >>= 1;
        }
}

/**
  Draw next block
*/
void
draw_next_block(void)
{
    int sx, sy;
    SDL_Rect dest;

    dest.x = NEXTOFFSETX2;
    dest.y = NEXTOFFSETY2;
    dest.w = NEXT_WIDTH;
    dest.h = NEXT_HEIGHT;
    SDL_FillRect(screen, &dest, 0);

    if (blocks[next_block][next_frame] & 0x1111)
        sx = 4;
    else if (blocks[next_block][next_frame] & 0x2222)
        sx = 3;
    else
        sx = 2;
    if (!(blocks[next_block][next_frame] & 0x8888))
        sx++;
    if (blocks[next_block][next_frame] & 0x000F)
        sy = 4;
    else if (blocks[next_block][next_frame] & 0x00F0)
        sy = 3;
    else
        sy = 2;
    if (!(blocks[next_block][next_frame] & 0xF000))
        sy++;

    draw_block(NEXTOFFSETX + (NEXT_HEIGHT - sx * BLOCK_HEIGHT) / 2,
               NEXTOFFSETY + (NEXT_WIDTH - sy * BLOCK_WIDTH) / 2, next_block,
               next_frame);
}

/**
  draw board
*/
void
draw_board(void)
{
    int i, j;
    SDL_Rect dest;

    dest.x = BOARDOFFSETX2;
    dest.y = BOARDOFFSETY2;
    dest.w = BOARDWIDTH;
    dest.h = BOARDHEIGHT;
    SDL_FillRect(screen, &dest, 0);

    for (i = 0; i < MAX_Y; i++)
        for (j = 0; j < MAX_X; j++)
            if (matrix[i][j])
                draw_piece(BOARDOFFSETX + j * BLOCK_HEIGHT,
                           BOARDOFFSETY + i * BLOCK_WIDTH, matrix[i][j]);



}

/**
  Draw numbers
*/
void
draw_number(int x, int value)
{
    SDL_Rect src, dest;
    char num[16], *b = num;

    // Update background
    dest.x = src.x = x;
    dest.y = src.y = SCOREOFFSETY;
    dest.w = src.w = NUMBER_WIDTH;
    dest.h = src.h = NUMBER_HEIGHT * 6;
    SDL_BlitSurface(background, &src, screen, &dest);

    // Offsets
    src.y = 0;
    src.w = NUMBER_WIDTH;
    src.h = NUMBER_HEIGHT;
    dest.h = dest.w = 0;
    sprintf(num, "%d", value);

    // Draw each number
    while (*b)
    {
        src.x = (*b - '0') * NUMBER_WIDTH;
        if (*b == '1')
            dest.y -= 2;
        SDL_BlitSurface(numbers, &src, screen, &dest);
        dest.y += NUMBER_HEIGHT - (*b == '1' ? 2 : 0);
        b++;
    }
    SDL_UpdateRect(screen, src.x, src.y, src.w, src.h);
    if (x != LEVELOFFSETX)
    {
        SDL_Flip(screen);
    }
}

/**
  Update panel
*/
void
update_game_values(int current_lines, int current_level, int current_score)
{
    draw_number(LINEOFFSETX, current_lines);
    draw_number(SCOREOFFSETX, current_score);
    draw_number(LEVELOFFSETX, current_level + 1);
}
