#include "talk/base/common.h"
#include "talk/base/httpclient.h"
#include "talk/base/logging.h"
#include "talk/base/physicalsocketserver.h"
#include "talk/base/socketadapters.h"
#include "talk/base/socketpool.h"
#include "talk/base/ssladapter.h"
#include "talk/base/asynchttprequest.h"

using namespace cricket;

HttpMonitor::HttpMonitor(SocketServer *ss) {
  ASSERT(cricket::Thread::Current() != NULL);
  ss_ = ss;
  reset();
}

void HttpMonitor::Connect(cricket::HttpClient *http) {
  http->SignalHttpClientComplete.connect(this,
    &HttpMonitor::OnHttpClientComplete);
}

void HttpMonitor::OnHttpClientComplete(cricket::HttpClient * http, int err) {
  complete_ = true;
  err_ = err;
  ss_->WakeUp();
}

cricket::Socket * SslSocketFactory::CreateSocket(int type) {
  return factory_->CreateSocket(type);
}

cricket::AsyncSocket * SslSocketFactory::CreateAsyncSocket(int type) {
  cricket::AsyncSocket * socket = factory_->CreateAsyncSocket(type);
  if (!socket)
    return 0;

  // Binary logging happens at the lowest level 
  if (!logging_label_.empty() && binary_mode_) {
    socket = new cricket::LoggingAdapter(socket, logging_level_, 
                                                 logging_label_.c_str(),
                                                 binary_mode_);
  }

  if (proxy_.type) {
    cricket::AsyncSocket * proxy_socket = 0;
    if (proxy_.type == cricket::PROXY_SOCKS5) {
      proxy_socket = new cricket::AsyncSocksProxySocket(socket, proxy_.address,
        proxy_.username, proxy_.password);
    } else {
      // Note: we are trying unknown proxies as HTTPS currently
      proxy_socket = new cricket::AsyncHttpsProxySocket(socket,
        agent_, proxy_.address,
        proxy_.username, proxy_.password);
    }
    if (!proxy_socket) {
      delete socket;
      return 0;
    }
    socket = proxy_socket;  // for our purposes the proxy is now the socket
  }

  if (!hostname_.empty()) {
    cricket::SSLAdapter * ssl_adapter = SSLAdapter::Create(socket);
    ssl_adapter->set_ignore_bad_cert(ignore_bad_cert_);
    ssl_adapter->StartSSL(hostname_.c_str(), true);
    socket = ssl_adapter;
  }

  // Regular logging occurs at the highest level
  if (!logging_label_.empty() && !binary_mode_) {
    socket = new cricket::LoggingAdapter(socket, logging_level_, 
                                                 logging_label_.c_str(),
                                                 binary_mode_);
  }
  return socket;
}

const int kDefaultHTTPTimeout = 5 * 1000; // 5 sec

AsyncHttpRequest::AsyncHttpRequest(const std::string &user_agent)
  : secure_(false), success_(false), error_(true), agent_(user_agent),
    timeout_(kDefaultHTTPTimeout), fail_redirect_(false), client_(user_agent.c_str(), NULL) {}

void AsyncHttpRequest::DoWork(bool *halt) {
  // Note that error starts as true.  If we finish in any way but a success,
  // it is an error.
  cricket::PhysicalSocketServer physical;
  cricket::SocketServer * ss = &physical;

  SslSocketFactory factory(ss, agent_);
  factory.SetProxy(proxy_);
  if (secure_)
    factory.UseSSL(host_.c_str());

  //factory.SetLogging("AsyncHttpRequest");

  cricket::ReuseSocketPool pool(&factory);
  client_.set_pool(&pool);
  
  HttpMonitor monitor(ss);
  monitor.Connect(&client_);

  bool transparent_proxy = (port_ == 80)
    && ((proxy_.type == cricket::PROXY_HTTPS)
        || (proxy_.type == cricket::PROXY_UNKNOWN));

  client_.set_transparent_proxy(transparent_proxy);
  client_.set_fail_redirect(fail_redirect_);

  buzz::scoped_ptr<AsyncHttpsProxySocket::AuthContext> context;
  
Retry:
  monitor.reset();
  if (client_.request().document.get() != NULL) {
    client_.request().document->Rewind();
    size_t length = 0;
    client_.request().document->GetSize(&length);
    LOG(LS_INFO) << "Uploading " << length << " bytes.";
  }
  client_.call(host_.c_str(), port_);
  
  ss->Wait(timeout_, true);
  if (!monitor.done()) {
    client_.reset();
    LOG(LS_INFO) << "AsyncHttpRequest request timed out";
    return;
  }
  
  if (monitor.error()) {
    client_.reset();
    LOG(LS_INFO) << "AsyncHttpRequest request error: " << monitor.error();
    if (monitor.error() == cricket::HE_AUTH) {
      //proxy_auth_required_ = true;
    }
    return;
  }

  if (transparent_proxy 
      && (client_.response().scode == cricket::HC_PROXY_AUTHENTICATION_REQUIRED)) {
    std::string response, auth_method;
    cricket::HttpData::const_iterator begin = client_.response().begin("Proxy-Authenticate");
    cricket::HttpData::const_iterator end = client_.response().end("Proxy-Authenticate");
    for (cricket::HttpData::const_iterator it = begin; it != end; ++it) {
      cricket::AsyncHttpsProxySocket::AuthResult res =
        cricket::AsyncHttpsProxySocket::Authenticate(
                    it->second.data(), it->second.size(),
                    proxy_.address,
                    ToString(client_.request().verb), client_.request().path,
                    proxy_.username, proxy_.password,
                    *context.use(), response, auth_method);
      if (res == cricket::AsyncHttpsProxySocket::AR_RESPONSE) {
        client_.request().setHeader("Proxy-Authorization", response);
        client_.response().clear();
        goto Retry;
      } else if (res == cricket::AsyncHttpsProxySocket::AR_IGNORE) {
        LOG(INFO) << "Ignoring Proxy-Authenticate: " << auth_method;
        continue;
      } else {
        break;
      }
    }
  }

  std::string value;
  if (client_.response().hasHeader("Location", value)) {
    response_redirect_ = value.c_str();
  }

  success_ = true;
  error_ = false;
}
