/*
 * Galago Status API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-private.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-enum-types.h>
#include <libgalago/galago-key-value.h>
#include <libgalago/galago-status.h>
#include <stdio.h>
#include <string.h>

struct _GalagoStatusPrivate
{
	GalagoStatusType primitive;

	GalagoPresence *presence;

	char *id;
	char *name;

	gboolean exclusive;

	GHashTable *attrs_table;
	GList *attrs;
};

enum
{
	PROP_0,
	PROP_TYPE,
	PROP_PRESENCE,
	PROP_ID,
	PROP_NAME,
	PROP_EXCLUSIVE
};

static GHashTable *status_id_map_table = NULL;

/**************************************************************************
 * Object/Class support
 **************************************************************************/
static void galago_status_destroy(GalagoObject *gobject);
static void galago_status_dbus_message_append(DBusMessageIter *iter,
											  const GalagoObject *object);
static void *galago_status_dbus_message_get(DBusMessageIter *iter);
static gchar *galago_status_dbus_get_signature(void);
static const gchar *galago_status_attr_dbus_get_signature(void);
static void galago_status_set_property(GObject *object, guint prop_id,
									   const GValue *value, GParamSpec *pspec);
static void galago_status_get_property(GObject *object, guint prop_id,
									   GValue *value, GParamSpec *pspec);
static void galago_status_set_attribute(GalagoObject *object,
										const char *name, GValue *value);
static gboolean galago_status_remove_attribute(GalagoObject *object,
											   const char *name);
const GValue *galago_status_get_attribute(const GalagoObject *object,
										  const char *name);
GList *galago_status_get_attributes(const GalagoObject *object);

static GalagoObjectClass *parent_class = NULL;

G_DEFINE_TYPE(GalagoStatus, galago_status, GALAGO_TYPE_OBJECT);

static void
galago_status_class_init(GalagoStatusClass *klass)
{
	GObjectClass      *gobject_class = G_OBJECT_CLASS(klass);
	GalagoObjectClass *object_class  = GALAGO_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->supports_attrs = TRUE;

	object_class->destroy             = galago_status_destroy;
	object_class->dbus_message_append = galago_status_dbus_message_append;
	object_class->dbus_message_get    = galago_status_dbus_message_get;
	object_class->dbus_get_signature  = galago_status_dbus_get_signature;
	object_class->set_attribute       = galago_status_set_attribute;
	object_class->remove_attribute    = galago_status_remove_attribute;
	object_class->get_attribute       = galago_status_get_attribute;
	object_class->get_attributes      = galago_status_get_attributes;

	gobject_class->set_property = galago_status_set_property;
	gobject_class->get_property = galago_status_get_property;

	g_object_class_install_property(gobject_class, PROP_TYPE,
		g_param_spec_enum("primitive", "Primitive",
						  "The primitive type this status is based on",
						  GALAGO_TYPE_STATUS_TYPE,
						  GALAGO_STATUS_UNSET,
						  G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
						  G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_PRESENCE,
		g_param_spec_object("presence", "Presence",
							"The presence object that this status belongs to",
							GALAGO_TYPE_PRESENCE,
							G_PARAM_READWRITE |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_ID,
		g_param_spec_string("id", "ID",
							"The status's ID",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_NAME,
		g_param_spec_string("name", "Name",
							"The status's descriptive name",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_EXCLUSIVE,
		g_param_spec_boolean("exclusive", "Exclusive",
							 "The status's exclusive state",
							 FALSE,
							 G_PARAM_READWRITE |
							 G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

static void
galago_status_init(GalagoStatus *status)
{
	status->priv = g_new0(GalagoStatusPrivate, 1);

	status->priv->attrs_table =
		g_hash_table_new_full(g_str_hash, g_str_equal, g_free,
							  (GFreeFunc)galago_key_value_destroy);
}

static void
galago_status_destroy(GalagoObject *object)
{
	GalagoStatus *status = GALAGO_STATUS(object);

	if (status->priv != NULL)
	{
		g_hash_table_destroy(status->priv->attrs_table);
		g_list_free(status->priv->attrs);

		if (status->priv->id != NULL)
			g_free(status->priv->id);

		if (status->priv->name != NULL)
			g_free(status->priv->name);

		g_free(status->priv);
		status->priv = NULL;
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->destroy != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->destroy(object);
}

static void
galago_status_dbus_message_append(DBusMessageIter *iter,
                                  const GalagoObject *object)
{
	GalagoStatus *status = (GalagoStatus *)object;
	GList *l;
	DBusMessageIter array_iter, struct_iter, value_iter;
	GalagoStatusType type;
	const char *id, *name;
	gboolean exclusive;

	type      = galago_status_get_primitive(status);
	id        = galago_status_get_id(status);
	name      = galago_status_get_name(status);
	exclusive = galago_status_is_exclusive(status);

	dbus_message_iter_append_basic(iter, DBUS_TYPE_BYTE,    &type);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_STRING,  &id);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_STRING,  &name);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_BOOLEAN, &exclusive);

	dbus_message_iter_open_container(iter, DBUS_TYPE_ARRAY,
									 galago_status_attr_dbus_get_signature(),
									 &array_iter);

	for (l = galago_object_get_attributes(GALAGO_OBJECT(status));
		 l != NULL;
		 l = l->next)
	{
		GalagoKeyValue *key_value = (GalagoKeyValue *)l->data;

		dbus_message_iter_open_container(&array_iter, DBUS_TYPE_STRUCT, NULL,
										 &struct_iter);
		dbus_message_iter_append_basic(&struct_iter, DBUS_TYPE_STRING,
									   &key_value->key);

		if (G_VALUE_HOLDS(key_value->value, G_TYPE_BOOLEAN))
		{
			gboolean value = g_value_get_boolean(key_value->value);
			dbus_message_iter_open_container(&struct_iter,
											 DBUS_TYPE_VARIANT,
											 DBUS_TYPE_BOOLEAN_AS_STRING,
											 &value_iter);
			dbus_message_iter_append_basic(&value_iter,
										   DBUS_TYPE_BOOLEAN, &value);
			dbus_message_iter_close_container(&struct_iter, &value_iter);
		}
		else if (G_VALUE_HOLDS(key_value->value, G_TYPE_STRING))
		{
			const char *value = g_value_get_string(key_value->value);
			dbus_message_iter_open_container(&struct_iter,
											 DBUS_TYPE_VARIANT,
											 DBUS_TYPE_STRING_AS_STRING,
											 &value_iter);
			dbus_message_iter_append_basic(&value_iter,
										   DBUS_TYPE_STRING, &value);
			dbus_message_iter_close_container(&struct_iter, &value_iter);
		}
		else if (G_VALUE_HOLDS(key_value->value, G_TYPE_INT))
		{
			int value = g_value_get_int(key_value->value);
			dbus_message_iter_open_container(&struct_iter,
											 DBUS_TYPE_VARIANT,
											 DBUS_TYPE_UINT32_AS_STRING,
											 &value_iter);
			dbus_message_iter_append_basic(&value_iter,
										   DBUS_TYPE_UINT32, &value);
			dbus_message_iter_close_container(&struct_iter, &value_iter);
		}
		else if (G_VALUE_HOLDS(key_value->value, G_TYPE_DOUBLE))
		{
			double value = g_value_get_double(key_value->value);
			dbus_message_iter_open_container(&struct_iter,
											 DBUS_TYPE_VARIANT,
											 DBUS_TYPE_DOUBLE_AS_STRING,
											 &value_iter);
			dbus_message_iter_append_basic(&value_iter,
										   DBUS_TYPE_DOUBLE, &value);
			dbus_message_iter_close_container(&struct_iter, &value_iter);
		}
		else
			g_assert_not_reached();

		dbus_message_iter_close_container(&array_iter, &struct_iter);
	}

	dbus_message_iter_close_container(iter, &array_iter);
}

static void *
galago_status_dbus_message_get(DBusMessageIter *iter)
{
	GalagoStatus *status;
	DBusMessageIter array_iter, struct_iter, value_iter;
	char type;
	const char *id, *name;
	gboolean exclusive;

	dbus_message_iter_get_basic(iter, &type);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &id);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &name);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &exclusive);
	dbus_message_iter_next(iter);

	status = galago_status_new((GalagoStatusType)type, id, name, exclusive);

	dbus_message_iter_recurse(iter, &array_iter);

	while (dbus_message_iter_get_arg_type(&array_iter) != DBUS_TYPE_INVALID)
	{
		const char *attr_id;

		dbus_message_iter_recurse(&array_iter, &struct_iter);
		dbus_message_iter_get_basic(&struct_iter, &attr_id);
		dbus_message_iter_next(&struct_iter);

		dbus_message_iter_recurse(&struct_iter, &value_iter);

		switch (dbus_message_iter_get_arg_type(&value_iter))
		{
			case DBUS_TYPE_BOOLEAN:
			{
				gboolean value;
				dbus_message_iter_get_basic(&value_iter, &value);
				galago_object_set_attr_bool(GALAGO_OBJECT(status),
											attr_id, value);
				break;
			}

			case DBUS_TYPE_STRING:
			{
				const char *value;
				dbus_message_iter_get_basic(&value_iter, &value);
				galago_object_set_attr_string(GALAGO_OBJECT(status),
											  attr_id, value);
				break;
			}

			case DBUS_TYPE_UINT32:
			{
				dbus_uint32_t value;
				dbus_message_iter_get_basic(&value_iter, &value);
				galago_object_set_attr_int(GALAGO_OBJECT(status),
										   attr_id, value);
				break;
			}

			case DBUS_TYPE_DOUBLE:
			{
				double value;
				dbus_message_iter_get_basic(&value_iter, &value);
				galago_object_set_attr_double(GALAGO_OBJECT(status),
											  attr_id, value);
				break;
			}

			default: /* This should never be reached. */
				g_assert_not_reached();
				break;
		}

		dbus_message_iter_next(&array_iter);
	}

	return status;
}

static const gchar *
galago_status_attr_dbus_get_signature(void)
{
	return DBUS_STRUCT_BEGIN_CHAR_AS_STRING
	       DBUS_TYPE_STRING_AS_STRING        // id
	       DBUS_TYPE_VARIANT_AS_STRING       // value
	       DBUS_STRUCT_END_CHAR_AS_STRING;
}

static gchar *
galago_status_dbus_get_signature(void)
{
	return g_strconcat(
		DBUS_TYPE_BYTE_AS_STRING,              // type
		DBUS_TYPE_STRING_AS_STRING,            // id
		DBUS_TYPE_STRING_AS_STRING,            // name
		DBUS_TYPE_BOOLEAN_AS_STRING,           // exclusive
		DBUS_TYPE_ARRAY_AS_STRING,             // array of attributes
		galago_status_attr_dbus_get_signature(),
		NULL);
}

static void
galago_status_set_property(GObject *object, guint prop_id,
						   const GValue *value, GParamSpec *pspec)
{
	GalagoStatus *status = GALAGO_STATUS(object);

	switch (prop_id)
	{
		case PROP_TYPE:
			status->priv->primitive =
				(GalagoStatusType)g_value_get_enum(value);
			g_object_notify(object, "primitive");
			break;

		case PROP_PRESENCE:
			galago_status_set_presence(status,
				GALAGO_PRESENCE(g_value_get_object(value)));
			break;

		case PROP_ID:
			status->priv->id = g_value_dup_string(value);
			g_object_notify(object, "id");
			break;

		case PROP_NAME:
			status->priv->name = g_value_dup_string(value);
			g_object_notify(object, "name");
			break;

		case PROP_EXCLUSIVE:
			status->priv->exclusive = g_value_get_boolean(value);
			g_object_notify(object, "exclusive");
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void
galago_status_get_property(GObject *object, guint prop_id,
						   GValue *value, GParamSpec *pspec)
{
	GalagoStatus *status = GALAGO_STATUS(object);

	switch (prop_id)
	{
		case PROP_TYPE:
			g_value_set_enum(value, galago_status_get_primitive(status));
			break;

		case PROP_PRESENCE:
			g_value_set_object(value, galago_status_get_presence(status));
			break;

		case PROP_ID:
			g_value_set_string(value, galago_status_get_id(status));
			break;

		case PROP_NAME:
			g_value_set_string(value, galago_status_get_name(status));
			break;

		case PROP_EXCLUSIVE:
			g_value_set_boolean(value, galago_status_is_exclusive(status));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}


/**************************************************************************
 * Status ID to Name mapping
 **************************************************************************/
#define ADD_STATUS_MAP(id, name) \
	g_hash_table_insert(status_id_map_table, g_strdup(id), g_strdup(name));

static gpointer
_init_status_id_map_table(gpointer unused)
{
	status_id_map_table =
		g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);

	ADD_STATUS_MAP(GALAGO_STATUS_ID_AVAILABLE,     _("Available"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_AWAY,          _("Away"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_BRB,           _("Be Right Back"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_BUSY,          _("Busy"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_DND,           _("Do Not Disturb"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_EXTENDED_AWAY, _("Extended Away"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_HIDDEN,        _("Hidden"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_OFFLINE,       _("Offline"));

	return NULL;
}

static const char *
_galago_statuses_map_id_to_name(const char *id)
{
	static GOnce map_init = G_ONCE_INIT;

	g_return_val_if_fail(id  != NULL, NULL);
	g_return_val_if_fail(*id != '\0', NULL);

	g_once(&map_init, _init_status_id_map_table, NULL);

	return (const char *)g_hash_table_lookup(status_id_map_table, id);
}

/**************************************************************************
 * GalagoStatus API
 **************************************************************************/

/**
 * galago_status_new
 * @type:      The type of status.
 * @id:        The status ID.
 * @name:      The name of the status.
 * @exclusive: TRUE if the status is exclusive.
 *
 * Creates a new status.
 *
 * If exclusive is TRUE, the status will be exclusive. Only one
 * exclusive status can be set at a time. If another exclusive status
 * is set, the previously set exclusive status will be removed.
 *
 * If exclusive is FALSE, the status will not be removed unless
 * manually removed.
 *
 * Returns: The status.
 */
GalagoStatus *
galago_status_new(GalagoStatusType type, const char *id, const char *name,
				  gboolean exclusive)
{
	const char *name2;

	g_return_val_if_fail(galago_is_initted(),           NULL);
	g_return_val_if_fail(type != GALAGO_STATUS_UNSET,   NULL);
	g_return_val_if_fail(id   != NULL,                  NULL);
	g_return_val_if_fail(*id  != '\0',                  NULL);

	if ((name2 = _galago_statuses_map_id_to_name(id)) != NULL)
		name = name2;

	g_return_val_if_fail(name  != NULL, NULL);
	g_return_val_if_fail(*name != '\0', NULL);

	return g_object_new(GALAGO_TYPE_STATUS,
						"primitive", type,
						"id", id,
						"name", name,
						"exclusive", exclusive,
						NULL);
}

/**
 * galago_status_duplicate
 * @status: The status to duplicate.
 *
 * Duplicates a status.
 *
 * Returns: The duplicate status.
 */
GalagoStatus *
galago_status_duplicate(const GalagoStatus *status)
{
	GalagoStatus *new_status;
	GList *l;

	g_return_val_if_fail(status != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	new_status = galago_status_new(galago_status_get_primitive(status),
								   galago_status_get_id(status),
								   galago_status_get_name(status),
								   galago_status_is_exclusive(status));

	for (l = galago_object_get_attributes(GALAGO_OBJECT(status));
		 l != NULL;
		 l = l->next)
	{
		GalagoKeyValue *key_value = (GalagoKeyValue *)l->data;

		if (G_VALUE_HOLDS(key_value->value, G_TYPE_BOOLEAN))
		{
			galago_object_set_attr_bool(GALAGO_OBJECT(new_status),
				key_value->key, g_value_get_boolean(key_value->value));
		}
		else if (G_VALUE_HOLDS(key_value->value, G_TYPE_STRING))
		{
			galago_object_set_attr_string(GALAGO_OBJECT(new_status),
				key_value->key, g_value_get_string(key_value->value));
		}
		else if (G_VALUE_HOLDS(key_value->value, G_TYPE_INT))
		{
			galago_object_set_attr_int(GALAGO_OBJECT(new_status),
				key_value->key, g_value_get_int(key_value->value));
		}
		else if (G_VALUE_HOLDS(key_value->value, G_TYPE_DOUBLE))
		{
			galago_object_set_attr_double(GALAGO_OBJECT(new_status),
				key_value->key, g_value_get_double(key_value->value));
		}
	}

	return new_status;
}

/**
 * galago_status_set_presence
 * @status:   The status.
 * @presence: The presence.
 *
 * Sets the parent presence of a status.
 *
 * This is intended for internal use only.
 */
void
galago_status_set_presence(GalagoStatus *status, GalagoPresence *presence)
{
	g_return_if_fail(status != NULL);
	g_return_if_fail(GALAGO_IS_STATUS(status));
	g_return_if_fail(presence == NULL || GALAGO_IS_PRESENCE(presence));

	if (status->priv->presence == presence)
		return;

	status->priv->presence = presence;
	g_object_notify(G_OBJECT(status), "presence");
}

/**
 * galago_status_get_presence
 * @status: The status.
 *
 * Returns the parent presence of a status.
 *
 * Returns: The presence.
 */
GalagoPresence *
galago_status_get_presence(const GalagoStatus *status)
{
	g_return_val_if_fail(status != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	return status->priv->presence;
}

/**
 * galago_status_get_primitive
 * @status: The status.
 *
 * Returns the primitive type of a status.
 *
 * Returns: The status's primitive type.
 */
GalagoStatusType
galago_status_get_primitive(const GalagoStatus *status)
{
	g_return_val_if_fail(status != NULL,           GALAGO_STATUS_UNSET);
	g_return_val_if_fail(GALAGO_IS_STATUS(status), GALAGO_STATUS_UNSET);

	return status->priv->primitive;
}

/**
 * galago_status_get_id
 * @status: The status.
 *
 * Returns the status's ID.
 *
 * Returns: The status's ID.
 */
const char *
galago_status_get_id(const GalagoStatus *status)
{
	g_return_val_if_fail(status != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	return status->priv->id;
}

/**
 * galago_status_get_name
 * @status: The status.
 *
 * Returns the status's name.
 *
 * Returns: The status's name.
 */
const char *
galago_status_get_name(const GalagoStatus *status)
{
	g_return_val_if_fail(status != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	return status->priv->name;
}

/**
 * galago_status_is_exclusive
 * @status: The status.
 *
 * Returns whether or not a status is exclusive.
 *
 * Returns: TRUE if the status is exclusive, or FALSE.
 */
gboolean
galago_status_is_exclusive(const GalagoStatus *status)
{
	g_return_val_if_fail(status != NULL,           FALSE);
	g_return_val_if_fail(GALAGO_IS_STATUS(status), FALSE);

	return status->priv->exclusive;
}

/**
 * galago_status_is_available
 * @status: The status.
 *
 * Returns whether or not a status is considered available.
 *
 * Returns: TRUE if the status is available, or FALSE.
 */
gboolean
galago_status_is_available(const GalagoStatus *status)
{
	GalagoStatusType type;

	g_return_val_if_fail(status != NULL,           FALSE);
	g_return_val_if_fail(GALAGO_IS_STATUS(status), FALSE);

	type = galago_status_get_primitive(status);

	return type == GALAGO_STATUS_AVAILABLE ||
	       type == GALAGO_STATUS_HIDDEN;
}

static void
galago_status_set_attribute(GalagoObject *object, const char *name,
							GValue *value)
{
	GalagoStatus *status = GALAGO_STATUS(object);
	GalagoPresence *presence;
	GalagoKeyValue *key_value;

	key_value = g_hash_table_lookup(status->priv->attrs_table, name);

	if (key_value == NULL)
	{
		key_value = galago_key_value_new(name, value);

		g_hash_table_insert(status->priv->attrs_table,
							g_strdup(name), key_value);
		status->priv->attrs = g_list_append(status->priv->attrs, key_value);
	}
	else if (G_VALUE_HOLDS(key_value->value, G_VALUE_TYPE(value)))
	{
		g_value_unset(key_value->value);
		g_free(key_value->value);
		key_value->value = value;
	}
	else
	{
		g_warning("Attempted to set existing attribute ID %s of type %s with "
				  "attribute of type %s",
				  name,
				  G_VALUE_TYPE_NAME(key_value->value),
				  G_VALUE_TYPE_NAME(value));

		return;
	}

	presence = galago_status_get_presence(status);

	if (presence != NULL)
		g_signal_emit_by_name(presence, "status-updated", 0, status, name);
}

static gboolean
galago_status_remove_attribute(GalagoObject *object, const char *name)
{
	GalagoStatus *status = GALAGO_STATUS(object);
	GalagoKeyValue *key_value;

	key_value = g_hash_table_lookup(status->priv->attrs_table, name);

	if (key_value == NULL)
		return FALSE;

	g_hash_table_remove(status->priv->attrs_table, name);
	status->priv->attrs = g_list_remove(status->priv->attrs, key_value);

	return TRUE;
}

const GValue *
galago_status_get_attribute(const GalagoObject *object, const char *name)
{
	GalagoStatus *status = GALAGO_STATUS(object);
	GalagoKeyValue *key_value;

	key_value = g_hash_table_lookup(status->priv->attrs_table, name);

	return (key_value == NULL ? NULL : key_value->value);
}

GList *
galago_status_get_attributes(const GalagoObject *object)
{
	return GALAGO_STATUS(object)->priv->attrs;
}
