/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "test.h"
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <string.h>

int test_chain_helper_0(CST * st, X509 * x, int count);
int test_chain_helper_1(CST * st, X509 * x, int count);
int test_chain_helper_2(CST * st, unsigned int certID, int count);

void test_verify(PARAM * param)
{
    TEST_TITLE("CST_is_valid()");
    TEST_DO_SIMPLE(CST_is_valid(param->st, param->cert));
    TEST_DO_SIMPLE(CST_is_valid(param->st, param->user_cert));
    TEST_DO_SIMPLE(CST_is_valid(param->st, param->hotbox_cert));
    TEST_DO_SIMPLE(!CST_is_valid(param->st, param->jboss_cert));
}

void test_chain(PARAM * param)
{
    TEST_TITLE("Test chain of certificates");
    TEST_DO_SIMPLE(test_chain_helper_0(param->st, param->cert, 1));
    TEST_DO_SIMPLE(test_chain_helper_1(param->st, param->cert, 1));
    TEST_DO_SIMPLE(test_chain_helper_2(param->st, param->cert_uid, 1));
    
    TEST_DO_SIMPLE(test_chain_helper_0(param->st, param->ca, 0));
    TEST_DO_SIMPLE(test_chain_helper_1(param->st, param->ca, 0));
    TEST_DO_SIMPLE(test_chain_helper_2(param->st, param->ca_uid, 0));
}

int test_chain_helper_1(CST * st, X509 * x, int count)
{
    GSList * chain = CST_get_chain_id(st, x);
    int result = (count == g_slist_length(chain));
    if (result && (count > 0))
    {
        result = CST_is_root_id(st, GPOINTER_TO_UINT(g_slist_nth_data(chain, count - 1)));
    }
    g_slist_free(chain);    
    return result;
}

int test_chain_helper_2(CST * st, unsigned int certID, int count)
{
    GSList * chain = CST_get_chain_id_by_id(st, certID);
    int result = (count == g_slist_length(chain));
    if (result && (count > 0))
    {
        result = CST_is_root_id(st, GPOINTER_TO_UINT(g_slist_nth_data(chain, count - 1)));
    }
    g_slist_free(chain);    
    return result;
}

int test_chain_helper_0(CST * st, X509 * x, int count)
{
    STACK_OF(X509) * chain = CST_get_chain(st, x);
 
    int result = FALSE;
    
    if (chain)
    {
        if (sk_X509_num(chain) == count)
        {
            result = TRUE;
        }
        
        free_stack_X509(chain); 
    }
    else
    {
        if (0 == count)
        {
            result = TRUE;
        }
    }

    return result;
}

void test_openssl(PARAM * param)
{
    TEST_TITLE("X509_NAME_cmp()");
    TEST_DO_SIMPLE(X509_NAME_cmp
                   (X509_get_subject_name(param->ca),
                    X509_get_subject_name(param->cert)) *
                   X509_NAME_cmp(X509_get_subject_name(param->cert),
                                 X509_get_subject_name(param->ca)) == -1);

    TEST_TITLE("X509_check_issued()");
    TEST_DO_SIMPLE(X509_check_issued(param->cert, param->ca) != 0);
    TEST_DO_SIMPLE(X509_check_issued(param->ca, param->cert) == 0);

    TEST_TITLE("X509_verify()");
    EVP_PKEY *verify_test_pkey = NULL;
    verify_test_pkey = X509_get_pubkey(param->ca);
    TEST_DO_SIMPLE(verify_test_pkey);
    TEST_DO_SIMPLE(X509_verify(param->ca, verify_test_pkey) > 0);
    EVP_PKEY_free(verify_test_pkey);

    TEST_DO_SIMPLE(compare_key(param->public_key, param->public_key, TRUE));
    TEST_DO_SIMPLE(compare_key(param->private_key, param->private_key, FALSE));
    TEST_DO_SIMPLE(!compare_key(param->private_key, param->user_private_key, FALSE));
}

void test_fingerprint(PARAM * param)
{
    TEST_TITLE("Fingerprint");
    TEST_DO_SIMPLE(test_cmp
                   ("6153EFA35DA3C7452079297044791B74", param->ca,
                    CST_get_fingerprint_MD5));
    TEST_DO_SIMPLE(test_cmp
                   ("073BF0070ABA0655C6379743EA9E7849CEBFF201", param->ca,
                    CST_get_fingerprint_SHA1));
    TEST_DO_SIMPLE(test_cmp
                   ("1BDA51E13F4D7FB300967AAEBE4AD821", param->cert,
                    CST_get_fingerprint_MD5));
    TEST_DO_SIMPLE(test_cmp
                   ("24EAC2805BE5018843048DAC04D2ABA3C4150C48",
                    param->cert, CST_get_fingerprint_SHA1));
}

void test_serial(PARAM * param)
{
    TEST_TITLE("Serial number in hex");
    TEST_DO_SIMPLE(test_cmp("00", param->ca, CST_get_serial_number_t));
    TEST_DO_SIMPLE(test_cmp("01", param->cert, CST_get_serial_number_t));
}

int test_cmp(char *must, X509 * cert, FFingerprint f)
{
    char *buf = (*f) (cert);
    int result = (0 == strcmp(buf, must));
    g_free(buf);
    return result;
}

void test_other_prop(PARAM * param)
{
    TEST_TITLE("Test email");
    TEST_DO_SIMPLE(test_cmp("ivanov@mail.ru", param->user_cert, CST_get_email));
    TEST_DO_SIMPLE(test_cmp("noc@pochta.ru", param->hotbox_cert, CST_get_email));
    
    TEST_TITLE("Test domain name");
    TEST_DO_SIMPLE(test_cmp("www.jboss.com", param->jboss_cert, CST_get_domain_name));

    TEST_TITLE("Test key alg name");
    TEST_DO_SIMPLE(test_cmp("rsaEncryption", param->jboss_cert, CST_get_public_key_alg));
    
#if 0
    TEST_TITLE("Test hex public key");
    char *temp = NULL;
    temp = CST_EVP_PKEY_to_hex(param->public_key);
    printf("Public key: %s\n", temp);
    g_free(temp);
#endif
}
